﻿namespace Opal.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading.Tasks;

    /// <summary>
    /// ワイルドカード指定のパスを解析するクラスです。
    /// </summary>
    public sealed class WildcardAnalyzer
    {
        private readonly string orgPath;
        private readonly string path;
        private readonly List<WildcardPart> parts =
            new List<WildcardPart>();

        private string basePath = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="path">解析するパスです。</param>
        public WildcardAnalyzer(string path)
        {
            this.orgPath = path;
            this.path = path.Replace(@"\", "/");
        }

        /// <summary>
        /// パスを解析します。
        /// </summary>
        public void Analyze()
        {
            var parts = this.path.Split('/');

            int index = 0;
            int skipIndex = 0;

            if (parts.Length > 0)
            {
                if (parts[0].IndexOf(":") >= 0)
                {
                    skipIndex = 1;
                    ++index;
                    this.basePath = parts[0] + @"\";
                }
            }

            foreach (var part in parts.Skip(skipIndex))
            {
                WildcardPart wildcardPart = new WildcardPart(part);

                if (wildcardPart.IsWildcard)
                {
                    break;
                }

                this.basePath = Path.Combine(this.basePath, wildcardPart.Value);

                ++index;
            }

            foreach (var part in parts.Skip(index))
            {
                WildcardPart wildcardPart = new WildcardPart(part);
                this.parts.Add(wildcardPart);
            }
        }

        /// <summary>
        /// 解析したパスから有効なパスを生成します。
        /// </summary>
        /// <returns>有効なパスのインスタンスを返します。</returns>
        public IList<string> CreateValidPaths()
        {
            List<string> results = new List<string>();

            // 分割要素がなかったので、basePath を返して終了。
            if (this.parts.Count == 0)
            {
                if (Directory.Exists(this.basePath))
                {
                    results.Add(this.basePath);
                }
                else if (File.Exists(this.basePath))
                {
                    results.Add(this.basePath);
                }

                return results;
            }

            List<string> basePaths = new List<string>();

            basePaths.Add(this.basePath);

            {
                foreach (var wildcardPart in this.parts.GetRange(0, this.parts.Count - 1))
                {
                    List<string> tmpPaths = new List<string>();

                    if (wildcardPart.IsWildcard)
                    {
                        foreach (var path in basePaths)
                        {
                            var paths = Directory.GetDirectories(path);
                            if (paths != null)
                            {
                                foreach (var wildcardPath in paths)
                                {
                                    DirectoryInfo dirInfo = new DirectoryInfo(wildcardPath);
                                    if (wildcardPart.IsMatch(dirInfo.Name))
                                    {
                                        tmpPaths.Add(wildcardPath);
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        foreach (var path in basePaths)
                        {
                            var paths = Directory.GetDirectories(path);
                            if (paths != null)
                            {
                                foreach (var normalPath in paths)
                                {
                                    DirectoryInfo dirInfo = new DirectoryInfo(normalPath);
                                    if (wildcardPart.Value == dirInfo.Name)
                                    {
                                        tmpPaths.Add(normalPath);
                                        break;
                                    }
                                }
                            }
                        }
                    }

                    if (tmpPaths.Count == 0)
                    {
                        break;
                    }

                    basePaths.Clear();
                    basePaths.AddRange(tmpPaths);
                }
            }

            {
                var wildcardPart = this.parts[this.parts.Count - 1];
                if (wildcardPart.IsWildcard)
                {
                    foreach (var path in basePaths)
                    {
                        var dirPaths = Directory.GetDirectories(path);
                        if (dirPaths != null)
                        {
                            foreach (var dirPath in dirPaths)
                            {
                                DirectoryInfo dirInfo = new DirectoryInfo(dirPath);
                                if (wildcardPart.IsMatch(dirInfo.Name))
                                {
                                    results.Add(dirPath);
                                }
                            }
                        }

                        var filePaths = Directory.GetFiles(path);
                        if (filePaths != null)
                        {
                            foreach (var filePath in filePaths)
                            {
                                FileInfo fileInfo = new FileInfo(filePath);
                                if (wildcardPart.IsMatch(fileInfo.Name))
                                {
                                    results.Add(filePath);
                                }
                            }
                        }
                    }
                }
                else
                {
                    foreach (var path in basePaths)
                    {
                        var dirPaths = Directory.GetDirectories(path);
                        if (dirPaths != null)
                        {
                            foreach (var dirPath in dirPaths)
                            {
                                DirectoryInfo dirInfo = new DirectoryInfo(dirPath);
                                if (wildcardPart.Value == dirInfo.Name)
                                {
                                    results.Add(dirPath);
                                    break;
                                }
                            }
                        }

                        var filePaths = Directory.GetFiles(path);
                        if (filePaths != null)
                        {
                            foreach (var filePath in filePaths)
                            {
                                FileInfo fileInfo = new FileInfo(filePath);
                                if (wildcardPart.Value == fileInfo.Name)
                                {
                                    results.Add(filePath);
                                    break;
                                }
                            }
                        }
                    }
                }
            }

            return results;
        }

        private class WildcardPart
        {
            private readonly string value;
            private readonly Regex regex = null;
            private bool isWildcard = false;

            public WildcardPart(string value)
            {
                this.value = value;

                string regexPattern = string.Empty;
                if (value.IndexOf('*') >= 0 || value.IndexOf('?') >= 0)
                {
                    regexPattern = Regex.Escape(value);
                }
                else
                {
                    regexPattern = value;
                }

                regexPattern = regexPattern.Replace(@"\*", ".*?");
                regexPattern = regexPattern.Replace(@"\?", ".");

                if (regexPattern != value)
                {
                    this.isWildcard = true;
                    regexPattern = regexPattern + "$";
                    this.regex = new Regex(regexPattern);
                }
            }

            public string Value
            {
                get
                {
                    return this.value;
                }
            }

            public bool IsWildcard
            {
                get
                {
                    return this.isWildcard;
                }
            }

            public bool IsMatch(string value)
            {
                if (this.regex != null)
                {
                    return this.regex.IsMatch(value);
                }

                return false;
            }
        }
    }
}
