﻿namespace Opal.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.ComponentModel;

    /// <summary>
    /// PropertyInfo のキャッシュクラスです。
    /// </summary>
    public sealed class PropertyInfoCacher : SynchronizableObject
    {
        private readonly Dictionary<Tuple<Type, string>, PropertyInfo> cache =
            new Dictionary<Tuple<Type, string>, PropertyInfo>();

        /// <summary>
        /// 指定インスタンスのPropertyInfo を追加します。
        /// </summary>
        /// <param name="target">追加対象です。指定のプロパティをインスタンスです。</param>
        /// <param name="propertyName">追加対象のプロパティ名です。</param>
        public void AddPropertyInfo(object target, string propertyName)
        {
            Debug.Assert(target != null);
            Debug.Assert(!string.IsNullOrWhiteSpace(propertyName));

            lock (this.SyncRoot)
            {
                var type = target.GetType();

                Tuple<Type, string> key = new Tuple<Type, string>(type, propertyName);
                if (this.cache.ContainsKey(key))
                {
                    return;
                }

                var propertyInfo = type.GetProperty(propertyName);
                if (propertyInfo != null)
                {
                    this.cache.Add(key, propertyInfo);
                }
            }
        }

        /// <summary>
        /// 指定インスタンスのPropertyInfoを取得します。
        /// </summary>
        /// <param name="target">取得対象のインスタンスです。</param>
        /// <param name="propertyName">取得対象のプロパティ名です。</param>
        /// <returns>指定インスタンスに対応するPropertyInfoのインスタンスを返します。</returns>
        public PropertyInfo GetPropertyInfo(object target, string propertyName)
        {
            Debug.Assert(target != null);
            Debug.Assert(!string.IsNullOrWhiteSpace(propertyName));

            lock (this.SyncRoot)
            {
                var type = target.GetType();
                Tuple<Type, string> key = new Tuple<Type, string>(type, propertyName);

                return this.cache[key];
            }
        }

        /// <summary>
        /// キャッシュをクリアします。
        /// </summary>
        public void Clear()
        {
            lock (this.SyncRoot)
            {
                this.cache.Clear();
            }
        }
    }
}
