﻿namespace Opal.Utilities
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.ComponentModel;

    /// <summary>
    /// 文字列グループ分けかつ文字列引きのキャッシュ管理クラスです。
    /// </summary>
    /// <typeparam name="TValue">キャッシュのテンプレートの型です。</typeparam>
    public abstract class GroupCacheManager<TValue> : SynchronizableObject
        where TValue : class
    {
        private readonly Dictionary<string, Dictionary<string, TValue>> cache =
            new Dictionary<string, Dictionary<string, TValue>>();

        /// <summary>
        /// グループ名を取得します。
        /// </summary>
        /// <returns>グループ名を返します。</returns>
        public IEnumerable<string> GetGroupNames()
        {
            lock (this.SyncRoot)
            {
                return this.cache.Keys;
            }
        }

        /// <summary>
        /// キーを取得します。
        /// </summary>
        /// <param name="groupName">取得対象のグループです。</param>
        /// <returns>キーを返します。</returns>
        public IEnumerable<string> GetKeys(string groupName)
        {
            Debug.Assert(!string.IsNullOrWhiteSpace(groupName));
            lock (this.SyncRoot)
            {
                Dictionary<string, TValue> groupValue = null;
                this.cache.TryGetValue(groupName, out groupValue);

                if (groupValue != null)
                {
                    foreach (var key in groupValue.Keys)
                    {
                        yield return key;
                    }
                }
            }
        }

        /// <summary>
        /// 値を追加します。（内部処理用）
        /// </summary>
        /// <param name="groupName">追加対象のグループ名です。</param>
        /// <param name="key">グループ内のキーです。</param>
        /// <param name="value">追加する値です。</param>
        protected void AddValueInternal(string groupName, string key, TValue value)
        {
            Debug.Assert(groupName != null);
            Debug.Assert(key != null);
            Debug.Assert(value != null);

            lock (this.SyncRoot)
            {
                Dictionary<string, TValue> groupValue = null;
                if (!this.cache.TryGetValue(groupName, out groupValue))
                {
                    groupValue = new Dictionary<string, TValue>();
                    this.cache.Add(groupName, groupValue);
                }

                groupValue.Add(key, value);
            }
        }

        /// <summary>
        /// 値を取得します。
        /// </summary>
        /// <param name="groupName">グループ名です。</param>
        /// <param name="key">グループ内のキーです。</param>
        /// <returns>指定したグループ、キーで値が存在すればインスタンスを返します。存在しない場合は、null を返します。</returns>
        protected TValue GetValueInternal(string groupName, string key)
        {
            Debug.Assert(groupName != null);
            Debug.Assert(key != null);

            lock (this.SyncRoot)
            {
                Dictionary<string, TValue> groupValue = null;
                if (!this.cache.TryGetValue(groupName, out groupValue))
                {
                    return null;
                }

                TValue value = null;
                groupValue.TryGetValue(key, out value);
                return value;
            }
        }

        /// <summary>
        /// 指定グループの値を取得します。
        /// </summary>
        /// <param name="groupName">グループ名です。</param>
        /// <returns>指定グループ値のインスタンスを返します。存在しない場合は null を返します。</returns>
        protected Dictionary<string, TValue> GetGroupValuesInternal(string groupName)
        {
            Debug.Assert(groupName != null);

            lock (this.SyncRoot)
            {
                Dictionary<string, TValue> groupValue = null;
                this.cache.TryGetValue(groupName, out groupValue);
                return groupValue;
            }
        }

        /// <summary>
        /// キャッシュクリアします。
        /// </summary>
        protected void ClearCacheInternal()
        {
            this.cache.Clear();
        }
    }
}
