﻿namespace Opal.Operations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// オペレーショントランザクションクラスです。
    /// </summary>
    public sealed class OperationTransaction : IDisposable
    {
        private readonly Dictionary<int, Operation> mergeableOperations = new Dictionary<int, Operation>();
        private readonly List<Operation> operations = new List<Operation>();
        private readonly object syncRoot = new object();
        private readonly OperationManager manager;
        private readonly bool merge;

        /// <summary>
        /// コンストラクタです。内部処理用
        /// </summary>
        /// <param name="manager">使用するオペレーション管理クラスです。</param>
        /// <param name="merge">オペレーション処理をマージするかのフラグです。</param>
        internal OperationTransaction(OperationManager manager, bool merge)
        {
            Debug.Assert(manager != null);
            this.manager = manager;
            this.merge = merge;
        }

        /// <summary>
        /// オペレーションを取得します。内部処理用
        /// </summary>
        internal ICollection<Operation> Operations
        {
            get
            {
                return this.operations;
            }
        }

        /// <summary>
        /// マージ可能なオペレーションを取得します。内部処理用
        /// </summary>
        internal ICollection<Operation> MergeableOperations
        {
            get
            {
                return this.mergeableOperations.Values;
            }
        }

        /// <summary>
        /// 追加します。
        /// </summary>
        /// <param name="operation">追加する IOperation です。</param>
        public void Add(Operation operation)
        {
            Debug.Assert(operation != null);
            lock (this.syncRoot)
            {
                if (this.merge)
                {
                    if (!operation.IsMergeable)
                    {
                        this.operations.Add(operation);
                        return;
                    }

                    if (this.mergeableOperations.ContainsKey(operation.GetMergeableHashCode()))
                    {
                        this.mergeableOperations[operation.GetMergeableHashCode()] = operation;
                    }
                    else
                    {
                        this.mergeableOperations.Add(operation.GetMergeableHashCode(), operation);
                    }
                }
                else
                {
                    this.operations.Add(operation);
                }
            }
        }

        /// <summary>
        /// 廃棄します。
        /// </summary>
        public void Dispose()
        {
            lock (this.syncRoot)
            {
                if (this.merge)
                {
                    this.operations.AddRange(this.mergeableOperations.Values);
                }

                this.manager.EndTransaction();
            }
        }
    }
}
