﻿namespace Opal.Logs
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.App;
    using Opal.Utilities;

    /// <summary>
    /// ログマネージャークラスです。
    /// </summary>
    public sealed class LogManager : ILogManager
    {
        private static readonly string ManagerKeyString;
        private readonly object syncRoot = new object();
        private readonly List<Log> logs = new List<Log>();
        private LogConductor logConductor;

        static LogManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Log);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public LogManager()
            : this(new NullLogConductor())
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="conductor">設定するロガーコンダクタです。</param>
        public LogManager(LogConductor conductor)
        {
            Debug.Assert(conductor != null);
            this.logConductor = conductor;
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// ログ処理クラスを取得します。
        /// </summary>
        /// <typeparam name="TConductor">ログ処理クラスのテンプレートの型です。</typeparam>
        /// <returns>ログ処理クラスのインスタンスを返します。</returns>
        public TConductor GetConductor<TConductor>() where TConductor : LogConductor
        {
            return this.logConductor as TConductor;
        }

        /// <summary>
        /// ロガーをプッシュします。
        /// </summary>
        /// <param name="logger">プッシュ対象のロガーです。</param>
        public void PushLogger(Logger logger)
        {
            Debug.Assert(logger != null);
            lock (this.syncRoot)
            {
                foreach (var log in logger.GetLogs())
                {
                    this.logs.Add(log);
                }
            }
        }

        /// <summary>
        /// ログ全体をフラッシュします。
        /// </summary>
        public void FlushAll()
        {
            Debug.Assert(this.logConductor != null);
            lock (this.syncRoot)
            {
                var readOnlyLogs = new ReadOnlyCollection<Log>(this.logs);
                if (this.logConductor.FlushLogs(readOnlyLogs))
                {
                    this.logs.Clear();
                }
            }
        }
    }
}
