﻿namespace Opal.Configurations
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using System.Text;
    using System.Threading.Tasks;
    using Opal.Utilities;

    /// <summary>
    /// コンフィグ管理クラスです。
    /// </summary>
    public sealed class ConfigManager : IConfigManager
    {
        private static readonly string ManagerKeyString;

        private readonly object syncRoot = new object();
        private readonly Dictionary<Type, Config> cache = new Dictionary<Type, Config>();

        static ConfigManager()
        {
            ManagerKeyString = Enum.GetName(typeof(ManagerKey), ManagerKey.Config);
        }

        /// <summary>
        /// マネージャーのキーを取得します。
        /// </summary>
        public string Key
        {
            get
            {
                return ManagerKeyString;
            }
        }

        /// <summary>
        /// コンフィグを取得します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <returns>指定したグループに存在するコンフィグのインスタンスを返します。</returns>
        public WeakReference<TConfig> GetConfig<TConfig>() where TConfig : Config
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TConfig);
                WeakReference<TConfig> weakConfig = new WeakReference<TConfig>((TConfig)this.cache[type]);
                return weakConfig;
            }
        }

        /// <summary>
        /// コンフィグを取得を試行します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <param name="config">取得成功ならば、コンフィグのインスタンスを格納します。存在しない場合は、null を格納します。</param>
        /// <returns>取得成功の場合は、true を失敗の場合は、false を返します。</returns>
        public bool TryGetConfig<TConfig>(out WeakReference<TConfig> config) where TConfig : Config
        {
            lock (this.syncRoot)
            {
                Config result = null;
                Type type = typeof(TConfig);
                if (this.cache.TryGetValue(type, out result))
                {
                    config = new WeakReference<TConfig>((TConfig)result);
                    return true;
                }

                config = null;
                return false;
            }
        }

        /// <summary>
        /// コンフィグを所有しているか判定します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <returns>所有している場合は、true 所有していない場合は、false を返します。</returns>
        public bool HasConfig<TConfig>() where TConfig : Config
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TConfig);
                return this.cache.ContainsKey(type);
            }
        }

        /// <summary>
        /// コンフィグを追加します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <param name="config">追加するコンフィグです。</param>
        public void AddConfig<TConfig>(TConfig config) where TConfig : Config
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TConfig);
                Config resultConfig = null;
                bool result = this.cache.TryGetValue(type, out resultConfig);
                Debug.Assert(!result);
                this.cache.Add(type, config);
            }
        }

        /// <summary>
        /// コンフィグを削除します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        public void RemoveConfig<TConfig>() where TConfig : Config
        {
            lock (this.syncRoot)
            {
                Type type = typeof(TConfig);
                Config resultConfig = null;
                bool result = this.cache.TryGetValue(type, out resultConfig);
                Debug.Assert(result);
                this.cache.Remove(type);
                resultConfig.Dispose();
            }
        }

        /// <summary>
        /// 状態をクリアします。
        /// </summary>
        public void Clear()
        {
            lock (this.syncRoot)
            {
                this.cache.Clear();
            }
        }

        /// <summary>
        /// コンフィグを取得します。
        /// </summary>
        /// <returns>コンフィグのインスタンスを返します。</returns>
        internal IEnumerable<Config> GetConfigs()
        {
            foreach (var config in this.cache.Values)
            {
                yield return config;
            }
        }
    }
}
