﻿namespace Opal.ComponentModel
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.Linq.Expressions;
    using System.Runtime.CompilerServices;
    using System.Threading;
    using Opal.Events;
    using Opal.Utilities;

    /// <summary>
    /// 通知機能付きオブジェクトです。
    /// </summary>
    public abstract class ObservableObject : SynchronizableObject, INotifyPropertyChanged
    {
        /// <summary>
        /// プロパティ変更イベント
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged;

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <param name="propertyName">変更したプロパティの名前です。</param>
        protected virtual void RaisePropertyChanged([CallerMemberName] string propertyName = null)
        {
            var threadSafeHandler = Interlocked.CompareExchange(ref this.PropertyChanged, null, null);
            if (threadSafeHandler != null)
            {
                var args = EventArgsFactory.GetPropertyChangedEventArgs(propertyName);
                threadSafeHandler(this, args);
            }
        }

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <typeparam name="TProperty">プロパティの型です。</typeparam>
        /// <param name="propertyNameExpression">変更したプロパティの名前の式です。</param>
        protected virtual void RaisePropertyChanged<TProperty>(
            Expression<Func<TProperty>> propertyNameExpression)
        {
            Debug.Assert(propertyNameExpression != null);

            MemberExpression memberExpression =
                propertyNameExpression.Body as MemberExpression;
            Debug.Assert(memberExpression != null);

            this.RaisePropertyChanged(memberExpression.Member.Name);
        }

        /// <summary>
        /// プロパティを設定します。
        /// </summary>
        /// <typeparam name="T">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetProperty<T>(ref T item, T value, [CallerMemberName] string propertyName = null)
        {
            if (!EqualityComparer<T>.Default.Equals(item, value))
            {
                item = value;
                this.RaisePropertyChanged(propertyName);
            }
        }

        /// <summary>
        /// プロパティを強制設定します。
        /// </summary>
        /// <typeparam name="T">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetPropertyForce<T>(ref T item, T value, [CallerMemberName] string propertyName = null)
        {
            item = value;
            this.RaisePropertyChanged(propertyName);
        }

        /// <summary>
        /// プロパティを設定します。
        /// </summary>
        /// <typeparam name="T">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="callback">設定後に実行されるコールバックです。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetProperty<T>(ref T item, T value, Action callback, [CallerMemberName] string propertyName = null)
        {
            Debug.Assert(callback != null);

            if (!EqualityComparer<T>.Default.Equals(item, value))
            {
                item = value;
                this.RaisePropertyChanged(propertyName);
                callback();
            }
        }

        /// <summary>
        /// プロパティを強制設定します。
        /// </summary>
        /// <typeparam name="T">設定対象のプロパティの型です。</typeparam>
        /// <param name="item">設定対象です。</param>
        /// <param name="value">設定する値です。</param>
        /// <param name="callback">設定後に実行されるコールバックです。</param>
        /// <param name="propertyName">設定するプロパティ名です。</param>
        protected void SetPropertyForce<T>(ref T item, T value, Action callback, [CallerMemberName] string propertyName = null)
        {
            Debug.Assert(callback != null);

            item = value;
            this.RaisePropertyChanged(propertyName);
            callback();
        }
    }
}
