﻿namespace Opal.App
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.CompilerServices;
    using System.Text;
    using System.Threading.Tasks;
    using System.Windows;
    using Opal.ComponentModel;
    using Opal.Configurations;
    using Opal.Logs;
    using Opal.Menus;
    using Opal.Operations;
    using Opal.Panes;
    using Opal.Services;
    using Opal.Storages;
    using Opal.Threading;
    using Opal.Utilities;
    using Opal.ViewModels;
    using Opal.Windows;
    using Opal.Windows.Documents;
    using Opal.Windows.Generic;
    using Opal.Windows.Tools;

    /// <summary>
    /// アプリケーションを管理するクラスです。
    /// </summary>
    public abstract class AppManager
    {
        private static readonly InternalData Instance = new InternalData();

        /// <summary>
        /// オーナーウィンドウを設定します。
        /// </summary>
        /// <param name="ownerWindow">設定するオーナーウィンドウです。</param>
        public static void SetOwnerWindow(Window ownerWindow)
        {
            Debug.Assert(ownerWindow != null);
            Debug.Assert(Instance.OwnerWindow == null, "OwnerWindowの2回以上の設定はサポート外です");

            Instance.OwnerWindow = ownerWindow;

            Instance.OwnerWindow.Closing += (s, e) =>
            {
                if (Instance.OwnerViewModel != null)
                {
                    e.Cancel = Instance.OwnerViewModel.ClosingWindow();
                }
            };
        }

        /// <summary>
        /// オーナービューモデルを設定します。
        /// </summary>
        /// <param name="ownerWindow">設定するオーナービューモデルです。</param>
        public static void SetOwnerViewModel(MainWindowViewModel ownerViewModel)
        {
            Debug.Assert(ownerViewModel != null);
            Debug.Assert(Instance.OwnerViewModel == null, "OwnerViewModelの2回以上の設定はサポート外です");

            Instance.OwnerViewModel = ownerViewModel;
            ownerViewModel.PropertyChanged += OwnerViewModel_PropertyChanged;
        }

        private static void OwnerViewModel_PropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "ActiveDocument")
            {
                Instance.SetActiveDocument(Instance.OwnerViewModel.ActiveDocument);
            }
        }

        /// <summary>
        /// オーナーウィンドウを閉じます。
        /// </summary>
        public static void CloseOwnerWindow()
        {
            Debug.Assert(Instance.OwnerWindow != null);
            Instance.OwnerWindow.Close();
        }

        /// <summary>
        /// オーナーウィンドウをアクティブにします。
        /// </summary>
        public static void ActivateOwnerWindow()
        {
            Debug.Assert(Instance.OwnerWindow != null);
            Instance.OwnerWindow.Activate();
            Instance.OwnerWindow.WindowState = WindowState.Normal;
        }

        /// <summary>
        /// ログマネージャーを取得します。
        /// </summary>
        /// <returns>ログマネージャーのインスタンスを返します。</returns>
        public static ILogManager GetLogManager()
        {
            return Instance.GetManager<ILogManager>(ManagerKey.Log);
        }

        /// <summary>
        /// ドキュメントマネージャーを取得します。
        /// </summary>
        /// <returns>ドキュメントマネージャーのインスタンスを返します。</returns>
        public static IDocumentManager GetDocumentManager()
        {
            return Instance.GetManager<IDocumentManager>(ManagerKey.Document);
        }

        /// <summary>
        /// ツールマネージャーを取得します。
        /// </summary>
        /// <returns>ツールマネージャーのインスタンスを返します。</returns>
        public static IToolManager GetToolManager()
        {
            return Instance.GetManager<IToolManager>(ManagerKey.Tool);
        }

        /// <summary>
        /// ウィンドウマネージャーを取得します。
        /// </summary>
        /// <returns>ウィンドウマネージャーのインスタンスを返します。</returns>
        public static IWindowManager GetWindowManager()
        {
            return Instance.GetManager<IWindowManager>(ManagerKey.Window);
        }

        /// <summary>
        /// メニューマネージャーを取得します。
        /// </summary>
        /// <returns>メニューマネージャーのインスタンスを返します。</returns>
        public static IMenuManager GetMenuManager()
        {
            return Instance.GetManager<IMenuManager>(ManagerKey.Menu);
        }

        /// <summary>
        /// ジョブマネージャーを取得します。
        /// </summary>
        /// <returns>ジョブマネージャーのインスタンスを返します。</returns>
        public static IJobManager GetJobManager()
        {
            return Instance.GetManager<IJobManager>(ManagerKey.Job);
        }

        /// <summary>
        /// デバッグメッセージを書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        [Conditional("DEBUG")]
        public static void WriteDebug(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            Instance.WriteDebug(message, fileName, fileLine, memberName);
        }

        /// <summary>
        /// エラーメッセージを書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージです。</param>
        /// <param name="exception">例外処理のインスタンスです。ログに例外時の情報を入力したい場合に使用します。</param>
        /// <param name="associatedAction">ログに関連付ける処理です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public static void WriteError(string message, Exception exception = null, Action associatedAction = null, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            Instance.WriteError(message, exception, associatedAction, fileName, fileLine, memberName);
        }

        /// <summary>
        /// ユーザ通知用メッセージを書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public static void WriteUserMessage(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            Instance.WriteUserMessage(message, fileName, fileLine, memberName);
        }

        /// <summary>
        /// 情報メッセージを書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージです。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public static void WriteInfo(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            Instance.WriteInfo(message, fileName, fileLine, memberName);
        }

        /// <summary>
        /// 警告メッセージを書き込みます。
        /// </summary>
        /// <param name="message">書き込むメッセージです。</param>
        /// <param name="exception">登録する例外です。</param>
        /// <param name="associatedAction">ログに関連付ける処理です。</param>
        /// <param name="fileName">ファイル名です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="fileLine">ファイル行です。記述した場所を自動的に入れるので設定不要です。</param>
        /// <param name="memberName">メンバー名です。記述した場所を自動的に入れるので設定不要です。</param>
        public static void WriteWarning(string message, Exception exception = null, Action associatedAction = null, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
        {
            Instance.WriteWarning(message, exception, associatedAction, fileName, fileLine, memberName);
        }

        /// <summary>
        /// ストレージを取得します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <returns>指定したグループに存在するストレージのインスタンスを返します。</returns>
        public static WeakReference<TStorage> GetStorage<TStorage>() where TStorage : Storage
        {
            return Instance.GetStorage<TStorage>();
        }

        /// <summary>
        /// ストレージを追加します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <param name="storage">追加するストレージです。</param>
        public static void AddStorage<TStorage>(TStorage storage) where TStorage : Storage
        {
            Debug.Assert(storage != null);
            Instance.AddStorage(storage);
        }

        /// <summary>
        /// ストレージが存在するか判定します。
        /// </summary>
        /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
        /// <returns>ストレージが存在すれば、true を返します。</returns>
        public static bool HasStorage<TStorage>() where TStorage : Storage
        {
            return Instance.HasStorage<TStorage>();
        }

        /// <summary>
        /// コンフィグを取得します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <returns>指定したグループに存在するコンフィグのインスタンスを返します。</returns>
        public static WeakReference<TConfig> GetConfig<TConfig>() where TConfig : Config
        {
            return Instance.GetConfig<TConfig>();
        }

        /// <summary>
        /// コンフィグを追加します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <param name="config">追加するコンフィグです。</param>
        public static void AddConfig<TConfig>(TConfig config) where TConfig : Config
        {
            Debug.Assert(config != null);
            Instance.AddConfig(config);
        }

        /// <summary>
        /// コンフィグが存在するか判定します。
        /// </summary>
        /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
        /// <returns>コンフィグが存在すれば、true を返します。</returns>
        public static bool HasConfig<TConfig>() where TConfig : Config
        {
            return Instance.HasConfig<TConfig>();
        }

        /// <summary>
        /// ウィンドウを表示します。
        /// </summary>
        /// <typeparam name="TWindow">表示ウィンドウのテンプレートの型です。</typeparam>
        /// <typeparam name="TArgs">ウィンドウ表示に使用する引数クラスのテンプレートの型です。</typeparam>
        /// <param name="args">ウィンドウ表示の引数です。</param>
        /// <returns>成功ならば、true を返します。</returns>
        public static bool ShowWindow<TWindow, TArgs>(TArgs args) where TArgs : WindowArgs
        {
            return Instance.ShowWindow<TWindow, TArgs>(args);
        }

        /// <summary>
        /// ウィンドウを表示します。
        /// </summary>
        /// <typeparam name="TWindow">表示ウィンドウのテンプレートの型です。</typeparam>
        /// <returns>成功ならば、true を返します。</returns>
        public static bool ShowWindow<TWindow>()
        {
            return Instance.ShowWindow<TWindow>();
        }

        /// <summary>
        /// オペレーションを実行します。
        /// </summary>
        /// <param name="operation">実行するオペレーションです。</param>
        public static void ExecuteOperation(Operation operation)
        {
            Instance.ExecuteOperation(operation);
        }

        /// <summary>
        /// ドキュメントを作成します。
        /// </summary>
        /// <param name="groupName">作成対象のドキュメントを特定するグループ名です。</param>
        /// <param name="key">作成対象のドキュメントを特定するキーです。</param>
        /// <returns>ドキュメントのインスタンスを返します。</returns>
        public static IDocument CreateDocument(string groupName, string key)
        {
            return Instance.CreateDocument(groupName, key);
        }

        /// <summary>
        /// ドキュメントを所有しているか判定します。
        /// </summary>
        /// <param name="document">判定対象のドキュメントです。</param>
        /// <returns>所有している場合は、true を返します。</returns>
        public static bool HasDocument(IDocument document)
        {
            Debug.Assert(document is DocumentViewModel);

            return Instance.HasDocument(document as DocumentViewModel);
        }

        /// <summary>
        /// ドキュメントを追加します。
        /// </summary>
        /// <param name="document">追加するドキュメントです。</param>
        public static void AddDocument(IDocument document)
        {
            Debug.Assert(document is DocumentViewModel);

            Instance.AddDocument(document as DocumentViewModel);
        }

        /// <summary>
        /// ドキュメントを削除します。
        /// </summary>
        /// <param name="document">削除するドキュメントです。</param>
        public static void RemoveDocument(IDocument document)
        {
            Debug.Assert(document is DocumentViewModel);

            Instance.RemoveDocument(document as DocumentViewModel);
        }

        /// <summary>
        /// デフォルトツールを生成します。
        /// </summary>
        public static void MakeDefaultTools()
        {
            var manager = GetPaneManager();
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        public static void Initialize()
        {
            var menuManager = Instance.GetManager<MenuManager>(ManagerKey.Menu);
            Debug.Assert(menuManager != null);

            // 表示メニューを作成します。
            foreach (var menu in CreateDisplayMenus())
            {
                menuManager.AddDisplayMenu(menu);
            }

            // ストレージにアクティブなドキュメント変更時のイベントを登録する。
            foreach (var storage in Instance.GetStorages())
            {
                Instance.StorageActiveDocumentChanged += storage.CallActiveDocumentChanged;
            }
        }

        /// <summary>
        /// アクティブ状態のドキュメントを設定します。
        /// </summary>
        /// <param name="document">アクティブ対象のドキュメントです。</param>
        public static void SetActiveDocument(IDocument document)
        {
            Instance.SetActiveDocument(document);
        }

        /// <summary>
        /// アクティブ状態のドキュメントを取得します。
        /// </summary>
        /// <returns>アクティブ状態のドキュメントを返します。</returns>
        public static IDocument GetActiveDocument()
        {
            return Instance.GetActiveDocument();
        }

        /// <summary>
        /// 特定の型のツールビューモデルを取得します。
        /// </summary>
        /// <typeparam name="ToolViewModelType">取得したいツールビューモデルをの型を指定します。</typeparam>
        /// <returns>ツールビューモデルを返します。</returns>
        public static ToolViewModelType GetToolViewModel<ToolViewModelType>()
            where ToolViewModelType : ToolViewModel
        {
            var toolPanes = Instance.GetManager<PaneManager>(ManagerKey.Pane).GetTools();
            foreach (var toolPane in toolPanes)
            {
                if (toolPane is ToolViewModelType)
                {
                    return toolPane as ToolViewModelType;
                }
            }

            return null;
        }

        /// <summary>
        /// ツールを追加します。
        /// </summary>
        /// <param name="key">追加するツールのキーです。</param>
        public static void AddTool(string toolKey)
        {
            var toolManager = Instance.GetManager<ToolManager>(ManagerKey.Tool);
            if (toolManager == null)
            {
                throw new InvalidOperationException("ToolManager was not found");
            }

            var paneManager = Instance.GetManager<PaneManager>(ManagerKey.Pane);
            if (paneManager == null)
            {
                throw new InvalidOperationException("PaneManager was not found");
            }

            var maker = toolManager.GetToolMakerFromKey(toolKey);
            AddTool(maker, paneManager);
        }

        /// <summary>
        /// オペレーションマネージャー取得します。
        /// </summary>
        /// <returns>オペレーションマネージャーのインスタンスを返します。</returns>
        internal static IOperationManager GetOperationManager()
        {
            return Instance.GetManager<IOperationManager>(ManagerKey.Operation);
        }

        /// <summary>
        /// ペインマネージャーを取得します。
        /// </summary>
        /// <returns>ペインマネージャーのインスタンスを返します。</returns>
        internal static PaneManager GetPaneManager()
        {
            return Instance.GetManager<PaneManager>(ManagerKey.Pane);
        }

        /// <summary>
        /// ツールメーカーを取得します。
        /// </summary>
        /// <param name="makerFullName">メーカーのフルネームです。</param>
        /// <returns>メーカーのインスタンスを返します。</returns>
        internal static ToolMaker GetToolMaker(string makerFullName)
        {
            ToolManager toolManager = Instance.GetManager<IToolManager>(ManagerKey.Tool) as ToolManager;

            return toolManager.GetToolMakerFromName(makerFullName);
        }

        /// <summary>
        /// ツールを追加します。
        /// </summary>
        /// <param name="maker">追加するツールのメーカークラスです。</param>
        /// <param name="paneManager">ツールを管理するマネージャークラスです。</param>
        /// <returns>ツールのインスタンスを返します。</returns>
        internal static ToolViewModel AddTool(ToolMaker maker, PaneManager paneManager)
        {
            Debug.Assert(maker != null);
            Debug.Assert(paneManager != null);

            var viewModel = maker.CreateTool();
            viewModel.IsActive = true;
            viewModel.IsVisible = true;

            if (!paneManager.ContainTool(viewModel))
            {
                Instance.ActiveDocumentChanged += viewModel.CallActiveDocumentChanged;

                foreach (var storage in Instance.GetStorages())
                {
                    storage.StorageChanged += viewModel.CallStorageChanged;
                }

                paneManager.AddTool(viewModel);
            }

            return viewModel;
        }

        /// <summary>
        /// マネージャーを追加します。(内部処理用）
        /// </summary>
        /// <param name="key">追加対象のキーです。</param>
        /// <param name="manager">追加するマネージャーです。</param>
        protected void AddManagerInternal(ManagerKey key, IAnyManager manager)
        {
            Debug.Assert(manager != null);
            Instance.AddManager(key, manager);
        }

        /// <summary>
        /// 状態を全てクリアします。（内部処理用）
        /// </summary>
        protected void ClearAllInternal()
        {
            Instance.Clear();
        }

        private static IEnumerable<Menu> CreateDisplayMenus()
        {
            var toolManager = Instance.GetManager<ToolManager>(ManagerKey.Tool);
            Debug.Assert(toolManager != null);

            var paneManager = Instance.GetManager<PaneManager>(ManagerKey.Pane);
            Debug.Assert(paneManager != null);

            foreach (var key in toolManager.GetKeys())
            {
                var maker = toolManager.GetToolMakerFromKey(key);
                Menu menu = new CommandableMenu(
                        maker.Key,
                        maker.Label,
                        param =>
                        {
                            AddTool(maker, paneManager);
                        },
                        param => { return true; });
                yield return menu;
            }
        }

        /// <summary>
        /// 内部データ管理用クラスです。
        /// </summary>
        private class InternalData : SynchronizableObject
        {
            private readonly Dictionary<ManagerKey, IAnyManager> managers = new Dictionary<ManagerKey, IAnyManager>();
            private readonly WeakReference<IDocument> weakActiveDocument = new WeakReference<IDocument>(null);
            private Window ownerWindow = null;
            private MainWindowViewModel ownerViewModel = null;

            private EventHandler<ActiveDocumentChangedEventArgs> storageActiveDocumentChanged;
            private EventHandler<ActiveDocumentChangedEventArgs> activeDocumentChanged;

            /// <summary>
            /// ストレージクラスに対して、アクティブなドキュメントが変更された時に実行されるイベントです。
            /// </summary>
            internal event EventHandler<ActiveDocumentChangedEventArgs> StorageActiveDocumentChanged
            {
                add
                {
                    this.storageActiveDocumentChanged += value.MakeWeakEventHandler(eventHandler => this.storageActiveDocumentChanged -= eventHandler);
                }

                remove
                {
                }
            }

            /// <summary>
            /// アクティブなドキュメントが変更された時に実行されるイベントです。
            /// </summary>
            internal event EventHandler<ActiveDocumentChangedEventArgs> ActiveDocumentChanged
            {
                add
                {
                    this.activeDocumentChanged += value.MakeWeakEventHandler(eventHandler => this.activeDocumentChanged -= eventHandler);
                }

                remove
                {
                }
            }

            /// <summary>
            /// オーナーウィンドウを取得設定します。
            /// </summary>
            internal Window OwnerWindow
            {
                get
                {
                    return this.ownerWindow;
                }

                set
                {
                    this.ownerWindow = value;
                }
            }

            /// <summary>
            /// オーナービューモデルを取得設定します。
            /// </summary>
            internal MainWindowViewModel OwnerViewModel
            {
                get
                {
                    return this.ownerViewModel;
                }

                set
                {
                    this.ownerViewModel = value;
                }
            }

            [Conditional("DEBUG")]
            internal void WriteDebug(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
            {
                Debug.Assert(!string.IsNullOrEmpty(message));

                var logManager = this.GetManager<LogManager>(ManagerKey.Log);
                Debug.Assert(logManager != null);

                Logger logger = new Logger();
                logger.AddDebug(message, fileName, fileLine, memberName);
                logManager.PushLogger(logger);
                logManager.FlushAll();
            }

            internal void WriteError(string message, Exception exception, Action associatedAction, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
            {
                Debug.Assert(!string.IsNullOrEmpty(message));

                var logManager = this.GetManager<LogManager>(ManagerKey.Log);
                Debug.Assert(logManager != null);

                Logger logger = new Logger();
                logger.AddError(message, exception, associatedAction, fileName, fileLine, memberName);
                logManager.PushLogger(logger);
                logManager.FlushAll();
            }

            internal void WriteUserMessage(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
            {
                Debug.Assert(!string.IsNullOrEmpty(message));

                var logManager = this.GetManager<LogManager>(ManagerKey.Log);
                Debug.Assert(logManager != null);

                Logger logger = new Logger();
                logger.AddUserMessage(message, fileName, fileLine, memberName);
                logManager.PushLogger(logger);
                logManager.FlushAll();
            }

            internal void WriteInfo(string message, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
            {
                Debug.Assert(!string.IsNullOrEmpty(message));

                var logManager = this.GetManager<LogManager>(ManagerKey.Log);
                Debug.Assert(logManager != null);

                Logger logger = new Logger();
                logger.AddInfo(message, fileName, fileLine, memberName);
                logManager.PushLogger(logger);
                logManager.FlushAll();
            }

            internal void WriteWarning(string message, Exception exception, Action associatedAction, [CallerFilePath] string fileName = null, [CallerLineNumber] int fileLine = 0, [CallerMemberName] string memberName = null)
            {
                Debug.Assert(!string.IsNullOrEmpty(message));

                var logManager = this.GetManager<LogManager>(ManagerKey.Log);
                Debug.Assert(logManager != null);

                Logger logger = new Logger();
                logger.AddWarning(message, exception, associatedAction, fileName, fileLine, memberName);
                logManager.PushLogger(logger);
                logManager.FlushAll();
            }

            /// <summary>
            /// オペレーションを実行します。
            /// </summary>
            /// <param name="operation">実行するオペレーションです。</param>
            internal void ExecuteOperation(Operation operation)
            {
                Debug.Assert(operation != null);

                var operationManager = this.GetManager<OperationManager>(ManagerKey.Operation);
                Debug.Assert(operationManager != null);

                operationManager.Execute(operation);
            }

            /// <summary>
            /// アクティブ状態のドキュメントを設定します。
            /// </summary>
            /// <param name="targetDocument">アクティブ対象のドキュメントです。</param>
            internal void SetActiveDocument(IDocument targetDocument)
            {
                this.weakActiveDocument.SetTarget(targetDocument);

                {
                    var paneManager = this.GetManager<PaneManager>(ManagerKey.Pane);
                    Debug.Assert(paneManager != null);
                    if (targetDocument != null)
                    {
                        foreach (DocumentViewModel document in paneManager.GetDocuments())
                        {
                            if (document != targetDocument)
                            {
                                document.IsActive = false;
                            }
                        }
                    }
                    else
                    {
                        foreach (DocumentViewModel document in paneManager.GetDocuments())
                        {
                            document.IsActive = false;
                        }
                    }
                }

                if (targetDocument != null)
                {
                    targetDocument.IsActive = true;
                }

                // ストレージの方を先にアクティブ通知を行う。
                if (this.storageActiveDocumentChanged != null)
                {
                    this.storageActiveDocumentChanged(this, new ActiveDocumentChangedEventArgs(targetDocument));
                }

                if (this.activeDocumentChanged != null)
                {
                    this.activeDocumentChanged(this, new ActiveDocumentChangedEventArgs(targetDocument));
                }

                if (this.OwnerViewModel != null)
                {
                    var activeDocumment = targetDocument as DocumentViewModel;
                    if (activeDocumment != null)
                    {
                        this.OwnerViewModel.ActiveDocument = activeDocumment;
                    }
                }
            }

            /// <summary>
            /// アクティブ状態のドキュメントを取得します。
            /// </summary>
            /// <returns>アクティブ状態のドキュメントを返します。</returns>
            internal IDocument GetActiveDocument()
            {
                IDocument activeDocument = null;
                this.weakActiveDocument.TryGetTarget(out activeDocument);
                return activeDocument;
            }

            /// <summary>
            /// マネージャーを追加します。
            /// </summary>
            /// <param name="key">追加対象のキーです。</param>
            /// <param name="manager">追加するマネージャーです。</param>
            internal void AddManager(ManagerKey key, IAnyManager manager)
            {
                Debug.Assert(manager != null);
                lock (this.SyncRoot)
                {
                    this.managers.Add(key, manager);
                }
            }

            /// <summary>
            /// マネージャーをクリアします。
            /// </summary>
            internal void Clear()
            {
                lock (this.SyncRoot)
                {
                    this.managers.Clear();
                }
            }

            /// <summary>
            /// マネージャーを取得します。
            /// </summary>
            /// <typeparam name="T">マネージャーのテンプレートの型です。</typeparam>
            /// <param name="key">マネージャーキーです。</param>
            /// <returns>指定の型のマネージャーインスタンスを返します。存在しない場合は、null を返します。</returns>
            internal T GetManager<T>(ManagerKey key) where T : class, IAnyManager
            {
                lock (this.SyncRoot)
                {
                    IAnyManager manager = null;
                    if (this.managers.TryGetValue(key, out manager))
                    {
                        return manager as T;
                    }

                    return null;
                }
            }

            /// <summary>
            /// ストレージを取得します。
            /// </summary>
            /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
            /// <returns>指定したグループに存在するストレージのインスタンスを返します。</returns>
            internal WeakReference<TStorage> GetStorage<TStorage>() where TStorage : Storage
            {
                var manager = this.GetManager<StorageManager>(ManagerKey.Storage);
                if (manager == null)
                {
                    return null;
                }

                return manager.GetStorage<TStorage>();
            }

            /// <summary>
            /// ストレージを追加します。
            /// </summary>
            /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
            /// <param name="storage">追加するストレージです。</param>
            internal void AddStorage<TStorage>(TStorage storage) where TStorage : Storage
            {
                Debug.Assert(storage != null);
                lock (this.SyncRoot)
                {
                    var manager = this.GetManager<StorageManager>(ManagerKey.Storage);
                    Debug.Assert(manager != null);
                    manager.AddStorage(storage);
                }
            }

            /// <summary>
            /// ストレージが存在するか判定します。
            /// </summary>
            /// <typeparam name="TStorage">ストレージのテンプレートの型です。</typeparam>
            /// <returns>ストレージが存在すれば、true を返します。</returns>
            internal bool HasStorage<TStorage>() where TStorage : Storage
            {
                lock (this.SyncRoot)
                {
                    var manager = this.GetManager<StorageManager>(ManagerKey.Storage);
                    Debug.Assert(manager != null);
                    return manager.HasStorage<TStorage>();
                }
            }

            /// <summary>
            /// ストレージを取得します。
            /// </summary>
            /// <returns>ストレージのインスタンスを返します。</returns>
            internal IEnumerable<Storage> GetStorages()
            {
                lock (this.SyncRoot)
                {
                    var manager = this.GetManager<StorageManager>(ManagerKey.Storage);
                    Debug.Assert(manager != null);
                    return manager.GetStorages();
                }
            }

            /// <summary>
            /// コンフィグを取得します。
            /// </summary>
            /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
            /// <returns>指定したグループに存在するコンフィグのインスタンスを返します。</returns>
            internal WeakReference<TConfig> GetConfig<TConfig>() where TConfig : Config
            {
                var manager = this.GetManager<ConfigManager>(ManagerKey.Config);
                Debug.Assert(manager != null);
                return manager.GetConfig<TConfig>();
            }

            /// <summary>
            /// コンフィグを追加します。
            /// </summary>
            /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
            /// <param name="config">追加するコンフィグです。</param>
            internal void AddConfig<TConfig>(TConfig config) where TConfig : Config
            {
                Debug.Assert(config != null);
                lock (this.SyncRoot)
                {
                    var manager = this.GetManager<ConfigManager>(ManagerKey.Config);
                    Debug.Assert(manager != null);
                    manager.AddConfig(config);
                }
            }

            /// <summary>
            /// コンフィグが存在するか判定します。
            /// </summary>
            /// <typeparam name="TConfig">コンフィグのテンプレートの型です。</typeparam>
            /// <returns>コンフィグが存在すれば、true を返します。</returns>
            internal bool HasConfig<TConfig>() where TConfig : Config
            {
                lock (this.SyncRoot)
                {
                    var manager = this.GetManager<ConfigManager>(ManagerKey.Config);
                    Debug.Assert(manager != null);
                    return manager.HasConfig<TConfig>();
                }
            }

            /// <summary>
            /// コンフィグを取得します。
            /// </summary>
            /// <returns>コンフィグのインスタンスを返します。</returns>
            internal IEnumerable<Config> GetConfigs()
            {
                lock (this.SyncRoot)
                {
                    var manager = this.GetManager<ConfigManager>(ManagerKey.Config);
                    Debug.Assert(manager != null);
                    return manager.GetConfigs();
                }
            }

            /// <summary>
            /// ウィンドウを表示します。
            /// </summary>
            /// <typeparam name="TWindow">表示ウィンドウのテンプレートの型です。</typeparam>
            /// <typeparam name="TArgs">ウィンドウ表示に使用する引数クラスのテンプレートの型です。</typeparam>
            /// <param name="args">ウィンドウ表示の引数です。</param>
            /// <returns>成功ならば、true を返します。</returns>
            internal bool ShowWindow<TWindow, TArgs>(TArgs args) where TArgs : WindowArgs
            {
                var windowManager = this.GetManager<WindowManager>(ManagerKey.Window);
                Debug.Assert(windowManager != null);

                lock (this.SyncRoot)
                {
                    var maker = windowManager.GetWindowMaker<TWindow>();
                    Debug.Assert(maker != null);

                    return ((WindowMaker<TArgs>)maker).Show(args);
                }
            }

            /// <summary>
            /// ウィンドウを表示します。
            /// </summary>
            /// <typeparam name="TWindow">表示ウィンドウのテンプレートの型です。</typeparam>
            /// <returns>成功ならば、true を返します。</returns>
            internal bool ShowWindow<TWindow>()
            {
                var windowManager = this.GetManager<WindowManager>(ManagerKey.Window);
                Debug.Assert(windowManager != null);

                lock (this.SyncRoot)
                {
                    var maker = windowManager.GetWindowMaker<TWindow>();
                    Debug.Assert(maker != null);

                    return maker.Show();
                }
            }

            /// <summary>
            /// ドキュメントを作成します。
            /// </summary>
            /// <param name="groupName">作成対象のドキュメントを特定するグループ名です。</param>
            /// <param name="key">作成対象のドキュメントを特定するキーです。</param>
            /// <returns>ドキュメントのインスタンスを返します。</returns>
            internal IDocument CreateDocument(string groupName, string key)
            {
                var documentManager = this.GetManager<Opal.Windows.Documents.DocumentManager>(ManagerKey.Document);

                lock (this.SyncRoot)
                {
                    var maker = documentManager.GetDocumentMaker(key);
                    Debug.Assert(maker != null);

                    var document = maker.CreateDocument();
                    document.DocumentClosed += this.CallDocumentClosed;

                    return document;
                }
            }

            /// <summary>
            /// ドキュメントを所有しているか判定します。
            /// </summary>
            /// <param name="document">判定対象のドキュメントです。</param>
            /// <returns>所有している場合は、true を返します。</returns>
            internal bool HasDocument(DocumentViewModel document)
            {
                Debug.Assert(document != null);

                var paneManager = this.GetManager<PaneManager>(ManagerKey.Pane);
                Debug.Assert(paneManager != null);

                lock (this.SyncRoot)
                {
                    return paneManager.ContainDocument(document);
                }
            }

            /// <summary>
            /// ドキュメントを追加します。
            /// </summary>
            /// <param name="document">追加するドキュメントです。</param>
            internal void AddDocument(DocumentViewModel document)
            {
                Debug.Assert(document != null);

                var paneManager = this.GetManager<PaneManager>(ManagerKey.Pane);
                Debug.Assert(paneManager != null);

                lock (this.SyncRoot)
                {
                    paneManager.AddDocument(document);
                }
            }

            /// <summary>
            /// ドキュメントを削除します。
            /// </summary>
            /// <param name="document">削除するドキュメントです。</param>
            internal void RemoveDocument(DocumentViewModel document)
            {
                Debug.Assert(document != null);

                var paneManager = this.GetManager<PaneManager>(ManagerKey.Pane);
                Debug.Assert(paneManager != null);

                lock (this.SyncRoot)
                {
                    paneManager.RemoveDocument(document);
                }
            }

            private void CallDocumentClosed(object sender, DocumentClosedEventArgs e)
            {
                Debug.Assert(sender != null);
                Debug.Assert(e != null);

                var paneManager = this.GetManager<PaneManager>(ManagerKey.Pane);
                Debug.Assert(paneManager != null);

                IDocument document = null;
                if (e.WeakDocument.TryGetTarget(out document))
                {
                    var viewModel = document as DocumentViewModel;
                    if (viewModel != null)
                    {
                        if (paneManager.ContainDocument(viewModel))
                        {
                            viewModel.DocumentClosed -= this.CallDocumentClosed;
                            viewModel.PaneView.DataContext = null;
                            paneManager.RemoveDocument(viewModel);
                        }
                    }
                }
            }
        }
    }
}
