﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Vertex : ObservableEntity<vertexType>, IChildEntity, IDeepCopyable<Vertex>, IDeepCopyFrom<Vertex>
    {
        private readonly LodOffset @lodOffset = new LodOffset();
        private readonly ObservableList<VtxAttrib> @vtxAttribs = new ObservableList<VtxAttrib>();
        private readonly ObservableList<VtxBuffer> @vtxBuffers = new ObservableList<VtxBuffer>();
        private int @vertexIndex;
        private Model parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Vertex()
        {
            this.@lodOffset.PropertyChanged += this.OnPropertyChanged;
            this.@vtxAttribs.CollectionChanged += OnCollectionChanged<VtxAttrib>;
            this.@vtxBuffers.CollectionChanged += OnCollectionChanged<VtxBuffer>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Vertex(Vertex source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Vertex(vertexType source)
        {
            this.@vtxAttribs.CollectionChanged += OnCollectionChanged<VtxAttrib>;
            this.@vtxBuffers.CollectionChanged += OnCollectionChanged<VtxBuffer>;
            if (source.@lod_offset != null)
            {
                this.@lodOffset = new LodOffset(source.@lod_offset);
            }
            else
            {
                this.@lodOffset = new LodOffset();
            }
            if (source.@vtx_attrib_array != null)
            {
                this.@vtxAttribs.Clear();
                foreach (var elem in source.@vtx_attrib_array.Items)
                {
                    this.@vtxAttribs.Add(new VtxAttrib(elem));
                }
            }
            if (source.@vtx_buffer_array != null)
            {
                this.@vtxBuffers.Clear();
                foreach (var elem in source.@vtx_buffer_array.Items)
                {
                    this.@vtxBuffers.Add(new VtxBuffer(elem));
                }
            }
            this.@vertexIndex = source.@vertex_index;
            this.@lodOffset.PropertyChanged += this.OnPropertyChanged;
        }

        public LodOffset LodOffset
        {
            get
            {
                return this.@lodOffset;
            }
        }

        public ObservableList<VtxAttrib> VtxAttribs
        {
            get
            {
                return this.@vtxAttribs;
            }
        }

        public ObservableList<VtxBuffer> VtxBuffers
        {
            get
            {
                return this.@vtxBuffers;
            }
        }

        public int VertexIndex
        {
            get
            {
                return this.@vertexIndex;
            }

            set
            {
                if (this.@vertexIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@vertexIndex, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Model : true);
                this.SetProperty(ref this.parent, value as Model, () => this.CalcCRC());
            }
        }

        public Model Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Model);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override vertexType CreateSerializableData()
        {
            var writeData = new vertexType();
            writeData.@lod_offset = this.@lodOffset.CreateSerializableData() as lod_offsetType;
            if (this.@VtxAttribs.Count > 0)
            {
                writeData.@vtx_attrib_array = new vtx_attrib_arrayType();
                writeData.@vtx_attrib_array.Items = this.@vtxAttribs.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@vtx_attrib_array.Items.Length == 0)
                {
                    writeData.@vtx_attrib_array = null;
                }
            }
            if (this.VtxBuffers.Count > 0)
            {
                writeData.vtx_buffer_array = new vtx_buffer_arrayType();
                writeData.vtx_buffer_array.Items = this.@vtxBuffers.Select(x => x.CreateSerializableData()).ToArray();
            }
            else
            {
                // 未定義の場合は全頂点属性を含む頂点バッファを作成
                List<inputType> inputs = new List<inputType>();
                for (int attrIndex = 0; attrIndex < this.VtxAttribs.Count; ++attrIndex)
                {
                    inputs.Add(new inputType()
                    {
                        attrib_index = attrIndex
                    });
                }

                writeData.vtx_buffer_array = new vtx_buffer_arrayType();
                writeData.vtx_buffer_array.Items = new vtx_bufferType[]
                {
                    new vtx_bufferType()
                    {
                        input_array = new input_arrayType()
                        {
                            Items = inputs.ToArray()
                        }
                    }
                };
            }
            writeData.@vertex_index = this.VertexIndex;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Vertex IDeepCopyable<Vertex>.DeepCopy()
        {
            return new Vertex(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Vertex source)
        {
            this.@lodOffset.DeepCopyFrom(source.@lodOffset);
            CopyElements(source.@vtxAttribs, this.@vtxAttribs);
            CopyElements(source.@vtxBuffers, this.@vtxBuffers);
            this.@vertexIndex = source.@vertexIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.LodOffset.HashValue)));
            foreach (var elem in this.VtxAttribs)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.VtxBuffers)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.VertexIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@lodOffset.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@vtxAttribs)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@vtxBuffers)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@lodOffset.Reset();
            foreach (var elem in this.@vtxAttribs)
            {
                elem.Reset();
            }
            foreach (var elem in this.@vtxBuffers)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@lodOffset.Refresh();
            foreach (var elem in this.@vtxAttribs)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@vtxBuffers)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
