﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class UserToolData : ObservableEntity<user_tool_dataType>, IDeepCopyable<UserToolData>, IDeepCopyFrom<UserToolData>
    {
        private readonly ObservableList<ToolDataElement> @elements = new ObservableList<ToolDataElement>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserToolData()
        {
            this.@elements.CollectionChanged += OnCollectionChanged<ToolDataElement>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public UserToolData(UserToolData source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public UserToolData(user_tool_dataType source)
        {
            this.@elements.CollectionChanged += OnCollectionChanged<ToolDataElement>;
            if (source.@Any != null)
            {
                this.@elements.Clear();
                foreach (var elem in source.@Any)
                {
                    this.@elements.Add(new ToolDataElement(elem));
                }
            }
        }

        public ObservableList<ToolDataElement> Elements
        {
            get
            {
                return this.@elements;
            }
        }

        /// <summary>
        /// 指定された型のツールデータオブジェクトを探します。
        /// </summary>
        /// <typeparam name="T">ツールデータオブジェクトの型を指定します。</typeparam>
        /// <returns>該当するデータを返します。見つからない場合は null を返します。</returns>
        public T FindToolDataObject<T>()
            where T : class, IToolDataObject
        {
            return FindToolDataObject(typeof(T)) as T;
        }

        public void AddToolDataObject(IToolDataObject toolDataObject)
        {
            Ensure.Argument.Null(FindToolDataObject(toolDataObject.GetType()));
            this.Elements.Add(new ToolDataElement(toolDataObject));
        }

        public void RemoveToolDataObject(IToolDataObject toolDataObject)
        {
            Ensure.Argument.True(this.elements.Any(x => x.ToolDataObject == toolDataObject));
            this.elements.Remove(this.elements.First(x => x.ToolDataObject == toolDataObject));
        }

        private IToolDataObject FindToolDataObject(Type toolDataObjectType)
        {
            foreach (var elem in this.Elements)
            {
                if (elem.ToolDataObject.GetType() == toolDataObjectType)
                {
                    return elem.ToolDataObject;
                }
            }

            return null;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override user_tool_dataType CreateSerializableData()
        {
            if (this.elements.Count == 0)
            {
                return null;
            }
            var writeData = new user_tool_dataType();
            if (this.@Elements.Count > 0)
            {
                writeData.@Any = this.@elements.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@Any.Length == 0)
                {
                    writeData.@Any = null;
                }
            }
            if (writeData.Any == null)
            {
                return null;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        UserToolData IDeepCopyable<UserToolData>.DeepCopy()
        {
            return new UserToolData(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(UserToolData source)
        {
            CopyElements(source.@elements, this.@elements);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.Elements)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@elements)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@elements)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@elements)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
