﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class SkeletonInfo : ObservableEntity<skeletonInfoType>, IDeepCopyable<SkeletonInfo>, IDeepCopyFrom<SkeletonInfo>
    {
        private bool @scaleEnable = true;
        private skeleton_info_scale_modeType @scaleMode = skeleton_info_scale_modeType.standard;
        private skeleton_info_rotate_modeType @rotateMode = skeleton_info_rotate_modeType.euler_xyz;
        private bool @motionMirroringEnable = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public SkeletonInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SkeletonInfo(SkeletonInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public SkeletonInfo(skeletonInfoType source)
        {
            this.@scaleEnable = source.@scale_enable;
            this.@scaleMode = source.@scale_mode;
            this.@rotateMode = source.@rotate_mode;
            this.@motionMirroringEnable = source.@motion_mirroring_enable;
        }

        public bool ScaleEnable
        {
            get
            {
                return this.@scaleEnable;
            }

            set
            {
                if (this.@scaleEnable == value)
                {
                    return;
                }

                this.SetProperty(ref this.@scaleEnable, value, () => this.CalcCRC());
            }
        }

        public skeleton_info_scale_modeType ScaleMode
        {
            get
            {
                return this.@scaleMode;
            }

            set
            {
                if (this.@scaleMode == value)
                {
                    return;
                }

                this.SetProperty(ref this.@scaleMode, value, () => this.CalcCRC());
            }
        }

        public skeleton_info_rotate_modeType RotateMode
        {
            get
            {
                return this.@rotateMode;
            }

            set
            {
                if (this.@rotateMode == value)
                {
                    return;
                }

                this.SetProperty(ref this.@rotateMode, value, () => this.CalcCRC());
            }
        }

        public bool MotionMirroringEnable
        {
            get
            {
                return this.@motionMirroringEnable;
            }

            set
            {
                if (this.@motionMirroringEnable == value)
                {
                    return;
                }

                this.SetProperty(ref this.@motionMirroringEnable, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override skeletonInfoType CreateSerializableData()
        {
            var writeData = new skeletonInfoType();
            writeData.@scale_enable = this.ScaleEnable;
            writeData.@scale_mode = this.ScaleMode;
            writeData.@rotate_mode = this.RotateMode;
            writeData.@motion_mirroring_enable = this.MotionMirroringEnable;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        SkeletonInfo IDeepCopyable<SkeletonInfo>.DeepCopy()
        {
            return new SkeletonInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(SkeletonInfo source)
        {
            this.@scaleEnable = source.@scaleEnable;
            this.@scaleMode = source.@scaleMode;
            this.@rotateMode = source.@rotateMode;
            this.@motionMirroringEnable = source.@motionMirroringEnable;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ScaleEnable)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ScaleMode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.RotateMode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MotionMirroringEnable)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
