﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShapeInfo : ObservableEntity<shape_infoType>, IChildEntity, IDeepCopyable<ShapeInfo>, IDeepCopyFrom<ShapeInfo>
    {
        private string @matName = string.Empty;
        private string @boneName = string.Empty;
        private string @originalMaterialName = string.Empty;
        private string @originalBoneName = string.Empty;
        private int @vertexIndex;
        private string @originalShapeHash = string.Empty;
        private int @originalShapeCount = 1;
        private string @deleteNearVertexMode = string.Empty;
        private string @divideSubmeshMode = string.Empty;
        private string @optimizePrimitiveMode = string.Empty;
        private string @polygonReductionMode = string.Empty;
        private Shape parent = null;
        private Vertex @vertex = null;
        private Bone @bone = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShapeInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShapeInfo(ShapeInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShapeInfo(shape_infoType source)
        {
            this.@matName = source.@mat_name;
            this.@boneName = source.@bone_name;
            this.@originalMaterialName = source.@original_material_name;
            this.@originalBoneName = source.@original_bone_name;
            this.@vertexIndex = source.@vertex_index;
            this.@originalShapeHash = source.@original_shape_hash;
            this.@originalShapeCount = source.@original_shape_count;
            this.@deleteNearVertexMode = source.@delete_near_vertex_mode;
            this.@divideSubmeshMode = source.@divide_submesh_mode;
            this.@optimizePrimitiveMode = source.@optimize_primitive_mode;
            this.@polygonReductionMode = source.@polygon_reduction_mode;
        }

        public string MatName
        {
            get
            {
                return this.@matName;
            }

            set
            {
                if (this.@matName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@matName, value, () => this.CalcCRC());
            }
        }

        internal string BoneName
        {
            get
            {
                return this.@boneName;
            }
        }

        public string OriginalMaterialName
        {
            get
            {
                return this.@originalMaterialName;
            }

            set
            {
                if (this.@originalMaterialName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalMaterialName, value, () => this.CalcCRC());
            }
        }

        public string OriginalBoneName
        {
            get
            {
                return this.@originalBoneName;
            }

            set
            {
                if (this.@originalBoneName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalBoneName, value, () => this.CalcCRC());
            }
        }

        internal int VertexIndex
        {
            get
            {
                if (this.Vertex != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    if (this.Parent == null)
                    {
                        return -1;
                    }
                    if (this.Parent.Parent == null)
                    {
                        return -1;
                    }
                    Ensure.Operation.True(this.Parent.Parent.Vertexes.Contains(this.Vertex));
                    return this.Parent.Parent.Vertexes.IndexOf(this.Vertex);
                }
                return this.@vertexIndex;
            }
        }

        public int VertexSkinningCount
        {
            get
            {
                Ensure.Operation.NotNull(this.Vertex);
                var i0 = this.Vertex.VtxAttribs.FirstOrDefault(x => x.Name == "_i0" );
                if (i0 == null)
                {
                    // リジッドボディー
                    return 0;
                }

                var w0 = this.Vertex.VtxAttribs.FirstOrDefault(x => x.Name == "_w0" );
                if (w0 == null)
                {
                    // リジッドスキニング
                    return 1;
                }

                // スムーススキニング
                var i1 = this.Vertex.VtxAttribs.FirstOrDefault(x => x.Name == "_i1" );
                if (i1 == null)
                {
                    return Utility.GetComponentCount(i0.Type);
                }

                return 4 + Utility.GetComponentCount(i1.Type);
            }
        }

        public bool LocalCoordinate
        {
            get
            {
                return this.VertexSkinningCount < 2;
            }
        }

        public string OriginalShapeHash
        {
            get
            {
                return this.@originalShapeHash;
            }

            set
            {
                if (this.@originalShapeHash == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalShapeHash, value, () => this.CalcCRC());
            }
        }

        public int OriginalShapeCount
        {
            get
            {
                return this.@originalShapeCount;
            }

            set
            {
                if (this.@originalShapeCount == value)
                {
                    return;
                }

                this.SetProperty(ref this.@originalShapeCount, value, () => this.CalcCRC());
            }
        }

        public string DeleteNearVertexMode
        {
            get
            {
                return this.@deleteNearVertexMode;
            }

            set
            {
                if (this.@deleteNearVertexMode == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@deleteNearVertexMode, value, () => this.CalcCRC());
            }
        }

        public string DivideSubmeshMode
        {
            get
            {
                return this.@divideSubmeshMode;
            }

            set
            {
                if (this.@divideSubmeshMode == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@divideSubmeshMode, value, () => this.CalcCRC());
            }
        }

        public string OptimizePrimitiveMode
        {
            get
            {
                return this.@optimizePrimitiveMode;
            }

            set
            {
                if (this.@optimizePrimitiveMode == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@optimizePrimitiveMode, value, () => this.CalcCRC());
            }
        }

        public string PolygonReductionMode
        {
            get
            {
                return this.@polygonReductionMode;
            }

            set
            {
                if (this.@polygonReductionMode == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@polygonReductionMode, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Shape : true);
                this.SetProperty(ref this.parent, value as Shape, () => this.CalcCRC());
            }
        }

        public Shape Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Shape);
            }
        }

        public Vertex Vertex
        {
            get
            {
                return this.@vertex;
            }

            set
            {
                if (this.@vertex == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@vertex, value, () => this.CalcCRC());
            }
        }

        public Bone Bone
        {
            get
            {
                return this.@bone;
            }

            set
            {
                if (this.@bone == value)
                {
                    return;
                }

                Bone oldBone = this.Bone;
                if (value != null)
                {
                    this.boneName = value.Name;
                }
                else
                {
                    // null のときは一時的に親から外されて参照が消えただけの場合があるので、
                    // 前に設定された名前を維持しておく
                }

                this.SetProperty(ref this.@bone, value, () => this.CalcCRC());

                if (oldBone != null)
                {
                    oldBone.UpdateShapeReferences();
                }
                if (this.Bone != null)
                {
                    this.Bone.UpdateShapeReferences();
                }
            }
        }

        // 前処理、後処理を呼び出さない純粋なセットのみ
        internal void SetBoneWithoutPrePostProcess(Bone value)
        {
            this.SetProperty(ref this.@bone, value, () => this.CalcCRC());
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shape_infoType CreateSerializableData()
        {
            var writeData = new shape_infoType();
            writeData.@mat_name = this.MatName;
            Ensure.Operation.NotNull(this.Bone, "Associated bone must not be null.");
            Ensure.Operation.True(this.Parent.Parent.Skeleton.EnumerateBones().Contains(this.Bone), "Associated bone must be in a file.");
            {
                if (this.Bone != null)
                {
                    writeData.bone_name = this.Bone.Name;
                }
                else
                {
                    writeData.bone_name = string.Empty;
                }
            }
            writeData.@original_material_name = this.OriginalMaterialName;
            writeData.@original_bone_name = this.OriginalBoneName;
            {
                Ensure.Operation.NotNull(this.Vertex);
                writeData.vertex_index = this.VertexIndex;
            }
            writeData.@vertex_skinning_count = this.VertexSkinningCount;
            writeData.@local_coordinate = this.LocalCoordinate;
            writeData.@original_shape_hash = this.OriginalShapeHash;
            writeData.@original_shape_count = this.OriginalShapeCount;
            writeData.@delete_near_vertex_mode = this.DeleteNearVertexMode;
            writeData.@divide_submesh_mode = this.DivideSubmeshMode;
            writeData.@optimize_primitive_mode = this.OptimizePrimitiveMode;
            writeData.@polygon_reduction_mode = this.PolygonReductionMode;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShapeInfo IDeepCopyable<ShapeInfo>.DeepCopy()
        {
            return new ShapeInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShapeInfo source)
        {
            this.@matName = source.@matName;
            this.@boneName = source.@boneName;
            this.@originalMaterialName = source.@originalMaterialName;
            this.@originalBoneName = source.@originalBoneName;
            this.@vertexIndex = source.@vertexIndex;
            this.@originalShapeHash = source.@originalShapeHash;
            this.@originalShapeCount = source.@originalShapeCount;
            this.@deleteNearVertexMode = source.@deleteNearVertexMode;
            this.@divideSubmeshMode = source.@divideSubmeshMode;
            this.@optimizePrimitiveMode = source.@optimizePrimitiveMode;
            this.@polygonReductionMode = source.@polygonReductionMode;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MatName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.BoneName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalMaterialName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalBoneName)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.VertexIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalShapeHash)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalShapeCount)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DeleteNearVertexMode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DivideSubmeshMode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OptimizePrimitiveMode)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.PolygonReductionMode)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
