﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShadingModel : ObservableEntity<shading_modelType>, IChildEntity, IDeepCopyable<ShadingModel>, IDeepCopyFrom<ShadingModel>, IDisposable
    {
        private VertexStage @vertexStage = null;
        private GeometryStage @geometryStage = null;
        private FragmentStage @fragmentStage = null;
        private ComputeStage @computeStage = null;
        private readonly ObservableList<Macro> @macros = new ObservableList<Macro>();
        private readonly ObservableList<OptionVar> @optionVars = new ObservableList<OptionVar>();
        private readonly ObservableList<AttribVar> @attribVars = new ObservableList<AttribVar>();
        private readonly ObservableList<SamplerVar> @samplerVars = new ObservableList<SamplerVar>();
        private readonly ObservableList<BlockVar> @blockVars = new ObservableList<BlockVar>();
        private readonly ObservableList<ShaderStorageBlockVar> @shaderStorageBlockVars = new ObservableList<ShaderStorageBlockVar>();
        private readonly ObservableList<RenderInfoSlot> @renderInfoSlots = new ObservableList<RenderInfoSlot>();
        private readonly ObservableList<Textblock> @textblocks = new ObservableList<Textblock>();
        private readonly ObservableList<Interleave> @interleaves = new ObservableList<Interleave>();
        private readonly ObservableList<Group> @groups = new ObservableList<Group>();
        private readonly ObservableList<Page> @pages = new ObservableList<Page>();
        private readonly Streamout @streamout = new Streamout();
        private string @name = string.Empty;
        private bool @materialShader;
        private int @revision;
        private ShaderDefinition parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShadingModel()
        {
            this.@macros.CollectionChanged += OnCollectionChanged<Macro>;
            this.@optionVars.CollectionChanged += OnCollectionChanged<OptionVar>;
            this.@attribVars.CollectionChanged += OnCollectionChanged<AttribVar>;
            this.@samplerVars.CollectionChanged += OnCollectionChanged<SamplerVar>;
            this.@blockVars.CollectionChanged += OnCollectionChanged<BlockVar>;
            this.@shaderStorageBlockVars.CollectionChanged += OnCollectionChanged<ShaderStorageBlockVar>;
            this.@renderInfoSlots.CollectionChanged += OnCollectionChanged<RenderInfoSlot>;
            this.@textblocks.CollectionChanged += OnCollectionChanged<Textblock>;
            this.@interleaves.CollectionChanged += OnCollectionChanged<Interleave>;
            this.@groups.CollectionChanged += OnCollectionChanged<Group>;
            this.@pages.CollectionChanged += OnCollectionChanged<Page>;
            this.@streamout.PropertyChanged += this.OnPropertyChanged;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShadingModel(ShadingModel source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShadingModel(shading_modelType source)
        {
            this.@macros.CollectionChanged += OnCollectionChanged<Macro>;
            this.@optionVars.CollectionChanged += OnCollectionChanged<OptionVar>;
            this.@attribVars.CollectionChanged += OnCollectionChanged<AttribVar>;
            this.@samplerVars.CollectionChanged += OnCollectionChanged<SamplerVar>;
            this.@blockVars.CollectionChanged += OnCollectionChanged<BlockVar>;
            this.@shaderStorageBlockVars.CollectionChanged += OnCollectionChanged<ShaderStorageBlockVar>;
            this.@renderInfoSlots.CollectionChanged += OnCollectionChanged<RenderInfoSlot>;
            this.@textblocks.CollectionChanged += OnCollectionChanged<Textblock>;
            this.@interleaves.CollectionChanged += OnCollectionChanged<Interleave>;
            this.@groups.CollectionChanged += OnCollectionChanged<Group>;
            this.@pages.CollectionChanged += OnCollectionChanged<Page>;
            if (source.@vertex_stage != null)
            {
                this.VertexStage = new VertexStage(source.@vertex_stage);
            }
            else
            {
                this.VertexStage = null;
            }
            if (source.@geometry_stage != null)
            {
                this.GeometryStage = new GeometryStage(source.@geometry_stage);
            }
            else
            {
                this.GeometryStage = null;
            }
            if (source.@fragment_stage != null)
            {
                this.FragmentStage = new FragmentStage(source.@fragment_stage);
            }
            else
            {
                this.FragmentStage = null;
            }
            if (source.@compute_stage != null)
            {
                this.ComputeStage = new ComputeStage(source.@compute_stage);
            }
            else
            {
                this.ComputeStage = null;
            }
            if (source.@macro_array != null)
            {
                this.@macros.Clear();
                foreach (var elem in source.@macro_array.Items)
                {
                    this.@macros.Add(new Macro(elem));
                }
            }
            if (source.@option_var_array != null)
            {
                this.@optionVars.Clear();
                foreach (var elem in source.@option_var_array.Items)
                {
                    this.@optionVars.Add(new OptionVar(elem));
                }
            }
            if (source.@attrib_var_array != null)
            {
                this.@attribVars.Clear();
                foreach (var elem in source.@attrib_var_array.Items)
                {
                    this.@attribVars.Add(new AttribVar(elem));
                }
            }
            if (source.@sampler_var_array != null)
            {
                this.@samplerVars.Clear();
                foreach (var elem in source.@sampler_var_array.Items)
                {
                    this.@samplerVars.Add(new SamplerVar(elem));
                }
            }
            if (source.@block_var_array != null)
            {
                this.@blockVars.Clear();
                foreach (var elem in source.@block_var_array.Items)
                {
                    this.@blockVars.Add(new BlockVar(elem));
                }
            }
            if (source.@shader_storage_block_var_array != null)
            {
                this.@shaderStorageBlockVars.Clear();
                foreach (var elem in source.@shader_storage_block_var_array.Items)
                {
                    this.@shaderStorageBlockVars.Add(new ShaderStorageBlockVar(elem));
                }
            }
            if (source.@render_info_slot_array != null)
            {
                this.@renderInfoSlots.Clear();
                foreach (var elem in source.@render_info_slot_array.Items)
                {
                    this.@renderInfoSlots.Add(DataModelConverter.Convert(elem));
                }
            }
            if (source.@textblock_array != null)
            {
                this.@textblocks.Clear();
                foreach (var elem in source.@textblock_array.Items)
                {
                    this.@textblocks.Add(new Textblock(elem));
                }
            }
            if (source.@interleave_array != null)
            {
                this.@interleaves.Clear();
                foreach (var elem in source.@interleave_array.Items)
                {
                    this.@interleaves.Add(new Interleave(elem));
                }
            }
            if (source.@group_array != null)
            {
                this.@groups.Clear();
                foreach (var elem in source.@group_array.Items)
                {
                    this.@groups.Add(new Group(elem));
                }
            }
            if (source.@page_array != null)
            {
                this.@pages.Clear();
                foreach (var elem in source.@page_array.Items)
                {
                    this.@pages.Add(new Page(elem));
                }
            }
            if (source.@streamout != null)
            {
                this.@streamout = new Streamout(source.@streamout);
            }
            else
            {
                this.@streamout = new Streamout();
            }
            this.@name = source.@name;
            this.@materialShader = source.@material_shader;
            this.@revision = source.@revision;
            this.@streamout.PropertyChanged += this.OnPropertyChanged;
        }

        public void Dispose()
        {
            if (this.@vertexStage != null)
            {
                this.@vertexStage.PropertyChanged -= this.OnPropertyChanged;
            }
            if (this.@geometryStage != null)
            {
                this.@geometryStage.PropertyChanged -= this.OnPropertyChanged;
            }
            if (this.@fragmentStage != null)
            {
                this.@fragmentStage.PropertyChanged -= this.OnPropertyChanged;
            }
            if (this.@computeStage != null)
            {
                this.@computeStage.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public VertexStage VertexStage
        {
            get
            {
                return this.@vertexStage;
            }

            set
            {
                if (this.@vertexStage == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(VertexStage).IsAssignableFrom(value.GetType()) : true);
                if (this.@vertexStage != null)
                {
                    (this.@vertexStage as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@vertexStage != null)
                {
                    this.@vertexStage.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@vertexStage, value, () => this.CalcCRC());
            }
        }

        public GeometryStage GeometryStage
        {
            get
            {
                return this.@geometryStage;
            }

            set
            {
                if (this.@geometryStage == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(GeometryStage).IsAssignableFrom(value.GetType()) : true);
                if (this.@geometryStage != null)
                {
                    (this.@geometryStage as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@geometryStage != null)
                {
                    this.@geometryStage.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@geometryStage, value, () => this.CalcCRC());
            }
        }

        public FragmentStage FragmentStage
        {
            get
            {
                return this.@fragmentStage;
            }

            set
            {
                if (this.@fragmentStage == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(FragmentStage).IsAssignableFrom(value.GetType()) : true);
                if (this.@fragmentStage != null)
                {
                    (this.@fragmentStage as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@fragmentStage != null)
                {
                    this.@fragmentStage.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@fragmentStage, value, () => this.CalcCRC());
            }
        }

        public ComputeStage ComputeStage
        {
            get
            {
                return this.@computeStage;
            }

            set
            {
                if (this.@computeStage == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(ComputeStage).IsAssignableFrom(value.GetType()) : true);
                if (this.@computeStage != null)
                {
                    (this.@computeStage as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@computeStage != null)
                {
                    this.@computeStage.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@computeStage, value, () => this.CalcCRC());
            }
        }

        public ObservableList<Macro> Macros
        {
            get
            {
                return this.@macros;
            }
        }

        public ObservableList<OptionVar> OptionVars
        {
            get
            {
                return this.@optionVars;
            }
        }

        public ObservableList<AttribVar> AttribVars
        {
            get
            {
                return this.@attribVars;
            }
        }

        public ObservableList<SamplerVar> SamplerVars
        {
            get
            {
                return this.@samplerVars;
            }
        }

        public ObservableList<BlockVar> BlockVars
        {
            get
            {
                return this.@blockVars;
            }
        }

        public ObservableList<ShaderStorageBlockVar> ShaderStorageBlockVars
        {
            get
            {
                return this.@shaderStorageBlockVars;
            }
        }

        public ObservableList<RenderInfoSlot> RenderInfoSlots
        {
            get
            {
                return this.@renderInfoSlots;
            }
        }

        public ObservableList<Textblock> Textblocks
        {
            get
            {
                return this.@textblocks;
            }
        }

        public ObservableList<Interleave> Interleaves
        {
            get
            {
                return this.@interleaves;
            }
        }

        public ObservableList<Group> Groups
        {
            get
            {
                return this.@groups;
            }
        }

        public ObservableList<Page> Pages
        {
            get
            {
                return this.@pages;
            }
        }

        public Streamout Streamout
        {
            get
            {
                return this.@streamout;
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        public bool MaterialShader
        {
            get
            {
                return this.@materialShader;
            }

            set
            {
                if (this.@materialShader == value)
                {
                    return;
                }

                this.SetProperty(ref this.@materialShader, value, () => this.CalcCRC());
            }
        }

        public int Revision
        {
            get
            {
                return this.@revision;
            }

            set
            {
                if (this.@revision == value)
                {
                    return;
                }

                this.SetProperty(ref this.@revision, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is ShaderDefinition : true);
                this.SetProperty(ref this.parent, value as ShaderDefinition, () => this.CalcCRC());
            }
        }

        public ShaderDefinition Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(ShaderDefinition);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shading_modelType CreateSerializableData()
        {
            var writeData = new shading_modelType();
            if (this.@vertexStage != null)
            {
                writeData.@vertex_stage = this.VertexStage.CreateSerializableData() as vertex_stageType;
            }
            if (this.@geometryStage != null)
            {
                writeData.@geometry_stage = this.GeometryStage.CreateSerializableData() as geometry_stageType;
            }
            if (this.@fragmentStage != null)
            {
                writeData.@fragment_stage = this.FragmentStage.CreateSerializableData() as fragment_stageType;
            }
            if (this.@computeStage != null)
            {
                writeData.@compute_stage = this.ComputeStage.CreateSerializableData() as compute_stageType;
            }
            if (this.@Macros.Count > 0)
            {
                writeData.@macro_array = new macro_arrayType();
                writeData.@macro_array.Items = this.@macros.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@macro_array.Items.Length == 0)
                {
                    writeData.@macro_array = null;
                }
            }
            if (this.@OptionVars.Count > 0)
            {
                writeData.@option_var_array = new option_var_arrayType();
                writeData.@option_var_array.Items = this.@optionVars.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@option_var_array.Items.Length == 0)
                {
                    writeData.@option_var_array = null;
                }
            }
            if (this.@AttribVars.Count > 0)
            {
                writeData.@attrib_var_array = new attrib_var_arrayType();
                writeData.@attrib_var_array.Items = this.@attribVars.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@attrib_var_array.Items.Length == 0)
                {
                    writeData.@attrib_var_array = null;
                }
            }
            if (this.@SamplerVars.Count > 0)
            {
                writeData.@sampler_var_array = new sampler_var_arrayType();
                writeData.@sampler_var_array.Items = this.@samplerVars.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@sampler_var_array.Items.Length == 0)
                {
                    writeData.@sampler_var_array = null;
                }
            }
            if (this.@BlockVars.Count > 0)
            {
                writeData.@block_var_array = new block_var_arrayType();
                writeData.@block_var_array.Items = this.@blockVars.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@block_var_array.Items.Length == 0)
                {
                    writeData.@block_var_array = null;
                }
            }
            if (this.@ShaderStorageBlockVars.Count > 0)
            {
                writeData.@shader_storage_block_var_array = new shader_storage_block_var_arrayType();
                writeData.@shader_storage_block_var_array.Items = this.@shaderStorageBlockVars.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_storage_block_var_array.Items.Length == 0)
                {
                    writeData.@shader_storage_block_var_array = null;
                }
            }
            if (this.@RenderInfoSlots.Count > 0)
            {
                writeData.@render_info_slot_array = new render_info_slot_arrayType();
                writeData.@render_info_slot_array.Items = this.@renderInfoSlots.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@render_info_slot_array.Items.Length == 0)
                {
                    writeData.@render_info_slot_array = null;
                }
            }
            if (this.@Textblocks.Count > 0)
            {
                writeData.@textblock_array = new textblock_arrayType();
                writeData.@textblock_array.Items = this.@textblocks.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@textblock_array.Items.Length == 0)
                {
                    writeData.@textblock_array = null;
                }
            }
            if (this.@Interleaves.Count > 0)
            {
                writeData.@interleave_array = new interleave_arrayType();
                writeData.@interleave_array.Items = this.@interleaves.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@interleave_array.Items.Length == 0)
                {
                    writeData.@interleave_array = null;
                }
            }
            if (this.@Groups.Count > 0)
            {
                writeData.@group_array = new group_arrayType();
                writeData.@group_array.Items = this.@groups.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@group_array.Items.Length == 0)
                {
                    writeData.@group_array = null;
                }
            }
            if (this.@Pages.Count > 0)
            {
                writeData.@page_array = new page_arrayType();
                writeData.@page_array.Items = this.@pages.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@page_array.Items.Length == 0)
                {
                    writeData.@page_array = null;
                }
            }
            writeData.@streamout = this.@streamout.CreateSerializableData() as streamoutType;
            writeData.@name = this.Name;
            writeData.@material_shader = this.MaterialShader;
            writeData.@revision = this.Revision;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShadingModel IDeepCopyable<ShadingModel>.DeepCopy()
        {
            return new ShadingModel(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShadingModel source)
        {
            if (source.@vertexStage == null)
            {
                this.VertexStage = null;
            }
            else
            {
                this.VertexStage.DeepCopyFrom(source.@vertexStage);
            }
            if (source.@geometryStage == null)
            {
                this.GeometryStage = null;
            }
            else
            {
                this.GeometryStage.DeepCopyFrom(source.@geometryStage);
            }
            if (source.@fragmentStage == null)
            {
                this.FragmentStage = null;
            }
            else
            {
                this.FragmentStage.DeepCopyFrom(source.@fragmentStage);
            }
            if (source.@computeStage == null)
            {
                this.ComputeStage = null;
            }
            else
            {
                this.ComputeStage.DeepCopyFrom(source.@computeStage);
            }
            CopyElements(source.@macros, this.@macros);
            CopyElements(source.@optionVars, this.@optionVars);
            CopyElements(source.@attribVars, this.@attribVars);
            CopyElements(source.@samplerVars, this.@samplerVars);
            CopyElements(source.@blockVars, this.@blockVars);
            CopyElements(source.@shaderStorageBlockVars, this.@shaderStorageBlockVars);
            CopyElements(source.@renderInfoSlots, this.@renderInfoSlots);
            CopyElements(source.@textblocks, this.@textblocks);
            CopyElements(source.@interleaves, this.@interleaves);
            CopyElements(source.@groups, this.@groups);
            CopyElements(source.@pages, this.@pages);
            this.@streamout.DeepCopyFrom(source.@streamout);
            this.@name = source.@name;
            this.@materialShader = source.@materialShader;
            this.@revision = source.@revision;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            if (this.VertexStage != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.VertexStage.HashValue)));
            }
            if (this.GeometryStage != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.GeometryStage.HashValue)));
            }
            if (this.FragmentStage != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.FragmentStage.HashValue)));
            }
            if (this.ComputeStage != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ComputeStage.HashValue)));
            }
            foreach (var elem in this.Macros)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.OptionVars)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.AttribVars)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.SamplerVars)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.BlockVars)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.ShaderStorageBlockVars)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.RenderInfoSlots)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Textblocks)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Interleaves)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Groups)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Pages)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Streamout.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MaterialShader)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Revision)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            if (this.@vertexStage != null)
            {
                this.@vertexStage.AutoCalc = this.AutoCalc;
            }
            if (this.@geometryStage != null)
            {
                this.@geometryStage.AutoCalc = this.AutoCalc;
            }
            if (this.@fragmentStage != null)
            {
                this.@fragmentStage.AutoCalc = this.AutoCalc;
            }
            if (this.@computeStage != null)
            {
                this.@computeStage.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@macros)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@optionVars)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@attribVars)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@samplerVars)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@blockVars)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shaderStorageBlockVars)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@renderInfoSlots)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@textblocks)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@interleaves)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@groups)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@pages)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            this.@streamout.AutoCalc = this.AutoCalc;
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            if (this.@vertexStage != null)
            {
                this.@vertexStage.Reset();
            }
            if (this.@geometryStage != null)
            {
                this.@geometryStage.Reset();
            }
            if (this.@fragmentStage != null)
            {
                this.@fragmentStage.Reset();
            }
            if (this.@computeStage != null)
            {
                this.@computeStage.Reset();
            }
            foreach (var elem in this.@macros)
            {
                elem.Reset();
            }
            foreach (var elem in this.@optionVars)
            {
                elem.Reset();
            }
            foreach (var elem in this.@attribVars)
            {
                elem.Reset();
            }
            foreach (var elem in this.@samplerVars)
            {
                elem.Reset();
            }
            foreach (var elem in this.@blockVars)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shaderStorageBlockVars)
            {
                elem.Reset();
            }
            foreach (var elem in this.@renderInfoSlots)
            {
                elem.Reset();
            }
            foreach (var elem in this.@textblocks)
            {
                elem.Reset();
            }
            foreach (var elem in this.@interleaves)
            {
                elem.Reset();
            }
            foreach (var elem in this.@groups)
            {
                elem.Reset();
            }
            foreach (var elem in this.@pages)
            {
                elem.Reset();
            }
            this.@streamout.Reset();
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            if (this.@vertexStage != null)
            {
                this.@vertexStage.Refresh();
            }
            if (this.@geometryStage != null)
            {
                this.@geometryStage.Refresh();
            }
            if (this.@fragmentStage != null)
            {
                this.@fragmentStage.Refresh();
            }
            if (this.@computeStage != null)
            {
                this.@computeStage.Refresh();
            }
            foreach (var elem in this.@macros)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@optionVars)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@attribVars)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@samplerVars)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@blockVars)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shaderStorageBlockVars)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@renderInfoSlots)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@textblocks)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@interleaves)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@groups)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@pages)
            {
                elem.Refresh();
            }
            this.@streamout.Refresh();
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
