﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderProgram : ObservableEntity<shader_programType>, IEquatable<ShaderProgram>, IDeepCopyable<ShaderProgram>, IDeepCopyFrom<ShaderProgram>
    {
        private readonly ObservableList<Option> @options = new ObservableList<Option>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ShaderProgram()
        {
            this.@options.CollectionChanged += OnCollectionChanged<Option>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderProgram(ShaderProgram source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderProgram(shader_programType source)
        {
            this.@options.CollectionChanged += OnCollectionChanged<Option>;
            if (source.@option_array != null)
            {
                this.@options.Clear();
                foreach (var elem in source.@option_array.Items)
                {
                    this.@options.Add(new Option(elem));
                }
            }
        }

        public ObservableList<Option> Options
        {
            get
            {
                return this.@options;
            }
        }

        /// <summary>
        /// 与えられた入力値と等しいかどうかを調べます。
        /// </summary>
        /// <param name="obj">比較対象です。</param>
        /// <returns>等しければ true、そうでなければ false を返します。</returns>
        public bool Equals(ShaderProgram obj)
        {
            if (obj == null)
            {
                return false;
            }

            if (this.Options.Count != obj.Options.Count)
            {
                return false;
            }

            // 順序の違いは無視して比較
            foreach (var elem in this.Options)
            {
                if (!obj.Options.Any(x => elem.Equals(x)))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_programType CreateSerializableData()
        {
            var writeData = new shader_programType();
            if (this.@Options.Count > 0)
            {
                writeData.@option_array = new option_arrayType();
                writeData.@option_array.Items = this.@options.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@option_array.Items.Length == 0)
                {
                    writeData.@option_array = null;
                }
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderProgram IDeepCopyable<ShaderProgram>.DeepCopy()
        {
            return new ShaderProgram(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderProgram source)
        {
            CopyElements(source.@options, this.@options);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.Options)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@options)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@options)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@options)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
