﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ShaderConfig : RootEntity<shader_configType>, IDeepCopyable<ShaderConfig>, IDeepCopyFrom<ShaderConfig>, IToolData
    {
        private readonly ToolData @toolData = new ToolData();
        private readonly UserToolData @userToolData = new UserToolData();
        private readonly ShaderConfigInfo @shaderConfigInfo = new ShaderConfigInfo();
        private readonly ObservableList<IncludePath> @includePathes = new ObservableList<IncludePath>();
        private readonly ObservableList<ForceIncludeFile> @forceIncludeFiles = new ObservableList<ForceIncludeFile>();
        private readonly ObservableList<Shader> @shaders = new ObservableList<Shader>();
        private readonly ObservableList<Stream> @streams = new ObservableList<Stream>();
        private shader_config_versionType @version;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        internal ShaderConfig()
        {
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@shaderConfigInfo.PropertyChanged += this.OnPropertyChanged;
            this.@includePathes.CollectionChanged += OnCollectionChanged<IncludePath>;
            this.@forceIncludeFiles.CollectionChanged += OnCollectionChanged<ForceIncludeFile>;
            this.@shaders.CollectionChanged += OnCollectionChanged<Shader>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderConfig(ShaderConfig source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ShaderConfig(shader_configType source)
        {
            this.@includePathes.CollectionChanged += OnCollectionChanged<IncludePath>;
            this.@forceIncludeFiles.CollectionChanged += OnCollectionChanged<ForceIncludeFile>;
            this.@shaders.CollectionChanged += OnCollectionChanged<Shader>;
            this.@streams.CollectionChanged += OnCollectionChanged<Stream>;
            if (source.@tool_data != null)
            {
                this.@toolData = new ToolData(source.@tool_data);
            }
            else
            {
                this.@toolData = new ToolData();
            }
            if (source.@user_tool_data != null)
            {
                this.@userToolData = new UserToolData(source.@user_tool_data);
            }
            else
            {
                this.@userToolData = new UserToolData();
            }
            if (source.@shader_config_info != null)
            {
                this.@shaderConfigInfo = new ShaderConfigInfo(source.@shader_config_info);
            }
            else
            {
                this.@shaderConfigInfo = new ShaderConfigInfo();
            }
            if (source.@include_path_array != null)
            {
                this.@includePathes.Clear();
                foreach (var elem in source.@include_path_array.Items)
                {
                    this.@includePathes.Add(new IncludePath(elem));
                }
            }
            if (source.@force_include_file_array != null)
            {
                this.@forceIncludeFiles.Clear();
                foreach (var elem in source.@force_include_file_array.Items)
                {
                    this.@forceIncludeFiles.Add(new ForceIncludeFile(elem));
                }
            }
            if (source.@shader_array != null)
            {
                this.@shaders.Clear();
                foreach (var elem in source.@shader_array.Items)
                {
                    this.@shaders.Add(new Shader(elem));
                }
            }
            if (source.@stream_array != null)
            {
                this.@streams.Clear();
                foreach (var elem in source.@stream_array.Items)
                {
                    this.@streams.Add(DataModelConverter.Convert(elem));
                }
            }
            this.@version = source.@version;
            this.@toolData.PropertyChanged += this.OnPropertyChanged;
            this.@userToolData.PropertyChanged += this.OnPropertyChanged;
            this.@shaderConfigInfo.PropertyChanged += this.OnPropertyChanged;
            this.ResolveInternalReferences();
        }

        public ToolData ToolData
        {
            get
            {
                return this.@toolData;
            }
        }

        public UserToolData UserToolData
        {
            get
            {
                return this.@userToolData;
            }
        }

        public ShaderConfigInfo ShaderConfigInfo
        {
            get
            {
                return this.@shaderConfigInfo;
            }
        }

        public ObservableList<IncludePath> IncludePathes
        {
            get
            {
                return this.@includePathes;
            }
        }

        public ObservableList<ForceIncludeFile> ForceIncludeFiles
        {
            get
            {
                return this.@forceIncludeFiles;
            }
        }

        public ObservableList<Shader> Shaders
        {
            get
            {
                return this.@shaders;
            }
        }

        public override ObservableList<Stream> Streams
        {
            get
            {
                return this.@streams;
            }
        }

        public shader_config_versionType Version
        {
            get
            {
                return this.@version;
            }

            set
            {
                if (this.@version == value)
                {
                    return;
                }

                this.SetProperty(ref this.@version, value, () => this.CalcCRC());
            }
        }

        public override void ResolveInternalReferences()
        {
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override shader_configType CreateSerializableData()
        {
            Ensure.Operation.Range(this.Shaders.Count, 1, int.MaxValue);
            var writeData = new shader_configType();
            writeData.@tool_data = this.@toolData.CreateSerializableData() as tool_dataType;
            writeData.@user_tool_data = this.@userToolData.CreateSerializableData() as user_tool_dataType;
            writeData.@shader_config_info = this.@shaderConfigInfo.CreateSerializableData() as shader_config_infoType;
            if (this.@IncludePathes.Count > 0)
            {
                writeData.@include_path_array = new include_path_arrayType();
                writeData.@include_path_array.Items = this.@includePathes.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@include_path_array.Items.Length == 0)
                {
                    writeData.@include_path_array = null;
                }
            }
            if (this.@ForceIncludeFiles.Count > 0)
            {
                writeData.@force_include_file_array = new force_include_file_arrayType();
                writeData.@force_include_file_array.Items = this.@forceIncludeFiles.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@force_include_file_array.Items.Length == 0)
                {
                    writeData.@force_include_file_array = null;
                }
            }
            if (this.@Shaders.Count > 0)
            {
                writeData.@shader_array = new shader_arrayType();
                writeData.@shader_array.Items = this.@shaders.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@shader_array.Items.Length == 0)
                {
                    writeData.@shader_array = null;
                }
            }
            if (this.IsStreamSerializationEnabled)
            {
                if (this.@Streams.Count > 0)
                {
                    writeData.@stream_array = new stream_arrayType();
                    writeData.@stream_array.Items = this.@streams.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                    if (writeData.@stream_array.Items.Length == 0)
                    {
                        writeData.@stream_array = null;
                    }
                }
            }
            writeData.@version = this.Version;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ShaderConfig IDeepCopyable<ShaderConfig>.DeepCopy()
        {
            return new ShaderConfig(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ShaderConfig source)
        {
            this.@toolData.DeepCopyFrom(source.@toolData);
            this.@userToolData.DeepCopyFrom(source.@userToolData);
            this.@shaderConfigInfo.DeepCopyFrom(source.@shaderConfigInfo);
            CopyElements(source.@includePathes, this.@includePathes);
            CopyElements(source.@forceIncludeFiles, this.@forceIncludeFiles);
            CopyElements(source.@shaders, this.@shaders);
            CopyElements(source.@streams, this.@streams);
            this.@version = source.@version;
            this.ResolveInternalReferences();
        }

        private protected override IRootEntity DeepCopy()
        {
            return new ShaderConfig(this);
        }

        private protected override void DeepCopyFrom(IRootEntity source)
        {
            Nintendo.ToolFoundation.Contracts.Ensure.Argument.True(source is ShaderConfig);
            this.DeepCopyFrom(source as ShaderConfig);
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UserToolData.HashValue)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.ShaderConfigInfo.HashValue)));
            foreach (var elem in this.IncludePathes)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.ForceIncludeFiles)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Shaders)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.Streams)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Version)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            this.@toolData.AutoCalc = this.AutoCalc;
            this.@userToolData.AutoCalc = this.AutoCalc;
            this.@shaderConfigInfo.AutoCalc = this.AutoCalc;
            foreach (var elem in this.@includePathes)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@forceIncludeFiles)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@shaders)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@streams)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            this.@toolData.Reset();
            this.@userToolData.Reset();
            this.@shaderConfigInfo.Reset();
            foreach (var elem in this.@includePathes)
            {
                elem.Reset();
            }
            foreach (var elem in this.@forceIncludeFiles)
            {
                elem.Reset();
            }
            foreach (var elem in this.@shaders)
            {
                elem.Reset();
            }
            foreach (var elem in this.@streams)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            this.@toolData.Refresh();
            this.@userToolData.Refresh();
            this.@shaderConfigInfo.Refresh();
            foreach (var elem in this.@includePathes)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@forceIncludeFiles)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@shaders)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@streams)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
