﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Page : ObservableEntity<pageType>, IUiElementHolder, IDeepCopyable<Page>, IDeepCopyFrom<Page>, IDisposable
    {
        private UiLabel @uiLabel = null;
        private UiComment @uiComment = null;
        private UiOrder @uiOrder = null;
        private string @name = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Page()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Page(Page source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Page(pageType source)
        {
            if (source.@ui_label != null)
            {
                this.UiLabel = new UiLabel(source.@ui_label);
            }
            else
            {
                this.UiLabel = null;
            }
            if (source.@ui_comment != null)
            {
                this.UiComment = new UiComment(source.@ui_comment);
            }
            else
            {
                this.UiComment = null;
            }
            if (source.@ui_order != null)
            {
                this.UiOrder = new UiOrder(source.@ui_order);
            }
            else
            {
                this.UiOrder = null;
            }
            this.@name = source.@name;
        }

        public void Dispose()
        {
            if (this.@uiLabel != null)
            {
                this.@uiLabel.PropertyChanged -= this.OnPropertyChanged;
            }
            if (this.@uiComment != null)
            {
                this.@uiComment.PropertyChanged -= this.OnPropertyChanged;
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.PropertyChanged -= this.OnPropertyChanged;
            }
        }

        public UiLabel UiLabel
        {
            get
            {
                return this.@uiLabel;
            }

            set
            {
                if (this.@uiLabel == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(UiLabel).IsAssignableFrom(value.GetType()) : true);
                if (this.@uiLabel != null)
                {
                    (this.@uiLabel as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@uiLabel != null)
                {
                    this.@uiLabel.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@uiLabel, value, () => this.CalcCRC());
            }
        }

        public UiComment UiComment
        {
            get
            {
                return this.@uiComment;
            }

            set
            {
                if (this.@uiComment == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(UiComment).IsAssignableFrom(value.GetType()) : true);
                if (this.@uiComment != null)
                {
                    (this.@uiComment as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@uiComment != null)
                {
                    this.@uiComment.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@uiComment, value, () => this.CalcCRC());
            }
        }

        public UiOrder UiOrder
        {
            get
            {
                return this.@uiOrder;
            }

            set
            {
                if (this.@uiOrder == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? typeof(UiOrder).IsAssignableFrom(value.GetType()) : true);
                if (this.@uiOrder != null)
                {
                    (this.@uiOrder as IChildEntity).Parent = null;
                }

                if (value != null)
                {
                    (value as IChildEntity).Parent = this;
                }

                if (this.@uiOrder != null)
                {
                    this.@uiOrder.PropertyChanged -= this.OnPropertyChanged;
                }

                if (value != null)
                {
                    value.PropertyChanged += this.OnPropertyChanged;
                }

                this.SetProperty(ref this.@uiOrder, value, () => this.CalcCRC());
            }
        }

        public string Name
        {
            get
            {
                return this.@name;
            }

            set
            {
                if (this.@name == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@name, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override pageType CreateSerializableData()
        {
            var writeData = new pageType();
            if (this.@uiLabel != null)
            {
                writeData.@ui_label = this.UiLabel.CreateSerializableData() as ui_labelType;
            }
            if (this.@uiComment != null)
            {
                writeData.@ui_comment = this.UiComment.CreateSerializableData() as ui_commentType;
            }
            if (this.@uiOrder != null)
            {
                writeData.@ui_order = this.UiOrder.CreateSerializableData() as ui_orderType;
            }
            writeData.@name = this.Name;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Page IDeepCopyable<Page>.DeepCopy()
        {
            return new Page(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Page source)
        {
            if (source.@uiLabel == null)
            {
                this.UiLabel = null;
            }
            else
            {
                this.UiLabel.DeepCopyFrom(source.@uiLabel);
            }
            if (source.@uiComment == null)
            {
                this.UiComment = null;
            }
            else
            {
                this.UiComment.DeepCopyFrom(source.@uiComment);
            }
            if (source.@uiOrder == null)
            {
                this.UiOrder = null;
            }
            else
            {
                this.UiOrder.DeepCopyFrom(source.@uiOrder);
            }
            this.@name = source.@name;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            if (this.UiLabel != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UiLabel.HashValue)));
            }
            if (this.UiComment != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UiComment.HashValue)));
            }
            if (this.UiOrder != null)
            {
                buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UiOrder.HashValue)));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Name)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            if (this.@uiLabel != null)
            {
                this.@uiLabel.AutoCalc = this.AutoCalc;
            }
            if (this.@uiComment != null)
            {
                this.@uiComment.AutoCalc = this.AutoCalc;
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            if (this.@uiLabel != null)
            {
                this.@uiLabel.Reset();
            }
            if (this.@uiComment != null)
            {
                this.@uiComment.Reset();
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            if (this.@uiLabel != null)
            {
                this.@uiLabel.Refresh();
            }
            if (this.@uiComment != null)
            {
                this.@uiComment.Refresh();
            }
            if (this.@uiOrder != null)
            {
                this.@uiOrder.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
