﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class OriginalMaterial : ObservableEntity<original_materialType>, IDeepCopyable<OriginalMaterial>, IDeepCopyFrom<OriginalMaterial>
    {
        private readonly ObservableList<OriginalColor> @originalColors = new ObservableList<OriginalColor>();
        private readonly ObservableList<OriginalTexsrt> @originalTexsrts = new ObservableList<OriginalTexsrt>();
        private string @matName = string.Empty;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OriginalMaterial()
        {
            this.@originalColors.CollectionChanged += OnCollectionChanged<OriginalColor>;
            this.@originalTexsrts.CollectionChanged += OnCollectionChanged<OriginalTexsrt>;
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalMaterial(OriginalMaterial source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalMaterial(original_materialType source)
        {
            this.@originalColors.CollectionChanged += OnCollectionChanged<OriginalColor>;
            this.@originalTexsrts.CollectionChanged += OnCollectionChanged<OriginalTexsrt>;
            if (source.@original_color_array != null)
            {
                this.@originalColors.Clear();
                foreach (var elem in source.@original_color_array.Items)
                {
                    this.@originalColors.Add(new OriginalColor(elem));
                }
            }
            if (source.@original_texsrt_array != null)
            {
                this.@originalTexsrts.Clear();
                foreach (var elem in source.@original_texsrt_array.Items)
                {
                    this.@originalTexsrts.Add(new OriginalTexsrt(elem));
                }
            }
            this.@matName = source.@mat_name;
        }

        public ObservableList<OriginalColor> OriginalColors
        {
            get
            {
                return this.@originalColors;
            }
        }

        public ObservableList<OriginalTexsrt> OriginalTexsrts
        {
            get
            {
                return this.@originalTexsrts;
            }
        }

        public string MatName
        {
            get
            {
                return this.@matName;
            }

            set
            {
                if (this.@matName == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@matName, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override original_materialType CreateSerializableData()
        {
            var writeData = new original_materialType();
            if (this.@OriginalColors.Count > 0)
            {
                writeData.@original_color_array = new original_color_arrayType();
                writeData.@original_color_array.Items = this.@originalColors.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@original_color_array.Items.Length == 0)
                {
                    writeData.@original_color_array = null;
                }
            }
            if (this.@OriginalTexsrts.Count > 0)
            {
                writeData.@original_texsrt_array = new original_texsrt_arrayType();
                writeData.@original_texsrt_array.Items = this.@originalTexsrts.Select(x => x.CreateSerializableData()).Where(x => x != null).ToArray();
                if (writeData.@original_texsrt_array.Items.Length == 0)
                {
                    writeData.@original_texsrt_array = null;
                }
            }
            writeData.@mat_name = this.MatName;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        OriginalMaterial IDeepCopyable<OriginalMaterial>.DeepCopy()
        {
            return new OriginalMaterial(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(OriginalMaterial source)
        {
            CopyElements(source.@originalColors, this.@originalColors);
            CopyElements(source.@originalTexsrts, this.@originalTexsrts);
            this.@matName = source.@matName;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            foreach (var elem in this.OriginalColors)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            foreach (var elem in this.OriginalTexsrts)
            {
                buffers.AddRange(BitConverter.GetBytes(elem.HashValue));
            }
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.MatName)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }

        /// <summary>
        /// 自動計算フラグを設定します。(内部処理用）
        /// </summary>
        protected override void SetAutoCalcFlagInternal()
        {
            foreach (var elem in this.@originalColors)
            {
                elem.AutoCalc = this.AutoCalc;
            }
            foreach (var elem in this.@originalTexsrts)
            {
                elem.AutoCalc = this.AutoCalc;
            }
        }

        /// <summary>
        /// エンティティの状態をリセットします。(内部処理用）
        /// </summary>
        protected override void ResetInternal()
        {
            foreach (var elem in this.@originalColors)
            {
                elem.Reset();
            }
            foreach (var elem in this.@originalTexsrts)
            {
                elem.Reset();
            }
        }

        /// <summary>
        /// エンティティの状態を更新します。(内部処理用）
        /// </summary>
        protected override void RefreshInternal()
        {
            foreach (var elem in this.@originalColors)
            {
                elem.Refresh();
            }
            foreach (var elem in this.@originalTexsrts)
            {
                elem.Refresh();
            }
        }

        private void OnPropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "HashValue")
            {
                this.CalcCRC();
            }
        }
    }
}
