﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class OriginalImage : ObservableEntity<original_imageType>, IChildEntity, IStreamIndex, IDeepCopyable<OriginalImage>, IDeepCopyFrom<OriginalImage>
    {
        private int @sliceIndex;
        private original_image_faceType @face;
        private original_image_formatType @format;
        private int @width;
        private int @height;
        private int @size;
        private string @originalPath = string.Empty;
        private int @streamIndex;
        private Texture parent = null;
        private Stream @stream = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OriginalImage()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalImage(OriginalImage source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public OriginalImage(original_imageType source)
        {
            this.@sliceIndex = source.@slice_index;
            this.@face = source.@face;
            this.@format = source.@format;
            this.@width = source.@width;
            this.@height = source.@height;
            this.@size = source.@size;
            this.@originalPath = source.@original_path;
            this.@streamIndex = source.@stream_index;
        }

        public int SliceIndex
        {
            get
            {
                return this.@sliceIndex;
            }

            set
            {
                if (this.@sliceIndex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@sliceIndex, value, () => this.CalcCRC());
            }
        }

        public original_image_faceType Face
        {
            get
            {
                return this.@face;
            }

            set
            {
                if (this.@face == value)
                {
                    return;
                }

                this.SetProperty(ref this.@face, value, () => this.CalcCRC());
            }
        }

        public original_image_formatType Format
        {
            get
            {
                return this.@format;
            }

            set
            {
                if (this.@format == value)
                {
                    return;
                }

                this.SetProperty(ref this.@format, value, () => this.CalcCRC());
            }
        }

        public int Width
        {
            get
            {
                return this.@width;
            }

            set
            {
                if (this.@width == value)
                {
                    return;
                }

                this.SetProperty(ref this.@width, value, () => this.CalcCRC());
            }
        }

        public int Height
        {
            get
            {
                return this.@height;
            }

            set
            {
                if (this.@height == value)
                {
                    return;
                }

                this.SetProperty(ref this.@height, value, () => this.CalcCRC());
            }
        }

        public int Size
        {
            get
            {
                return this.@size;
            }

            set
            {
                if (this.@size == value)
                {
                    return;
                }

                this.SetProperty(ref this.@size, value, () => this.CalcCRC());
            }
        }

        public string OriginalPath
        {
            get
            {
                return this.@originalPath;
            }

            set
            {
                if (this.@originalPath == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@originalPath, value, () => this.CalcCRC());
            }
        }

        int IStreamIndex.StreamIndex
        {
            get
            {
                if (this.Stream != null)
                {
                    // 親が見つからない場合はインデックスを求められないので -1 を返す
                    IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                    if (rootEntity == null)
                    {
                        return -1;
                    }

                    var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                    return sourceProp.IndexOf(this.Stream);
                }
                return this.@streamIndex;
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Texture : true);
                this.SetProperty(ref this.parent, value as Texture, () => this.CalcCRC());
            }
        }

        public Texture Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Texture);
            }
        }

        public Stream Stream
        {
            get
            {
                return this.@stream;
            }

            set
            {
                if (this.@stream == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@stream, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override original_imageType CreateSerializableData()
        {
            var writeData = new original_imageType();
            writeData.@slice_index = this.SliceIndex;
            writeData.@face = this.Face;
            writeData.@format = this.Format;
            writeData.@width = this.Width;
            writeData.@height = this.Height;
            writeData.@size = this.Size;
            writeData.@original_path = this.OriginalPath;
            {
                IRootEntity rootEntity = Utility.FindRootEntity((this as IChildEntity).Parent);
                Ensure.Operation.NotNull(rootEntity);
                var sourceProp = rootEntity.GetType().GetProperty("Streams").GetValue(rootEntity) as ObservableList<Stream>;
                // 読み取り高速化のために不要なデータを読み飛ばす場合があるので、参照先が存在しない場合を許容します
                Ensure.Operation.True((sourceProp.Count == 0) || (Stream != null));
                writeData.stream_index = (this as IStreamIndex).StreamIndex;
            }
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        OriginalImage IDeepCopyable<OriginalImage>.DeepCopy()
        {
            return new OriginalImage(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(OriginalImage source)
        {
            this.@sliceIndex = source.@sliceIndex;
            this.@face = source.@face;
            this.@format = source.@format;
            this.@width = source.@width;
            this.@height = source.@height;
            this.@size = source.@size;
            this.@originalPath = source.@originalPath;
            this.@streamIndex = source.@streamIndex;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.SliceIndex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Face)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Format)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Width)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Height)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Size)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.OriginalPath)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32((this as IStreamIndex).StreamIndex)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
