﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class ModelInfo : ObservableEntity<model_infoType>, IChildEntity, IDeepCopyable<ModelInfo>, IDeepCopyFrom<ModelInfo>
    {
        private int @totalProcessVertex;
        private bool @unitePosQuantize;
        private string @dccPreset = string.Empty;
        private float @dccMagnify = 1;
        private int @dccStartFrame;
        private Model parent = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ModelInfo()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ModelInfo(ModelInfo source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public ModelInfo(model_infoType source)
        {
            this.@totalProcessVertex = source.@total_process_vertex;
            this.@unitePosQuantize = source.@unite_pos_quantize;
            this.@dccPreset = source.@dcc_preset;
            this.@dccMagnify = source.@dcc_magnify;
            this.@dccStartFrame = source.@dcc_start_frame;
        }

        public int MaterialCount
        {
            get
            {
                return this.Parent.Materials.Count;
            }
        }

        public int BoneCount
        {
            get
            {
                return this.Parent.Skeleton.CountBones();
            }
        }

        public int ShapeCount
        {
            get
            {
                return this.Parent.Shapes.Count;
            }
        }

        public int SmoothSkinningShape
        {
            get
            {
                return this.Parent.Shapes.Count(x => x.ShapeInfo.VertexSkinningCount > 1);
            }
        }

        public int RigidSkinningShape
        {
            get
            {
                return this.Parent.Shapes.Count(x => x.ShapeInfo.VertexSkinningCount == 1);
            }
        }

        public int SmoothSkinningMatrix
        {
            get
            {
                return this.Parent.Skeleton.EnumerateBones().Count(x => x.SmoothSkinningMatrixIndex >= 0);
            }
        }

        public int RigidSkinningMatrix
        {
            get
            {
                return this.Parent.Skeleton.EnumerateBones().Count(x => x.RigidSkinningMatrixIndex >= 0);
            }
        }

        public int TotalProcessVertex
        {
            get
            {
                return this.@totalProcessVertex;
            }

            set
            {
                if (this.@totalProcessVertex == value)
                {
                    return;
                }

                this.SetProperty(ref this.@totalProcessVertex, value, () => this.CalcCRC());
            }
        }

        public bool UnitePosQuantize
        {
            get
            {
                return this.@unitePosQuantize;
            }

            set
            {
                if (this.@unitePosQuantize == value)
                {
                    return;
                }

                this.SetProperty(ref this.@unitePosQuantize, value, () => this.CalcCRC());
            }
        }

        public string DccPreset
        {
            get
            {
                return this.@dccPreset;
            }

            set
            {
                if (this.@dccPreset == value)
                {
                    return;
                }

                Ensure.Argument.NotNull(value);
                this.SetProperty(ref this.@dccPreset, value, () => this.CalcCRC());
            }
        }

        public float DccMagnify
        {
            get
            {
                return this.@dccMagnify;
            }

            set
            {
                if (this.@dccMagnify == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccMagnify, value, () => this.CalcCRC());
            }
        }

        public int DccStartFrame
        {
            get
            {
                return this.@dccStartFrame;
            }

            set
            {
                if (this.@dccStartFrame == value)
                {
                    return;
                }

                this.SetProperty(ref this.@dccStartFrame, value, () => this.CalcCRC());
            }
        }

        IEntity IChildEntity.Parent
        {
            get
            {
                return this.parent as IEntity;
            }

            set
            {
                if (this.parent == value)
                {
                    return;
                }

                Ensure.Argument.True(value != null ? value is Model : true);
                this.SetProperty(ref this.parent, value as Model, () => this.CalcCRC());
            }
        }

        public Model Parent
        {
            get
            {
                return this.parent;
            }
        }

        Type IChildEntity.ParentType
        {
            get
            {
                return typeof(Model);
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override model_infoType CreateSerializableData()
        {
            var writeData = new model_infoType();
            writeData.@material_count = this.MaterialCount;
            writeData.@bone_count = this.BoneCount;
            writeData.@shape_count = this.ShapeCount;
            writeData.@smooth_skinning_shape = this.SmoothSkinningShape;
            writeData.@rigid_skinning_shape = this.RigidSkinningShape;
            writeData.@smooth_skinning_matrix = this.SmoothSkinningMatrix;
            writeData.@rigid_skinning_matrix = this.RigidSkinningMatrix;
            writeData.@total_process_vertex = this.TotalProcessVertex;
            writeData.@unite_pos_quantize = this.UnitePosQuantize;
            writeData.@dcc_preset = this.DccPreset;
            writeData.@dcc_magnify = this.DccMagnify;
            writeData.@dcc_start_frame = this.DccStartFrame;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        ModelInfo IDeepCopyable<ModelInfo>.DeepCopy()
        {
            return new ModelInfo(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(ModelInfo source)
        {
            this.@totalProcessVertex = source.@totalProcessVertex;
            this.@unitePosQuantize = source.@unitePosQuantize;
            this.@dccPreset = source.@dccPreset;
            this.@dccMagnify = source.@dccMagnify;
            this.@dccStartFrame = source.@dccStartFrame;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.TotalProcessVertex)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.UnitePosQuantize)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccPreset)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccMagnify)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.DccStartFrame)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
