﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using nw.g3d.nw4f_3dif;
using Opal.Security.Cryptography;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using Nintendo.G3dTool.Entities.Internal;

namespace Nintendo.G3dTool.Entities
{
    public class Lod : ObservableEntity<lodType>, IDeepCopyable<Lod>, IDeepCopyFrom<Lod>
    {
        private float @min = 0.0f;
        private float @max = 13.0f;
        private float @bias = 0.0f;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Lod()
        {
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Lod(Lod source)
            : this()
        {
            this.DeepCopyFrom(source);
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="source">設定するデータです。</param>
        public Lod(lodType source)
        {
            this.@min = source.@min;
            this.@max = source.@max;
            this.@bias = source.@bias;
        }

        public float Min
        {
            get
            {
                return this.@min;
            }

            set
            {
                if (this.@min == value)
                {
                    return;
                }

                this.SetProperty(ref this.@min, value, () => this.CalcCRC());
            }
        }

        public float Max
        {
            get
            {
                return this.@max;
            }

            set
            {
                if (this.@max == value)
                {
                    return;
                }

                this.SetProperty(ref this.@max, value, () => this.CalcCRC());
            }
        }

        public float Bias
        {
            get
            {
                return this.@bias;
            }

            set
            {
                if (this.@bias == value)
                {
                    return;
                }

                this.SetProperty(ref this.@bias, value, () => this.CalcCRC());
            }
        }

        /// <summary>
        /// 出力データを作成します。
        /// </summary>
        /// <returns>出力データのインスタンスを返します。</returns>
        public override lodType CreateSerializableData()
        {
            var writeData = new lodType();
            writeData.@min = this.Min;
            writeData.@max = this.Max;
            writeData.@bias = this.Bias;
            return writeData;
        }

        /// <summary>
        /// 現在のインスタンスをディープコピーで複製した新規インスタンスを返します。
        /// </summary>
        Lod IDeepCopyable<Lod>.DeepCopy()
        {
            return new Lod(this);
        }

        /// <summary>
        /// 入力ファイルからディープコピーします。
        /// </summary>
        /// <param name="source">コピー元となる入力ファイルです。</param>
        public void DeepCopyFrom(Lod source)
        {
            this.@min = source.@min;
            this.@max = source.@max;
            this.@bias = source.@bias;
        }

        /// <summary>
        /// エンティティの CRC を作成します。（内部処理用）
        /// </summary>
        /// <returns>CRC の値を返します。</returns>
        protected override uint CreateCRCInternal()
        {
            CRC32 crc = new CRC32();
            List<byte> buffers = new List<byte>();
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Min)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Max)));
            buffers.AddRange(BitConverter.GetBytes(crc.ComputeHashUInt32(this.Bias)));
            return crc.ComputeHashUInt32(buffers.ToArray());
        }
    }
}
