﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Utility.h"

#include <intrin.h>
const WCHAR* global_IntermediateAsciiFileExtentions[] =
{
    L".fmda",
    L".ftxa",
    L".fska",
    L".fspa",
    L".fcla",
    L".ftsa",
    L".ftpa",
    L".fvba",
    L".fvma",
    L".fsha",
    L".fsna",
    L".fsca",
    L".fsda",
    L".fsva",
    NULL, // WARNING: the last element must always be NULL
};

const WCHAR* global_IntermediateBinaryFileExtentions[] =
{
    L".fmdb",
    L".ftxb",
    L".fskb",
    L".fspb",
    L".fclb",
    L".ftsb",
    L".ftpb",
    L".fvbb",
    L".fvmb",
    L".fshb",
    L".fsnb",
    L".fscb",
    L".fsdb",
    L".fsvb",
    NULL,  // WARNING: the last element must always be NULL
};

//==============================================================================
//
// GUID to string in format of xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx
//
//==============================================================================
std::wstring NWCreateUUIDString( const GUID &id )
{
    WCHAR szTemp[256];
    _snwprintf_s(szTemp,255,_TRUNCATE,L"%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X",
             (int)id.Data1,
             (int)id.Data2,
             (int)id.Data3,
             (int)id.Data4[0],
             (int)id.Data4[1],
             (int)id.Data4[2],
             (int)id.Data4[3],
             (int)id.Data4[4],
             (int)id.Data4[5],
             (int)id.Data4[6],
             (int)id.Data4[7]);

    std::wstring str = szTemp;
    return str;
} // End of NWCreateUUIDString


//==============================================================================
//
// Compare GUID
//
//==============================================================================
bool NWIsSameGUID( const GUID &id1, const GUID &id2  )
{
    if ( (id1.Data1==id2.Data1) &&
         (id1.Data2==id2.Data2) &&
         (id1.Data3==id2.Data3) &&
         (id1.Data4[0]==id2.Data4[0]) &&
         (id1.Data4[1]==id2.Data4[1]) &&
         (id1.Data4[2]==id2.Data4[2]) &&
         (id1.Data4[3]==id2.Data4[3]) &&
         (id1.Data4[4]==id2.Data4[4]) &&
         (id1.Data4[5]==id2.Data4[5]) &&
         (id1.Data4[6]==id2.Data4[6]) &&
         (id1.Data4[7]==id2.Data4[7]) )
    {
        return true;
    } // End if

    return false;
} // End of NWIsSameGUID


//==============================================================================
//
// Find child node from xml
//
//==============================================================================
MSXML2::IXMLDOMNodePtr NWXMLFindChildNode( const MSXML2::IXMLDOMNodePtr node,
                                           const char* name )
{
    MSXML2::IXMLDOMNodeListPtr children = node->childNodes;
    for (;;)
    {
        MSXML2::IXMLDOMNodePtr child = children->nextNode();
        if (child == NULL)
        {
            break;
        } // End if

        if (strcmp(child->nodeName,name)==0)
        {
            return child;
        } // End if
    } // End for

    return NULL;
} // End of NWXMLFindChildNode


//==============================================================================
//
// Get attribute from XML node
//
//==============================================================================
std::string NWXMLGetAttrValue( const MSXML2::IXMLDOMNodePtr node,
                               const char* attrName )
{
    MSXML2::IXMLDOMNamedNodeMapPtr attrs = node->attributes;
    MSXML2::IXMLDOMNodePtr attr = attrs->getNamedItem(attrName);
    if (attr != NULL)
    {
        return std::string(attr->text);
    } // End if

    return "";
} // End of NWXMLGetAttrValue


//==============================================================================
//
// Get encoder CLSID for saving image for debugging
//
//==============================================================================
int NWGetEncoderClsid( const WCHAR* format, CLSID* pClsid )
{
    UINT  num  = 0;         // number of image encoders
    UINT  size = 0;         // size of the image encoder array in bytes

    Gdiplus::ImageCodecInfo* pImageCodecInfo = NULL;

    Gdiplus::GetImageEncodersSize(&num, &size);
    if(size == 0)
        return -1;  // Failure

    pImageCodecInfo = new Gdiplus::ImageCodecInfo[size];
    if(pImageCodecInfo == NULL)
        return -1;  // Failure

    Gdiplus::GetImageEncoders(num, size, pImageCodecInfo);

    for(UINT j = 0; (j < num) && (j<size); ++j)
    {
        if( wcscmp(pImageCodecInfo[j].MimeType, format) == 0 )
        {
            *pClsid = pImageCodecInfo[j].Clsid;
            delete[] pImageCodecInfo;
            return j;  // Success
        } // End if
    } // End for

    delete[] pImageCodecInfo;
    return -1;  // Failure
} // End of NWGetEncoderClsid


//==============================================================================
//
// Used for format <-> string conversion
//
//==============================================================================
const char* g_formatNames[] =
{
    "L4",
    "L8",
    "A4",
    "A8",
    "LA4",
    "LA8",
    "HILO8",
    "RGB565",
    "RGB8",
    "RGB5_A1",
    "RGBA4",
    "RGBA8",
    "ETC1",
    "ETC1_A4",

    // Revolution
    "i4",
    "i8",
    "ia4",
    "ia8",
    "rgb565",
    "rgb5a3",
    "rgba8",
    "c4",
    "c8",
    "c14",
    "cmpr",

    // For cafe
    "UNORM_8",
    "UINT_8",
    "SNORM_8",
    "SINT_8",
    "UNORM_4_4",
    "UNORM_16",
    "UINT_16",
    "SINT_16",
    "FLOAT_16",
    "UNORM_8_8",
    "UINT_8_8",
    "SNORM_8_8",
    "SINT_8_8",
    "UNORM_5_6_5",
    "UNORM_6_5_5",
    "SNORM_6_5_5",
    "UNORM_1_5_5_5",
    "UNORM_4_4_4_4",
    "UNORM_5_5_5_1",
    "UINT_32",
    "SINT_32",
    "FLOAT_32",
    "UNORM_16_16",
    "UINT_16_16",
    "SNORM_16_16",
    "SINT_16_16",
    "FLOAT_16_16",
    "FLOAT_11_11_10",
    "UNORM_2_10_10_10",
    "UNORM_8_8_8_8",
    "UINT_8_8_8_8",
    "SNORM_8_8_8_8",
    "SINT_8_8_8_8",
    "SRGB_8_8_8_8",
    "UNORM_10_10_10_2",
    "UINT_10_10_10_2",
    "UINT_32_32",
    "SINT_32_32",
    "FLOAT_32_32",
    "UNORM_16_16_16_16",
    "UINT_16_16_16_16",
    "SNORM_16_16_16_16",
    "SINT_16_16_16_16",
    "FLOAT_16_16_16_16",
    "UINT_32_32_32_32",
    "SINT_32_32_32_32",
    "FLOAT_32_32_32_32",
    "SHAREDEXP_5_9_9_9",
    "UINT_32_32_32",
    "SINT_32_32_32",
    "FLOAT_32_32_32",
    "UNORM_BC1",
    "SRGB_BC1",
    "UNORM_BC2",
    "SRGB_BC2",
    "UNORM_BC3",
    "SRGB_BC3",
    "UNORM_BC4",
    "SNORM_BC4",
    "UNORM_BC5",
    "SNORM_BC5",
    "UFLOAT_BC6",
    "FLOAT_BC6",
    "UNORM_BC7",
    "SRGB_BC7",
    "UNORM_ETC1",
    "UNORM_ETC2",
    "SRGB_ETC2",
    "UNORM_ETC2_MASK",
    "SRGB_ETC2_MASK",
    "UNORM_ETC2_ALPHA",
    "SRGB_ETC2_ALPHA",
    "UNORM_EAC_11",
    "SNORM_EAC_11",
    "UNORM_EAC_11_11",
    "SNORM_EAC_11_11",
    "UNORM_PVRTC1_2bpp",
    "SRGB_PVRTC1_2bpp",
    "UNORM_PVRTC1_4bpp",
    "SRGB_PVRTC1_4bpp",
    "UNORM_PVRTC1_ALPHA_2bpp",
    "SRGB_PVRTC1_ALPHA_2bpp",
    "UNORM_PVRTC1_ALPHA_4bpp",
    "SRGB_PVRTC1_ALPHA_4bpp",
    "UNORM_PVRTC2_ALPHA_2bpp",
    "SRGB_PVRTC2_ALPHA_2bpp",
    "UNORM_PVRTC2_ALPHA_4bpp",
    "SRGB_PVRTC2_ALPHA_4bpp",
    "UNORM_ASTC_4x4",
    "SRGB_ASTC_4x4",
    "UNORM_ASTC_5x4",
    "SRGB_ASTC_5x4",
    "UNORM_ASTC_5x5",
    "SRGB_ASTC_5x5",
    "UNORM_ASTC_6x5",
    "SRGB_ASTC_6x5",
    "UNORM_ASTC_6x6",
    "SRGB_ASTC_6x6",
    "UNORM_ASTC_8x5",
    "SRGB_ASTC_8x5",
    "UNORM_ASTC_8x6",
    "SRGB_ASTC_8x6",
    "UNORM_ASTC_8x8",
    "SRGB_ASTC_8x8",
    "UNORM_ASTC_10x5",
    "SRGB_ASTC_10x5",
    "UNORM_ASTC_10x6",
    "SRGB_ASTC_10x6",
    "UNORM_ASTC_10x8",
    "SRGB_ASTC_10x8",
    "UNORM_ASTC_10x10",
    "SRGB_ASTC_10x10",
    "UNORM_ASTC_12x10",
    "SRGB_ASTC_12x10",
    "UNORM_ASTC_12x12",
    "SRGB_ASTC_12x12",

    "UNKNOWN",
    "UNKNOWN"
}; // End of g_formatNames

const WCHAR* g_formatNamesW[] =
{
    L"L4",
    L"L8",
    L"A4",
    L"A8",
    L"LA4",
    L"LA8",
    L"HILO8",
    L"RGB565",
    L"RGB8",
    L"RGB5_A1",
    L"RGBA4",
    L"RGBA8",
    L"ETC1",
    L"ETC1_A4",

    // Revolution
    L"i4",
    L"i8",
    L"ia4",
    L"ia8",
    L"rgb565",
    L"rgb5a3",
    L"rgba8",
    L"c4",
    L"c8",
    L"c14",
    L"cmpr",

    // For Cafe
    L"UNORM_8",
    L"UINT_8",
    L"SNORM_8",
    L"SINT_8",
    L"UNORM_4_4",
    L"UNORM_16",
    L"UINT_16",
    L"SINT_16",
    L"FLOAT_16",
    L"UNORM_8_8",
    L"UINT_8_8",
    L"SNORM_8_8",
    L"SINT_8_8",
    L"UNORM_5_6_5",
    L"UNORM_6_5_5",
    L"SNORM_6_5_5",
    L"UNORM_1_5_5_5",
    L"UNORM_4_4_4_4",
    L"UNORM_5_5_5_1",
    L"UINT_32",
    L"SINT_32",
    L"FLOAT_32",
    L"UNORM_16_16",
    L"UINT_16_16",
    L"SNORM_16_16",
    L"SINT_16_16",
    L"FLOAT_16_16",
    L"FLOAT_11_11_10",
    L"UNORM_2_10_10_10",
    L"UNORM_8_8_8_8",
    L"UINT_8_8_8_8",
    L"SNORM_8_8_8_8",
    L"SINT_8_8_8_8",
    L"SRGB_8_8_8_8",
    L"UNORM_10_10_10_2",
    L"UINT_10_10_10_2",
    L"UINT_32_32",
    L"SINT_32_32",
    L"FLOAT_32_32",
    L"UNORM_16_16_16_16",
    L"UINT_16_16_16_16",
    L"SNORM_16_16_16_16",
    L"SINT_16_16_16_16",
    L"FLOAT_16_16_16_16",
    L"UINT_32_32_32_32",
    L"SINT_32_32_32_32",
    L"FLOAT_32_32_32_32",
    L"SHAREDEXP_5_9_9_9",
    L"UINT_32_32_32",
    L"SINT_32_32_32",
    L"FLOAT_32_32_32",
    L"UNORM_BC1",
    L"SRGB_BC1",
    L"UNORM_BC2",
    L"SRGB_BC2",
    L"UNORM_BC3",
    L"SRGB_BC3",
    L"UNORM_BC4",
    L"SNORM_BC4",
    L"UNORM_BC5",
    L"SNORM_BC5",
    L"UFLOAT_BC6",
    L"FLOAT_BC6",
    L"UNORM_BC7",
    L"SRGB_BC7",
    L"UNORM_ETC1",
    L"UNORM_ETC2",
    L"SRGB_ETC2",
    L"UNORM_ETC2_MASK",
    L"SRGB_ETC2_MASK",
    L"UNORM_ETC2_ALPHA",
    L"SRGB_ETC2_ALPHA",
    L"UNORM_EAC_11",
    L"SNORM_EAC_11",
    L"UNORM_EAC_11_11",
    L"SNORM_EAC_11_11",
    L"UNORM_PVRTC1_2bpp",
    L"SRGB_PVRTC1_2bpp",
    L"UNORM_PVRTC1_4bpp",
    L"SRGB_PVRTC1_4bpp",
    L"UNORM_PVRTC1_ALPHA_2bpp",
    L"SRGB_PVRTC1_ALPHA_2bpp",
    L"UNORM_PVRTC1_ALPHA_4bpp",
    L"SRGB_PVRTC1_ALPHA_4bpp",
    L"UNORM_PVRTC2_ALPHA_2bpp",
    L"SRGB_PVRTC2_ALPHA_2bpp",
    L"UNORM_PVRTC2_ALPHA_4bpp",
    L"SRGB_PVRTC2_ALPHA_4bpp",
    L"UNORM_ASTC_4x4",
    L"SRGB_ASTC_4x4",
    L"UNORM_ASTC_5x4",
    L"SRGB_ASTC_5x4",
    L"UNORM_ASTC_5x5",
    L"SRGB_ASTC_5x5",
    L"UNORM_ASTC_6x5",
    L"SRGB_ASTC_6x5",
    L"UNORM_ASTC_6x6",
    L"SRGB_ASTC_6x6",
    L"UNORM_ASTC_8x5",
    L"SRGB_ASTC_8x5",
    L"UNORM_ASTC_8x6",
    L"SRGB_ASTC_8x6",
    L"UNORM_ASTC_8x8",
    L"SRGB_ASTC_8x8",
    L"UNORM_ASTC_10x5",
    L"SRGB_ASTC_10x5",
    L"UNORM_ASTC_10x6",
    L"SRGB_ASTC_10x6",
    L"UNORM_ASTC_10x8",
    L"SRGB_ASTC_10x8",
    L"UNORM_ASTC_10x10",
    L"SRGB_ASTC_10x10",
    L"UNORM_ASTC_12x10",
    L"SRGB_ASTC_12x10",
    L"UNORM_ASTC_12x12",
    L"SRGB_ASTC_12x12",


    L"UNKNOWN",
    L"UNKNOWN"
}; // End of g_formatNamesW

//==============================================================================
//
// Get format from string
//
//==============================================================================
NW_IMG_FMT NWGetTextureFormatFromString( const char* szStr )
{
    int i = 0;
    while (g_formatNames[i]!=NULL)
    {

        if (_stricmp(g_formatNames[i],szStr)==0)
        {
            return (NW_IMG_FMT)i;
        } // End if

        i++;
    } // End while

    return NW_IMG_FMT_NONE;
} // End of NWGetTextureFormatFromString

//==============================================================================
//
// Get format string from format
//
//==============================================================================
const char* NWGetTextureFormatName( NW_IMG_FMT format )
{
    return g_formatNames[(int)format];
} // End of NWGetTextureFormatName


//==============================================================================
//
// Get format string from format for Unicode
//
//==============================================================================
const WCHAR* NWGetTextureFormatNameW( NW_IMG_FMT format )
{
    return g_formatNamesW[(int)format];
} // End of NWGetTextureFormatNameW


//==============================================================================
//
// Get texture type name
//
//==============================================================================
const char* NWGetTextureTypeName( NW_IMG_TYPE texType )
{
    switch (texType)
    {
        case NW_IMG_TYPE_1D :
            return "1D";

        case NW_IMG_TYPE_2D :
            return "2D";

        case NW_IMG_TYPE_3D :
            return "3D";

        case NW_IMG_TYPE_CUBE :
            return "Cube";

        case NW_IMG_TYPE_1D_ARRAY :
            return "1DArray";

        case NW_IMG_TYPE_2D_ARRAY :
            return "2DArray";

        case NW_IMG_TYPE_3D_ARRAY :
            return "3DArray";
    } // End of switch

    return "None";
} // End of NWGetTextureTypeName


//==============================================================================
//
// Get texture type name for unicode
//
//==============================================================================
const WCHAR* NWGetTextureTypeNameW( NW_IMG_TYPE texType )
{
    switch (texType)
    {
        case NW_IMG_TYPE_1D :
            return L"1D";

        case NW_IMG_TYPE_2D :
            return L"2D";

        case NW_IMG_TYPE_3D :
            return L"3D";

        case NW_IMG_TYPE_CUBE :
            return L"Cube";

        case NW_IMG_TYPE_1D_ARRAY :
            return L"1DArray";

        case NW_IMG_TYPE_2D_ARRAY :
            return L"2DArray";

        case NW_IMG_TYPE_3D_ARRAY :
            return L"3DArray";
    } // End of switch

    return L"None";
} // End of NWGetTextureTypeName


//==============================================================================
//
// Get number of bits per pixel
//
//==============================================================================
int NWGetTextureFormatBitsPerPixel( NW_IMG_FMT format )
{
    switch (format)
    {
        case NW_IMG_FMT_L4 :
            return 4;

        case NW_IMG_FMT_L8 :
            return 8;

        case NW_IMG_FMT_A4 :
            return 4;

        case NW_IMG_FMT_A8 :
            return 8;

        case NW_IMG_FMT_LA4 :
            return 8;

        case NW_IMG_FMT_LA8 :
            return 16;

        case NW_IMG_FMT_HILO8 :
            return 16;

        case NW_IMG_FMT_RGB565 :
            return 16;

        case NW_IMG_FMT_RGB8 :
            return 24;

        case NW_IMG_FMT_RGB5_A1 :
            return 16;

        case NW_IMG_FMT_RGBA4 :
            return 16;

        case NW_IMG_FMT_RGBA8 :
            return 32;

        // Revolution
        case NW_IMG_FMT_R_I4:
            return 4;

        case NW_IMG_FMT_R_I8:
            return 8;

        case NW_IMG_FMT_R_IA4:
            return 8;

        case NW_IMG_FMT_R_IA8:
            return 16;

        case NW_IMG_FMT_R_R5G6B5:
            return 16;

        case NW_IMG_FMT_R_RGB5A3:
            return 16;

        case NW_IMG_FMT_R_RGBA8:
            return 32;

        case NW_IMG_FMT_R_C4:
            return 4;

        case NW_IMG_FMT_R_C8:
            return 8;

        case NW_IMG_FMT_R_C14:
            return 16;

        case NW_IMG_FMT_R_CMPR:
            return 0; // Compressed format, bbp makes no sense

        case NW_IMG_FMT_GX2_UNORM_8:
            return 8;

        case NW_IMG_FMT_GX2_UINT_8:
            return 8;

        case NW_IMG_FMT_GX2_SNORM_8:
            return 8;

        case NW_IMG_FMT_GX2_SINT_8:
            return 8;

        case NW_IMG_FMT_GX2_UNORM_4_4:
            return 8;

        case NW_IMG_FMT_GX2_UNORM_16:
            return 16;

        case NW_IMG_FMT_GX2_UINT_16:
            return 16;

        case NW_IMG_FMT_GX2_SINT_16:
            return 16;

        case NW_IMG_FMT_GX2_FLOAT_16:
            return 16;

        case NW_IMG_FMT_GX2_UNORM_8_8:
            return 16;

        case NW_IMG_FMT_GX2_UINT_8_8:
            return 16;

        case NW_IMG_FMT_GX2_SNORM_8_8:
            return 16;

        case NW_IMG_FMT_GX2_SINT_8_8:
            return 16;

        case NW_IMG_FMT_GX2_UNORM_5_6_5:
            return 16;

        case NW_IMG_FMT_GX2_UNORM_6_5_5:
            return 16;

        case NW_IMG_FMT_GX2_SNORM_6_5_5:
            return 16;

        case NW_IMG_FMT_GX2_UNORM_1_5_5_5:
            return 16;

        case NW_IMG_FMT_GX2_UNORM_4_4_4_4:
            return 16;

        case NW_IMG_FMT_GX2_UNORM_5_5_5_1:
            return 16;

        case NW_IMG_FMT_GX2_UINT_32:
            return 32;

        case NW_IMG_FMT_GX2_SINT_32:
            return 32;

        case NW_IMG_FMT_GX2_FLOAT_32:
            return 32;

        case NW_IMG_FMT_GX2_UNORM_16_16:
            return 32;

        case NW_IMG_FMT_GX2_UINT_16_16:
            return 32;

        case NW_IMG_FMT_GX2_SNORM_16_16:
            return 32;

        case NW_IMG_FMT_GX2_SINT_16_16:
            return 32;

        case NW_IMG_FMT_GX2_FLOAT_16_16:
            return 32;

        case NW_IMG_FMT_GX2_FLOAT_11_11_10:
            return 32;

        case NW_IMG_FMT_GX2_UNORM_2_10_10_10:
            return 32;

        case NW_IMG_FMT_GX2_UNORM_8_8_8_8:
            return 32;

        case NW_IMG_FMT_GX2_UINT_8_8_8_8:
            return 32;

        case NW_IMG_FMT_GX2_SNORM_8_8_8_8:
            return 32;

        case NW_IMG_FMT_GX2_SINT_8_8_8_8:
            return 32;

        case NW_IMG_FMT_GX2_SRGB_8_8_8_8:
            return 32;

        case NW_IMG_FMT_GX2_UNORM_10_10_10_2:
            return 32;

        case NW_IMG_FMT_GX2_UINT_10_10_10_2:
            return 32;

        case NW_IMG_FMT_GX2_UINT_32_32:
            return 64;

        case NW_IMG_FMT_GX2_SINT_32_32:
            return 64;

        case NW_IMG_FMT_GX2_FLOAT_32_32:
            return 64;

        case NW_IMG_FMT_GX2_UNORM_16_16_16_16:
            return 64;

        case NW_IMG_FMT_GX2_UINT_16_16_16_16:
            return 64;

        case NW_IMG_FMT_GX2_SNORM_16_16_16_16:
            return 64;

        case NW_IMG_FMT_GX2_SINT_16_16_16_16:
            return 64;

        case NW_IMG_FMT_GX2_FLOAT_16_16_16_16:
            return 64;

        case NW_IMG_FMT_GX2_UINT_32_32_32_32:
            return 128;

        case NW_IMG_FMT_GX2_SINT_32_32_32_32:
            return 128;

        case NW_IMG_FMT_GX2_FLOAT_32_32_32_32:
            return 128;

        case NW_IMG_FMT_GX2_SHAREDEXP_5_9_9_9:
            return 32;

        case NW_IMG_FMT_GX2_UINT_32_32_32:
            return 96;

        case NW_IMG_FMT_GX2_SINT_32_32_32:
            return 96;

        case NW_IMG_FMT_GX2_FLOAT_32_32_32:
            return 96;

        case NW_IMG_FMT_GX2_UNORM_BC1:
            return 4;

        case NW_IMG_FMT_GX2_SRGB_BC1:
            return 4;

        case NW_IMG_FMT_GX2_UNORM_BC2:
            return 8;

        case NW_IMG_FMT_GX2_SRGB_BC2:
            return 8;

        case NW_IMG_FMT_GX2_UNORM_BC3:
            return 8;

        case NW_IMG_FMT_GX2_SRGB_BC3:
            return 8;

        case NW_IMG_FMT_GX2_UNORM_BC4:
            return 4;

        case NW_IMG_FMT_GX2_SNORM_BC4:
            return 4;

        case NW_IMG_FMT_GX2_UNORM_BC5:
            return 8;

        case NW_IMG_FMT_GX2_SNORM_BC5:
            return 8;



    } // End switch

    return 0;
} // End of NWGetTextureFormatBitsPerPixel


//==============================================================================
//
// Check if texture format is linear
//
//==============================================================================
int NWTextureFormatIsLinear( NW_IMG_FMT format )
{
    switch (format)
    {
    case NW_IMG_FMT_GX2_SRGB_8_8_8_8:
    case NW_IMG_FMT_GX2_SRGB_BC1:
    case NW_IMG_FMT_GX2_SRGB_BC2:
    case NW_IMG_FMT_GX2_SRGB_BC3:
        return true;
    default:
        return false;
    }
} // End of NWTextureFormatIsLinear


//==============================================================================
//
// Check if texture format has alpha channel
//
//==============================================================================
bool NWTextureFormatHasAlpha( NW_IMG_FMT format, NW_IMG_FMT paletteFormat )
{
    switch (format)
    {
        case NW_IMG_FMT_L4 :
            return false;

        case NW_IMG_FMT_L8 :
            return false;

        case NW_IMG_FMT_A4 :
            return true;

        case NW_IMG_FMT_A8 :
            return true;

        case NW_IMG_FMT_LA4 :
            return true;

        case NW_IMG_FMT_LA8 :
            return true;

        case NW_IMG_FMT_HILO8 :
            return false;

        case NW_IMG_FMT_RGB565 :
            return false;

        case NW_IMG_FMT_RGB8 :
            return false;

        case NW_IMG_FMT_RGB5_A1 :
            return true;

        case NW_IMG_FMT_RGBA4 :
            return true;

        case NW_IMG_FMT_RGBA8 :
            return true;

        case NW_IMG_FMT_ETC1 :
            return false;

        case NW_IMG_FMT_ETCA :
            return true;


        // Revolution
        case NW_IMG_FMT_R_I4:
        case NW_IMG_FMT_R_I8:
            return false;

        case NW_IMG_FMT_R_IA4:
        case NW_IMG_FMT_R_IA8:
            return true;

        case NW_IMG_FMT_R_R5G6B5:
            return false;

        case NW_IMG_FMT_R_RGB5A3:
            return true;

        case NW_IMG_FMT_R_RGBA8:
            return true;

        case NW_IMG_FMT_R_C4:
        case NW_IMG_FMT_R_C8:
        case NW_IMG_FMT_R_C14:
            return NWTextureFormatHasAlpha(paletteFormat, paletteFormat);

        case NW_IMG_FMT_R_CMPR:
            return true;

        case NW_IMG_FMT_GX2_UNORM_8 :
        case NW_IMG_FMT_GX2_UINT_8 :
        case NW_IMG_FMT_GX2_SNORM_8 :
        case NW_IMG_FMT_GX2_SINT_8 :
        case NW_IMG_FMT_GX2_UNORM_4_4 :
        case NW_IMG_FMT_GX2_UNORM_16 :
        case NW_IMG_FMT_GX2_UINT_16 :
        case NW_IMG_FMT_GX2_SINT_16 :
        case NW_IMG_FMT_GX2_FLOAT_16 :
        case NW_IMG_FMT_GX2_UNORM_8_8 :
        case NW_IMG_FMT_GX2_UINT_8_8 :
        case NW_IMG_FMT_GX2_SNORM_8_8 :
        case NW_IMG_FMT_GX2_SINT_8_8 :
        case NW_IMG_FMT_GX2_UNORM_5_6_5 :
        case NW_IMG_FMT_GX2_UNORM_6_5_5 :
        case NW_IMG_FMT_GX2_SNORM_6_5_5 :
            return false;

        case NW_IMG_FMT_GX2_UNORM_1_5_5_5 :
        case NW_IMG_FMT_GX2_UNORM_4_4_4_4 :
        case NW_IMG_FMT_GX2_UNORM_5_5_5_1 :
            return true;

        case NW_IMG_FMT_GX2_UINT_32 :
        case NW_IMG_FMT_GX2_SINT_32 :
        case NW_IMG_FMT_GX2_FLOAT_32 :
        case NW_IMG_FMT_GX2_UNORM_16_16 :
        case NW_IMG_FMT_GX2_UINT_16_16 :
        case NW_IMG_FMT_GX2_SNORM_16_16 :
        case NW_IMG_FMT_GX2_SINT_16_16 :
        case NW_IMG_FMT_GX2_FLOAT_16_16 :
        case NW_IMG_FMT_GX2_FLOAT_11_11_10 :
            return false;

        case NW_IMG_FMT_GX2_UNORM_2_10_10_10 :
            return true;

        case NW_IMG_FMT_GX2_UNORM_8_8_8_8 :
        case NW_IMG_FMT_GX2_UINT_8_8_8_8 :
        case NW_IMG_FMT_GX2_SNORM_8_8_8_8 :
        case NW_IMG_FMT_GX2_SINT_8_8_8_8 :
        case NW_IMG_FMT_GX2_SRGB_8_8_8_8 :
            return true;

        case NW_IMG_FMT_GX2_UNORM_10_10_10_2 :
        case NW_IMG_FMT_GX2_UINT_10_10_10_2 :
            return true;

        case NW_IMG_FMT_GX2_UINT_32_32 :
        case NW_IMG_FMT_GX2_SINT_32_32 :
        case NW_IMG_FMT_GX2_FLOAT_32_32 :
            return false;

        case NW_IMG_FMT_GX2_UNORM_16_16_16_16 :
        case NW_IMG_FMT_GX2_UINT_16_16_16_16 :
        case NW_IMG_FMT_GX2_SNORM_16_16_16_16 :
        case NW_IMG_FMT_GX2_SINT_16_16_16_16 :
        case NW_IMG_FMT_GX2_FLOAT_16_16_16_16 :
        case NW_IMG_FMT_GX2_UINT_32_32_32_32 :
        case NW_IMG_FMT_GX2_SINT_32_32_32_32 :
        case NW_IMG_FMT_GX2_FLOAT_32_32_32_32 :
        case NW_IMG_FMT_GX2_SHAREDEXP_5_9_9_9 :
            return true;

        case NW_IMG_FMT_GX2_UINT_32_32_32 :
        case NW_IMG_FMT_GX2_SINT_32_32_32 :
        case NW_IMG_FMT_GX2_FLOAT_32_32_32 :
            return false;

        case NW_IMG_FMT_GX2_UNORM_BC1 :
        case NW_IMG_FMT_GX2_SRGB_BC1 :
            // AlphaImageを作ってからCNWTexture::MakeHasAlphaFromAlphaImageWhenBC1で判定する
            return true;

        case NW_IMG_FMT_GX2_UNORM_BC2 :
        case NW_IMG_FMT_GX2_SRGB_BC2 :
        case NW_IMG_FMT_GX2_UNORM_BC3 :
        case NW_IMG_FMT_GX2_SRGB_BC3 :
            return true;

        case NW_IMG_FMT_GX2_UNORM_BC4 :
        case NW_IMG_FMT_GX2_SNORM_BC4 :
        case NW_IMG_FMT_GX2_UNORM_BC5 :
        case NW_IMG_FMT_GX2_SNORM_BC5 :
            return false;
    } // End switch

    return false;
} // End of NWTextureFormatHasAlpha


//==============================================================================
//
// String to linear flag array
//
//==============================================================================
bool NWStringToLinearFlagArray( const std::string &str, USHORT* linearFlags, int flagCount )
{
    // using a bool array breaks the stack! (-> use USHORT array)

    memset(linearFlags, 0, sizeof(*linearFlags) * flagCount);

    if (flagCount == 0)
        return true;

    if (flagCount < 0 || str.empty())
        return false;

    char* separators = " ";
    char* token;
    char* tokenContext = NULL;
    char* cstr = const_cast<char*>(str.c_str()); // first time cstr is set to string containing tokens

    for (int i = 0; i < flagCount; i++)
    {
        token = strtok_s(cstr, separators, &tokenContext);

        if (token == NULL)
            break;

        linearFlags[i] = _strnicmp(token, "true", 4) == 0 ? 1 : 0;

        cstr = NULL; // cstr (first parameter of strtok_s function) is then NULL for all next iterations
    }

    return true;
} // End of NWStringToLinearFlagArray

//==============================================================================
//
// String to component selector
//
//==============================================================================
NW_COMP_SEL NWStringToCompSel( const std::string &compSelStr )
{
    NW_COMP_SEL compSel;

    int index = 0;
    int i     = 0;
    while ( (i<(int)compSelStr.size()) && (index<4) )
    {
        char ch = compSelStr[i];
        if (ch==' ')
        {
            i++;
            continue;
        } // End if

        if ( (ch=='r') || (ch=='R') )
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_R;
        else if ( (ch=='g') || (ch=='G') )
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_G;
        else if ( (ch=='b') || (ch=='B') )
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_B;
        else if ( (ch=='a') || (ch=='A') )
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_A;
        else if (ch=='0')
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_0;
        else if (ch=='1')
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_1;

        index++;
        i++;
    } // End while

    return compSel;
} // End of NWStringToCompSel

NW_COMP_SEL NWStringToCompSelW(const std::wstring &compSelStr)
{
    NW_COMP_SEL compSel;

    int index = 0;
    int i = 0;
    while ((i<(int)compSelStr.size()) && (index<4))
    {
        wchar_t ch = compSelStr[i];
        if (ch == ' ')
        {
            i++;
            continue;
        } // End if

        if ((ch == 'r') || (ch == 'R'))
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_R;
        else if ((ch == 'g') || (ch == 'G'))
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_G;
        else if ((ch == 'b') || (ch == 'B'))
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_B;
        else if ((ch == 'a') || (ch == 'A'))
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_A;
        else if (ch == '0')
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_0;
        else if (ch == '1')
            compSel.sel[index] = NW_COMP_SEL_ELEMENT_1;

        index++;
        i++;
    } // End while

    return compSel;
} // End of NWStringToCompSel


//==============================================================================
//
// Component selector to string
//
//==============================================================================
std::wstring NWCompSelToString( const NW_COMP_SEL& compSel )
{
    WCHAR szTempStr[256];

    static const char compSelChars[6] =
    {
        'R',
        'G',
        'B',
        'A',
        '0',
        '1'
    }; // End of compSelChars

    _snwprintf_s(szTempStr,255,_TRUNCATE,L"CompSel : %c%c%c%c",
        compSelChars[compSel.sel[0]],
        compSelChars[compSel.sel[1]],
        compSelChars[compSel.sel[2]],
        compSelChars[compSel.sel[3]]);

    return std::wstring(szTempStr);
} // End of NWCompSelToString


//==============================================================================
//
// Unicode string from ASCII string
//
//==============================================================================
std::wstring NWStringToWString( const std::string& s )
{
    std::wstring temp(s.length(),L' ');
    std::copy(s.begin(), s.end(), temp.begin());
    return temp;
} // End of NWStringToWString


//==============================================================================
//
// ASCII string from Unicode string
// ( We can not do perfect conversion, just truncate to ASCII )
//
//==============================================================================
std::string NWWStringToString( const std::wstring& s )
{
    std::string temp(s.length(),' ');
    int i;
    for (i=0;i<(int)s.length();i++)
    {
        temp[i] = (char)s[i];
    } // End for

    return temp;
} // End of NWStringToWString


//==============================================================================
//
// Convert UTF8 to W String
//
//==============================================================================
std::wstring NWUTF8toWString( /*[in]*/ const unsigned char* szSrc )
{
    std::wstring str;
    if (szSrc==NULL)
    {
        return str;
    } // End if

    int i             = 0;
    unsigned char  ch = szSrc[i];
    while ( ch > 0  )
    {
        if ( ch < 0x80 )
        {
            str += (WCHAR) ch;
            ++i;
        } // End if
        else if ( ch < 0xE0 )
        {
            str += (WCHAR) (((ch & 0x1F) << 6 | (szSrc[i+1] & 0x3F)));
            i += 2;
        } // End else if
        else if ( ch < 0xF0 )
        {
            str += (WCHAR) (((ch & 0x0F) << 12) | ((szSrc[i+1] & 0x3F) << 6) | (szSrc[i+2] & 0x3F));
            i += 3;
        } // End else if
        else
        {
            // 4-byte Unicode character, skip this character
            i += 4;
        } // End else

        ch = szSrc[i];
    } // End while

    return str;
} // End of NWUTF8toWString

  //==============================================================================
  //
  // Convert MultiByte String to W String
  //
  //==============================================================================
std::wstring NWMultiByteStringToWString(const char* szSrc)
{
    auto size = MultiByteToWideChar(CP_ACP, 0, szSrc, -1, 0, 0);
    std::unique_ptr<wchar_t[]> buffer(new wchar_t[size+1]);
    auto result = MultiByteToWideChar(CP_ACP, 0, szSrc, -1, buffer.get(), size);
    if (result != size)
    {
        return L"";
    }

    buffer[size] = 0;

    return std::wstring(buffer.get());
}


//==============================================================================
//
// Get maximum mipmap counts with given size and format
//
//==============================================================================
int NWGetMaxMipmapLevels( NW_IMG_FMT format, int width, UINT height )
{
    int mipCount = 1;

    // NW4F format?
    if ( (format>=NW_IMG_FMT_GX2_UNORM_8) &&
         (format<=NW_IMG_FMT_GX2_SNORM_BC5) )
    {
        width  >>= 1;
        height >>= 1;

        while ( (width>0) && (height>0) )
        {
            mipCount++;

            width  >>= 1;
            height >>= 1;
        } // End while

        return mipCount;
    } // End if

    // NW4C format or Revolution
    if (format==NW_IMG_FMT_ETC1)
    {
        width  >>= 1;
        height >>= 1;

        // Smallest mipmap is 16x16
        while ( (width>=16) && (height>=16) )
        {
            mipCount++;

            width  >>= 1;
            height >>= 1;
        } // End while
    } // End if
    else
    {
        width  >>= 1;
        height >>= 1;

        // Smallest mipmap is 8x8
        while ( (width>=8) && (height>=8) )
        {
            mipCount++;

            width  >>= 1;
            height >>= 1;
        } // End while
    } // End if

    return mipCount;
} // End of NWGetMaxMipmapLevels


//==============================================================================
//
// Get extension of file path
//
//==============================================================================
std::wstring NWGetExtension( const WCHAR* szPath )
{
    std::wstring extStr = ::PathFindExtensionW(szPath);
    return extStr;
} // End of NWGetExtension


//==============================================================================
///
/// Format big numbers by inserting comma
///
//==============================================================================
void NW4FormatNumberStringWithComma( WCHAR *szTempStr, int maxStrLen, __int64 number )
{
    int numDigits = 0;
    __int64 num = number;

    int numStacks[256];

    while ( (num>0) && (numDigits<256) )
    {
        int n = num % 10;
        numStacks[numDigits] = n;
        numDigits++;
        num /= 10;
    } // End while

    // Special case for zero
    if (numDigits==0)
    {
        numStacks[numDigits] = 0;
        numDigits++;
    } // End if

    int i;
    int chPos = 0;
    int commaInsert;
    for (i=0;i<numDigits;i++)
    {
        int offset = numDigits - i -1;
        szTempStr[chPos] = (WCHAR)((int)L'0' + numStacks[offset]);
        chPos++;

        if (chPos>=maxStrLen)
            break;

        if (i==numDigits-1)
            break;

        commaInsert = offset % 3;

        if (commaInsert==0) // Need to insert comma?
        {
            szTempStr[chPos] = L',';
            chPos++;
        } // End if
    } // End for

    szTempStr[chPos] = (WCHAR)0;
} // End of NW4FormatNumberStringWithComma


//==============================================================================
///
/// Get file name from path
///
//==============================================================================
std::wstring NWGetFileNameFromPath( const WCHAR *szPath )
{
    std::wstring str;

    if (szPath==NULL)
        return str;

    // Find the length of path
    int len = (int)wcslen(szPath);
    if (len<=0)
        return str;

    // Find last slash from end
    int iPos = len - 1;
    while (iPos>=0)
    {
        if ( (szPath[iPos]==L'\\') ||
             (szPath[iPos]==L'/') )
        {
            break;
        } // End if

        iPos--;
    } // End while

    iPos++;

    // Copy to file name
    str.reserve(len-iPos+1);

    int i;
    for (i=iPos;i<len;i++)
    {
        str += szPath[i];
    } // End for

    return str;
} // End of NWGetFileNameFromPath


//==============================================================================
///
/// Get directory name from path
///
//==============================================================================
std::wstring NWGetDirectoryNameFromPath( const WCHAR *szPath )
{
    std::wstring str;

    if (szPath==NULL)
        return str;

    // Find the length of path
    int len = (int)wcslen(szPath);
    if (len<=0)
        return str;

    // Find last slash from end
    int iPos = len - 1;
    while (iPos>=0)
    {
        if ( (szPath[iPos]==L'\\') ||
             (szPath[iPos]==L'/') )
        {
            break;
        } // End if

        iPos--;
    } // End while

    if (iPos<0)
        return str;

    // Copy to file name
    str.reserve(iPos+1);

    int i;
    for (i=0;i<iPos;i++)
    {
        str += szPath[i];
    } // End for

    return str;
} // End of NWGetDirectoryNameFromPath


//==============================================================================
///
/// Get resource content as buffer.
// ( Caller is responsible for freeing the memory returned by this function )
///
//==============================================================================
unsigned char* NWGetResourceFileContent( HMODULE hModule, int resourceID,
                                         const WCHAR* szResourceType,
                                         int *pSizeOfFile )
{
    if (pSizeOfFile!=NULL)
        *pSizeOfFile = 0;

    // Find & get resource
    HRSRC hRsc = ::FindResource(hModule,MAKEINTRESOURCE(resourceID),szResourceType);
    if (hRsc==NULL)
    {
        return NULL;
    } // End if

    // According to Win32 SDK doc, you shouldn't call unload on resource data.
    // Just get handle by LoadResource, but releasing this handle is not necessary.
    HGLOBAL hGlobal = ::LoadResource(hModule,hRsc);
    if (hGlobal==NULL)
    {
        return NULL;
    } // End if

    // Get file
    int memSize = ::SizeofResource(hModule,hRsc);
    if (memSize==NULL)
    {
        return NULL;
    } // End if

    // According to Win32 SDK doc, you shouldn't call unlock on resource data.
    // Just get resource data by LockResource, but do not call unlock.
    unsigned char *pBuffer = (unsigned char*)(::LockResource(hGlobal));
    if (pBuffer==NULL)
    {
        return NULL;
    } // End if

    unsigned char *pDestBuffer = new unsigned char[memSize];
    if (pDestBuffer==NULL)
    {
        return NULL;
    } // End if

    memcpy(pDestBuffer,pBuffer,memSize);

    if (pSizeOfFile!=NULL)
        *pSizeOfFile = memSize;

    return pDestBuffer;
} // End of NWGetResourceFileContent


//==============================================================================
//
// Find next power of 2
//
//==============================================================================
int NWFindNextPowerOf2( int val )
{
    int v = 1;
    while (v<val)
    {
        v <<= 1;
    } // End while

    return v;
} // End of NWFindNextPowerOf2


//==============================================================================
//
// Compute image data size
//
//==============================================================================
__int64 NWComputeImageDataSize( NW_IMG_FMT format, NW_IMG_TYPE type,
                                int width, int height, int mipMap )
{
    __int64 textureDataSize = 0;
    if ( (format == NW_IMG_FMT_ETC1) ||
         (format == NW_IMG_FMT_ETCA) )
    {
        __int64 bytesPerBlock = 8;
        if (format == NW_IMG_FMT_ETCA)
            bytesPerBlock = 16;

        __int64 wd = (__int64)width;
        __int64 ht = (__int64)height;

        int i;
        for ( i = 0; i < mipMap; i++ )
        {
            __int64 numBlocksX    = (__int64)max(1,wd / 4);
            __int64 numBlocksY    = (__int64)max(1,ht / 4);

            textureDataSize += (bytesPerBlock * numBlocksX * numBlocksY);

            wd >>= 1;
            ht >>= 1;
        } // End for
    } // End if
    else if ( (format == NW_IMG_FMT_R_CMPR) )
    {
        int bytesPerBlock = 8;

        __int64 wd = width;
        __int64 ht = height;

        int i;
        for ( i = 0; i < mipMap; i++ )
        {
            __int64 numBlocksX    = (__int64)max(1,wd / 4);
            __int64 numBlocksY    = (__int64)max(1,ht / 4);

            textureDataSize += (bytesPerBlock * numBlocksX * numBlocksY);

            wd >>= 1;
            ht >>= 1;
        } // End for
    } // End if
    else
    {
        __int64 wd = (__int64)width;
        __int64 ht = (__int64)height;

        __int64 bitsPerPixel = (__int64)NWGetTextureFormatBitsPerPixel( format );

        int i;
        for ( i = 0; i < mipMap ; i++ )
        {
            __int64 numPixelsX    = (__int64)max(1,wd);
            __int64 numPixelsY    = (__int64)max(1,ht);

            textureDataSize += ((bitsPerPixel * numPixelsX * numPixelsY) / 8);

            wd >>= 1;
            ht >>= 1;
        } // End for
    } // End else

    if (type==NW_IMG_TYPE_CUBE)
    {
        textureDataSize *= 6;
    } // End if

    return textureDataSize;
} // End of NWComputeImageDataSize


//==============================================================================
//
// Get directory name from path
//
//==============================================================================
std::wstring NWGetDirectoryName( const std::wstring& pathName )
{
    WCHAR separators[] = { (WCHAR)'/',
                           (WCHAR)':',
                           (WCHAR)'\\',
                           (WCHAR)NULL };
    int numSeparators = sizeof(separators) / sizeof(WCHAR);

    std::wstring dirName;

    // Find any of separator from end
    int i,j;
    int cutIndex = -1;
    int strLen = (int)pathName.size();
    for (i=strLen-1;i>=0 && cutIndex<0;i--)
    {
        WCHAR ch = pathName[i];
        for (j=0;j<numSeparators;j++)
        {
            if (ch==separators[j])
            {
                cutIndex = i;
                break;
            } // End if
        } // End for
    } // End for

    if (cutIndex<0) // If separators not found
        return dirName;

    dirName.reserve(cutIndex+1);
    for (i=0;i<cutIndex;i++)
    {
        dirName += pathName[i];
    } // End for

    return dirName;
} // End of NWGetDirectoryName


//==============================================================================
//
// Get procedure name
//
//==============================================================================
void* NWGetProcAddress( HMODULE hModule, const char *szProcName )
{
   FARPROC proc = ::GetProcAddress(hModule,szProcName);
   return proc;
} // End of NWGetProcAddress


//==============================================================================
///
/// NWLogOutput
///
//==============================================================================
void NWLogOutput( const WCHAR* szMessage,
                    ... )
{
    static WCHAR  szStr[4096];

    va_list params;
    va_start ( params, szMessage );

    vswprintf_s ( (WCHAR*)szStr, 4096, (WCHAR*)szMessage, params );
    wcscat_s((WCHAR*)szStr,4096,L"\n");

    va_end( params );

    OutputDebugStringW ( (WCHAR*)szStr );
} // End of NWLogOutput


//==============================================================================
//
// Determine if the file is a known ascii intermediate file
//    It determines it by checking file extension
//
//==============================================================================
bool NWIsIntermediateAsciiFile( const WCHAR* filename )
{
    return NWIsIntermediateFile(filename,
        global_IntermediateAsciiFileExtentions,
        (sizeof(global_IntermediateAsciiFileExtentions) / sizeof(*global_IntermediateAsciiFileExtentions)) - 1); // -1 is to skip the NULL termating array element
} // End of NWIsIntermediateAsciiFile


//==============================================================================
//
// Determine if the file is a known binary intermediate file
//    It determines it by checking file extension
//
//==============================================================================
bool NWIsIntermediateBinaryFile( const WCHAR* filename )
{
    return NWIsIntermediateFile(filename,
        global_IntermediateBinaryFileExtentions,
        (sizeof(global_IntermediateBinaryFileExtentions) / sizeof(*global_IntermediateBinaryFileExtentions)) - 1); // -1 is to skip the NULL termating array element
} // End of NWIsIntermediateBinaryFile


//==============================================================================
//
// Determine if the file is a known intermediate file
//    It determines it by checking file extension
//
//==============================================================================
bool NWIsIntermediateFile( const WCHAR* filename, const WCHAR** extensionArray, int extensionArrayLength )
{
    // ensure input parameters are correct
    if (filename == NULL || extensionArray == NULL || extensionArrayLength <= 0)
        return false;

    int fileLength = (int)wcslen(filename);

    for (int i = 0; i < extensionArrayLength; i++)
    {
        const WCHAR* extension = extensionArray[i];
        int extLength = (int)wcslen(extension);

        if (extLength > fileLength - 2) // -2 to exclude the quotes
            continue;

        // point to extension part of the filename
        const WCHAR* fileExtension = filename + fileLength - extLength - 1; // to exclude the closing quote

        if (_wcsnicmp(fileExtension, extension, extLength) == 0)
            return true;
    }

    return false;
} // End of NWIsIntermediateFile

//==============================================================================
//
// Determine the number of channels from the image format
//
//==============================================================================
int NWGetChannelCountFromImageFormat(NW_IMG_FMT format)
{
    switch (format)
    {
        case NW_IMG_FMT_L4: return 1;
        case NW_IMG_FMT_L8: return 1;
        case NW_IMG_FMT_A4: return 1;
        case NW_IMG_FMT_A8: return 1;
        case NW_IMG_FMT_LA4: return 2;
        case NW_IMG_FMT_LA8: return 2;
        case NW_IMG_FMT_HILO8: return 2;
        case NW_IMG_FMT_RGB565: return 3;
        case NW_IMG_FMT_RGB8: return 3;
        case NW_IMG_FMT_RGB5_A1: return 4;
        case NW_IMG_FMT_RGBA4: return 4;
        case NW_IMG_FMT_RGBA8: return 4;
        case NW_IMG_FMT_ETC1: return 3;
        case NW_IMG_FMT_ETCA: return 4;

        // Revolution
        case NW_IMG_FMT_R_I4: return 1;
        case NW_IMG_FMT_R_I8: return 1;
        case NW_IMG_FMT_R_IA4: return 2;
        case NW_IMG_FMT_R_IA8: return 2;
        case NW_IMG_FMT_R_R5G6B5: return 3;
        case NW_IMG_FMT_R_RGB5A3: return 4;
        case NW_IMG_FMT_R_RGBA8: return 4;
        case NW_IMG_FMT_R_C4: return 1;
        case NW_IMG_FMT_R_C8: return 1;
        case NW_IMG_FMT_R_C14: return 1;
        case NW_IMG_FMT_R_CMPR: return 0;

        // For Cafe Format
        case NW_IMG_FMT_GX2_UNORM_8: return 1;
        case NW_IMG_FMT_GX2_UINT_8: return 1;
        case NW_IMG_FMT_GX2_SNORM_8: return 1;
        case NW_IMG_FMT_GX2_SINT_8: return 1;
        case NW_IMG_FMT_GX2_UNORM_4_4: return 2;
        case NW_IMG_FMT_GX2_UNORM_16: return 1;
        case NW_IMG_FMT_GX2_UINT_16: return 1;
        case NW_IMG_FMT_GX2_SINT_16: return 1;
        case NW_IMG_FMT_GX2_FLOAT_16: return 1;
        case NW_IMG_FMT_GX2_UNORM_8_8: return 2;
        case NW_IMG_FMT_GX2_UINT_8_8: return 2;
        case NW_IMG_FMT_GX2_SNORM_8_8: return 2;
        case NW_IMG_FMT_GX2_SINT_8_8: return 2;
        case NW_IMG_FMT_GX2_UNORM_5_6_5: return 3;
        case NW_IMG_FMT_GX2_UNORM_6_5_5: return 3;
        case NW_IMG_FMT_GX2_SNORM_6_5_5: return 3;
        case NW_IMG_FMT_GX2_UNORM_1_5_5_5: return 4;
        case NW_IMG_FMT_GX2_UNORM_4_4_4_4: return 4;
        case NW_IMG_FMT_GX2_UNORM_5_5_5_1: return 4;
        case NW_IMG_FMT_GX2_UINT_32: return 1;
        case NW_IMG_FMT_GX2_SINT_32: return 1;
        case NW_IMG_FMT_GX2_FLOAT_32: return 1;
        case NW_IMG_FMT_GX2_UNORM_16_16: return 2;
        case NW_IMG_FMT_GX2_UINT_16_16: return 2;
        case NW_IMG_FMT_GX2_SNORM_16_16: return 2;
        case NW_IMG_FMT_GX2_SINT_16_16: return 2;
        case NW_IMG_FMT_GX2_FLOAT_16_16: return 2;
        case NW_IMG_FMT_GX2_FLOAT_11_11_10: return 3;
        case NW_IMG_FMT_GX2_UNORM_2_10_10_10: return 4;
        case NW_IMG_FMT_GX2_UNORM_8_8_8_8: return 4;
        case NW_IMG_FMT_GX2_UINT_8_8_8_8: return 4;
        case NW_IMG_FMT_GX2_SNORM_8_8_8_8: return 4;
        case NW_IMG_FMT_GX2_SINT_8_8_8_8: return 4;
        case NW_IMG_FMT_GX2_SRGB_8_8_8_8: return 4;
        case NW_IMG_FMT_GX2_UNORM_10_10_10_2: return 4;
        case NW_IMG_FMT_GX2_UINT_10_10_10_2: return 4;
        case NW_IMG_FMT_GX2_UINT_32_32: return 2;
        case NW_IMG_FMT_GX2_SINT_32_32: return 2;
        case NW_IMG_FMT_GX2_FLOAT_32_32: return 2;
        case NW_IMG_FMT_GX2_UNORM_16_16_16_16: return 4;
        case NW_IMG_FMT_GX2_UINT_16_16_16_16: return 4;
        case NW_IMG_FMT_GX2_SNORM_16_16_16_16: return 4;
        case NW_IMG_FMT_GX2_SINT_16_16_16_16: return 4;
        case NW_IMG_FMT_GX2_FLOAT_16_16_16_16: return 4;
        case NW_IMG_FMT_GX2_UINT_32_32_32_32: return 4;
        case NW_IMG_FMT_GX2_SINT_32_32_32_32: return 4;
        case NW_IMG_FMT_GX2_FLOAT_32_32_32_32: return 4;
        case NW_IMG_FMT_GX2_SHAREDEXP_5_9_9_9: return 4;
        case NW_IMG_FMT_GX2_UINT_32_32_32: return 3;
        case NW_IMG_FMT_GX2_SINT_32_32_32: return 3;
        case NW_IMG_FMT_GX2_FLOAT_32_32_32: return 3;
        case NW_IMG_FMT_GX2_UNORM_BC1: return 3;
        case NW_IMG_FMT_GX2_SRGB_BC1: return 3;
        case NW_IMG_FMT_GX2_UNORM_BC2: return 4;
        case NW_IMG_FMT_GX2_SRGB_BC2: return 4;
        case NW_IMG_FMT_GX2_UNORM_BC3: return 4;
        case NW_IMG_FMT_GX2_SRGB_BC3: return 4;
        case NW_IMG_FMT_GX2_UNORM_BC4: return 4;
        case NW_IMG_FMT_GX2_SNORM_BC4: return 4;
        case NW_IMG_FMT_GX2_UNORM_BC5: return 2;
        case NW_IMG_FMT_GX2_SNORM_BC5: return 2;
    }
    return 0;
}


static volatile LONG g_moduleLockCount = 0;

//==============================================================================
//
// Lock DLL from unloading
//
//==============================================================================
void NWShellLockModule()
{
    ::InterlockedIncrement(&g_moduleLockCount);
} // End of NWShellLockModule


//==============================================================================
//
// Release Lock DLL from unloading
//
//==============================================================================
void NWShellUnlockModule()
{
    ::InterlockedDecrement(&g_moduleLockCount);
} // End of NWShellUnlockModule


//==============================================================================
//
// Wait until lock is free
//
//==============================================================================
void NWShellWaitUnlockModule()
{
    while (::_InterlockedAnd(&g_moduleLockCount,0xFFFFFFFF)!=0)
    {
        ::Sleep(1);
    } // End while
} // End of NWShellWaitUnlockModule

namespace {
    int GetBase64Value(int c)
    {
        if (c >= (int)'A' && c <= (int)'Z')
        {
            return c - (int)'A';
        } // End if
        else if (c >= (int)'a' && c <= (int)'z')
        {
            return c - (int)'a' + 26;
        } // End else if
        else if (c >= (int)'0' && c <= (int)'9')
        {
            return c - (int)'0' + 52;
        } // End else if
        else if ((int)'+' == c)
        {
            return 62;
        } // End else if
        else if ((int)'/' == c)
        {
            return 63;
        } // End else if
        else if ((int)'=' == c)
        {
            return 0;
        } // End else if

        return -1;
    } // End of GetBase64Value for CNWCTexLoader
}
//==============================================================================
//
// Base64 文字列からバイナリへ変換
//
//==============================================================================
std::unique_ptr<unsigned char[]> NWDecodeBase64(const unsigned char* srcBuf, int srcSize, int& dstSize)
{
    //--------------------------------------------------------------------------
    // alloc dst buffer
    //--------------------------------------------------------------------------
    int allocSize = srcSize;
    if (allocSize <= 0)
        return NULL;

    std::unique_ptr<unsigned char[]> dstBuf(new unsigned char[allocSize]);
    if (dstBuf == NULL)
        return NULL;

    //--------------------------------------------------------------------------
    // decode
    //--------------------------------------------------------------------------
    const unsigned char* src = srcBuf;
    const unsigned char* srcEnd = srcBuf + srcSize;
    unsigned char* dst = dstBuf.get();
    while (src < srcEnd)
    {
        int ichar = 0;
        unsigned int base64 = 0;
        while (src < srcEnd)
        {
            int c = *src++;
            if (c == '=')
            {
                break;
            } /// End if

            int val = GetBase64Value(c);
            if (val != -1)
            {
                base64 = (base64 << 6) | val;
                ++ichar;
                if (ichar >= 4)
                {
                    break;
                } // End if
            } // End if
        } // End while

        if (ichar == 0)
        {
            break;
        } // End if

        int outSize = 3;
        if (ichar < 4)
        {
            base64 <<= (4 - ichar) * 6;
            outSize = (ichar == 2) ? 1 : 2;
        } // End if

        for (int idst = 0; idst < outSize; ++idst)
        {
            *dst++ = static_cast<unsigned char>((base64 >> ((2 - idst) * 8)) & 0xff);
        } // End for
    } // End while

      //--------------------------------------------------------------------------
      // End
      //--------------------------------------------------------------------------
    dstSize = static_cast<int>(dst - dstBuf.get());

    return dstBuf;
}


