﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"
#include "ShellExtension_Resource.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_FileData.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_Utility.h"

#ifndef NW_FOR_FXX
#include "Service/Preview/ShellExtension_Preview.h"

#include "ImageLoader/ShellExtension_ImageLoader.h"
#include "ImageLoader/ShellExtension_CTexLoader.h"
#include "ImageLoader/ShellExtension_FTXLoader.h"
#endif // NW_FOR_FXX


//==============================================================================
//
// Implementation of CShellExtensionManager
//
//==============================================================================
CShellExtensionManager::CShellExtensionManager() :
    m_hInstance(NULL),
    m_GDIToken((ULONG_PTR)0),
    m_pFileDataCS(NULL),
    m_pTextureLoadingCS(NULL)
{
} // End of Constructor for CShellExtensionManager


//------------------------------------------------------------------------------
CShellExtensionManager::~CShellExtensionManager()
{
    Deinit();
} // End of Destructor for CShellExtensionManager


//------------------------------------------------------------------------------
// Init and Deinit
//------------------------------------------------------------------------------
bool CShellExtensionManager::Init( HINSTANCE hInstance )
{
    m_pFileDataCS = new CNWCriticalSection();
    m_pTextureLoadingCS = new CNWCriticalSection();

    m_hInstance = hInstance;
    WCHAR modulePathName[MAX_PATH];
    ::GetModuleFileName(m_hInstance,modulePathName,MAX_PATH);
    m_moduleName = modulePathName;

    m_moduleDirectory = NWGetDirectoryName(m_moduleName);

    ::GetModuleFileName(NULL,modulePathName,MAX_PATH);
    std::wstring appPath = modulePathName;

    int lastSeparator = (int)appPath.find_last_of((WCHAR)'/');
    if (lastSeparator<0)
        lastSeparator = (int)appPath.find_last_of((WCHAR)'\\');
    if (lastSeparator>=0)
    {
        if (lastSeparator<(int)appPath.size()-1)
        {
            m_appName = &(appPath.c_str()[lastSeparator+1]);
        } // End if
    } // End if

    InitGDIPlus();

#ifndef NW_FOR_FXX
    CNWCTexLoader::InitBlockToLinearTable();
    CShellExtensionPreview::InitPreview();
#endif // NW_FOR_FXX

#ifdef NW_FOR_FTX
#endif


    // Locale
    setlocale(LC_CTYPE, "");

    return true;
} // End of Init for CShellExtensionManager


//------------------------------------------------------------------------------
void CShellExtensionManager::Deinit()
{
    NWShellWaitUnlockModule();

    ClearAllFileData();
    DeinitGDIPlus();

    if (m_pTextureLoadingCS==NULL)
    {
        delete m_pTextureLoadingCS;
        m_pTextureLoadingCS = NULL;
    } // End if

    if (m_pFileDataCS!=NULL)
    {
        delete m_pFileDataCS;
        m_pFileDataCS = NULL;
    } // End if
} // End of Deinit for CShellExtensionManager


//------------------------------------------------------------------------------
// Check calling app name is equal to given
//------------------------------------------------------------------------------
bool CShellExtensionManager::IsAppName( const WCHAR* szAppName ) const
{
    if (_wcsicmp(m_appName.c_str(),szAppName)==0)
        return true;

    return false;
} // End of IsAppName for CShellExtensionManager


//------------------------------------------------------------------------------
// Initialize and Uninitialize GDI
//------------------------------------------------------------------------------
bool CShellExtensionManager::InitGDIPlus()
{
    if (m_GDIToken!=(ULONG_PTR)0) // Already initialized
        return true;

    Gdiplus::GdiplusStartupInput gdiPlusStartupInput;
    GdiplusStartup(&m_GDIToken,&gdiPlusStartupInput,NULL);

    return true;
} // End of InitGDIPlus for CShellExtensionManager


//------------------------------------------------------------------------------
void CShellExtensionManager::DeinitGDIPlus()
{
    //GdiplusShutdown(g_GDIToken);
    m_GDIToken = (ULONG_PTR)0;
} // End of DeinitGDIPlus for CShellExtensionManager


//------------------------------------------------------------------------------
// Get file data with given file path
//------------------------------------------------------------------------------
CShellExtensionFileData* CShellExtensionManager::GetFileData( const WCHAR* szFileName, const WCHAR *szCaller )
{
    CNWSingleLock lock(m_pFileDataCS);

    CShellExtensionFileData *pData = NULL;

    FileDataMap::iterator itr = m_fileDataMap.find(szFileName);
    if (itr==m_fileDataMap.end())
    {
        pData = new CShellExtensionFileData(this, szFileName);

        m_fileDataMap.insert(FileDataMap::value_type(pData->m_fileName.c_str(),pData));
        pData->AddRef(szCaller);

        return pData;
    } // End if

    pData = (*itr).second;
    pData->AddRef(szCaller);

    return pData;
} // End of GetFileData for CShellExtensionManager

//------------------------------------------------------------------------------
// Remove filedata from map
//------------------------------------------------------------------------------
void CShellExtensionManager::RemoveFileData( CShellExtensionFileData *pData )
{
    CNWSingleLock lock(m_pFileDataCS);

    FileDataMap::iterator itr = m_fileDataMap.find(pData->m_fileName.c_str());
    if (itr==m_fileDataMap.end())
    {
        NW_LOG(L"CShellExtensionManager::RemoveFileData --> Must not reach here!!!");
        return;
    } // End if

    m_fileDataMap.erase(itr);
} // End of RemoveFileData for CShellExtensionManager


//------------------------------------------------------------------------------
// Clear all data
//------------------------------------------------------------------------------
void CShellExtensionManager::ClearAllFileData()
{
    CNWSingleLock lock(m_pFileDataCS);

    FileDataMap::iterator itr = m_fileDataMap.begin();
    while (itr!=m_fileDataMap.end())
    {
        FileDataMap::iterator nextItr = itr;
        ++nextItr;

        CShellExtensionFileData *pData = (*itr).second;
        pData->m_pManager = NULL;

        m_fileDataMap.erase(itr);

        itr = nextItr;
    } // End while
} // End of ClearAllFileData for CShellExtensionManager


//------------------------------------------------------------------------------
// Accessor for Singleton Instance
//------------------------------------------------------------------------------
CShellExtensionManager* CShellExtensionManager::Instance()
{
    static CShellExtensionManager myManager;
    return &myManager;
} // End of Instance for CShellExtensionManager

