﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShellExtension_PCH.h"

#include "ShellExtension_Resource.h"
#include "ShellExtension_Type.h"
#include "ShellExtension_SyncObject.h"
#include "ShellExtension_Texture.h"
#include "ShellExtension_FileData.h"
#include "ShellExtension_Manager.h"
#include "ShellExtension_Utility.h"

#include "Service/InfoTip/ShellExtension_InfoTip.h"

//==============================================================================
//
// CShellExtensionInfoTip Implementation
//
//==============================================================================
CShellExtensionInfoTip::CShellExtensionInfoTip() :
    m_pFileData(NULL)
{
    NWShellLockModule();
} // End of Constructor for CShellExtensionInfoTip


//------------------------------------------------------------------------------
CShellExtensionInfoTip::~CShellExtensionInfoTip()
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"InfoTip");
        m_pFileData = NULL;
    } // End if

    NWShellUnlockModule();
} // End of Destructor for CShellExtensionInfoTip


//------------------------------------------------------------------------------
// Load
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionInfoTip::Load( LPCOLESTR wszFile, DWORD dwMode )
{
    if (m_pFileData!=NULL)
    {
        m_pFileData->Release(L"InfoTip::Load");
        m_pFileData = NULL;
    } // End if

    NW_USE_VAR(dwMode);

    USES_CONVERSION;
    const WCHAR* szFileName = W2CT(wszFile);
    m_pFileData = CShellExtensionManager::Instance()->GetFileData(szFileName,L"InfoTip");
    if (m_pFileData==NULL)
        return E_FAIL;

    return S_OK;
} // End of Load for CShellExtensionInfoTip


//------------------------------------------------------------------------------
// Get InfoTip Flag
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionInfoTip::GetInfoFlags(DWORD *pdwFlags)
{
    NW_USE_VAR(pdwFlags);

    return E_NOTIMPL;
} // End of GetInfoTip for CShellExtensionInfoTip


//------------------------------------------------------------------------------
// Get InfoTip
//------------------------------------------------------------------------------
STDMETHODIMP CShellExtensionInfoTip::GetInfoTip(DWORD dwFlags, LPWSTR *ppwszTip)
{
    NW_USE_VAR(dwFlags);
    NW_USE_VAR(ppwszTip);

    if (m_pFileData==NULL)
        return E_FAIL;

    CNWSingleLock lock(m_pFileData->GetCriticalSection());

    CNWTexture *pTexImage = m_pFileData->GetTexture();
    if (pTexImage==NULL)
    {
        if (m_pFileData->UpdateImage(false)==false)
            return S_FALSE;

        pTexImage = m_pFileData->GetTexture();
    } // End if

    IMalloc* pMalloc = NULL;
    if ( FAILED(SHGetMalloc(&pMalloc)) )
        return E_FAIL;

    auto deleter = [](IMalloc* x) { x->Release();  };
    std::unique_ptr<IMalloc, decltype(deleter)> p(pMalloc, deleter);

    if (ppwszTip!=NULL)
    {
        WCHAR szTemp[1024] = {0};
        WCHAR* sxPtr = szTemp;

        int numOutput       = 0;
        size_t curStrPos    = 0;
        size_t maxStrCount  = 0;
        size_t maxOneStrLen = 128;

        // Comment
        std::wstring commentText(pTexImage->GetComment());
        if (!commentText.empty())
        {
            if (numOutput>0)
            {
                maxStrCount = (1023 - curStrPos);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
                curStrPos = wcslen(sxPtr);
            }

            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Comment : %s",
                         commentText.c_str());
            curStrPos = wcslen(sxPtr);

            numOutput++;
        } // End if

        // Lay Comp
        if (pTexImage->GetLayerCompsSize() > 0)
        {
            if (numOutput>0)
            {
                maxStrCount = 1023 - (int)(sxPtr - szTemp);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
                curStrPos = wcslen(sxPtr);
            }

            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"LayerComps : %d",
                                (int)pTexImage->GetLayerCompsSize());
            curStrPos = wcslen(sxPtr);

            numOutput++;
        } // End if

        // Format
        if (numOutput>0)
        {
            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
            curStrPos = wcslen(sxPtr);
        }

        maxStrCount = 1023 - (int)(sxPtr - szTemp);
        maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
        _snwprintf_s(sxPtr+curStrPos, maxStrCount, _TRUNCATE, L"Format : %s", pTexImage->GetFormatName());
        curStrPos = wcslen(sxPtr);

        if ( ( (pTexImage->GetFormat()==NW_IMG_FMT_ETC1) ||
                (pTexImage->GetFormat()==NW_IMG_FMT_ETCA) ) &&
             (pTexImage->GetCompressTypeText().size()>0) )
        {
            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos, maxStrCount, _TRUNCATE, L"(%s)", pTexImage->GetCompressTypeText().c_str());
            curStrPos = wcslen(sxPtr);
        } // End if

        // Size
        maxStrCount = 1023 - (int)(sxPtr - szTemp);
        maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
        _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
        curStrPos = wcslen(sxPtr);

        switch (pTexImage->GetTextureType())
        {
            case NW_IMG_TYPE_1D :
            case NW_IMG_TYPE_1D_ARRAY :
                maxStrCount = 1023 - (int)(sxPtr - szTemp);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Size : %d",
                        (int)pTexImage->GetWidth());
                curStrPos = wcslen(sxPtr);
                break;

            case NW_IMG_TYPE_2D :
            case NW_IMG_TYPE_2D_ARRAY :
            case NW_IMG_TYPE_CUBE :
            case NW_IMG_TYPE_CUBE_ARRAY :
                maxStrCount = 1023 - (int)(sxPtr - szTemp);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Size : %d x %d",
                        (int)pTexImage->GetWidth(),
                        (int)pTexImage->GetHeight());
                curStrPos = wcslen(sxPtr);
                break;

            case NW_IMG_TYPE_3D :
            case NW_IMG_TYPE_3D_ARRAY :
                maxStrCount = 1023 - (int)(sxPtr - szTemp);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Size : %d x %d x %d",
                        (int)pTexImage->GetWidth(),
                        (int)pTexImage->GetHeight(),
                        (int)pTexImage->GetDepth());
                curStrPos = wcslen(sxPtr);
                break;
            default:
                szTemp[0] = L'\0';
                break;
        } // End switch

        WCHAR szTempStr[256];

        // Mipmaps
        if (pTexImage->HasMipmaps())
        {
            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
            curStrPos = wcslen(sxPtr);

            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Mipmaps : %d", (int)pTexImage->GetNumMipmaps());
            curStrPos = wcslen(sxPtr);
        } // end

        // Data Size
        NW4FormatNumberStringWithComma(szTempStr,256,pTexImage->GetTextureDataSize());
        maxStrCount = 1023 - (int)(sxPtr - szTemp);
        maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
        _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
        curStrPos = wcslen(sxPtr);

        maxStrCount = 1023 - (int)(sxPtr - szTemp);
        maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
        _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Data Size : %s Bytes", szTempStr);
        curStrPos = wcslen(sxPtr);

        // Comp Sel
        if (pTexImage->IsForNW4F())
        {
            if (pTexImage->HasCompSel())
            {
                maxStrCount = 1023 - (int)(sxPtr - szTemp);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
                curStrPos = wcslen(sxPtr);

                const NW_COMP_SEL &compSel = pTexImage->GetCompSel();
                std::wstring compSelStr = NWCompSelToString(compSel);

                maxStrCount = 1023 - (int)(sxPtr - szTemp);
                maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
                _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"%s", compSelStr.c_str());
                curStrPos = wcslen(sxPtr);
            } // End if
        } // End if

        // Linear
        std::wstring linearText;
        linearText.reserve(5);

        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_R))
            linearText += L'R';
        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_G))
            linearText += L'G';
        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_B))
            linearText += L'B';
        if (pTexImage->GetUILinearFlag(NW_COMP_SEL_ELEMENT_A))
            linearText += L'A';

        maxStrCount = 1023 - (int)(sxPtr - szTemp);
        maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
        _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
        curStrPos = wcslen(sxPtr);

        maxStrCount = 1023 - (int)(sxPtr - szTemp);
        maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
        _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Linear : %s",
                          linearText.c_str());
        curStrPos = wcslen(sxPtr);

        // Hint
        std::wstring hintText(pTexImage->GetHint());
        if (!hintText.empty())
        {
            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"\n");
            curStrPos = wcslen(sxPtr);

            maxStrCount = 1023 - (int)(sxPtr - szTemp);
            maxStrCount = min(maxOneStrLen,max(0,maxStrCount));
            _snwprintf_s(sxPtr+curStrPos,maxStrCount,_TRUNCATE,L"Hint : %s",
                              hintText.c_str());
            curStrPos = wcslen(sxPtr);
        } // End if

        int strLen = (int)wcslen(szTemp);
        *ppwszTip = (LPWSTR)pMalloc->Alloc((strLen + 1) * sizeof(WCHAR));
        wcscpy_s(*ppwszTip, strLen+1, szTemp);
    } // End if

    return S_OK;
} // End of GetInfoTip for CShellExtensionInfoTip


//------------------------------------------------------------------------------
// Register
//------------------------------------------------------------------------------
bool CShellExtensionInfoTip::RegisterService( REGISTRATION_PARAM &param )
{
    WCHAR szGUIDStr[128];
    WCHAR szValStr[128];

    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionInfoTip).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open( HKEY_CLASSES_ROOT,
                       _T("CLSID"),
                       KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        _snwprintf_s(szValStr,127,_TRUNCATE,_T("NintendoWare %s Infotip Shell Extension"), param.fileTypeName.c_str() );

        CRegKey GUIDKey;
        if (GUIDKey.Create(CLSIDKey.m_hKey,szGUIDStr)==ERROR_SUCCESS)
        {
            GUIDKey.SetStringValue(NULL,szValStr);

            //------------------------------------------------------------------
            // Inproc server
            //------------------------------------------------------------------
            CRegKey InpProc32Key;
            if (InpProc32Key.Create(GUIDKey.m_hKey,L"InprocServer32")==ERROR_SUCCESS)
            {
                InpProc32Key.SetStringValue(NULL,param.moduleName.c_str());
                InpProc32Key.SetStringValue(L"ThreadingModel", L"Apartment");
                InpProc32Key.Close();
            } // End if

            GUIDKey.Close();
        } // End if

        CLSIDKey.Close();
    } // End if

    //--------------------------------------------------------------------------
    // ProgID
    //--------------------------------------------------------------------------
    int i;
    for (i=0;i<(int)param.supportedExtensions.size();i++)
    {
        CRegKey progIDKey;
        if (progIDKey.Open( HKEY_CLASSES_ROOT,
                            param.progIDNames[i].c_str(),
                            KEY_SET_VALUE )==ERROR_SUCCESS)
        {
            //------------------------------------------------------------------
            // ShellEx Icon
            //------------------------------------------------------------------
            CRegKey shellExKey;
            if (shellExKey.Open(progIDKey.m_hKey,
                                L"ShellEx",
                                KEY_SET_VALUE)==ERROR_SUCCESS)
            {
                //--------------------------------------------------------------
                // ShellEx Icon
                //--------------------------------------------------------------
                CRegKey handlerKey;
                if (handlerKey.Create(shellExKey.m_hKey,L"{00021500-0000-0000-C000-000000000046}")==ERROR_SUCCESS)
                {
                    handlerKey.SetStringValue(NULL,szGUIDStr);
                    handlerKey.Close();
                } // End if

                shellExKey.Close();
            } // End if

            progIDKey.Close();
        } // End if
    } // End for

    return true;
} // End of RegisterService for CShellExtensionInfoTip


//------------------------------------------------------------------------------
// Unregister
//------------------------------------------------------------------------------
bool CShellExtensionInfoTip::UnregisterService( REGISTRATION_PARAM &param )
{
    NW_USE_VAR(param.supportedExtensions.size());

    WCHAR szGUIDStr[128];
    _snwprintf_s(szGUIDStr,127,_TRUNCATE,L"{%s}", NWCreateUUIDString(CLSID_ShellExtensionInfoTip).c_str());

    //--------------------------------------------------------------------------
    // CLSID
    //--------------------------------------------------------------------------
    CRegKey CLSIDKey;
    if (CLSIDKey.Open ( HKEY_CLASSES_ROOT,
                        _T("CLSID"),
                        KEY_SET_VALUE )==ERROR_SUCCESS)
    {
        //----------------------------------------------------------------------
        // My Interface ID
        //----------------------------------------------------------------------
        CLSIDKey.RecurseDeleteKey(szGUIDStr);

        CLSIDKey.Close();
    } // End if

    return true;
} // End of UnregisterService for CShellExtensionInfoTip
