﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;
using Microsoft.Win32;

namespace NintendoWare.ExtensionManager
{
    //==========================================================================
    /// <summary>
    /// MainForm class
    /// </summary>
    //==========================================================================
    public partial class MainForm : Form
    {
        //----------------------------------------------------------------------
        /// <summary>
        /// Constructor
        /// </summary>
        //----------------------------------------------------------------------
        public MainForm()
        {
            InitializeComponent();

            m_windowTitle = this.Text; // Cache window title here

            ApplyWindowPositionFromConfig();

            InitEditingExtensionTree();

            RefreshUI();

        }

        #region MainForm Config
        //----------------------------------------------------------------------
        /// <summary>
        /// Apply size and location from config
        /// </summary>
        //----------------------------------------------------------------------
        protected void ApplyWindowPositionFromConfig()
        {
            MainApp app = MainApp.Instance;
            if (app == null)
                return;

            ConfigManager configManager = app.ConfigManager;
            if ((configManager != null) && (configManager.SessionConfig != null))
            {
                System.Drawing.Point loc = this.Location;
                System.Drawing.Size size = this.Size;

                SessionConfig sessionConfig = configManager.SessionConfig;
                if (sessionConfig != null)
                {
                    Utility.WindowConfig mainFormConfig = sessionConfig.MainFormConfig;

                    // Ignore size
                    mainFormConfig.width  = -1;
                    mainFormConfig.height = -1;

                    Utility.ApplyWindowConfig(this, mainFormConfig, true);
                }
            }
        }
        #endregion

        #region Refresh UI
        private bool m_bInsideRefreshUI = false;

        //----------------------------------------------------------------------
        /// <summary>
        /// Refresh UI
        /// </summary>
        //----------------------------------------------------------------------
        public void RefreshUI()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo registrationInfo = registrationManager.EditingRegistration;
            if (registrationInfo == null)
            {
                m_bInsideRefreshUI = true;

                this.m_usePlugIn_Icon.Enabled          = false;
                this.m_usePlugIn_Preview.Enabled       = false;
                this.m_usePlugIn_InfoTip.Enabled       = false;

                this.m_installAllButton.Enabled        = false;
                this.m_uninstallAllButton.Enabled      = false;

                this.m_browseApplicationButton.Enabled = false;
                this.m_applicationPathName.Enabled     = false;

                this.m_applicationListBox.Enabled      = false;

                this.m_applicationGroup.Enabled = false;

                m_bInsideRefreshUI = false;
                return;
            }

            m_bInsideRefreshUI = true;

            UpdateRegisteredApplicationBox();

            this.m_installAllButton.Enabled   = true;
            this.m_uninstallAllButton.Enabled = true;

            this.m_usePlugIn_Icon.Enabled        = GetEnabledForPlugInService(ShellExtensionService.IconService, registrationInfo);
            this.m_usePlugIn_Preview.Enabled     = GetEnabledForPlugInService(ShellExtensionService.PreviewService, registrationInfo);
            this.m_usePlugIn_InfoTip.Enabled     = GetEnabledForPlugInService(ShellExtensionService.InfoTipService, registrationInfo);
            this.m_applicationGroup.Enabled = GetEnabledForPlugInService(ShellExtensionService.ShellCommandService, registrationInfo);

            this.m_usePlugIn_Icon.CheckState        = GetCheckStateForPlugInService(ShellExtensionService.IconService);
            this.m_usePlugIn_Preview.CheckState     = GetCheckStateForPlugInService(ShellExtensionService.PreviewService);
            this.m_usePlugIn_InfoTip.CheckState     = GetCheckStateForPlugInService(ShellExtensionService.InfoTipService);

            this.m_applicationListBox.Enabled = true;

            int curAppIndex = this.m_applicationListBox.SelectedIndex;
            this.m_browseApplicationButton.Enabled = curAppIndex == m_customAppIndex;
            this.m_applicationPathName.Enabled = curAppIndex == m_customAppIndex;

            m_bInsideRefreshUI = false;
        }

        public void CheckAllInstall()
        {
            OnInstallAll(this, null);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Get enabled flag for plugin service
        /// </summary>
        //----------------------------------------------------------------------
        private bool GetEnabledForPlugInService(ShellExtensionService service,
                                                RegistrationInfo registrationInfo)
        {
            if (service == ShellExtensionService.None)
                return true; // a service is considered as available by default

            return registrationInfo.IsServiceAvailable(service);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Get check status for plugin service
        /// </summary>
        //----------------------------------------------------------------------
        CheckState GetCheckStateForPlugInService(ShellExtensionService service)
        {
            CheckState state = CheckState.Unchecked;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return state;

            if (m_activeExtensionTreeNodeData == null)
            {
                return state;
            }

            if (m_activeExtensionTreeNodeData.ForAllFiles)
            {
                var values = (from info in registrationManager.RegistrationInfo
                              where info.IsServiceAvailable(service) && !info.RemoveCustomExtension
                              select info.CurrentStatus.IsUsingService(service))
                              .Distinct().ToArray();

                if (values.Length == 0)
                {
                    return state;
                }

                if (values.Length == 1)
                {
                    state = values[0] ? CheckState.Checked : CheckState.Unchecked;
                }
                else
                {
                    state = CheckState.Indeterminate;
                }
            }
            else
            {
                RegistrationInfo registrationInfo = registrationManager.EditingRegistration;
                if (registrationInfo == null)
                    return state;

                if (registrationInfo.IsServiceAvailable(service) && registrationInfo.CurrentStatus.IsUsingService(service))
                    state = CheckState.Checked;
                else
                    state = CheckState.Unchecked;
            }

            return state;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Get shell command for displaying in UI
        /// </summary>
        //----------------------------------------------------------------------
        private void GetShellCommandForUI( ref String commandName, ref String commandPath )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            commandName = "";
            commandPath = "";

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            if (m_activeExtensionTreeNodeData == null)
            {
                return;
            }

            if (m_activeExtensionTreeNodeData.ForAllFiles)
            {
                if (registrationManager.NumRegistrationInfos <= 0)
                    return;

                int i;
                RegistrationInfo regInfo1st = registrationManager.GetRegistrationInfo(0);

                commandName = regInfo1st.CurrentStatus.ShellCommandName;
                commandPath = regInfo1st.CurrentStatus.ShellCommandPath;

                // Check if true or false is not consistent, then set it to intermediate
                for (i = 1; i < registrationManager.NumRegistrationInfos; i++)
                {
                    RegistrationInfo registrationInfo = registrationManager.GetRegistrationInfo(i);
                    if (commandName != registrationInfo.CurrentStatus.ShellCommandName)
                    {
                        // Reset to nothing if different
                        commandName = "";
                        commandPath = "";
                        break;
                    }
                }
            }
            else
            {
                RegistrationInfo registrationInfo = registrationManager.EditingRegistration;
                if (registrationInfo == null)
                    return;

                commandName = registrationInfo.CurrentStatus.ShellCommandName;
                commandPath = registrationInfo.CurrentStatus.ShellCommandPath;
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Refresh WindowTitle
        /// </summary>
        //----------------------------------------------------------------------
        private String m_windowTitle = "";

        public void RefreshWindowTitle()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            if (registrationManager.IsSavingRequired())
                this.Text = m_windowTitle + " (*)";
            else
                this.Text = m_windowTitle;
        }
        #endregion

        #region Active Extension Tree
        //----------------------------------------------------------------------
        /// <summary>
        /// ExtensionTreeNodeData
        /// </summary>
        //----------------------------------------------------------------------
        class ExtensionTreeNodeData
        {
            public ExtensionTreeNodeData( bool bAll )
            {
                m_bAll = bAll;
                DisplayName = "";
            }

            public String DisplayName
            {
                set;
                get;
            }

            public bool ForAllFiles
            {
                get { return m_bAll; }
            }

            public List<String> ExtensionList
            {
                get { return m_extensionList; }
            }

            private bool         m_bAll = false;
            private List<String> m_extensionList = new List<String>();
        };

        private ExtensionTreeNodeData m_activeExtensionTreeNodeData = null;

        //----------------------------------------------------------------------
        /// <summary>
        /// Init extension control tree
        /// </summary>
        //----------------------------------------------------------------------
        private void InitEditingExtensionTree()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);


            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            TreeNode rootNode = m_extensionTree.Nodes.Add("All Files");

            // All
            {
                ExtensionTreeNodeData data = new ExtensionTreeNodeData(true);
                data.DisplayName = "All Files";

                rootNode.Text = data.DisplayName;
                rootNode.Tag  = data;
            }

            var items = new []
            {
                new ExtensionTreeNodeData(false) { DisplayName = "FTX", ExtensionList = { "FTXA", "FTXB" } },
                new ExtensionTreeNodeData(false) { DisplayName = "TGA", ExtensionList = { "TGA" } },
                new ExtensionTreeNodeData(false) { DisplayName = "PSD", ExtensionList = { "PSD" } },
                new ExtensionTreeNodeData(false) { DisplayName = "AI", ExtensionList = { "AI" } },
                new ExtensionTreeNodeData(false) { DisplayName = "EPS", ExtensionList = { "EPS" } },
                new ExtensionTreeNodeData(false) { DisplayName = "ESET", ExtensionList = { "ESET" } },
                new ExtensionTreeNodeData(false) { DisplayName = "FLYT", ExtensionList = { "FLYT" } },
                new ExtensionTreeNodeData(false) { DisplayName = "FMD", ExtensionList = { "FMDA", "FMDB" } },
            }
            .Concat(registrationManager.RegistrationInfo
                .Where(x => x.IsCustomExtension && !x.RemoveCustomExtension)
                .Select(x => new ExtensionTreeNodeData(false) {
                    DisplayName = x.ExtensionName.ToUpper(),
                    ExtensionList = { x.ExtensionName.ToUpper() } })
                .OrderBy(x => x.DisplayName));

            foreach (var data in items)
            {
                rootNode.Nodes.Add(new TreeNode()
                {
                    Text = data.DisplayName,
                    Tag = data,
                });
            }

            rootNode.ExpandAll();

            m_extensionTree.SelectedNode = rootNode;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Copy current registration status to given extension name
        /// </summary>
        //----------------------------------------------------------------------
        private void CopyRegistrationStatus( ExtensionTreeNodeData data,
                                             ShellExtensionService service)
        {
            if (data==null)
                return;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo curRegInfo = registrationManager.EditingRegistration;

            if (data.ForAllFiles)
            {
                int i;
                for (i = 0; i < registrationManager.NumRegistrationInfos; i++)
                {
                    RegistrationInfo regInfo = registrationManager.GetRegistrationInfo(i);
                    if (regInfo == curRegInfo)
                        continue;

                    regInfo.CurrentStatus.CopyFrom(curRegInfo.CurrentStatus, service, regInfo);
                }
            }
            else
            {
                foreach (String extStr in data.ExtensionList)
                {
                    RegistrationInfo regInfo = registrationManager.FindRegistrationInfo(extStr);
                    if (regInfo == curRegInfo)
                        continue;

                    regInfo.CurrentStatus.CopyFrom(curRegInfo.CurrentStatus, service, regInfo);
                }
            }
        }
        #endregion

        #region Application
        private int m_noneAppIndex   = 0;
        private int m_resetAppIndex  = 0;
        private int m_customAppIndex = 0;

        //----------------------------------------------------------------------
        /// <summary>
        /// Application Box
        /// </summary>
        //----------------------------------------------------------------------
        private void PopulateRegisteredApplicationBox()
        {
            if (m_activeExtensionTreeNodeData == null)
            {
                return;
            }

            m_bInsideRefreshUI = true;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            this.m_applicationListBox.Items.Clear();

            m_noneAppIndex  = this.m_applicationListBox.Items.Count;
            m_resetAppIndex = m_noneAppIndex+1;
            this.m_applicationListBox.Items.Add(Properties.Resources.TEXT_NONE);
            this.m_applicationListBox.Items.Add(Properties.Resources.TEXT_RESET);

            if (m_activeExtensionTreeNodeData.ForAllFiles)
            {
                int i;
                for (i = 0; i < registrationManager.NumRegisteredApplications; i++)
                {
                    RegistrationManager.AppInfo appInfo = registrationManager.GetRegisteredApplicationInfo(i);

                    // For "All Files", only application that can be assigned to ALL extensions are added to combo box
                    bool bSupported = true;
                    int j;
                    for (j = 0; j < registrationManager.NumRegistrationInfos; j++)
                    {
                        RegistrationInfo thisRegInfo = registrationManager.GetRegistrationInfo(j);
                        if (appInfo.IsExtensionSupported(thisRegInfo.ExtensionName) == false)
                        {
                            // This should not be added to combobox
                            bSupported = false;
                            break;
                        }
                    }

                    if (bSupported)
                    {
                        this.m_applicationListBox.Items.Add(appInfo.Name);
                    }
                }
            }
            else
            {
                int i;
                for (i = 0; i < registrationManager.NumRegisteredApplications; i++)
                {
                    RegistrationManager.AppInfo appInfo = registrationManager.GetRegisteredApplicationInfo(i);
                    if (appInfo.IsExtensionSupported(regInfo.ExtensionName))
                    {
                        this.m_applicationListBox.Items.Add(appInfo.Name);
                    }
                }
            }

            m_customAppIndex = this.m_applicationListBox.Items.Count;
            this.m_applicationListBox.Items.Add(Properties.Resources.TEXT_CUSTOM);

            this.m_applicationListBox.SelectedIndex = 0;

            UpdateRegisteredApplicationBox();

            m_bInsideRefreshUI = false;
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Update registered application box
        /// </summary>
        //----------------------------------------------------------------------
        private void UpdateRegisteredApplicationBox()
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            String commandName = "";
            String commandPath = "";

            GetShellCommandForUI(ref commandName, ref commandPath);

            if (String.Compare(commandName, Properties.Resources.TEXT_NONE, StringComparison.CurrentCultureIgnoreCase) == 0)
            {
                this.m_browseApplicationButton.Enabled = false;
                this.m_applicationPathName.Enabled     = false;
                this.m_applicationPathName.Text        = "";

                if (this.m_applicationListBox.Items.Count>0)
                    this.m_applicationListBox.SelectedIndex = m_noneAppIndex;
            }
            else if (String.Compare(commandName, Properties.Resources.TEXT_RESET, StringComparison.CurrentCultureIgnoreCase) == 0)
            {
                this.m_browseApplicationButton.Enabled = false;
                this.m_applicationPathName.Enabled     = false;
                this.m_applicationPathName.Text        = "";

                if (this.m_applicationListBox.Items.Count > 0)
                    this.m_applicationListBox.SelectedIndex = m_resetAppIndex;
            }
            else if (String.Compare(commandName, Properties.Resources.TEXT_CUSTOM, StringComparison.CurrentCultureIgnoreCase) == 0)
            {
                if (this.m_applicationListBox.Items.Count > 0)
                    this.m_applicationListBox.SelectedIndex = m_customAppIndex;

                this.m_browseApplicationButton.Enabled = true;
                this.m_applicationPathName.Enabled     = true;
                this.m_applicationPathName.Text        = commandPath;
            }
            else
            {
                if (commandName.Length > 0)
                {
                    int appInfoIndex = registrationManager.FindRegisterAppName(regInfo.CurrentStatus.ShellCommandName);
                    if (appInfoIndex >= 0)
                    {
                        RegistrationManager.AppInfo appInfo = registrationManager.GetRegisteredApplicationInfo(appInfoIndex);

                        // Search in ComboBox ( ignore indices for None and Custom )
                        int appComboBoxIndex = -1;
                        int i;
                        for (i = 1; i < m_applicationListBox.Items.Count - 1; i++)
                        {
                            if (String.Compare(m_applicationListBox.Items[i] as String, appInfo.Name, StringComparison.CurrentCultureIgnoreCase) == 0)
                            {
                                appComboBoxIndex = i;
                                break;
                            }
                        }

                        if (appComboBoxIndex >= 0)
                        {
                            this.m_browseApplicationButton.Enabled = false;
                            this.m_applicationPathName.Enabled     = false;
                            this.m_applicationPathName.Text        = "";

                            this.m_applicationListBox.SelectedIndex = appComboBoxIndex;
                        }
                    }
                    else
                    {
                        this.m_browseApplicationButton.Enabled = false;
                        this.m_applicationPathName.Enabled     = false;
                        this.m_applicationPathName.Text        = "";

                        this.m_applicationListBox.SelectedIndex = -1;
                    }
                }
                else
                {
                    this.m_browseApplicationButton.Enabled = false;
                    this.m_applicationPathName.Enabled     = false;
                    this.m_applicationPathName.Text        = "";

                    this.m_applicationListBox.SelectedIndex = -1;
                }
            }
        }
        #endregion

        #region  Event Handling

        /// <summary>
        /// 選択変更
        /// </summary>
        private void m_extensionTree_AfterSelect(object sender, TreeViewEventArgs e)
        {
            if (m_bInsideRefreshUI)
                return;

            ExtensionTreeNodeData data = e.Node.Tag as ExtensionTreeNodeData;
            if (data == null)
                return;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            if (data.ExtensionList.Count > 0)
            {
                registrationManager.SetEditingRegistration(data.ExtensionList[0]);
            }
            else
            {
                if (registrationManager.NumRegistrationInfos <= 0)
                    return;

                registrationManager.SetEditingRegistration(registrationManager.GetRegistrationInfo(0).ExtensionName);
            }

            m_activeExtensionTreeNodeData = data;

            PopulateRegisteredApplicationBox();

            RefreshUI();
        }

        #region  Check boxes
        //----------------------------------------------------------------------
        private void OnUseIconService(object sender, EventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            bool bEnabled = m_usePlugIn_Icon.Checked;
            if (regInfo.CurrentStatus.UseIconService != bEnabled)
                regInfo.CurrentStatus.UseIconService = bEnabled;

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.IconService);

            RefreshWindowTitle();
        }

        //----------------------------------------------------------------------
        private void OnUseInfoTipService(object sender, EventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            bool bEnabled = m_usePlugIn_InfoTip.Checked;
            if (regInfo.CurrentStatus.UseInfoTipService != bEnabled)
                regInfo.CurrentStatus.UseInfoTipService = bEnabled;

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.InfoTipService);

            RefreshWindowTitle();
        }

        //----------------------------------------------------------------------
        private void OnUsePreviewService(object sender, EventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            bool bEnabled = m_usePlugIn_Preview.Checked;
            if (regInfo.CurrentStatus.UsePreviewService != bEnabled)
                regInfo.CurrentStatus.UsePreviewService = bEnabled;

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.PreviewService);

            RefreshWindowTitle();
        }

        #endregion

        #region  Application choosing
        //----------------------------------------------------------------------
        /// <summary>
        /// Handle application selection in the list
        /// </summary>
        //----------------------------------------------------------------------
        private void OnSelectApplication(object sender, EventArgs e)
        {
            if (m_bInsideRefreshUI)
                return;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            int selected = m_applicationListBox.SelectedIndex;
            if ((selected < 0) || (selected >= m_applicationListBox.Items.Count))
                return;

            String appPath = "";
            if (selected == m_noneAppIndex) // None?
            {
                this.m_browseApplicationButton.Enabled = false;
                this.m_applicationPathName.Enabled     = false;

                regInfo.CurrentStatus.ShellCommandName = Properties.Resources.TEXT_NONE;
                regInfo.CurrentStatus.ShellCommandPath = "";
            }
            else if (selected == m_resetAppIndex) // None?
            {
                this.m_browseApplicationButton.Enabled = false;
                this.m_applicationPathName.Enabled     = false;

                regInfo.CurrentStatus.ShellCommandName = Properties.Resources.TEXT_RESET;
                regInfo.CurrentStatus.ShellCommandPath = "";
            }
            else if (selected < m_customAppIndex)
            {
                // -2 since first two are None and Reset
                if ((selected-2) < registrationManager.NumRegisteredApplications)
                {
                    int appInfoIndex = registrationManager.FindRegisterAppName(m_applicationListBox.Items[selected] as String);

                    RegistrationManager.AppInfo appInfo = registrationManager.GetRegisteredApplicationInfo(appInfoIndex);

                    regInfo.CurrentStatus.ShellCommandName = appInfo.Name;
                    regInfo.CurrentStatus.ShellCommandPath = appInfo.Path;
                }

                this.m_browseApplicationButton.Enabled = false;
                this.m_applicationPathName.Enabled     = false;
            }
            else
            {
                this.m_browseApplicationButton.Enabled = true;
                this.m_applicationPathName.Enabled     = true;

                regInfo.CurrentStatus.ShellCommandName = Properties.Resources.TEXT_CUSTOM;
                regInfo.CurrentStatus.ShellCommandPath = "";
            }

            m_bInsideRefreshUI = true; // Make sure changing m_applicationPathName does not cause status change
            m_applicationPathName.Text = appPath;
            m_bInsideRefreshUI = false;

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.ShellCommandService);

            RefreshWindowTitle();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle browse command
        /// </summary>
        //----------------------------------------------------------------------
        private void OnBrowse(object sender, EventArgs e)
        {
            if (m_bInsideRefreshUI)
                return;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            OpenFileDialog browseDialog = new OpenFileDialog();

            browseDialog.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.ProgramFiles);
            browseDialog.Title = "Open with...";
            browseDialog.Filter = "Programs (*.exe;*.pif;*.com;*.bat)|*.exe;*.pif;*.com;*.bat|All files (*.*)|*.*";
            browseDialog.FilterIndex = 1;
            browseDialog.RestoreDirectory = true;

            if (browseDialog.ShowDialog() == DialogResult.OK)
            {
                string shellCommand = browseDialog.FileName;

                m_applicationListBox.SelectedIndex = m_customAppIndex;
                m_applicationPathName.Text = shellCommand;

                regInfo.CurrentStatus.ShellCommandName = Properties.Resources.TEXT_CUSTOM;
                regInfo.CurrentStatus.ShellCommandPath = shellCommand;

                CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.ShellCommandService);

                RefreshWindowTitle();
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle application path
        /// </summary>
        //----------------------------------------------------------------------
        private void OnDefineApplicationPath(object sender, EventArgs e)
        {
            if (m_bInsideRefreshUI)
                return;

            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            String shellCommand = m_applicationPathName.Text;

            regInfo.CurrentStatus.ShellCommandName = Properties.Resources.TEXT_CUSTOM;
            regInfo.CurrentStatus.ShellCommandPath = shellCommand;

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.ShellCommandService);

            RefreshWindowTitle();
        }
        #endregion

        #region  Apply / Cancel / Close
        //----------------------------------------------------------------------
        /// <summary>
        /// Handle closing
        /// </summary>
        //----------------------------------------------------------------------
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            app.OnExitingApp();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle apply
        /// </summary>
        //----------------------------------------------------------------------
        private void OnApply( object sender, EventArgs e )
        {
            MainApp app = MainApp.Instance;
            Debug.Assert( app != null );

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            registrationManager.Apply(false);
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle Ok
        /// </summary>
        //----------------------------------------------------------------------
        private void OnOk(object sender, EventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            if (app.SetupRegistraition(false, false, MainApp.SetupType.GUI) != ReturnCode.Success)
                return;

            Close();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle cancel
        /// </summary>
        //----------------------------------------------------------------------
        private void OnCancel( object sender, EventArgs e )
        {
            Close();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle Install all
        /// </summary>
        //----------------------------------------------------------------------
        private void OnInstallAll(object sender, EventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            if (m_usePlugIn_Icon.Enabled)
            {
                regInfo.CurrentStatus.UseIconService = true;
            }

            if (m_usePlugIn_InfoTip.Enabled)
            {
                regInfo.CurrentStatus.UseInfoTipService = true;
            }

            if (m_usePlugIn_Preview.Enabled)
            {
                regInfo.CurrentStatus.UsePreviewService = true;
            }

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.AllServices);

            RefreshUI();
            RefreshWindowTitle();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle Uninstall all
        /// </summary>
        //----------------------------------------------------------------------
        private void OnUninstallAll(object sender, EventArgs e)
        {
            MainApp app = MainApp.Instance;
            Debug.Assert(app != null);

            RegistrationManager registrationManager = app.RegistrationManager;
            if (registrationManager == null)
                return;

            RegistrationInfo regInfo = registrationManager.EditingRegistration;
            if (regInfo == null)
                return;

            if (m_usePlugIn_Icon.Enabled)
            {
                regInfo.CurrentStatus.UseIconService = false;
            }

            if (m_usePlugIn_InfoTip.Enabled)
            {
                regInfo.CurrentStatus.UseInfoTipService = false;
            }

            if (m_usePlugIn_Preview.Enabled)
            {
                regInfo.CurrentStatus.UsePreviewService = false;
            }

            CopyRegistrationStatus(m_activeExtensionTreeNodeData, ShellExtensionService.AllServices);

            RefreshUI();
            RefreshWindowTitle();
        }

        #endregion

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle File Exit
        /// </summary>
        //----------------------------------------------------------------------
        private void OnFileExit(object sender, EventArgs e)
        {
            Close();
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle Help
        /// </summary>
        //----------------------------------------------------------------------
        private void OnHelp(object sender, EventArgs e)
        {
            String dirName = System.IO.Path.GetDirectoryName(System.Reflection.Assembly.GetEntryAssembly().Location);
            String pathToHelpFile = dirName + "\\Thumbnail_Setup.html";

            if (System.IO.File.Exists(pathToHelpFile))
            {
                System.Diagnostics.Process p = new System.Diagnostics.Process();
                p.StartInfo.FileName = pathToHelpFile;
                p.Start();
            }
        }

        //----------------------------------------------------------------------
        /// <summary>
        /// Handle Help Version
        /// </summary>
        //----------------------------------------------------------------------
        private void OnHelpVersion(object sender, EventArgs e)
        {
            AboutBox aboutBox = new AboutBox();
            aboutBox.ShowDialog();
        }
        #endregion

    }
}
