﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using nw.g3d.iflib;
using nw.g3d.toollib;
using nw.g3d.nw4f_3dif;
using System.Windows.Forms;
using System.Threading;
using nw.g3d.bifedit.IPC;
using System.ServiceModel;

namespace nw.g3d.bifedit
{
    /// <summary>
    /// Main program
    /// </summary>
    class Program
    {
        /// <summary>
        /// サーバーのミューテックス
        /// </summary>
        private static readonly Mutex MasterMutex = new Mutex(false, "{6775E1AA-7047-472D-81A4-CE22C35D91C5}");

        /// <summary>
        /// サーバとクライアントを同期させるためのミューテックス
        /// </summary>
        private static readonly Mutex ServiceMutex = new Mutex(false, "{B4222B60-20F4-4485-A365-B43607D3031F}");

        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        [STAThread]
        static void Main(string[] args)
        {
            Run(args);
        }

        private static void Run(string[] args)
        {
            // パラメータにファイル比較モードが指定されているかどうか取得する
            bool isCompare = CompareProcess.IsComparasion(args);

            // パラメータに指定されているファイル名を取得する
            string fileToOpen = null;
            if (isCompare == false)
            {
                fileToOpen = PathManager.GetFileToOpen(args);
            }

            // Serviceミューテックスをロック
            try
            {
                ServiceMutex.WaitOne();
            }
            catch (AbandonedMutexException)
            {
                // アプリケーションを強制終了するなど、Mutexがアンロックされず削除されたときに例外が発生する
                // ミューテックスのロック自体は正常に行われているので無視する
            }

            bool isMaster = false;

            // Masterミューテックスをロック
            // ロックできればIPCサーバーを起動、できなければIPCクライアントを起動
            try
            {
                isMaster = MasterMutex.WaitOne(0, true);
            }
            catch (AbandonedMutexException)
            {
                // アプリケーションを強制終了するなど、Mutexがアンロックされず削除されたときに例外が発生する
                // ミューテックスのロック自体は正常に行われているので無視する
                isMaster = true;  // ミューテックスをロックできた…はず
            }

            var communicationService = new CommunicationService();

            if (isMaster)
            {
                // Master process

                // メインフォームを作成
                Application.EnableVisualStyles();
                var mainForm = new MainForm(isCompare == false && fileToOpen == null);

                // IPCサーバーにイベント処理メソッドを登録
                communicationService.ShowUserInterfaceRequested += () => mainForm.BeginInvoke((Action) mainForm.ShowWindow);
                communicationService.EditorProcessFileRequested += RunEditorProcess;
                communicationService.CompaleProcessFileRequested += RunCompareProcess;

                // IPCサーバーを起動
                communicationService.Start();

                // Serviceミューテックスをアンロック
                ServiceMutex.ReleaseMutex();

                if (isCompare)
                {
                    // ファイル比較モードが指定されていたとき、ファイル比較を行う
                    RunCompareProcess(args);
                }
                else if (fileToOpen != null)
                {
                    // ファイル名が指定されていたとき、ファイルを開く
                    RunEditorProcess(fileToOpen);
                }

                Application.Run(mainForm);

                // IPCサーバーを停止
                communicationService.Stop();
                G3dParallel.Cleanup(true);

                // Masterミューテックスをアンロック
                MasterMutex.ReleaseMutex();
            }
            else
            {
                // Slave process

                // IPCクライアントを生成
                IRemoteController clientSideController = communicationService.GenerateClientSideProxy();
                if (clientSideController == null)
                    return; // throw an exception ?

                try
                {
                    if (isCompare)
                    {
                        // ファイル比較モードが指定されていたとき、ファイル比較コマンドをサーバーに送る
                        clientSideController.CompaleProcessFile(args);
                    }
                    else if (fileToOpen != null)
                    {
                        // ファイル名が指定されていたとき、ファイルオープンコマンドをサーバーに送る
                        clientSideController.EditorProcessFile(fileToOpen);
                    }
                    else
                    {
                        // その他の場合、UI表示コマンドをサーバーに送る
                        clientSideController.ShowUserInterface();
                    }
                }
                catch (EndpointNotFoundException)
                {
                    // エクスプローラでg3difedit.exeを選択してEnterを連打したときなど、
                    // IPCサーバーの起動が完了する前にクライアントを生成しようとしたときに例外が発生する
                    // IPCクライアントの生成に失敗したかどうかは直接調べるのが難しい
                    System.Diagnostics.Debug.Assert(false, "IPCクライアント生成失敗");
                }
                finally
                {
                    // Serviceミューテックスをアンロック
                    ServiceMutex.ReleaseMutex();
                }
            }
        }

        private static void RunEditorProcess(string filename)
        {
            if (EditorProcess.Run(filename) == false)
            {
                CustomMessageBox.Show(Properties.Resources.NW4F_EDIT_FILE_NO_EDITOR, Properties.Resources.NW4F_EDIT_FILE_NO_EDITOR_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private static void RunCompareProcess(string[] args)
        {
            if (CompareProcess.Run(args) == false)
            {
                CustomMessageBox.Show(Properties.Resources.NW4F_COMPARE_ERROR, Properties.Resources.NW4F_COMPARE_ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
    }
} // End of nw.g3d.bifedit
