﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.IO;

namespace nw.g3d.bifedit
{
    public class PathManager
    {
        public string EditorApplication { get; set; }
        public string EditorCommandArguments { get; set; }

        public static string ExecutablePath { get; private set; }
        public static string WorkingPath    { get; private set; }

        public string XSDFilePath
        {
            get
            {
                return m_config.XSDFilePath;
            }
        }

        public bool IsXSDFilePathValid
        {
            get
            {
                return m_config.IsValidXSDPath;
            }
        }

        public ICollection<AppInfo> ApplicationList
        {
            get
            {
                return m_config.ApplicationList;
            }
        }

        public int SelectedAppIndex { get { return m_selectedAppIndex; } }

        private int m_selectedAppIndex = -1;

        public Config Config
        {
            get { return m_config; }
        }

        private Config m_config      = new Config();

        static PathManager()
        {
            InitializePaths();
        }

        public PathManager()
        {
            Initialize();
        }

        /// <summary>
        /// Initialize directory settings. Override this to customize your directory settings
        /// </summary>
        private static void InitializePaths()
        {
            Assembly mainAssembly = Assembly.GetEntryAssembly();

            ExecutablePath = Path.GetDirectoryName(mainAssembly.Location);
            if (ExecutablePath != null)
                ExecutablePath = ExecutablePath.Replace("/", "\\"); // Fix slash

            WorkingPath = Directory.GetCurrentDirectory();
            if (WorkingPath != null)
                WorkingPath = WorkingPath.Replace("/", "\\"); // Fix slash
        }

        public static string GetFileToOpen(string[] args)
        {
            if (args == null || args.Length == 0 || File.Exists(args[0]) == false)
                return null;

            return args[0];
        }

        public void SetSelectedEditor(int index)
        {
            if (index < 0)
            {
                m_selectedAppIndex = -1;
                m_config.SelectedAppName = "";
                EditorApplication = null;
                EditorCommandArguments = null;
                return;
            }

            if (index >= m_config.ApplicationList.Count)
            {
                m_selectedAppIndex = -1;
                m_config.SelectedAppName = "";
                EditorApplication = m_config.CustomAppPath;
                EditorCommandArguments = m_config.CustomAppCommandArguments;
            }
            else
            {
                m_selectedAppIndex = index;
                EditorApplication = m_config.ApplicationList[index].Path;
                EditorCommandArguments = m_config.ApplicationList[index].CommandArguments;
                m_config.SelectedAppName = EditorApplication;
            }
        }

        private void Initialize()
        {
            if (ExecutablePath != null)
                m_config.LoadConfig(ExecutablePath);

            EditorApplication = null;

            m_selectedAppIndex = -1;

            if (m_config.SelectedAppName.Length > 0)
            {
                // Find selected application
                int i;
                for (i = 0; i < m_config.ApplicationList.Count; i++)
                {
                    if (String.Equals(m_config.ApplicationList[i].Name, m_config.SelectedAppName, StringComparison.CurrentCultureIgnoreCase))
                    {
                        m_selectedAppIndex = i;
                        EditorApplication = m_config.ApplicationList[i].Path;
                        break;
                    }
                }
            }
            else if (m_config.CustomAppPath.Length > 0)
            {
                EditorApplication = m_config.CustomAppPath;
            }

        }

        public void SaveStatusConfig()
        {
            if (ExecutablePath != null)
            {
                if (EditorApplication == null)
                {
                    m_config.SelectedAppName = "";
                    m_config.CustomAppPath   = "";
                }
                else if (m_selectedAppIndex >= 0)
                {
                    m_config.SelectedAppName = m_config.ApplicationList[m_selectedAppIndex].Name;
                    m_config.CustomAppPath = "";
                }

                m_config.SaveStatusConfig(ExecutablePath);
            }
        }

        private static PathManager s_Instance = null;

        public static PathManager Instance
        {
            get
            {
                if (s_Instance == null)
                    s_Instance = new PathManager();

                return s_Instance;
            }
        }
    }
}
