﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Xml;
using System.Xml.Serialization;
using System.Xml.Schema;

namespace NW4F.LayoutBinaryConverter
{
    using LYT = Schema.Flyt;
    using LAN = Schema.Flan;
    using CPX = Schema.Fcpx;

    public class ComprexFontLoader
    {
        readonly string _fcpxSchemaFileName = null;
        readonly XmlSerializer _xmlSerializerCpx = new XmlSerializer(typeof(CPX::ComplexFontDescription));

        public ComprexFontLoader(bool bXsdValidate)
        {
            if (bXsdValidate)
            {
                string schemaDir = Path.Combine(FileUtil.GetNwToolPath(), @"LayoutEditor\XMLSchemata\ver_1_5");

                _fcpxSchemaFileName = Path.Combine(schemaDir, "fcpx.xsd");
            }
        }

        public CpxInfo LoadCPX(string inFileName)
        {
            CPX::ComplexFontDescription rootEle = (CPX::ComplexFontDescription)Deserialize(inFileName, _fcpxSchemaFileName, _xmlSerializerCpx);
            return new CpxInfo(rootEle, inFileName);
        }

        static Object Deserialize(string fileName, string schemaFile, XmlSerializer xmlSerializer)
        {
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.CloseInput = false;    // fileStreamを閉じない。(usingブロックで閉じるため)
            if (schemaFile != null)
            {
                settings.ValidationType = ValidationType.Schema;
                XmlSchemaSet sc = new XmlSchemaSet();
                sc.Add(null, schemaFile);
                settings.Schemas = sc;

                settings.ValidationEventHandler += new ValidationEventHandler(delegate(object sender, ValidationEventArgs args){
                    Report.Err.WriteLine(string.Format("XML Validation error: {0}:{1}", fileName, args.Exception.LineNumber));
                    Report.Err.WriteLine("\t" + args.Message);
                    Report.Err.WriteLine();
                });
            }
            else
            {
                settings.ValidationType = ValidationType.None;
            }

            using (FileStream fileStream = new FileStream(fileName, FileMode.Open, FileAccess.Read))
            {
                XmlDocument xmlDocument = new XmlDocument();
                xmlDocument.PreserveWhitespace = true; // 空白を保存する
                xmlDocument.Load(fileStream);
                using (XmlNodeReader xmlNodeReader = new XmlNodeReader(xmlDocument))
                {
                    using (XmlReader xmlReader = XmlReader.Create(xmlNodeReader, settings))
                    {
                        return xmlSerializer.Deserialize(xmlReader);
                    }
                }
            }
        }
    }

    class LayoutFileLoader
    {
        readonly bool _bBannerLibBugFix;
        readonly bool _bNoCheckVer;
        readonly bool _bExtraFrameForOneTime;
        readonly string _rlytSchemaFileName = null;
        readonly string _rlanSchemaFileName = null;
        readonly string _rlpjSchemaFileName = null;
        readonly string _rlctSchemaFileName = null;
        readonly XmlSerializer _xmlSerializerLyt = new XmlSerializer(typeof(LYT::Document));
        readonly XmlSerializer _xmlSerializerLan = new XmlSerializer(typeof(LAN::Document));

        public LayoutFileLoader(bool bBannerLibBugFix, bool bNoCheckVer, bool bXsdValidate, bool bExtraFrameForOneTime)
        {
            _bBannerLibBugFix = bBannerLibBugFix;
            _bNoCheckVer = bNoCheckVer;
            _bExtraFrameForOneTime = bExtraFrameForOneTime;
            if (bXsdValidate)
            {
                string schemaDir = Path.Combine(FileUtil.GetNwToolPath(), @"LayoutEditor\XMLSchemata\ver_1_5");

                _rlytSchemaFileName = Path.Combine(schemaDir, "flyt.xsd");
                _rlanSchemaFileName = Path.Combine(schemaDir, "flan.xsd");
                _rlpjSchemaFileName = Path.Combine(schemaDir, "clpj.xsd");
                _rlctSchemaFileName = Path.Combine(schemaDir, "clct.xsd");
            }
        }

        public LytInfo LoadLYT(string inFileName)
        {
            LYT::Document rootEle = (LYT::Document)Deserialize(inFileName, _rlytSchemaFileName, _xmlSerializerLyt);
            return new LytInfo(rootEle, inFileName, _bBannerLibBugFix, _bNoCheckVer);
        }

        public LanInfo LoadLAN(string inFileName, LytInfo lytInfo)
        {
            LAN::Document rootEle = (LAN::Document)Deserialize(inFileName, _rlanSchemaFileName, _xmlSerializerLan);
            return new LanInfo(rootEle, inFileName, _bNoCheckVer, _bExtraFrameForOneTime, lytInfo);
        }

        static Object Deserialize(string fileName, string schemaFile, XmlSerializer xmlSerializer)
        {
            XmlReaderSettings settings = new XmlReaderSettings();
            settings.CloseInput = false;    // fileStreamを閉じない。(usingブロックで閉じるため)
            if (schemaFile != null)
            {
                settings.ValidationType = ValidationType.Schema;
                XmlSchemaSet sc = new XmlSchemaSet();
                sc.Add(null, schemaFile);
                settings.Schemas = sc;

                ValidationEventHandleObject valiObj = new ValidationEventHandleObject(fileName);
                settings.ValidationEventHandler += new ValidationEventHandler(valiObj.ValidationEventHandle);
            }
            else
            {
                settings.ValidationType = ValidationType.None;
            }

            using (FileStream fileStream = new FileStream(fileName, FileMode.Open, FileAccess.Read))
            {
                XmlDocument xmlDocument = new XmlDocument();
                xmlDocument.PreserveWhitespace = true; // 空白を保存する
                xmlDocument.Load(fileStream);
                using (XmlNodeReader xmlNodeReader = new XmlNodeReader(xmlDocument))
                {
                    using (XmlReader xmlReader = XmlReader.Create(xmlNodeReader, settings))
                    {
#if true
                        return xmlSerializer.Deserialize(xmlReader);
#else
                try
                {
                    return xmlSerializer.Deserialize(xmlReader);
                }
                catch (InvalidOperationException ex)
                {
                    Report.Err.WriteLine(ex.Message);
                    if (ex.InnerException != null)
                    {
                        Report.Err.WriteLine(ex.InnerException.Message);
                    }
                    return null;
                }
#endif
                    }
                }
            }
        }

        public static void Serialize(string fileName, object obj)
        {
            XmlSerializer xmlSerializer = new XmlSerializer(obj.GetType());

            using (FileStream stream = new FileStream(fileName, FileMode.Create, FileAccess.Write))
            {
                xmlSerializer.Serialize(stream, obj);
            }
        }

        class ValidationEventHandleObject
        {
            readonly string _fileName;

            public ValidationEventHandleObject(string fileName)
            {
                _fileName = fileName;
            }

            public void ValidationEventHandle(object sender, ValidationEventArgs args)
            {
                Report.Err.WriteLine(string.Format("XML Validation error: {0}:{1}", _fileName, args.Exception.LineNumber));
                Report.Err.WriteLine("\t" + args.Message);
                Report.Err.WriteLine();
            }
        }

    }
}
