﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace NW4F.LayoutBinaryConverter
{
    /// <summary>
    ///
    /// </summary>
    class CpxWriter
    {
        /// <summary>
        /// 文字列テーブルへのオフセットを計算します。
        /// </summary>
        int CalcOffsetToScalableFontDescriptionTail_(IEnumerable<Schema.Fcpx.ScalableFontDescription> scalableFontDescriptionSet)
        {
            int sizeOfResMultiScalableFontHeaderTotal = 0;

            // scalableFontDescriptionSet
            sizeOfResMultiScalableFontHeaderTotal += 32; // sizeof(ResMultiScalableFont)

            foreach (var scalableFontDescription in scalableFontDescriptionSet)
            {
                // ResScalableFontDescription
                sizeOfResMultiScalableFontHeaderTotal += 40 - 8; // sizeof(ResScalableFontDescription) - sizeof(ResCharCodeRangeSet)

                // ResCharCodeRangeSet
                sizeOfResMultiScalableFontHeaderTotal += 4 + 4;
                //sizeOfResMultiScalableFontHeaderTotal += scalableFontDescription.charCodeRangeSet.Count() * (4 + 4);
            }

            return sizeOfResMultiScalableFontHeaderTotal;
        }

        /// <summary>
        /// charCodeRange テーブルへのオフセットを計算します。
        /// </summary>
        int CalcCharCodeRangeTableSize_(IEnumerable<Schema.Fcpx.ScalableFontDescription> scalableFontDescriptionSet)
        {
            int sizeOfCharCodeRangeTable = 0;
            foreach (var scalableFontDescription in scalableFontDescriptionSet)
            {
                sizeOfCharCodeRangeTable += scalableFontDescription.charCodeRangeSet.Count() * (4 + 4);
            }

            return sizeOfCharCodeRangeTable;
        }

        /// <summary>
        /// ファイルパスから、ファイル名を取得します。
        /// </summary>
        string GetFontNameFromFilePath_(string fontPath)
        {
            return Path.GetFileName(fontPath);
        }

        /// <summary>
        /// ResComplexFontDescription を出力します。
        /// </summary>
        void WriteResComplexFontDescription_(BinaryCpxWriter cpxWriter)
        {
            // ResComplexFontDescription
            cpxWriter.WriteU32(4);
        }

        /// <summary>
        /// マルチスケーラブルフォントを出力します。
        /// </summary>
        void WriteResMultiScalableFont_(BinaryCpxWriter cpxWriter, float fontSize, IEnumerable<Schema.Fcpx.ScalableFontDescription> scalableFontDescriptionSet, uint alternateChar, int lineFeedOffset)
        {
            int charCodeRangeSetOffset = CalcOffsetToScalableFontDescriptionTail_(scalableFontDescriptionSet);
            int stringTableStartOffset = charCodeRangeSetOffset + CalcCharCodeRangeTableSize_(scalableFontDescriptionSet);
            const int SizeofResCharCodeRange = 4 + 4;

            var fontNames = new List<string>();

            // ResMultiScalableFontHeader
            cpxWriter.WriteResMultiScalableFontHeader(fontSize, alternateChar, lineFeedOffset, scalableFontDescriptionSet.Count());

            // ScalableFontDescription を出す
            foreach (var scdesc in scalableFontDescriptionSet)
            {
                cpxWriter.WriteResScalableFontDesc(
                    scdesc.boldWeight,
                    Convert.ToUInt32(scdesc.ttcFontIndex),
                    Convert.ToUInt32(stringTableStartOffset),
                    (byte)scdesc.borderWidth,
                    scdesc.scaleWidth,
                    scdesc.scaleHeight,
                    scdesc.ignorePalt,
                    scdesc.deleteBearingX,
                    scdesc.bearingOffsetX,
                    scdesc.forceMonospacedEnabled,
                    scdesc.forceMonospacedWidth,
                    scdesc.baselineOffset);

                // ResCharCodeRangeSet
                cpxWriter.WriteU32(Convert.ToUInt32(scdesc.charCodeRangeSet.Count()));
                cpxWriter.WriteU32(Convert.ToUInt32(charCodeRangeSetOffset));

                charCodeRangeSetOffset += scdesc.charCodeRangeSet.Count() * SizeofResCharCodeRange;

                string fontName = GetFontNameFromFilePath_(scdesc.path);
                stringTableStartOffset += fontName.Length + 1;
                stringTableStartOffset += 3 - (stringTableStartOffset + 3) % 4;
                fontNames.Add(fontName);
            }

            // ResCharCodeRange テーブル
            foreach (var scdesc in scalableFontDescriptionSet)
            {
                foreach (var crs in scdesc.charCodeRangeSet)
                {
                    cpxWriter.WriteResCharMap(crs.first, crs.last);
                }
            }

            // 文字列テーブル
            foreach(var fontName in fontNames)
            {
                byte[] dataSet = System.Text.Encoding.ASCII.GetBytes(fontName);
                foreach(var data in dataSet)
                {
                    cpxWriter.WriteU8(data);
                }

                cpxWriter.WriteU8(0);

                // アライメントを 4byte に揃える
                while (cpxWriter.Stream.Position % 4 != 0)
                {
                    cpxWriter.WriteU8(0);
                }
            }
        }

        /// <summary>
        /// ペアフォントを出力します。
        /// </summary>
        void WriteResPairFont_(BinaryCpxWriter cpxWriter, Schema.Fcpx.PairFont pairFont, CpxInfo cpxInfo)
        {
            long basePosition = cpxWriter.Stream.Position; // オフセットの基底となる位置

            // ResPairFontHeader
            cpxWriter.WriteResPairFontHeader();
            long pos = cpxWriter.Stream.Position; // firstFont、secondFont のオフセットを書き込む位置
            cpxWriter.WriteU32(0); // firstFont のオフセット(ダミー)
            cpxWriter.WriteU32(0); // secondFont のオフセット(ダミー)

            uint firstFontOffset = Convert.ToUInt32(cpxWriter.Stream.Position - basePosition);
            WriteFontTree_(cpxWriter, pairFont.firstFont, cpxInfo);

            uint secondFontOffset = Convert.ToUInt32(cpxWriter.Stream.Position - basePosition);
            WriteFontTree_(cpxWriter, pairFont.secondFont, cpxInfo);

            long currentPosition = cpxWriter.Stream.Position;
            cpxWriter.Stream.Position = pos;
            cpxWriter.WriteU32(firstFontOffset); // firstFont のオフセット
            cpxWriter.WriteU32(secondFontOffset); // secondFont のオフセット
            cpxWriter.Stream.Position = currentPosition;
        }

        /// <summary>
        /// ビットマップフォントを出力します。
        /// </summary>
        void WriteResBitmapFont_(BinaryCpxWriter cpxWriter, Schema.Fcpx.BitmapFont bitmapFont)
        {
            const int charCodeRangeSetOffset = 4 + 4 + 4 + 4;
            int stringTableOffset = charCodeRangeSetOffset + bitmapFont.charCodeRangeSet.Count() * (4 + 4);

            // ResMultiScalableFontHeader
            cpxWriter.WriteResBitmapFontHeader(Convert.ToUInt32(stringTableOffset));

            // ResCharCodeRangeSet
            cpxWriter.WriteU32(Convert.ToUInt32(bitmapFont.charCodeRangeSet.Count()));
            cpxWriter.WriteU32(Convert.ToUInt32(charCodeRangeSetOffset));

            // ResCharCodeRange テーブル
            foreach (var crs in bitmapFont.charCodeRangeSet)
            {
                cpxWriter.WriteResCharMap(crs.first, crs.last);
            }

            // 文字列テーブル
            {
                string fontName = GetFontNameFromFilePath_(bitmapFont.path);

                // .ffnt の場合は .bffnt に読み替える
                if (Path.GetExtension(fontName).ToLowerInvariant() == ".ffnt")
                {
                    fontName = Path.GetFileNameWithoutExtension(fontName) + ".bffnt";
                }

                byte[] dataSet = System.Text.Encoding.ASCII.GetBytes(fontName);
                foreach (var data in dataSet)
                {
                    cpxWriter.WriteU8(data);
                }

                cpxWriter.WriteU8(0);

                // アライメントを 4byte に揃える
                while (cpxWriter.Stream.Position % 4 != 0)
                {
                    cpxWriter.WriteU8(0);
                }
            }
        }

        /// <summary>
        /// フォントの木構造を出力します。
        /// </summary>
        public void WriteFontTree_(BinaryCpxWriter cpxWriter, Schema.Fcpx.SubFont subFont, CpxInfo cpxInfo)
        {
            if (subFont.Item is Schema.Fcpx.ScalableFont)
            {
                var scFont = subFont.Item as Schema.Fcpx.ScalableFont;
                WriteResMultiScalableFont_(cpxWriter, scFont.size, new Schema.Fcpx.ScalableFontDescription[] { scFont.scalableFontDescription }, scFont.alternateChar, scFont.lineFeedOffset);
            }
            else if (subFont.Item is Schema.Fcpx.MultiScalableFont)
            {
                var multiScFont = subFont.Item as Schema.Fcpx.MultiScalableFont;
                WriteResMultiScalableFont_(cpxWriter, multiScFont.size, multiScFont.scalableFontDescriptionSet, multiScFont.alternateChar, multiScFont.lineFeedOffset);
            }
            else if (subFont.Item is Schema.Fcpx.PairFont)
            {
                var pairFont = subFont.Item as Schema.Fcpx.PairFont;
                WriteResPairFont_(cpxWriter, pairFont, cpxInfo);
            }
            else if (subFont.Item is Schema.Fcpx.BitmapFont)
            {
                var bitmapFont = subFont.Item as Schema.Fcpx.BitmapFont;
                WriteResBitmapFont_(cpxWriter, bitmapFont);
            }
            else
            {
                // エラー（対応していない種類の複合フォント）
                string typeName = cpxInfo.ComplexFontDescription.subFont.Item != null ? cpxInfo.ComplexFontDescription.subFont.Item.GetType().Name : string.Empty;
                throw new LayoutDataException(string.Format(Properties.Resources.ErrorUnknownComplexFontType, typeName));
            }
        }

        /// <summary>
        /// バイナリファイルを出力します。
        /// </summary>
        public void Write(string outDirName, CpxInfo cpxInfo)
        {
            string dirName = FileUtil.MakeResourceDirectory(outDirName, BinaryCpxWriter.TypeOfComplexFontResource);
            string outFileName = FileUtil.MakeOutFileNameStr(dirName, null, cpxInfo.DocFilePath, null);
            using (FileStream fs = new FileStream(outFileName, FileMode.Create, FileAccess.Write, FileShare.None))
            {
                BinaryCpxWriter cpxWriter = new BinaryCpxWriter(fs);

                cpxWriter.WriteBinaryFileHeader(0, 0);      // バイナリファイルヘッダ

                // 0 だとデータが無いとみなされて assert に引っかかるため
                // bfcpx のブロック数は便宜上 1 とみなす
                ushort dataBlockNum = 1;

                WriteResComplexFontDescription_(cpxWriter);
                WriteFontTree_(cpxWriter, cpxInfo.ComplexFontDescription.subFont, cpxInfo);

                // この時点でファイル書き込みが全て終了したので、バイナリファイルヘッダを更新
                // ファイルポインタの現在位置がファイルサイズ
                long fileSize = cpxWriter.Stream.Position;
                cpxWriter.Stream.Position = 0;
                cpxWriter.WriteBinaryFileHeader((uint)fileSize, dataBlockNum);
            }
        }
    }
}
