﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections.Generic;
using System.Diagnostics;
using System.Reflection;
using System.Linq;
using Nintendo.Foundation.IO;

namespace NW4F.LayoutBinaryConverter
{
    /// <summary>
    /// コマンドライン文字列を解析します。
    ///
    /// 同じ機能を持つ異なるスイッチが指定された場合、最後に指定されたものが有効になります。
    /// </summary>
    public class CommandLineParser
    {
        public string[] inFileNames;
        public string outDirName;

        // 指定不可
        public bool IsImageFileWriteTimeSameOriginal;

        [CommandLineValues(CountMin = 2, Description = "OptionPaths", ValueName = "paths", DescriptionConverterName = "DescriptionConverter")]
        public string[] Paths { get; set; }

        // 削除予定
        [CommandLineOption('p', "platform", Description = "OptionPlatform", DescriptionConverterName = "DescriptionConverter", IsRequired = false)]
        public string TargetPlatformName { get; set; }

        // 削除予定
        [CommandLineOption('a', "api-type", Description = "OptionTargetApiType", DescriptionConverterName = "DescriptionConverter", IsRequired = false)]
        public string TargetApiTypeName { get; set; }

        // 削除予定
        [CommandLineOption('c', "code-type", Description = "OptionTargetCodeType", DescriptionConverterName = "DescriptionConverter", IsRequired = false)]
        public string TargetCodeTypeName { get; set; }

        [CommandLineOption('t', "tile-mode", Description = "OptionTargetTileMode", DescriptionConverterName = "DescriptionConverter", IsRequired = false)]
        public string TargetTileMode { get; set; }

        [CommandLineOption('u', Description = "OptionUpdate", DescriptionConverterName = "DescriptionConverter")]
        public bool IsUpdateMode { get; set; }

        [CommandLineOption('g', Description = "OptionTagSplit", DescriptionConverterName = "DescriptionConverter")]
        public bool IsTagSplit { get; set; }

        [CommandLineOption("tag-split-and-whole", Description = "OptionTagSplitAndWhole", DescriptionConverterName = "DescriptionConverter")]
        public bool IsTagSplitAndWhole { get; set; }

        [CommandLineOption("no-convert-cvtrchar", Description = "OptionNoConvertCvtrChar", DescriptionConverterName = "DescriptionConverter")]
        public bool IsNoConvertTagChar { get; set; }

        [CommandLineOption("discard-textboxtext", Description = "OptionDiscardTextboxText", DescriptionConverterName = "DescriptionConverter")]
        public bool IsDiscardTextboxText { get; set; }

        [CommandLineOption("omit-samekey", Description = "OptionOmitSameKey", DescriptionConverterName = "DescriptionConverter")]
        public bool IsOmitSameKey { get; set; }

        [CommandLineOption("omit-samekey-all", Description = "OptionOmitSameKeyAll", DescriptionConverterName = "DescriptionConverter")]
        public bool IsOmitSameKeyAll { get; set; }

        [CommandLineOption("omit-nokey-all", Description = "OptionOmitNoKeyAll", DescriptionConverterName = "DescriptionConverter")]
        public bool IsOmitNoKeyAll { get; set; }

        [CommandLineOption("bake-infinity", Description = "OptionBakeInfinityAreaKey", DescriptionConverterName = "DescriptionConverter")]
        public bool IsBakeInfinityAreaKey { get; set; }

        [CommandLineOption("no-taginfo", Description = "OptionNoTagInfo", DescriptionConverterName = "DescriptionConverter")]
        public bool IsNoOutTagInfo { get; set; }

        [CommandLineOption("no-check-version", Description = "OptionNoCheckVersion", DescriptionConverterName = "DescriptionConverter")]
        public bool IsNoCheckVersion { get; set; }

        [CommandLineOption("cvtr-all-tex", Description = "OptionCvtrAllTex", DescriptionConverterName = "DescriptionConverter")]
        public bool IsConvertAllTex { get; set; }

        [CommandLineOption("no-copy-font", Description = "OptionNoCopyFont", DescriptionConverterName = "DescriptionConverter")]
        public bool IsNoCopyFont { get; set; }

        [CommandLineOption("no-copy-ffnt", Description = "OptionNoCopyFfnt", DescriptionConverterName = "DescriptionConverter")]
        public bool IsNoCopyFfnt { get; set; }

        [CommandLineOption("extra-frame-for-onetime", Description = "OptionExtraFrameForOnetime", DescriptionConverterName = "DescriptionConverter")]
        public bool IsExtraFrameForOneTime { get; set; }

        [CommandLineOption("cvtr-all-anim-content", Description = "OptionConvertAllAnimContent", DescriptionConverterName = "DescriptionConverter")]
        public bool IsConvertAllAnimContent { get; set; }

        [CommandLineOption("allow-nogroup-animtag", Description = "OptionAllowNoGroupAnimTag", DescriptionConverterName = "DescriptionConverter")]
        public bool IsAllowedNoGroupAnimTag { get; set; }

        [CommandLineOption("degamma-texture", Description = "OptionDegammaTexture", DescriptionConverterName = "DescriptionConverter")]
        public bool IsDegammaTexture { get; set; }

        [CommandLineOption("degamma-parameter", Description = "OptionDegammaParameter", DescriptionConverterName = "DescriptionConverter")]
        public bool IsDegammaParameter { get; set; }

        [CommandLineOption("skip-if-srctexture-not-exist", Description = "OptionSkipIfSrcTextureNotExist", DescriptionConverterName = "DescriptionConverter")]
        public bool IsSkipIfSrcTextureNotExist { get; set; }

        [CommandLineOption('s', "silent", Description = "OptionQuiet", DescriptionConverterName = "DescriptionConverter")]
        public bool IsQuiet { get; set; }

        [CommandLineOption("gpu-encoding", Description = "OptionGpuEncodigEnabled", DescriptionConverterName = "DescriptionConverter", DefaultValue ="true")]
        public string IsGpuEncodigEnabled { get; set; }

        [CommandLineOption("mesh-trimming-optimization", Description = "OptionMeshTrimmingOptimization", DescriptionConverterName = "DescriptionConverter")]
        public string FillOptimizationMode{ get; set; }

        [CommandLineOption("mesh-trimming-optimization-margin", Description = "OptionMeshTrimmingOptimizationMargin", DescriptionConverterName = "DescriptionConverter")]
        public string FillOptimizationMargin{ get; set; }

        [CommandLineOption("output-optimization-statistics", Description = "OptionOutputOptimizationStatistics", DescriptionConverterName = "DescriptionConverter")]
        public bool IsOutputOptimizationStatistics{ get; set; }

        [CommandLineOption('i', Description = "", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsEnableViewerHidden { get; set; }

        [CommandLineOption("no-output-layout-anim-tag-name", Description = "OptionNoOutputLayoutTagNameExport", DescriptionConverterName = "DescriptionConverter")]
        public bool IsNoOutputLayoutTagNameExport { get; set; }

        // 指定不可
        public bool IsBannerCompatible { get; set; }

        [CommandLineOption("compatible-0-12-x", Description = "OptionCompatible_0_12_x", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsCompatible_0_12_x { get; set; }

        [CommandLineOption("xsd-validate", Description = "", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsXsdValidate { get; set; }

        [CommandLineOption("degamma-parameter-old-bug", Description = "", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsDegammaParameterOldBug { get; set; }

        [CommandLineOption("hide-identify-information-in-statistics", Description = "", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsHideIdentifyInformationInStatistics{ get; set; }

        [CommandLineOption("logfile", Description = "", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsOutLogFile { get; set; }

        [CommandLineOption("ftxb-cache-directory", Description = "OptionFtxbCacheDirectory", DescriptionConverterName = "DescriptionConverter")]
        public string FtxbCacheDirectory { get; set; }

        [CommandLineArgFileOption("args-file", Description = "OptionArgsFile", DescriptionConverterName = "DescriptionConverter")]
        public string ArgsFile { get; set; }

        [CommandLineOption("shader-env-directories", Description = "OptionShaderEnvDirectories", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public string ShaderEnvDirectories { get; set; }

        [CommandLineOption("combiner-editor-path", Description = "OptionCombinerEditorPath", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public string CombinerEditorPath { get; set; }

        [CommandLineOption("ecmb-empty-error", Description = "OptionEcmbEmptyError", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsEcmbEmptyErr { get; set; }

        [CommandLineOption("keep-font-scale", Description = "OptionKeepingScaleFont", DescriptionConverterName = "DescriptionConverter", IsHidden = true)]
        public bool IsKeepingFontScaleEnabled { get; set; }

        public static string DescriptionConverter(string description, string name)
        {
            var value = Properties.Resources.ResourceManager.GetString(description);
            Debug.Assert(!string.IsNullOrEmpty(value) || description == string.Empty);

            // 隠しオプションでも空や null 以外を返さないとエラーになる。
            return value ?? "Hidden Option";
        }

        public void ValidateAndConvertArguments()
        {
            if (IsOmitSameKey && IsOmitSameKeyAll)
            {
                throw new CommandLineParserException(string.Format(Properties.Resources.ErrorExclusiveOptions, "--omit-samekey", "--omit-samekey-all"));
            }

            if (IsOmitSameKey && IsOmitNoKeyAll)
            {
                throw new CommandLineParserException(string.Format(Properties.Resources.ErrorExclusiveOptions, "--omit-samekey", "--omit-nokey-all"));
            }

            if (IsOmitSameKeyAll && IsOmitNoKeyAll)
            {
                throw new CommandLineParserException(string.Format(Properties.Resources.ErrorExclusiveOptions, "--omit-samekey-all", "--omit-nokey-all"));
            }

            if (IsDegammaParameter && IsDegammaParameterOldBug)
            {
                throw new CommandLineParserException(string.Format(Properties.Resources.ErrorExclusiveOptions, "--degamma-parameter", "--degamma-parameter-old-bug"));
            }

            if (IsDegammaParameterOldBug)
            {
                // 後の処理の都合上ここで設定
                IsDegammaParameter = true;
            }

            Debug.Assert(Paths.Length >= 2);

            inFileNames = Paths.Take(Paths.Length - 1).ToArray();
            outDirName = Paths[Paths.Length - 1];

            if (FileUtil.IsDirectorySeparatorEnd(outDirName))
            {
                // 末尾の'\'を取る
                outDirName = FileUtil.TrimDirectorySeparatorEnd(outDirName);
            }
        }

        static string GetExecFileName()
        {
            return Path.GetFileNameWithoutExtension(Environment.GetCommandLineArgs()[0]);
        }

        public static void PrintProductInfo()
        {
            Assembly mainAssembly = Assembly.GetEntryAssembly();

            // プロダクト情報を出力
            object[] ProductArray = mainAssembly.GetCustomAttributes(typeof(AssemblyProductAttribute), false);
            if (ProductArray != null && ProductArray.Length > 0)
            {
                Report.Out.WriteLine(((AssemblyProductAttribute)ProductArray[0]).Product);
            }

            // バージョン情報を出力
            Version appVersion = mainAssembly.GetName().Version;
            Report.Out.WriteLine(string.Format("Version {0}.{1}.{2}", appVersion.Major, appVersion.Minor, appVersion.Build));

            // コピーライト情報を出力
            object[] CopyrightArray = mainAssembly.GetCustomAttributes(typeof(AssemblyCopyrightAttribute), false);
            if (CopyrightArray != null && CopyrightArray.Length > 0)
            {
                Report.Out.WriteLine(((AssemblyCopyrightAttribute)CopyrightArray[0]).Copyright);
            }

            Report.Out.WriteLine();
        }

    }

    public class CommandLineParserException : Exception
    {
        public CommandLineParserException(string message)
            : base(message)
        { }
    }
}
