﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using sch = NW4F.LayoutBinaryConverter.Schema.Flyt;

namespace NW4F.LayoutBinaryConverter
{
    /// <summary>
    /// レイアウトライブラリのバグ対策用コード
    /// バナーデータの互換のために、単純にライブラリのコードを修正できない場合に、
    /// データによるバグ対策を行うクラス
    /// </summary>
    class BugBuster
    {
        LytInfo _rlyt;
        sch.Material _dmyMaterial;

        public BugBuster(LytInfo rlyt)
        {
            _rlyt = rlyt;
        }

#if true
        List<sch.Pane> _repairPane = new List<sch.Pane>();

        /// <summary>
        /// バグ:テキストボックスペインの文字列描画において、アルファコンペアの値をセットしていない。
        ///
        /// アルファコンペアを初期値に戻すダミーのピクチャペインを、既定では無い設定をして、
        /// かつ以後描画されるテキストボックスペインに影響を与えるピクチャ/ウィンドウペインに、
        /// 直下の子供として挿入する。
        /// </summary>
        public void InsertDummyPicturePane()
        {
            DrawContext drawContext = CreateDrawContext(_rlyt.RootPane, false);   // ルートペインはNULLペインであるため、描画状態を何も更新しない。そのため不定であるため、安全のため既定の設定でないとみなす。
            CheckPane(_rlyt.RootPane, drawContext);

            InsertDummyPane();
        }

        void CheckPane(sch.Pane pane, DrawContext parentDrawContext)
        {
            // 自分のチェック
            DrawContext drawContext = CheckSelf(pane, parentDrawContext);

            // 子供のペインを再帰的に処理する
            if (pane.ChildList != null)
            {
                foreach (sch.Pane childPane in pane.ChildList)
                {
                    CheckPane(childPane, drawContext);
                }
            }

            // ダミーペインの挿入が行われず、既定ではない設定にするペインのリストが残っているときは、
            // 親のDrawContext にマージして、以後の描画に影響があることを示す。
            if (drawContext != parentDrawContext && ! drawContext.bAlphaCompareDefault)
            {
                parentDrawContext.Add(drawContext);
            }
        }

        DrawContext CheckSelf(sch.Pane pane, DrawContext parentDrawContext)
        {
            switch (pane.kind)
            {
            case sch.PaneKind.Picture:
                {
                    sch.Picture picture = (sch.Picture)pane.Item;
                    return CreateDrawContext(pane, new MatInfo(picture.detailSetting, picture.materialCtr).IsAlphaCompareDefaultSetting);
                }

            case sch.PaneKind.Window:
                return CreateDrawContext(
                        pane,
                        GetLatestDrawMaterial((sch.Window)pane.Item).IsAlphaCompareDefaultSetting);

            case sch.PaneKind.TextBox:
                if (! parentDrawContext.bAlphaCompareDefault)   // 直前までの描画の設定でアルファコンペアが既定で無い場合
                {
                    parentDrawContext.MoveRepairPanes(_repairPane);
                }
                return parentDrawContext;

            default:
                return parentDrawContext;
            }
        }

        void InsertDummyPane()
        {
            foreach (sch.Pane pane in _repairPane)
            {
                sch.Pane dummyPane = CreateDummyPane(_rlyt.PaneDic);
                LytInfo.AddPaneDic(_rlyt.PaneDic, dummyPane);
                if (pane.ChildList == null)
                {
                    pane.ChildList = new List<sch.Pane>();
                }
                pane.ChildList.Insert(0, dummyPane);        // 最初の子供の位置に挿入

                Debug.WriteLine(string.Format("insert dummy pane at {0}", pane.name));
            }
        }

        static DrawContext CreateDrawContext(sch.Pane pane, bool bAlphaCompareDefault)
        {
            DrawContext drawContext = new DrawContext();

            if (! bAlphaCompareDefault)
            {
                drawContext.Add(pane);
            }

            return drawContext;
        }

        class DrawContext
        {
            List<sch.Pane> noDefPaneList;

            public bool bAlphaCompareDefault
            {
                get
                {
                    return noDefPaneList == null;
                }
            }

            public void Add(sch.Pane pane)
            {
                Debug.Assert(pane != null);

                if (noDefPaneList == null)
                {
                    noDefPaneList = new List<sch.Pane>();
                }

                noDefPaneList.Add(pane);
            }

            public void Add(DrawContext other)
            {
                Debug.Assert(!other.bAlphaCompareDefault);

                if (noDefPaneList == null)
                {
                    noDefPaneList = new List<sch.Pane>();
                }

                noDefPaneList.AddRange(other.noDefPaneList);
            }

            public void MoveRepairPanes(List<sch.Pane> repairPaneList)
            {
                Debug.Assert(!bAlphaCompareDefault);

                repairPaneList.AddRange(noDefPaneList);

                noDefPaneList = null;
            }
        }

#else
        /// <summary>
        /// バグ:テキストボックスペインの文字列描画において、アルファコンペアの値をセットしていない。
        ///
        /// アルファコンペアを初期値に戻す、ダミーのピクチャペインを、影響があるテキストボックスペインの
        /// 前に挿入する。
        ///
        /// 影響を受けるテキストボックスペインは、アルファコンペアの値を既定と異なる、
        /// 設定にしたピクチャペイン・ウィンドウペインの後に描画されるテキストボックスペイン。
        /// </summary>
        public void InsertDummyPicturePane()
        {
            bool bParentDefaultAlpComp = false;
            CheckPane(_rlyt.RootPane, ref bParentDefaultAlpComp);
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="bParentModAlphaCompare">ペインがアルファ比較を既定と異なる値に設定する場合に真</param>
        bool CheckPane(sch.Pane pane, ref bool bParentDefaultAlpComp)
        {
            bool bInsertDummyPane;
            bool bDefaultAlpComp = CheckSelf(pane, ref bParentDefaultAlpComp, out bInsertDummyPane);

            // 子供のペインを再帰的に処理する
            if (pane.ChildList != null)
            {
                for (int i = 0; i < pane.ChildList.Count; ++i)
                {
                    sch.Pane childPane = pane.ChildList[i];
                    if (CheckPane(childPane, ref bDefaultAlpComp))
                    {
                        sch.Pane dummyPane = CreateDummyPane(_rlyt.PaneDic);
                        RlytUtil.AddPaneDic(_rlyt.PaneDic, dummyPane);
                        pane.ChildList.Insert(i, dummyPane);        // テキストボックスペインの直前に挿入
                        ++i;    // paneを直前に挿入したので、1つづらす

                        Debug.WriteLine(string.Format("insert dummy pane {0} at {1}", pane.ChildList[i - 1].name, pane.ChildList[i].name));
                    }
                }
            }

            return bInsertDummyPane;
        }

        bool CheckSelf(sch.Pane pane, ref bool bParentDefaultAlpComp, out bool bInsertDummyPane)
        {
            bInsertDummyPane = false;

            switch (pane.kind)
            {
            case sch.PaneKind.Picture:
                {
                    sch.Picture picture = (sch.Picture)pane.Item;
                    return CheckMaterial(new MatInfo(picture.detailSetting, picture.materialRevo), ref bParentDefaultAlpComp);
                }

            case sch.PaneKind.TextBox:
                if (! bParentDefaultAlpComp)
                {
                    bInsertDummyPane = true;
                    bParentDefaultAlpComp = true;  // ダミーペインを挿入するので、既定に戻す
                }
                return true;

            case sch.PaneKind.Window:
                return CheckWindowPane((sch.Window)pane.Item, ref bParentDefaultAlpComp);

            default:
                return bParentDefaultAlpComp;
            }
        }

        /// <summary>
        /// ウィンドウペインのチェック
        /// 最後に描画されるマテリアルを調べる
        /// </summary>
        /// <param name="window"></param>
        /// <param name="bInheritDefaultAlpComp"></param>
        /// <returns></returns>
        static bool CheckWindowPane(sch.Window window, ref bool bInheritDefaultAlpComp)
        {
            return CheckMaterial(GetLatestDrawMaterial(window), ref bInheritDefaultAlpComp);
        }

        static bool CheckMaterial(MatInfo matInfo, ref bool bParentDefaultAlpComp)
        {
            if (matInfo.IsAlphaCompareDefaultSetting)
            {
                return true;
            }

            bParentDefaultAlpComp = false;
            return bParentDefaultAlpComp;
        }

#endif
        static MatInfo GetLatestDrawMaterial(sch.Window window)
        {
            sch.WindowFrame[] frames = new sch.WindowFrame[window.frame.Length];
            Array.Copy(window.frame, frames, frames.Length);

            // 描画の後に来るもの順に並べる。
            Array.Sort(
                frames,
                delegate(sch.WindowFrame x, sch.WindowFrame y)
                {
                    return GetFrameOrder(y.frameType) - GetFrameOrder(x.frameType);
                });

            // 最後に描画するフレームから、実際に描画されるフレームを見つける
            // (テクスチャが無いフレームは描画されない)
            sch.WindowFrame lastFrame = Array.Find(
                frames,
                delegate(sch.WindowFrame obj)
                {
                    sch.TexMap[] texMaps = obj.detailSetting ? obj.materialCtr.texMap : obj.material.texMap;
                    return texMaps != null; // TexMap配列が null の場合はテクスチャが1枚も無い
                });

            if (lastFrame != null)  // 描画されるフレームがあればそのフレームのマテリアル
            {
                return new MatInfo(lastFrame.detailSetting, lastFrame.materialCtr);
            }
            else    // コンテントのマテリアル
            {
                return new MatInfo(window.content.detailSetting, window.content.materialCtr);
            }
        }

        /// <summary>
        /// ライブラリ側での描画順から小さい値を返す
        /// </summary>
        /// <param name="type"></param>
        /// <returns></returns>
        static int GetFrameOrder(sch.WindowFrameType type)
        {
            switch (type)
            {
            case sch.WindowFrameType.CornerLT:  return 0;
            case sch.WindowFrameType.FrameT:    return 1;
            case sch.WindowFrameType.CornerRT:  return 2;
            case sch.WindowFrameType.FrameR:    return 3;
            case sch.WindowFrameType.CornerRB:  return 4;
            case sch.WindowFrameType.FrameB:    return 5;
            case sch.WindowFrameType.CornerLB:  return 6;
            case sch.WindowFrameType.FrameL:    return 7;
            default: return int.MaxValue;
            }
        }

        /// <summary>
        /// ダミーペインを作る
        /// </summary>
        /// <returns></returns>
        sch.Pane CreateDummyPane(Dictionary<string, sch.Pane> dictionary)
        {
            sch.Pane pane = new sch.Pane(MakeDummyPaneName(dictionary), new sch.Vec2(16, 16));
            pane.kind = sch.PaneKind.Picture;
            if (_dmyMaterial == null)   // リソースサイズ削減のため、ダミーマテリアルは1つだけ作成し共有する
            {
                _dmyMaterial = new sch.Material(pane.name);
            }
            _dmyMaterial.whiteColor = new sch.WhiteColor(0, 0, 0, 0);   // 描画されないように透明にする
            pane.Item = new sch.Picture(_dmyMaterial);
            return pane;
        }

        static string MakeDummyPaneName(Dictionary<string, sch.Pane> dictionary)
        {
            for (int i = 0; i <= ushort.MaxValue; ++i)  // 最大16bit 分まわす
            {
                string resName = string.Format("___INITPANE{0:d5}", i);
                if (!dictionary.ContainsKey(resName))
                {
                    return resName;
                }
            }

            throw new LayoutDataException("Init pane name over.");
        }

        struct MatInfo
        {
            public MatInfo(bool detailSetting, sch.Material_CTR matCTR)
            {
                this.bDetailSetting = detailSetting;
                this.materialCTR = matCTR;
            }

            bool bDetailSetting;
            sch.Material_CTR materialCTR;

            /// <summary>
            /// 影響を与えるピクチャペインの直下の子供にダミーペインを追加する。
            ///
            /// 影響が出るテキストボックスペインとは、影響を与えるピクチャペイン以降に描画されるテキストボックスペインの中で、
            /// ピクチャペイン、ウィンドウペインの子でないテキストボックスペイン
            /// </summary>
            public bool IsAlphaCompareDefaultSetting
            {
                get
                {
                    if (! bDetailSetting)
                    {
                        // 詳細でない場合は、アルファ比較は常に既定の設定
                        return true;
                    }

                    return _IsAlphaCompareDefaultSetting(materialCTR.alphaCompare);
                }
            }

            /// <summary>
            /// アルファ比較の設定において、常にパスする設定の場合は、真を返します。
            /// </summary>
            /// <param name="alphaComp"></param>
            /// <returns></returns>
            static bool _IsAlphaCompareDefaultSetting(sch.Material_CTRAlphaCompare alphaComp)
            {
                return alphaComp.comp == sch.Compare.Always;
            }
        }

    }
}
