﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace FcpxMaker.DataModel
{
    using static FcpxMaker.Type;

    /// <summary>
    /// スケーラブルフォントの設定です。
    /// </summary>
    [Serializable]
    public class ScalableFontDescriptionDataModel : DataModelBase
    {
        DataModelBase _charCodeRange = new CharCodeRangeDataModel();
        DataModelBase _fontDataModel = null;

        /// <summary>
        /// フォントのデータモデルです。
        /// </summary>
        public DataModelBase FontDataModel
        {
            get { return _fontDataModel; }
            set { _fontDataModel = value; }
        }

        /// <summary>
        /// 文字範囲のデータモデルです。
        /// </summary>
        public DataModelBase CharCodeRangeDataModel
        {
            get { return _charCodeRange; }
            set { _charCodeRange = value; }
        }

        /// <summary>
        /// フォントファイルのパスを指定します。
        /// </summary>
        public string FontFilePath
        {
            get
            {
                if (_fontDataModel == null) return string.Empty;

                return (_fontDataModel as FontDataModel).RelativeFontFilePath;
            }
        }

        /// <summary>
        /// ボールドウェイトを指定します(0.0で通常の太さです)。
        /// </summary>
        public float BoldWeight { get; set; }

        /// <summary>
        /// 縁取りの太さを指定します(0で縁取り描画がOFFになります)。
        /// </summary>
        public int BorderWidth { get; set; }

        /// <summary>
        /// ttcフォントの場合に使用するフォントのインデクスを指定します。
        /// </summary>
        public int TtcFontIndex { get; set; }

        /// <summary>
        /// 文字幅のスケールを指定します。
        /// 省略した場合は 1.0 になります。
        /// </summary>
        public float ScaleWidth { get; set; } = 1.0f;

        /// <summary>
        /// 文字の高さのスケールを指定します。
        /// 省略した場合は 1.0 になります。
        /// </summary>
        public float ScaleHeight { get; set; } = 1.0f;

        /// <summary>
        /// OpenType フォントの全角文字をプロポーショナルにするための情報を無視して等幅で描画するか否かを指定します。
        /// 省略した場合は false になります。
        /// </summary>
        public bool IgnorePalt { get; set; } = false;

        /// <summary>
        /// グリフの左右のベアリング(余白)を削除するか否かを指定します。
        /// 省略した場合は false になります。
        /// </summary>
        public bool DeleteBearingX { get; set; } = false;

        /// <summary>
        /// グリフの左右のベアリング(余白)を調整します。
        /// 省略した場合は 0 になります。
        /// </summary>
        public int BearingOffsetX { get; set; } = 0;

        /// <summary>
        /// 強制的に等幅で描画するか否かを指定します。
        /// 省略した場合は false になります。
        /// </summary>
        public bool ForceMonospacedEnabled { get; set; } = false;

        /// <summary>
        /// 強制的に等幅で描画する際の文字幅を調整します。
        /// 省略した場合は 0 になります。
        /// </summary>
        public int ForceMonospacedWidth { get; set; } = 0;

        /// <summary>
        /// ベースラインの上下を調整します。
        /// 省略した場合は 0 になります。
        /// </summary>
        public int BaseLineOffset { get; set; } = 0;

        /// <summary>
        /// フォントを追加します。
        /// </summary>
        public void SetFontDataModel(DataModelBase dataModel)
        {
            _fontDataModel = dataModel;
        }

        /// <summary>
        /// フォントを削除します。
        /// </summary>
        public void RemoveFontDataModel()
        {
            _fontDataModel = null;
        }

        /// <summary>
        /// データの種類です。
        /// </summary>
        override public DataType DataType
        {
            get { return DataType.ScalableFontDescription; }
        }

        /// <summary>
        /// データの検証結果です。
        /// </summary>
        override public bool Validate
        {
            get
            {
                return _charCodeRange.Validate;
            }
        }

        /// <summary>
        /// データモデルを比較します。
        /// </summary>
        override public bool Equals(DataModelBase dataModel)
        {
            ScalableFontDescriptionDataModel dm = dataModel as ScalableFontDescriptionDataModel;
            if (dm == null) return false;

            if (this.FontDataModel == null && dm.FontDataModel != null)
            {
                return false;
            }

            if (this.CharCodeRangeDataModel == null && dm.CharCodeRangeDataModel != null)
            {
                return false;
            }

            if ((this.FontDataModel != null && !this.FontDataModel.Equals(dm.FontDataModel)) ||
                (this.CharCodeRangeDataModel != null && !this.CharCodeRangeDataModel.Equals(dm.CharCodeRangeDataModel)) ||
                (this.BoldWeight != dm.BoldWeight) ||
                (this.BorderWidth != dm.BorderWidth) ||
                (this.TtcFontIndex != dm.TtcFontIndex) ||
                (this.ScaleWidth != dm.ScaleWidth) ||
                (this.ScaleHeight != dm.ScaleHeight) ||
                (this.IgnorePalt != dm.IgnorePalt) ||
                (this.DeleteBearingX != dm.DeleteBearingX) ||
                (this.BearingOffsetX != dm.BearingOffsetX) ||
                (this.ForceMonospacedEnabled != dm.ForceMonospacedEnabled) ||
                (this.ForceMonospacedWidth != dm.ForceMonospacedWidth) ||
                (this.BaseLineOffset != dm.BaseLineOffset))
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// データモデルを複製します。
        /// </summary>
        override public DataModelBase Clone()
        {
            ScalableFontDescriptionDataModel newDataModel = new ScalableFontDescriptionDataModel();
            newDataModel.FontDataModel = this.FontDataModel?.Clone();
            newDataModel.CharCodeRangeDataModel = this.CharCodeRangeDataModel?.Clone();
            newDataModel.BoldWeight = this.BoldWeight;
            newDataModel.BorderWidth = this.BorderWidth;
            newDataModel.TtcFontIndex = this.TtcFontIndex;
            newDataModel.ScaleWidth = this.ScaleWidth;
            newDataModel.ScaleHeight = this.ScaleHeight;
            newDataModel.IgnorePalt = this.IgnorePalt;
            newDataModel.DeleteBearingX = this.DeleteBearingX;
            newDataModel.BearingOffsetX = this.BearingOffsetX;
            newDataModel.ForceMonospacedEnabled = this.ForceMonospacedEnabled;
            newDataModel.ForceMonospacedWidth = this.ForceMonospacedWidth;
            newDataModel.BaseLineOffset = this.BaseLineOffset;

            return newDataModel;
        }

        /// <summary>
        /// データモデルをコピーします。
        /// </summary>
        override public bool CopyParam(DataModelBase dataModel, bool copyChild)
        {
            ScalableFontDescriptionDataModel dm = dataModel as ScalableFontDescriptionDataModel;
            if (dm == null) return false;

            if (copyChild)
            {
                this.FontDataModel = dm.FontDataModel;
            }

            this.CharCodeRangeDataModel.CopyParam(dm.CharCodeRangeDataModel, copyChild);
            this.BoldWeight = dm.BoldWeight;
            this.BorderWidth = dm.BorderWidth;
            this.TtcFontIndex = dm.TtcFontIndex;
            this.ScaleWidth = dm.ScaleWidth;
            this.ScaleHeight = dm.ScaleHeight;
            this.IgnorePalt = dm.IgnorePalt;
            this.DeleteBearingX = dm.DeleteBearingX;
            this.BearingOffsetX = dm.BearingOffsetX;
            this.ForceMonospacedEnabled = dm.ForceMonospacedEnabled;
            this.ForceMonospacedWidth = dm.ForceMonospacedWidth;
            this.BaseLineOffset = dm.BaseLineOffset;

            return true;
        }

        /// <summary>
        /// 子ノードとして追加可能かどうかを返します。
        /// </summary>
        override public bool CanAddChild(DataModelBase dataModel)
        {
            if (dataModel is FontDataModel)
            {
                return
                    ((dataModel as FontDataModel).FontType == FontFileType.Scalable &&
                     _fontDataModel == null);
            }

            return false;
        }

        /// <summary>
        /// 表示名です。
        /// </summary>
        override public string ToString()
        {
            return DataType.ToString();
        }
    }
}
