﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using LECore.Structures.Core;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Imaging;
using System.Linq;

namespace LayoutEditor.Utility
{
    /// <summary>
    /// カラーコンバータクラス。
    /// </summary>
    public class AnimationMarkColorConverter
    {
        private static Dictionary<AnimationStatusKind, Color> _converList = new Dictionary<AnimationStatusKind, Color>()
        {
            { AnimationStatusKind.None,       SystemColors.Control },
            { AnimationStatusKind.SameKey,    Color.Red            },
            { AnimationStatusKind.DiffKey,    Color.Pink           },
            { AnimationStatusKind.HasKey,     Color.HotPink        },
            { AnimationStatusKind.DiffValue,  Color.Pink           },
        };

        /// <summary>
        /// デフォルトの色です。
        /// </summary>
        public static Color DefaultColor
        {
            get
            {
                return _converList[AnimationStatusKind.None];
            }
        }

        /// <summary>
        /// AnimationColorTypeをColorに変換します。
        /// </summary>
        public static Color Convert(AnimationStatusKind type)
        {
            Color ret = SystemColors.Control;

            if (_converList.ContainsKey(type))
            {
                ret = _converList[type];
            }

            return ret;
        }

        /// <summary>
        /// Colorを取得します。
        /// </summary>
        public static Color GetColor(IAnmAttribute attr, bool isSeparateMode)
        {
            if (attr.NumSubAttribute == 0)
            {
                return Convert(AnmAttributeHelper.GetMarkColor(attr, isSeparateMode));
            }
            else
            {
                return Convert(AnmAttributeHelper.GetMarkColorRecursive(attr, isSeparateMode));
            }
        }

        /// <summary>
        /// 複数のColorから優先度の高い色を選択します。
        /// </summary>
        public static Color SelectColor(Color[] colors)
        {
            if (colors.Where(c => c == Color.Pink).Count() > 0)
            {
                return Color.Pink;
            }
            else if (colors.Where(c => c == Color.Red).Count() > 0)
            {
                return Color.Red;
            }
            else if (colors.Where(c => c == Color.HotPink).Count() > 0)
            {
                return Color.HotPink;
            }
            else
            {
                return SystemColors.Control;
            }
        }
    }
}
