﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using LayoutEditor.Controls.UI;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow;

namespace LayoutEditor.Plugin
{
    /// <summary>
    /// プラグインを管理します。
    /// </summary>
    public class PluginManager
    {
        /// <summary>
        /// シングルトンインスタンスです。
        /// </summary>
        public static readonly PluginManager Instance = new PluginManager();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PluginManager()
        {
            this.ButtonPlugins = new List<ButtonPluginBase>();
            this.PagePlugins = new List<PagePluginInfo>();
        }

        /// <summary>
        /// ボタンプラグインリストを取得または設定します。
        /// </summary>
        public List<ButtonPluginBase> ButtonPlugins { get; private set; }

        /// <summary>
        /// ページプラグインリストを取得または設定します。
        /// </summary>
        public List<PagePluginInfo> PagePlugins { get; private set; }

        /// <summary>
        /// プラグインをロードします。
        /// </summary>
        public void LoadPlugins()
        {
            // Pluginフォルダにあるdllリストを取得
            string appDir = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            string pluginDir = Path.Combine(appDir, "Plugins");
            string[] dllFiles = Directory.Exists(pluginDir) ? Directory.GetFiles(pluginDir) : new string[0];

            foreach (string dllFile in dllFiles)
            {
                // 拡張子をチェック
                if (Path.GetExtension(dllFile).ToLower() != ".dll")
                {
                    continue;
                }

                try
                {
                    // dllを読み込む
                    Assembly assembly = Assembly.LoadFrom(dllFile);
                    Type[] types = assembly.GetTypes();

                    foreach (Type type in types)
                    {
                        if (type.IsSubclassOf(typeof(ButtonPluginBase)))
                        {
                            // ボタンプラグインをロード
                            object plugin = assembly.CreateInstance(type.FullName);
                            this.ButtonPlugins.Add((ButtonPluginBase)plugin);
                        }
                        else if (type.IsSubclassOf(typeof(PagePluginBase)))
                        {
                            // ページプラグインをロード
                            PagePluginInfo pluginInfo = new PagePluginInfo();

                            // NodeNameプロパティを取得
                            PropertyInfo nodeNameProperty = type.GetProperty("NodeName");
                            if (nodeNameProperty == null)
                            {
                                System.Diagnostics.Debug.Assert(false, "NodeNameプロパティが未実装");
                                continue;
                            }

                            pluginInfo.NodeName = nodeNameProperty.GetValue(null) as string;

                            // CreateInstanceDelegateプロパティを取得
                            PropertyInfo createInstanceDelegateProperty = type.GetProperty("CreateInstanceDelegate");
                            if (createInstanceDelegateProperty == null)
                            {
                                System.Diagnostics.Debug.Assert(false, "CreateInstanceDelegateプロパティが未実装");
                                continue;
                            }

                            pluginInfo.CreatePropertyFunction = createInstanceDelegateProperty.GetValue(null) as CreatePropertyPageFunction;

                            this.PagePlugins.Add(pluginInfo);
                        }
                    }
                }
                catch (Exception)
                {
                }
            }
        }
    }

    /// <summary>
    /// ページプラグインの情報です。
    /// </summary>
    public struct PagePluginInfo
    {
        /// <summary>
        /// ノード名を取得または設定します。
        /// </summary>
        public string NodeName { get; set; }

        /// <summary>
        /// ページ初期化用の関数を取得または設定します。
        /// </summary>
        public CreatePropertyPageFunction CreatePropertyFunction { get; set; }
    }
}
