﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using LECore;
using LECore.Structures;
using System;

namespace LayoutEditor.Forms.ToolWindows
{
    /// <summary>
    /// ツリーノード状態を保持するクラスです。
    /// </summary>
    public class TreeNodeStateHolder<TTreeNodeTagType> where TTreeNodeTagType : class
    {
        private readonly Dictionary<TTreeNodeTagType, NodeStateDictionary> _nodeStateDictionaries = new Dictionary<TTreeNodeTagType, NodeStateDictionary>();
        private readonly Predicate<TTreeNodeTagType> _checkStateCacheIsObsolete;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TreeNodeStateHolder(Predicate<TTreeNodeTagType> checkStateCacheIsObsolete)
        {
            _checkStateCacheIsObsolete = checkStateCacheIsObsolete;
        }

        /// <summary>
        /// ツリーノード状態を保存するクラスです。
        /// </summary>
        private class NodeState
        {
            public bool Expanded { get; set; }
        }

        /// <summary>
        /// ツリー状態を保存するクラスです。
        /// </summary>
        private class NodeStateDictionary : Dictionary<TTreeNodeTagType, NodeState>
        {
        }

        /// <summary>
        /// ツリーノード状態を保存します。
        /// </summary>
        public void PreserveNodeState(TreeNode rootNode, TTreeNodeTagType tagObject)
        {
            NodeStateDictionary nodeStates = null;

            // rootNode.Tag をキー情報とします。
            if (_nodeStateDictionaries.TryGetValue(tagObject, out nodeStates) == false)
            {
                nodeStates = new NodeStateDictionary();
                _nodeStateDictionaries.Add(tagObject, nodeStates);
            }

            PreserveNodeState_(nodeStates, rootNode);
        }

        /// <summary>
        /// ツリーノード状態を復旧します。
        /// </summary>
        public void RestoreNodeState(TreeNode rootNode, TTreeNodeTagType tagObject)
        {
            NodeStateDictionary nodeStates = null;
            if (_nodeStateDictionaries.TryGetValue(tagObject, out nodeStates) != false)
            {
                RestoreNodeState_(nodeStates, rootNode);
            }
        }

        /// <summary>
        /// 不要なツリー状態を削除します。
        /// </summary>
        public void RemoveUnnecessaryNodeState()
        {
            TTreeNodeTagType[] keys = _nodeStateDictionaries.Keys.ToArray();
            foreach (TTreeNodeTagType key in keys)
            {
                if (_checkStateCacheIsObsolete(key))
                {
                    _nodeStateDictionaries.Remove(key);
                }
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// ツリーノード状態を保存します。
        /// 再帰的に処理します。
        /// </summary>
        private void PreserveNodeState_(NodeStateDictionary nodeStates, TreeNode node)
        {
            TTreeNodeTagType tagObject = node.Tag as TTreeNodeTagType;
            NodeState nodeState = null;

            if (tagObject != null)
            {
                if (nodeStates.TryGetValue(tagObject, out nodeState) == false)
                {
                    nodeState = new NodeState();
                    nodeStates.Add(tagObject, nodeState);
                }

                nodeState.Expanded = node.IsExpanded || node.Nodes.Count == 0;
            }

            foreach (TreeNode childNode in node.Nodes)
            {
                PreserveNodeState_(nodeStates, childNode);
            }
        }

        /// <summary>
        /// ツリーノード状態を復旧します。
        /// 再帰的に処理します。
        /// </summary>
        private void RestoreNodeState_(NodeStateDictionary nodeStates, TreeNode node)
        {
            TTreeNodeTagType tagObject = node.Tag as TTreeNodeTagType;
            NodeState nodeState = null;
            if (tagObject != null)
            {
                if (nodeStates.TryGetValue(tagObject, out nodeState) != false)
                {
                    if (nodeState.Expanded != false)
                    {
                        node.Expand();
                    }
                    else
                    {
                        node.Collapse();
                    }
                }
                else
                {
                    node.Expand();
                }
            }

            foreach (TreeNode childNode in node.Nodes)
            {
                RestoreNodeState_(nodeStates, childNode);
            }
        }
    }
}
