﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.common
{
    static public class ToolStripMenuItemHelper
    {
        /// <summary>
        /// IMessageFilter では、
        /// メッセージキューの内容のみが処理される。
        /// そのため、SendMessage系のメッセージがフックできない。
        ///
        /// ここでは、NativeWindow クラスを派生クラスにカスタム処理を記述し、
        /// そのクラスに、対象コントロールのプロシージャをフックさせている。
        ///
        /// 以下Web掲示板からの引用：
        ///
        /// PreFilterMessage filters messages that comes through the application message
        /// queue in other words they are posted rather than sent. There are not a lot
        /// of messages like this. Besically these are messages from the mouse, keyboard
        /// and few others.
        ///
        /// On the other hand the WndProc is the target for all messages, so you are
        /// going to receive them all there.
        ///
        /// You can use the Spy++ tool to spy on all messages send or posted to an
        /// application. For all messages in the log there there is a mark if the
        /// message has been sent or posted. If I remember correctly posted messages are
        /// marked with P and sent send with S. Only messages marked with P can be
        /// filtered in PreFilterMessage.
        ///
        /// CodeProject　の　Trapping windows messages も 参考になる。
        /// http://www.codeproject.com/dotnet/devicevolumemonitor.asp?df=100&forumid=15152&exp=0&select=1306974&tid=920143
        /// </summary>
        public class ToolStripMessageFilter : NativeWindow
        {
            protected Control _owner;
            protected bool _isHandleAssigned = false;
            protected ToolStripMessageFilter() { throw new InvalidOperationException(); }
            protected ToolStripMessageFilter( Control owner )
            {
                _owner = owner;

                _owner.HandleCreated += new EventHandler( Event_Owner_HandleCreated_ );

                // IsHandleCreated が false の状態で Handle を取得するとコントロールの配置が崩れることがある。
                // 詳しい理由は不明。
                if (_owner.IsHandleCreated && !_isHandleAssigned)
                {
                    _isHandleAssigned = true;
                    AssignHandle(_owner.Handle);
                }

                _owner.HandleDestroyed += new EventHandler( Event_Owner_HandleDestroyed_ );
            }

            /// <summary>
            /// 生成します。
            /// </summary>
            static public void BindMessageFilter( Control owner )
            {
                ToolStripMessageFilter instance = new ToolStripMessageFilter( owner);
            }

            /// <summary>
            ///
            /// </summary>
            void Event_Owner_HandleCreated_(object sender, EventArgs e)
            {
                if (!_isHandleAssigned)
                {
                    _isHandleAssigned = true;
                    AssignHandle((sender as Control).Handle);
                }
            }

            delegate void ReleaseHandleFunction();

            /// <summary>
            ///
            /// </summary>
            void Event_Owner_HandleDestroyed_( object sender, EventArgs e )
            {
                _owner.BeginInvoke( new ReleaseHandleFunction( ReleaseHandle ) );
                // ReleaseHandle();
            }

            /// <summary>
            ///
            /// </summary>
            protected override void WndProc( ref Message m )
            {
                if( !_owner.IsDisposed &&
                    _owner.Visible &&
                    _owner.Enabled )
                {
                    switch( m.Msg )
                    {
                        // マウスアクティベート
                        // 既定では戻り値が MA_ACTIVATEANDEAT なので以降のマウス処理がされない
                        case LECore.Win32.WM.WM_MOUSEACTIVATE:
                        m.Result = (IntPtr)LECore.Win32.MA.MA_ACTIVATE;
                        return;
                    }
                }
                base.WndProc( ref m );
            }
        }

        #region MenuItemの複製

        /// <summary>
        /// ToolStripMenuItemの複製から呼び出される
        /// クリックイベントハンドラ
        /// </summary>
        static void CloneMenuItemHandler_( object sender, EventArgs e )
        {
            ToolStripMenuItem mi = sender as ToolStripMenuItem;
            ToolStripMenuItem cloneSrcMenuItem = mi.Tag as ToolStripMenuItem;

            // クローン元のクリックを呼び出す。
            cloneSrcMenuItem.PerformClick();
        }

        /// <summary>
        /// ToolStripMenuItemの複製を作成します。
        /// Clickイベントハンドラも実行されます。
        /// </summary>
        static ToolStripMenuItem CloneToolStripMenuItem_( ToolStripMenuItem src )
        {
            ToolStripMenuItem dst = new ToolStripMenuItem();

            // クリックイベントを設定
            dst.Click += CloneMenuItemHandler_;
            dst.DropDownOpened += CloneMenuItemHandler_;

            // 複製元のToolStripMenuItemを記憶します。
            dst.Tag = src;

            dst.AccessibleName = src.AccessibleName;
            dst.AccessibleRole = src.AccessibleRole;
            dst.Alignment = src.Alignment;
            dst.AllowDrop = src.AllowDrop;
            dst.Anchor = src.Anchor;
            dst.AutoSize = src.AutoSize;
            dst.AutoToolTip = src.AutoToolTip;
            dst.BackColor = src.BackColor;
            dst.BackgroundImage = src.BackgroundImage;
            dst.BackgroundImageLayout = src.BackgroundImageLayout;
            dst.Checked = src.Checked;
            dst.CheckOnClick = src.CheckOnClick;
            dst.CheckState = src.CheckState;
            dst.DisplayStyle = src.DisplayStyle;
            dst.Dock = src.Dock;
            dst.DoubleClickEnabled = src.DoubleClickEnabled;
            dst.Enabled = src.Enabled;
            dst.Font = src.Font;
            dst.ForeColor = src.ForeColor;
            dst.Image = src.Image;
            dst.ImageAlign = src.ImageAlign;
            dst.ImageScaling = src.ImageScaling;
            dst.ImageTransparentColor = src.ImageTransparentColor;
            dst.Margin = src.Margin;
            dst.MergeAction = src.MergeAction;
            dst.MergeIndex = src.MergeIndex;
            dst.Name = src.Name;
            dst.Overflow = src.Overflow;
            dst.Padding = src.Padding;
            dst.RightToLeft = src.RightToLeft;

            dst.ShortcutKeys = src.ShortcutKeys;
            dst.ShowShortcutKeys = src.ShowShortcutKeys;
            dst.ShortcutKeyDisplayString = src.ShortcutKeyDisplayString;
            dst.Text = src.Text;
            dst.TextAlign = src.TextAlign;
            dst.TextDirection = src.TextDirection;
            dst.TextImageRelation = src.TextImageRelation;
            dst.ToolTipText = src.ToolTipText;

            dst.Available = src.Available;

            if( !src.AutoSize )
            {
                dst.Size = src.Size;
            }
            return dst;
        }

        /// <summary>
        /// ToolStripMenuItem 階層構造全体をコピーします。
        /// </summary>
        static ToolStripItem CloneToolStripMenuItemRecursive_( ToolStripItem src )
        {
            ToolStripItem dst = null;
            if( src is ToolStripMenuItem )
            {
                ToolStripMenuItem srcMenuItem = src as ToolStripMenuItem;

                dst = CloneToolStripMenuItem_( srcMenuItem );
                ToolStripMenuItem dstMenuItem = dst as ToolStripMenuItem;
                foreach( ToolStripItem child in srcMenuItem.DropDownItems )
                {
                    ToolStripItem childClone = CloneToolStripMenuItemRecursive_( child );
                    dstMenuItem.DropDownItems.Add( childClone );
                }
            }
            else if( src is ToolStripSeparator )
            {
                dst = new ToolStripSeparator();
            }
            else
            {
                Debug.Assert( false, "Unkown ToolStripItem Type in CloneToolStripMenuItemRecursive_()." );
            }

            return dst;
        }

        /// <summary>
        /// ToolStripMenuItem 以下を ContextMenuStrip　として複製します。
        /// </summary>
        public static ContextMenuStrip MakeContextMenuStripFromToolStripMenuItem( ToolStripItem[] srcSet )
        {
            ContextMenuStrip cm = new ContextMenuStrip();

            foreach( ToolStripItem src in srcSet )
            {
                cm.Items.Add( CloneToolStripMenuItemRecursive_( src ) );
            }

            return cm;
        }
        #endregion MenuItemの複製
    }
}
