﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.TextureMgrWindow
{
    using System.ComponentModel;
    using common;
    using Controls;
    using LayoutEditor.Structures.SerializableObject;
    using LayoutEditor.Utility;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.LECoreInterface;
    using CoreData = LECore.Structures;
    using LECore;
    using NW4C.Tga;

    /// <summary>
    /// TextureMgrWindow の概要の説明です。
    /// </summary>
    public partial class TextureMgrWindow :
        LEToolWindow,
        IInitializeAfterLoadSetting
    {
        BackgroundWorker _backgroundUnusedTexUIUpdater = new BackgroundWorker();
        const int _ThubmnailSize = 64;

        #region InternalTexImage
        /// <summary>
        /// テクスチャ画像
        /// リストアイテムが、Tagで参照するデータ型として使用します。
        /// ITextureImage と
        /// そのソース画像ファイルの最後に書き込みが行われた時間を管理します。
        ///
        /// 書き込みが行われた時間をもとに、最新状態への更新処理を行います。
        /// </summary>
        struct InternalTexImage
        {
            readonly CoreData.ITextureImage _textureImage;
            readonly DateTime _timeSrcFileLastWrite;
            readonly TexImagePixelFmt _texImagePixelFmt;
            readonly string _filePath;
            readonly Bitmap _bitmap;

            /// <summary>
            /// テクスチャ
            /// </summary>
            public CoreData.ITextureImage ITextureImage
            {
                get { return _textureImage; }
            }

            /// <summary>
            /// ファイルの最終更新時間を取得します。
            /// </summary>
            static DateTime GetLastWriteTime_(CoreData.ITextureImage textureImage)
            {
                if (textureImage.Valid)
                {
                    switch (textureImage.SourceType)
                    {
                        case TextureSourceType.File:
                            return File.Exists(textureImage.FilePath) ? File.GetLastWriteTime(textureImage.FilePath) : DateTime.MinValue;
                        case TextureSourceType.Dynamic:
                        default:
                            return textureImage.UpdateTime;
                    }
                }
                else
                {
                    return DateTime.MinValue;
                }
            }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            public InternalTexImage(CoreData.ITextureImage textureImage)
            {
                Debug.Assert(textureImage != null);

                _textureImage = textureImage;
                _timeSrcFileLastWrite = GetLastWriteTime_(textureImage);
                _texImagePixelFmt = textureImage.PixelFmt;
                _filePath = textureImage.FilePath;
                _bitmap = textureImage.GDIBitmap;
            }

            /// <summary>
            /// 情報が無効か判定します。
            /// </summary>
            public bool Obsolete
            {
                get
                {
                    DateTime time = GetLastWriteTime_(_textureImage);
                    return
                        time != _timeSrcFileLastWrite ||
                        _texImagePixelFmt != _textureImage.PixelFmt ||
                        _filePath != _textureImage.FilePath ||
                        _bitmap != _textureImage.GDIBitmap;
                }
            }

            /// <summary>
            /// ファイルパス
            /// _textureImage.FilePath と同じとは限らない
            /// </summary>
            public string FilePath { get { return _filePath; } }
        }
        #endregion InternalTexImage

        #region プロパティ

        private TextureMgrManipulator _textureMgrManipulator = new TextureMgrManipulator();
        private string _lastFileLoadedDirectoryName = string.Empty;
        private Timer _errorMessageTimer = new Timer();

        /// <summary>
        /// テクスチャマネージャを取得します。
        /// </summary>
        CoreData.ITextureMgr _ITextureMgr
        {
            get
            {
                if (_CurrentSubScene != null)
                {
                    return _CurrentSubScene.ITextureMgr;
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// テクスチャマネージャ操作クラス
        /// </summary>
        TextureMgrManipulator _TextureMgrManipulator
        {
            get
            {
                _textureMgrManipulator.BindTarget(_ITextureMgr);
                return _textureMgrManipulator;
            }
        }

        /// <summary>
        /// 現在のシーンを取得します。
        /// </summary>
        CoreData.ISubScene _CurrentSubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// 新規ペインを生成可能か？
        /// </summary>
        bool _CanCreateNewPane
        {
            get { return _CurrentSubScene != null && _lvwTexture.SelectedIndices.Count != 0; }
        }

        /// <summary>
        /// "フォーマット"列のインデックスを返します。
        /// </summary>
        int _TextureFormatListViewSubItemIndex
        {
            get { return _cmhTexFmt.Index; }
        }

        /// <summary>
        /// レイアウトが参照しているテクスチャを表示するか
        /// </summary>
        private bool _ShowTexturesInSubScene { get; set; } = true;

        /// <summary>
        /// レイアウトに含まれないテクスチャイメージ
        /// </summary>
        private Dictionary<string, ITextureImage> pathToImage { get; set; } = new Dictionary<string, ITextureImage>();

        private bool _UpdatingState { get; set; } = false;

        /// <summary>
        /// ディレクトリのパス
        /// </summary>
        private string _DirectoryPath { get; set; }

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TextureMgrWindow()
        {
            //
            // Windows フォーム デザイナ サポートに必要です。
            //
            InitializeComponent();

            _cmbDirectory.ComboBox.DrawMode = DrawMode.OwnerDrawFixed;
            _cmbDirectory.ComboBox.DrawItem += ComboBox_DrawItem;

            // アイコンイメージリストの初期化
            _lvwTexture.LargeImageList = new ImageList();
            _lvwTexture.LargeImageList.ColorDepth = ColorDepth.Depth32Bit;
            _lvwTexture.LargeImageList.ImageSize = new Size(64, 64);
            _lvwTexture.SmallImageList = new ImageList();
            _lvwTexture.SmallImageList.ColorDepth = ColorDepth.Depth32Bit;
            _lvwTexture.SmallImageList.ImageSize = new Size(16, 16);

            _lvwTexture.CheckNodeSubItemEditingChanged = CheckTextureFormatChanged_;
            _lvwTexture.IsEditTargetSubItemIdxValid = IsEditTargetSubItemIdxValid_;

            string[] typeStrSet = Enum.GetNames(typeof(TexImagePixelFmt));

            InitializeFormatList_();

            // リストビュー項目文字列の設定
            _cmhTexName.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_NAME");
            _cmhTexSizeX.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_W");
            _cmhTexSizeY.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_H");
            _cmhTexDataSize.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_PADDING");
            _cmhTexNativeSizeNX.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_NATIVESIZE_NX");
            _cmhTexFmt.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_FMT");
            _cmhTexFilePath.Text = LayoutEditor.StringResMgr.Get("LISTVIEW_TEXMGR_FILEPATH");

            // NXライブラリの読み込みに失敗している場合は"サイズ(NX)"を非表示にします
            if (LayoutEditorCore.PlatformDetail.IsNXDllLoaded == false)
            {
                _lvwTexture.Columns.RemoveAt(_cmhTexNativeSizeNX.Index);
            }

            // メッセージフィルタの設定
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_tspMain);

            // バックグラウンドでのUI更新
            _backgroundUnusedTexUIUpdater.DoWork += new DoWorkEventHandler(Event_backgroundUnusedTexFinder_DoWork);
            _backgroundUnusedTexUIUpdater.WorkerSupportsCancellation = true;

            _lvwTexture.GetEditResultTextFunc = new Func<Control, string>(
                (ctrl) =>
                {
                    var texItem = ((ctrl as UIComboBox).SelectedItem as TextureFormatItem);
                    if (texItem == null) { return string.Empty; }

                    return texItem.Format.ToString();
                });

            // リストクリック後に、リストビューの選択変更メッセージが起こりエラーメッセージが上書きされる
            // のでタイミングをずらすために、少し待機時間を作るためのタイマーです。
            _errorMessageTimer.Tick += ErrorMessageTimer_Tick;
            _errorMessageTimer.Interval = 1000;
        }

        private void ComboBox_DrawItem(object sender, DrawItemEventArgs e)
        {
            if (e.Index < 0)
            {
                // 何らかの要素が選択されているはず
                Debug.Assert(false);
                return;
            }

            e.DrawBackground();
            var item = ((UIListControlItem)_cmbDirectory.Items[e.Index]);
            var data = (DirectoryDropDownData)item.Data;
            var text = item.ToString();
            Brush b = null;
            if (e.Index > 0 && data.path == null)
            {
                b = new SolidBrush(SystemColors.GrayText);
            }
            else
            {
                b = new SolidBrush(e.ForeColor);
            }

            e.Graphics.DrawString(text, e.Font, b, e.Bounds);

            b.Dispose();
        }

        /// <summary>
        ///
        /// </summary>
        public class TextureFormatItem
        {
            public TexImagePixelFmt Format { get; private set; }

            public TextureFormatItem(TexImagePixelFmt fmt)
            {
                Format = fmt;
            }

            public override string ToString()
            {
                return GetPixcelFormatString_(Format);
            }
        }

        /// <summary>
        /// リストビューの項目
        /// </summary>
        private class TextureListViewItem : ListViewItem
        {
            public TextureListViewItem(string text) : base(text)
            {
            }
            public InternalTexImage InternalTexImage { get; set; }
            //public string ImageIndexKey { get; set; }
            public bool InSubScene { get; set; }
            public TextureDirectory Directory { get; set; }
            public string SubDirectory { get; set; }
            public Bitmap Thumbnail { get; set; }
        }

        private void SetFormatCombo_(UIComboBox combo, IEnumerable<TexImagePixelFmt> fmtList)
        {
            foreach (var texFmt in fmtList)
            {
                if (texFmt == TexImagePixelFmt.Unknown)
                {
                    combo.Items.Add("----------");
                    continue;
                }

                combo.Items.Add(new TextureFormatItem(texFmt));
            }
        }

        private void InitializeFormatList_()
        {
            // グレー画像用フォーマット選択肢
            {
                _cmbTextureFmtForGray.Items.Add("====Alpha Format====");
                SetFormatCombo_(_cmbTextureFmtForGray, LECore.LayoutEditorCore.PlatformDetail.GetGrayScaleFormatList(true));

                _cmbTextureFmtForGray.Items.Add("====Color Format====");
                SetFormatCombo_(_cmbTextureFmtForGray, LECore.LayoutEditorCore.PlatformDetail.GetGrayScaleFormatList(false));

                _cmbTextureFmtForGray.ExpandDropDownWidth();
                _cmbTextureFmtForGray.SelectedValueChanged += Event_CmbTextureFmtFull_DropDownClosed;
            }

            // 非グレー アルファ無、画像用フォーマット選択肢
            {

                _cmbTextureFmtForNoGrayNoAplha.Items.Add("====Color Format====");
                SetFormatCombo_(_cmbTextureFmtForNoGrayNoAplha, LECore.LayoutEditorCore.PlatformDetail.GetFormatList(false, false));

                _cmbTextureFmtForNoGrayNoAplha.Items.Add("====Alpha Format====");
                SetFormatCombo_(_cmbTextureFmtForNoGrayNoAplha, LECore.LayoutEditorCore.PlatformDetail.GetFormatList(false, true));

                _cmbTextureFmtForNoGrayNoAplha.ExpandDropDownWidth();
                _cmbTextureFmtForNoGrayNoAplha.SelectedValueChanged += Event_CmbTextureFmtFull_DropDownClosed;
            }

            // 非グレー アルファ有り、画像用フォーマット選択肢
            {
                _cmbTextureFmtForGrayAlpha.Items.Add("====Alpha Format====");
                SetFormatCombo_(_cmbTextureFmtForGrayAlpha, LECore.LayoutEditorCore.PlatformDetail.GetFormatList(true, false));

                _cmbTextureFmtForGrayAlpha.SelectedValueChanged += Event_CmbTextureFmtFull_DropDownClosed;
                _cmbTextureFmtForGrayAlpha.ExpandDropDownWidth();
            }

            // 動的テクスチャ用フォーマット選択肢
            {
                _cmbTextureFmtForDynamic.Items.Add(new TextureFormatItem(TexImagePixelFmt.RGBA8));
                _cmbTextureFmtForDynamic.Items.Add(new TextureFormatItem(TexImagePixelFmt.R10G10B10A2));

                _cmbTextureFmtForDynamic.SelectedValueChanged += Event_CmbTextureFmtFull_DropDownClosed;
                _cmbTextureFmtForDynamic.ExpandDropDownWidth();
            }
        }

        /// <summary>
        /// ツリービューの内容が更新が必要かチェックします。
        /// </summary>
        bool IsTextureTreeViewObsolete_(CoreData.ITextureMgr texMgr)
        {
            // マネージャが発見できない場合
            if (texMgr == null)
            {
                // リセットの必要がありそうなら true を返します。
                return _lvwTexture.Items.Count != 0;
            }

            if (_lvwTexture.Items.Count == texMgr.ITextureImageSet.Count())
            {
                // 要素があれば true を返す。
                // もっと条件を厳しくしてもいいかもしれない。
                return _lvwTexture.Items.Count != 0;
            }
            return true;
        }

        /// <summary>
        /// UIに表示する文字列を取得します。
        /// </summary>
        internal static string GetPixcelFormatString_(TexImagePixelFmt fmt)
        {
            return LECore.LayoutEditorCore.PlatformDetail.GetPixcelFormatString(fmt);
        }

        /// <summary>
        /// バイト数文字列を取得
        /// </summary>
        string ByteString_(int byteNum)
        {
            float resultValue = byteNum;
            return resultValue.ToString();
        }

        /// <summary>
        /// ITextureImage の内容をノードに設定します。
        /// </summary>
        void SetTextureToItem_(TextureListViewItem dstItem, CoreData.ITextureImage srcTexImg, int imageIndex, bool inSubScene, TextureDirectory directory)
        {
            bool isReadonly = _ITextureMgr?.CheckReadOnlyByName(srcTexImg.Name) ?? false;
            // 更新
            dstItem.InternalTexImage = new InternalTexImage(srcTexImg);
            dstItem.Name = srcTexImg.Name;
            dstItem.Text = srcTexImg.Name;
            dstItem.ImageIndex = imageIndex;
            dstItem.StateImageIndex = 0;
            dstItem.ForeColor = isReadonly ? LECore.AppConstants.InactiveForeColor : LECore.AppConstants.ActiveForeColor;
            dstItem.InSubScene = inSubScene;
            dstItem.Directory = directory;

            // キャプチャテクスチャは背景を専用の色に変えて識別しやすくする
            if (srcTexImg.SourceType == TextureSourceType.Dynamic)
            {
                if (isReadonly)
                {
                    dstItem.BackColor = LECore.AppConstants.InactiveCaptureTextureBackColor;
                }
                else
                {
                    dstItem.BackColor = LECore.AppConstants.ActiveCatpureTextureBackColor;
                }
            }
            else
            {
                if (isReadonly)
                {
                    dstItem.BackColor = LECore.AppConstants.InactiveBackColor;
                }
                else
                {
                    dstItem.BackColor = LECore.AppConstants.ActiveBackColor;
                }
            }

            for (int i = dstItem.SubItems.Count; i < _lvwTexture.Columns.Count; i++)
            {
                dstItem.SubItems.Add("");
            }

            dstItem.SubItems[_TextureFormatListViewSubItemIndex].Text = dstItem.InSubScene ? GetPixcelFormatString_(srcTexImg.PixelFmt) : string.Empty;
            dstItem.SubItems[_TextureFormatListViewSubItemIndex].Tag = dstItem.InSubScene ? srcTexImg.PixelFmtIsFixed : true;
            dstItem.SubItems[_cmhTexSizeX.Index].Text = srcTexImg.Size.X.ToString();
            dstItem.SubItems[_cmhTexSizeY.Index].Text = srcTexImg.Size.Y.ToString();
            dstItem.SubItems[_cmhTexDataSize.Index].Text = dstItem.InSubScene ? ByteString_(srcTexImg.NativeTextureDataSize) : string.Empty;
            if (LayoutEditorCore.PlatformDetail.IsNXDllLoaded)
            {
                dstItem.SubItems[_cmhTexNativeSizeNX.Index].Text = dstItem.InSubScene ? ByteString_(srcTexImg.NativeTextureDataSizeNX) : string.Empty;
            }
            dstItem.SubItems[_cmhTexFilePath.Index].Text = srcTexImg.FilePath;
        }

        /// <summary>
        /// ディレクトリ選択コンボボックスの Tag に設定する情報
        /// </summary>
        private class DirectoryDropDownData
        {
            // フルパス
            public string path;

            // 相対パス
            public string relativePath;

            // サブディレクトリ用の示すラベル
            public string subdirectoryLabel;

            public TextureDirectory directory;
            public string error;
            public bool allDirectories;

            public bool IsEquivalent(DirectoryDropDownData data)
            {
                return path == data.path &&
                    subdirectoryLabel == data.subdirectoryLabel &&
                    directory == data.directory &&
                    error == data.error &&
                    allDirectories == data.allDirectories;
            }

            /// <summary>
            /// ラベルの取得
            /// </summary>
            public string GetLabel()
            {
                string label;
                if (directory == null)
                {
                    label = StringResMgr.Get("TEXTUREMGRWINDOW_IN_LAYOUT");
                }
                else
                {
                    if (subdirectoryLabel != null)
                    {
                        label = subdirectoryLabel;
                    }
                    else
                    {
                        label = directory.Label ?? string.Empty;
                    }
                }
                return label;
            }

            /// <summary>
            /// 項目を識別するための ID
            /// </summary>
            public string GetId()
            {
                if (directory == null)
                {
                    return "";
                }
                else
                {
                    return (directory.Label ?? string.Empty) + ":" + (relativePath ?? string.Empty) + ":" + allDirectories;
                }

            }
        }

        /// <summary>
        /// コンボボックスの更新
        /// selectItem が指定されていれば、選択も行う
        /// </summary>
        private void UpdateDirectoryCombobox(string selectedId)
        {
            var items = Enumerable.Repeat<TextureDirectory>(null, 1)
                .Concat(ViewManager?.AppSetting?.ProjectSettings?.TextureDirectories ?? Enumerable.Empty<TextureDirectory>())
                .SelectMany(x => GetDirectoryData(x)).ToArray();

            if (items.Length != _cmbDirectory.Items.Count ||
                items.Zip(
                    _cmbDirectory.Items.OfType<UIListControlItem>().Select(x => (DirectoryDropDownData)x.Data),
                    (x, y) => x.IsEquivalent(y)).Any(x => !x))
            {
                if (string.IsNullOrEmpty(selectedId) && _cmbDirectory.SelectedIndex != -1)
                {

                    selectedId = ((DirectoryDropDownData)((UIListControlItem)_cmbDirectory.SelectedItem).Data).GetId();
                }

                _cmbDirectory.BeginUpdate();
                _cmbDirectory.Items.Clear();

                foreach (var item in items)
                {
                    string label = item.GetLabel();

                    _cmbDirectory.Items.Add(new UIListControlItem(
                        label,
                        item));
                }

                // 幅を調整
                UIComboBox.ExpandDropDownWidth(_cmbDirectory.ComboBox);

                _cmbDirectory.SelectByCondition(x => selectedId == ((DirectoryDropDownData)((UIListControlItem)x).Data).GetId());
                if (_cmbDirectory.SelectedIndex == -1)
                {
                    _cmbDirectory.SelectedIndex = 0;
                }

                if (_cmbDirectory.Items.Count == 1)
                {
                    _cmbDirectory.Visible = false;
                }
                else
                {
                    _cmbDirectory.Visible = true;
                }

                _cmbDirectory.EndUpdate();
            }
        }

        /// <summary>
        /// directory 以下のディレクトリの情報の取得
        /// </summary>
        private IEnumerable<DirectoryDropDownData> GetDirectoryData(TextureDirectory directory)
        {
            var items = new List<DirectoryDropDownData>();
            DirectoryDropDownData rootItem = new DirectoryDropDownData { directory = directory, allDirectories = directory?.AllDirectories ?? false};
            try
            {
                string error;
                var rootPath = ExpandPath(directory, out error);
                if (rootPath != null && !rootPath.EndsWith(@"\"))
                {
                    rootPath += @"\";
                }

                rootItem.path = rootPath;
                rootItem.error = error;
                items.Add(rootItem);

                if (rootPath != null && directory.AllDirectories && directory.ShowDirectories)
                {
                    // *.tga だけだと余計なファイルも列挙される
                    var files = Directory.EnumerateFiles(rootPath, "*.tga", SearchOption.AllDirectories).Where(x => Path.GetExtension(x).ToLower() == ".tga");

                    var groups = files.GroupBy(x => Path.GetDirectoryName(x)).Select(x => x.Key.Substring(rootPath.Length)).OrderBy(x => x).ToArray();

                    var root = new Trie();
                    foreach (var group in groups)
                    {
                        root.Add(((IEnumerable<string>)group.Split(@"/\".ToArray())).GetEnumerator());
                    }

                    foreach (var node in root.Children)
                    {
                        node.Enumerate(1, rootPath, "", directory, items);
                    }
                }
            }
            catch (Exception e)
            {
                rootItem.error = e.Message;
                return Enumerable.Repeat(rootItem, 1);
            }

            return items;
        }

        // サブディレクトリ表示のための Trie
        private class Trie
        {
            public string Name;
            public List<Trie> Children = new List<Trie>();

            public void Add(IEnumerator<string> enumerator)
            {
                if (enumerator.MoveNext())
                {
                    if (Children.Count == 0 || Children[Children.Count - 1].Name != enumerator.Current)
                    {
                        Children.Add(new Trie() { Name = enumerator.Current });
                    }

                    Children[Children.Count - 1].Add(enumerator);
                }
                else
                {
                    // 末尾を示す要素
                    Children.Add(new Trie() { });
                }
            }

            // 要素を追加
            public void Enumerate(int depth, string basePath, string relativePath, TextureDirectory directory, List<DirectoryDropDownData> items)
            {
                string name = "";
                Trie branch = this;
                string splitter = "";

                // 分岐がないところはまとめる
                while(branch.Children.Count == 1)
                {
                    if (!string.IsNullOrEmpty(branch.Name))
                    {
                        name += splitter + branch.Name;
                        splitter = @"\";
                        relativePath += branch.Name + @"\";
                    }
                    branch = branch.Children[0];
                }


                if (!string.IsNullOrEmpty(branch.Name))
                {
                    name += splitter + branch.Name;
                    relativePath += branch.Name + @"\";
                }

                if (name == string.Empty && branch.Children.Count == 0)
                {
                    name = ".";
                }

                name = new string(' ', depth * 2) + name;

                items.Add(
                    new DirectoryDropDownData
                    {
                        path = basePath + relativePath,
                        relativePath = relativePath,
                        allDirectories = branch.Children.Count > 0,
                        directory = directory,
                        subdirectoryLabel = name,
                    }
                    );

                foreach (var child in branch.Children)
                {
                    child.Enumerate(depth + 1, basePath, relativePath, directory, items);
                }
            }
        }


        /// <summary>
        /// 更新します。
        /// マネージャ更新イベントハンドラで呼び出されます。
        /// </summary>
        public void UpdateState(string directoryId = null)
        {
            if (_UpdatingState)
            {
                return;
            }

            _UpdatingState = true;
            LEToolWindow.DbgReportUpdate(this);

            // ディレクトリ選択コンボボックスの更新
            UpdateDirectoryCombobox(directoryId);

            _lvwTexture.BeginUpdate();
            var data = ((UIListControlItem)_cmbDirectory.SelectedItem).Data as DirectoryDropDownData;
            var textureDirectory = data.directory;

            // 表示対象の列挙
            ITextureImage[] textureImages;

            string errorMessage = _ITextureMgr != null ? data.error: string.Empty;

            if (textureDirectory != null)
            {
                _ShowTexturesInSubScene = false;
                try
                {
                    // Layout で使用しているテクスチャはそのまま使用する
                    Dictionary<string, ITextureImage> pathToRegisteredTextuers = _ITextureMgr?.ITextureImageSet?.Where(x => x.ICaptureTexture == null).ToDictionary(x => x.FilePath.ToLower()) ?? new Dictionary<string, ITextureImage>();

                    // 項目も更新する
                    string path = data.path;

                    if (!string.IsNullOrEmpty(path) && Directory.Exists(path))
                    {
                        var files = Directory.EnumerateFiles(path, "*.tga", data.allDirectories ? SearchOption.AllDirectories : SearchOption.TopDirectoryOnly).ToArray();
                        textureImages = files.Where(x => Path.GetExtension(x).ToLower() == ".tga").Select(x =>
                        {
                            ITextureImage image;
                            if (!pathToRegisteredTextuers.TryGetValue(x.ToLower(), out image))
                            {
                                if (!pathToImage.TryGetValue(x.ToLower(), out image))
                                {
                                    image = TextureImageFactory.BuildTextureImageFromTga(x);
                                    pathToImage[x.ToLower()] = image;
                                }
                            }
                            return image;
                        }).ToArray();
                        _DirectoryPath = path;
                    }
                    else
                    {
                        textureImages = new ITextureImage[0];
                        _DirectoryPath = null;
                    }
                }
                catch (Exception e)
                {
                    textureImages = new ITextureImage[0];
                    _DirectoryPath = null;
                    errorMessage = e.Message;
                }
            }
            else
            {
                _ShowTexturesInSubScene = true;
                if (_ITextureMgr != null)
                {
                    textureImages = _ITextureMgr.ITextureImageSet.Where(x => !x.IsHidingFromList).ToArray();
                }
                else
                {
                    textureImages = new ITextureImage[0];
                }
                _DirectoryPath = null;
            }

            // エラーのアイコンの更新
            if (!string.IsNullOrEmpty(errorMessage))
            {
                lblError.ToolTipText = errorMessage;
                lblError.Visible = true;
            }
            else
            {
                lblError.Visible = false;
            }

            // エラーがあった場合やレイアウトが開かれていないと項目が表示できない場合は無効にする
            _lvwTexture.Enabled = string.IsNullOrEmpty(errorMessage) &&
                ((textureDirectory != null && textureDirectory.Base != TextureDirectory.BaseType.Layout) || _ITextureMgr != null);

            var selectedItems = new HashSet<TextureListViewItem>(_lvwTexture.SelectedItems.OfType<TextureListViewItem>());

            bool updateImages = false;

            HashSet<ITextureImage> textureImagesInManager = new HashSet<ITextureImage>(_ITextureMgr?.ITextureImageSet ?? Enumerable.Empty<ITextureImage>());

            //---------------------------------------------------
            // 削除されたテクスチャに対応するノードを取り除きます。
            Func<TextureListViewItem, ITextureImage, bool> isSame = (x, y) =>
            {
                return (_ShowTexturesInSubScene ? x.Name == y.Name : x.InternalTexImage.FilePath == y.FilePath) && x.Directory == textureDirectory;
            };

            Dictionary<string, TextureListViewItem> oldItems = _lvwTexture.Items.OfType<TextureListViewItem>()
                .Where(x => x.Directory == textureDirectory)
                .ToDictionary(x => _ShowTexturesInSubScene ? x.Name : x.InternalTexImage.FilePath);
            Dictionary<string, ITextureImage> idToTextureImage = textureImages.ToDictionary(x => _ShowTexturesInSubScene ? x.Name : x.FilePath);

            var deletedItems = _lvwTexture.Items.OfType<TextureListViewItem>()
                .Where(x => x.Directory != textureDirectory || !idToTextureImage.ContainsKey(_ShowTexturesInSubScene ? x.Name : x.InternalTexImage.FilePath))
                .OrderByDescending(x => x.ImageIndex).ToArray();
            foreach (var item in deletedItems)
            {
                updateImages = true;
                _lvwTexture.Items.Remove(item);
            }

            //---------------------------------------------------
            // すべてのテクスチャについて
            foreach (CoreData.ITextureImage texImg in textureImages)
            {
                // テクスチャ名から登録済みか検索します。
                TextureListViewItem item;

                oldItems.TryGetValue(_ShowTexturesInSubScene ? texImg.Name: texImg.FilePath, out item);

                if (item != null)
                {
                    // 登録済みの場合は、更新の必要があれば更新します。
                    InternalTexImage internalTexImage = item.InternalTexImage;
                    if (internalTexImage.Obsolete ||
                        internalTexImage.ITextureImage != texImg)
                    {
                        updateImages = true;

                        // 更新
                        SetTextureToItem_(item, texImg, item.ImageIndex, textureImagesInManager.Contains(texImg), textureDirectory);

                        item.Thumbnail = TextureThumbnailHelper.CreateThumbnail(texImg.Size.X, texImg.Size.Y, texImg.GDIBitmap, _ThubmnailSize, _ThubmnailSize);
                    }
                }
                else
                {
                    updateImages = true;

                    // 登録済みではない場合は、新規登録します。
                    item = new TextureListViewItem(texImg.Name);
                    SetTextureToItem_(item, texImg, 0, textureImagesInManager.Contains(texImg), textureDirectory);

                    item.Thumbnail = TextureThumbnailHelper.CreateThumbnail(texImg.Size.X, texImg.Size.Y, texImg.GDIBitmap, _ThubmnailSize, _ThubmnailSize);

                    _lvwTexture.Items.Add(item);
                }
            }

            if (updateImages)
            {
                if (_lvwTexture.LargeImageList.Images.Count > 0)
                {
                    _lvwTexture.LargeImageList.Images.Clear();
                    _lvwTexture.SmallImageList.Images.Clear();
                }

                // リストビューのバグ回避らしき動作を回避するために一度要素数を取得する。
                // 内部で、ListView.ApplyUpdateCachedItems が呼ばれる。
                var c = _lvwTexture.Items.Count;

                var images = _lvwTexture.Items.OfType<TextureListViewItem>().Select(x => x.Thumbnail).ToArray();
                if (images.Any())
                {
                    _lvwTexture.SmallImageList.Images.AddRange(images);
                    _lvwTexture.LargeImageList.Images.AddRange(images);

                    // 画像と同じ順になっているので、インデックスを割り当てる
                    for (int i = 0; i < _lvwTexture.Items.Count; i++)
                    {
                        _lvwTexture.Items[i].ImageIndex = i;
                        Debug.Assert(((TextureListViewItem)_lvwTexture.Items[i]).Thumbnail != _lvwTexture.LargeImageList.Images[i]);
                    }
                }
            }

            // 選択ノードの復元を試みます。
            foreach (TextureListViewItem lvi in _lvwTexture.Items)
            {
                if (selectedItems.Contains(lvi))
                {
                    Debug.Assert(lvi.Selected);
                    lvi.Selected = true;
                }
            }
            _lvwTexture.EndUpdate();

            // -------------- ボタン状態の更新
            UpdateState_ButtonState_();
            UpdateStatusBar_();

            // PictureBoxの画像を更新します。
            UpdateTexturePictureBox_();

            _UpdatingState = false;
        }



        /// <summary>
        /// 環境変数を展開して相対パスを展開する。
        /// </summary>
        private string ExpandPath(TextureDirectory directory, out string error)
        {
            error = null;
            try
            {
                string path;
                if (directory == null)
                {
                    return null;
                }

                if (directory.Base == TextureDirectory.BaseType.Layout &&
                    !(_CurrentSubScene?.OwnerDocument?.IsFileSaved ?? false))
                {
                    error = StringResMgr.Get("TEXTUREMGRWINDOW_NOLAYOUTPATH");
                    return null;
                }

                var root = directory.Base == TextureDirectory.BaseType.Project ? Path.GetDirectoryName(ViewManager.AppSetting.ProjectSettings.FilePath) :
                    directory.Base == TextureDirectory.BaseType.Layout && (_CurrentSubScene?.OwnerDocument?.IsFileSaved ?? false) ? Path.GetDirectoryName(_CurrentSubScene.OwnerDocument.LastSavedFilePath) :
                    null;

                path = directory.Path ?? string.Empty;
                path = Environment.ExpandEnvironmentVariables(path);

                if (!string.IsNullOrEmpty(root))
                {
                    path = Path.Combine(root, path);
                }

                path = Path.GetFullPath(path);

                if (!Directory.Exists(path))
                {
                    error = StringResMgr.Get("FLOLDER_NOT_FOUND", path);
                    return null;
                }

                return path;
            }
            catch (Exception e)
            {
                error = e.Message;
                return null;
            }
        }

        /// <summary>
        /// ボタン状態の更新
        /// </summary>
        void UpdateState_ButtonState_()
        {
            bool readOnlyNotSelected = !ReadOnlySelected();
            _tsbLoad.Enabled = _CurrentSubScene != null && _ShowTexturesInSubScene;
            _tsbDeleteSelected.Enabled = _lvwTexture.SelectedItems.Count > 0 && readOnlyNotSelected && _ShowTexturesInSubScene;
            _tsbCreatePane.Enabled = _CanCreateNewPane;
            _tsbUpdate.Enabled = _lvwTexture.Items.Count > 0 && _CurrentSubScene != null;
            _tsbSelectUnused.Enabled = ExsitsUnusedTexture_() && _ShowTexturesInSubScene;

            bool filePathOperationEnabled = false;
            if (_lvwTexture.SelectedItems.Count == 1)
            {
                var internalTexImg = ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage;
                if (internalTexImg.ITextureImage.SourceType == TextureSourceType.File)
                {
                    filePathOperationEnabled = true;
                }
            }

            _tsbResetFilePath.Enabled = filePathOperationEnabled && readOnlyNotSelected && _ShowTexturesInSubScene;
            _tsbOpenByExplorer.Enabled = filePathOperationEnabled || (_lvwTexture.SelectedItems.Count == 0&& !string.IsNullOrEmpty(_DirectoryPath));
            _tsbOpenSrcFileByExplorer.Enabled = filePathOperationEnabled && IsSourceFileOfSigleSelectedTgaFound_();
        }

        /// <summary>
        /// バックグラウンドで未使用テクスチャの UI 状態を更新する。
        /// </summary>
        void Event_backgroundUnusedTexFinder_DoWork(object sender, DoWorkEventArgs e)
        {
            if (_backgroundUnusedTexUIUpdater.CancellationPending)
            {
                e.Cancel = true;
                return;
            }

            bool bExsitsUnusedTexture = ExsitsUnusedTexture_();
            this.BeginInvoke(new Action(() => _tsbSelectUnused.Enabled = bExsitsUnusedTexture));
        }

        /// <summary>
        /// 未使用テクスチャがあるかどうか？
        /// </summary>
        private bool ExsitsUnusedTexture_()
        {
            var currentSubScene = _CurrentSubScene;
            if (currentSubScene == null)
            {
                return false;
            }

            if (currentSubScene.ITextureMgr.ITextureImageSet.Count() <= 0)
            {
                return false;
            }

            foreach (ITextureImage image in currentSubScene.ITextureMgr.ITextureImageSet)
            {
                if (SubSceneHelper.GetUseTexturePaneSet(currentSubScene, image.Name).Length == 0)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 選択セットが読み込み専用のテクスチャを持っているか？
        /// </summary>
        private bool ReadOnlySelected()
        {
            if (_ITextureMgr != null)
            {
                foreach (TextureListViewItem selectedItem in _lvwTexture.SelectedItems)
                {
                    var texImage = selectedItem?.InternalTexImage.ITextureImage;
                    if (texImage != null)
                    {
                        if (_ITextureMgr.CheckReadOnlyByName(texImage.Name))
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// ステータスバーの更新
        /// </summary>
        void UpdateStatusBar_()
        {
            if (_lvwTexture.SelectedIndices.Count != 0)
            {
                int index = _lvwTexture.SelectedIndices[0];
                InternalTexImage texImg = ((TextureListViewItem)_lvwTexture.Items[index]).InternalTexImage;
                _lblSelectedItemInfo.Text = string.Format("{0} ({1})", texImg.ITextureImage.Name, texImg.ITextureImage.FilePath);
            }
            else
            {
                _lblSelectedItemInfo.Text = _DirectoryPath ?? " ";
            }
        }

        #region イベントハンドラ

        /// <summary>
        /// マネージャにテクスチャを読み込みます。
        /// </summary>
        /// <param name="fileNameSet"></param>
        /// <returns>読み込まれたファイルの数</returns>
        int LoadTexturesToManager_(string[] fileNameSet)
        {
            Array.Sort(fileNameSet, delegate (string lhs, string rhs)
           {
               return String.Compare(lhs, rhs, true);
           });

            CoreData.ITextureImage[] resultSet =
                _TextureMgrManipulator.RegisterITextureImageSetFromFile(fileNameSet);

            if (fileNameSet.Length != resultSet.Length)
            {
                try
                {
                    var errorFiles = fileNameSet.Where(
                        x => !resultSet.Any(y => y.IsSameName(Path.GetFileNameWithoutExtension(x))));

                    ShowLoadTextureErrorDialog(errorFiles);
                }
                catch
                {
                }
            }

            return resultSet.Length;
        }

        /// <summary>
        /// 読み込みボタンハンドラ
        /// ファイルダイアログを表示して、
        /// tga ファイルを選択し、新規にテクスチャを登録します。
        /// </summary>
        private void Event_BtnLoad_Click(object sender, System.EventArgs e)
        {
            OpenFileDialog openFileDialog = new OpenFileDialog();

            openFileDialog.Filter = "tga files (*.tga)|*.tga";
            openFileDialog.RestoreDirectory = true;
            openFileDialog.Multiselect = true;

            if (!string.IsNullOrEmpty(_lastFileLoadedDirectoryName) &&
                Directory.Exists(_lastFileLoadedDirectoryName))
            {
                openFileDialog.InitialDirectory = SaveLoadUIHelper.GetAbsolutePathForExistDirectory(_lastFileLoadedDirectoryName);
            }

            if (openFileDialog.ShowDialog() == DialogResult.OK)
            {
                LoadTexturesToManager_(openFileDialog.FileNames);

                var filePath = openFileDialog.FileNames.FirstOrDefault();
                if (filePath != null)
                {
                    _lastFileLoadedDirectoryName = Path.GetDirectoryName(filePath);
                }
            }
        }

        /// <summary>
        /// 選択項目から、ペインを新規登録します。
        /// </summary>
        void RegisterNewPanes_()
        {
            if (_CanCreateNewPane)
            {
                var errors = new List<string>();
                _CurrentSubScene.BeginMassiveModify();
                foreach (int itemIdx in _lvwTexture.SelectedIndices)
                {
                    var item = (TextureListViewItem)_lvwTexture.Items[itemIdx];
                    if (!item.InSubScene)
                    {
                        var result = _TextureMgrManipulator.RegisterITextureImageSetFromFile(new[] { item.InternalTexImage.ITextureImage.FilePath });
                        if (result.Length == 0)
                        {
                            errors.Add(item.InternalTexImage.ITextureImage.FilePath);
                            continue;
                        }
                    }

                    string textureName = item.Text;

                    SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                    subSceneMnp.BindTarget(_CurrentSubScene);
                    subSceneMnp.AddPicturePane(
                        ViewManager.AppSetting.ProjectSettings.GetNewPicturePaneName(),
                        textureName,
                        ViewManager.AppSetting.ProjectSettings.DefaultPaneNames.UseTextureNameAsPicturePaneName);
                }
                _CurrentSubScene.EndMassiveModify();

                if (errors.Count > 0)
                {
                    ShowLoadTextureErrorDialog(errors);
                }
            }
        }

        private void ShowLoadTextureErrorDialog(IEnumerable<string> files)
        {
            MessageReportDlg dlg = new MessageReportDlg(LayoutEditor.Forms.MessageReportDlg.ButtonTypes.Ok, false);
            dlg.Title = StringResMgr.Get("ADDTEXTURE");
            dlg.Message = StringResMgr.Get("SAMENAMETEXTURE") + "\r\n\r\n" + string.Join("\r\n", files);
            dlg.ShowDialog(this);

        }

        /// <summary>
        /// ペイン作成ボタンハンドラ
        /// 選択中のテクスチャを使用して、Pictureペインを作成します。
        /// </summary>
        private void Event_BtnCreatePane_Click(object sender, System.EventArgs e)
        {
            RegisterNewPanes_();
        }

        /// <summary>
        /// 更新時注意メッセージの生成
        /// </summary>
        string GetTextureUpdateNotification_(ITextureImage texImg, bool hasAlphaBrofore, bool isGrayBrofore)
        {
            string notification = string.Empty;

            if (texImg.SrcImageHasAlphaBit != hasAlphaBrofore)
            {
                notification += " ---- ";
                notification += hasAlphaBrofore ? "alpha[YES => NO]" : "alpha[NO => YES]";
            }

            if (texImg.SrcImageIsGrayScale != isGrayBrofore)
            {
                notification += " ---- ";
                notification += isGrayBrofore ? "grayscale[YES => NO]" : "grayscale[NO => YES]";
            }

            return notification;
        }

        private static string GetErrorMessageIdForTexturePathChange_(SubSceneHelper.ChangeTexturePathError error)
        {
            switch (error)
            {
                case SubSceneHelper.ChangeTexturePathError.FailToChange: return "TEXTURE_ERROR_RESETFILE";
                case SubSceneHelper.ChangeTexturePathError.InvalidSameNameTexture: return "TEXTURE_ERROR_RESETFILE_DUPLICATED";
                default: Debug.Assert(false); return string.Empty;
            }
        }

        /// <summary>
        /// ファイルパスの再設定
        /// </summary>
        private void Event_BtnResetFilePath_Click(object sender, EventArgs e)
        {
            // 選択アイテムのテクスチャがひとつの時のみ
            if (_lvwTexture.SelectedItems.Count != 1)
            {
                return;
            }

            var internalTexImg = ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage;
            var texImage = internalTexImg.ITextureImage;

            // ファイルダイアログを表示して、
            // 新しいファイル名を選択してもらう
            var opd = new OpenFileDialog { Multiselect = false };

            if (!string.IsNullOrEmpty(texImage.FilePath))
            {
                opd.InitialDirectory = Path.GetDirectoryName(texImage.FilePath);
                opd.FileName = Path.GetFileName(texImage.FilePath);
            }

            if (opd.ShowDialog(this) == DialogResult.OK
                && Path.GetFullPath(opd.FileName) != Path.GetFullPath(texImage.FilePath))
            {
                var newFilePath = opd.FileName;
                var hasAlphaBrofore = texImage.SrcImageHasAlphaBit;
                var isGrayBrofore = texImage.SrcImageIsGrayScale;
                // ダイアログ
                var reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);

                SubSceneHelper.ChangeTexturePathError error;
                var bResult = SubSceneHelper.ChangeTexturePath(LECore.LayoutEditorCore.Scene.CurrentISubScene, newFilePath, texImage, out error);

                // ダイアログ
                reportDlg.Title = StringResMgr.Get("TEXTURE_MSG_DLG_TITLE");
                if (bResult)
                {
                    reportDlg.Message = StringResMgr.Get("TEXTURE_MSG_TEXUPDATED", texImage.Name);
                    reportDlg.ShowDialog(this);

                    var notification = GetTextureUpdateNotification_(texImage, hasAlphaBrofore, isGrayBrofore);
                    if (!string.IsNullOrEmpty(notification))
                    {
                        reportDlg.Message = StringResMgr.Get(
                            "TEXTURE_MSG_TEXUPDATED_FMTCHANGE",
                            texImage.Name + notification);
                        reportDlg.ShowDialog(this);
                    }
                }
                else
                {
                    string errorMessageId = GetErrorMessageIdForTexturePathChange_(error);

                    reportDlg.Message = StringResMgr.Get(errorMessageId);
                    reportDlg.ShowDialog(this);
                }
            }

            Event_LvwTexture_SelectedIndexChanged(null, EventArgs.Empty);
        }

        /// <summary>
        /// 最新の情報に更新
        /// </summary>
        private void Event_BtnUpdate_Click(object sender, System.EventArgs e)
        {
            if (_cmbDirectory.SelectedIndex != 0)
            {
                // 一旦ディレクトリ内の項目を更新する
                UpdateState();
            }

            Debug.Assert(_ITextureMgr != null);

            List<string> updatedTexSet = new List<string>();
            List<string> fmtChangeTexSet = new List<string>();

            _CurrentSubScene.BeginMassiveModify();
            _TextureMgrManipulator.BeginUpdate();
            // サブシーン内のテクスチャの更新
            foreach (ITextureImage texImg in _ITextureMgr.ITextureImageSet)
            {
                bool hasAlphaBrofore = texImg.SrcImageHasAlphaBit;
                bool isGrayBrofore = texImg.SrcImageIsGrayScale;

                if (_TextureMgrManipulator.UpdateTextureImage(texImg))
                {
                    string notification = GetTextureUpdateNotification_(texImg, hasAlphaBrofore, isGrayBrofore);
                    updatedTexSet.Add(texImg.Name);

                    if (!string.IsNullOrEmpty(notification))
                    {
                        fmtChangeTexSet.Add(texImg.Name + notification);
                    }
                }
            }
            _TextureMgrManipulator.EndUpdate();

            // サブシーン外のテクスチャの更新
            foreach (var treeViewItem in _lvwTexture.Items.OfType<TextureListViewItem>().Where(x => !x.InSubScene))
            {
                // 数が多くなるのでダイアログ表示には含めない
                TextureImageFactory.UpdateTextureImage(treeViewItem.InternalTexImage.ITextureImage, treeViewItem.InternalTexImage.FilePath);
            }

            _CurrentSubScene.EndMassiveModify();

            // 正しく、更新されたテクスチャがあれば、レポートします。
            if (updatedTexSet.Count > 0)
            {
                StringBuilder sb = new StringBuilder();


                foreach (string name in updatedTexSet)
                {
                    sb.Append("    " + name + System.Environment.NewLine);
                }

                MessageReportDlg reportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                reportDlg.Title = StringResMgr.Get("TEXTURE_MSG_DLG_TITLE");

                // 更新したテクスチャ名を報告します。
                reportDlg.Message = StringResMgr.Get("TEXTURE_MSG_TEXUPDATED", sb.ToString());
                reportDlg.ShowDialog(this);

                // 形式変更があったテクスチャを報告します。
                if (fmtChangeTexSet.Count > 0)
                {
                    StringBuilder sb2 = new StringBuilder();
                    foreach (string name in fmtChangeTexSet)
                    {
                        sb2.Append("    " + name + System.Environment.NewLine);
                    }

                    reportDlg.Message = StringResMgr.Get("TEXTURE_MSG_TEXUPDATED_FMTCHANGE", sb2.ToString());
                    reportDlg.ShowDialog(this);
                }
            }
        }

        /// <summary>
        /// リストボックス選択アイテム変更ハンドラ
        ///
        /// リストボックス選択アイテムの状況に対応して、
        /// ペイン作成ボタンの状態を更新します。
        /// </summary>
        private void Event_LvwTexture_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            if (_UpdatingState)
            {
                // 内部で更新中は処理しない
                return;
            }

            // -------------- ボタン状態の更新
            UpdateState_ButtonState_();
            UpdateStatusBar_();

            // PictureBoxの画像を更新します。
            UpdateTexturePictureBox_();
        }

        /// <summary>
        /// ピクチャボックスを更新します。
        /// </summary>
        private void UpdateTexturePictureBox_()
        {
            if (_lvwTexture.SelectedIndices.Count != 0)
            {
                var texImg = ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage.ITextureImage;

                if (texImg != null)
                {
                    Debug.Assert(texImg.Size.X != 0.0f && texImg.Size.Y != 0.0f);
                    Bitmap newImage = TextureThumbnailHelper.CreateThumbnail(
                        texImg.Size.X, texImg.Size.Y, texImg.GDIBitmap, _pctTexture.Width, _pctTexture.Height); ;

                    PictureBoxHelper.SetImageWithDispose(_pctTexture, newImage);
                }
            }
            else
            {
                PictureBoxHelper.SetImageWithDispose(_pctTexture, null);
            }
        }

        /// <summary>
        /// 未使用テクスチャを選択。
        /// </summary>
        private void Event_BtnSelectUnused_Click(object sender, System.EventArgs e)
        {
            foreach (TextureListViewItem ivi in _lvwTexture.Items)
            {
                InternalTexImage internalTexImg = ivi.InternalTexImage;
                IPane[] userPaneSet = SubSceneHelper.GetUseTexturePaneSet(_CurrentSubScene, internalTexImg.ITextureImage.Name);
                ivi.Selected = userPaneSet.Length == 0;
            }
            _lvwTexture.Select();
        }


        #region 選択ペインの消去
        /// <summary>
        /// ペインを消去する旨をユーザに通知します。
        /// </summary>
        DialogResult ShowMessageBoxPaneRemove_(
                                               InternalTexImage internalTexImg,
                                               Hashtable subScenePaneTable)
        {
            // ペインを消去する旨をユーザに通知します。
            // TEXTURE_MSG_DELTETEXTURE テクスチャ{0}を削除することで、以下のペインが削除されます。\nやり直しが行えませんが削除してもよろしいですか。\n
            string msg = LayoutEditor.StringResMgr.Get("TEXTURE_MSG_DELTETEXTURE", internalTexImg.ITextureImage.Name);

            IPane[] paneSet = subScenePaneTable[_CurrentSubScene] as IPane[];
            if (paneSet != null)
            {
                msg += "\n";
                foreach (IPane pane in paneSet)
                {
                    msg += string.Format("\t{0}\n", pane.PaneName);
                }
            }

            // TEXTURE_MSG_DLG_TITLE  テクスチャ - メッセージ
            return MessageBox.Show(this, msg, LayoutEditor.StringResMgr.Get("TEXTURE_MSG_DLG_TITLE"), MessageBoxButtons.YesNoCancel);

        }

        /// <summary>
        /// テクスチャを使用しているペインセットを消去します。
        /// </summary>
        void ClearPaneSetUsingTextureImage_(Hashtable subScenePaneTable)
        {
            // ペインを消去したのち、
            foreach (ISubScene subScene in subScenePaneTable.Keys)
            {
                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget(subScene);
                subSceneMnp.ISubScene.BeginMassiveModify();

                subSceneMnp.BeginSelectSetChange();
                subSceneMnp.ResetSelectedSet();

                IPane[] paneSet = subScenePaneTable[subScene] as IPane[];
                foreach (IPane pane in paneSet)
                {
                    subSceneMnp.SelectPanesByPaneRef(pane);
                }
                subSceneMnp.EndSelectSetChange();

                subSceneMnp.DeleteSelectedPanes();
                subSceneMnp.ISubScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// テクスチャをマネージャから消去します。
        /// </summary>
        void ClearTextureImageSet_(ArrayList internalTexImg)
        {
            // テクスチャ、を消去します。
            TextureMgrManipulator textureMgrMnp = new TextureMgrManipulator();
            textureMgrMnp.BindTarget(_ITextureMgr);

            // 引数として適切な型の配列に、コピーします。
            List<ITextureImage> textureImageSet = new List<ITextureImage>();
            for (int i = 0; i < internalTexImg.Count; i++)
            {
                InternalTexImage internalTexImage = (InternalTexImage)internalTexImg[i];

                // キャプチャテクスチャはペン削除時に呼び出される TryToDeleteCaptureTexture 内で削除済みのため、ここでは処理しない。
                if (internalTexImage.ITextureImage.SourceType == TextureSourceType.File)
                {
                    textureImageSet.Add(internalTexImage.ITextureImage);
                }
            }

            textureMgrMnp.RemoveTextureSet(textureImageSet.ToArray());
        }

        /// <summary>
        /// 選択を削除
        /// </summary>
        private void Event_BtnDeleteSelected_Click(object sender, System.EventArgs e)
        {
            ArrayList clearedTextureSet = new ArrayList();

            // 全ての選択アイテムのテクスチャについて
            foreach (TextureListViewItem ivi in _lvwTexture.Items)
            {
                if (!ivi.Selected)
                {
                    continue;
                }

                Hashtable subScenePaneTable = new Hashtable();
                InternalTexImage internalTexImg = ivi.InternalTexImage;

                ISubScene subScene = _CurrentSubScene;
                // テクスチャを使用しているかキャプチャもとになっているペインを取得します。
                IPane[] paneSet =
                    SubSceneHelper.GetUseTextureAndCapturePaneSet(subScene, internalTexImg.ITextureImage.Name);

                if (paneSet.Length > 0)
                {
                    subScenePaneTable.Add(subScene, paneSet);
                }

                // テクスチャを使用しているペインが存在する場合は
                if (subScenePaneTable.Keys.Count > 0)
                {
                    // ペインを消去する旨をユーザに通知します。
                    DialogResult res = ShowMessageBoxPaneRemove_(internalTexImg, subScenePaneTable);

                    // ユーザがペイン消去を承諾した場合には、
                    if (res == DialogResult.Yes)
                    {
                        ClearPaneSetUsingTextureImage_(subScenePaneTable);

                        // 消去項目に登録します。
                        clearedTextureSet.Add(internalTexImg);
                    }
                }
                else
                {
                    // テクスチャ、を消去します。
                    // 消去項目に登録します。
                    clearedTextureSet.Add(internalTexImg);
                }
            }

            // テクスチャの消去があった場合は、
            // コマンドバッファ、クリップボードの内容を消去します。
            if (clearedTextureSet.Count != 0)
            {
                // 実際に消去します。
                ClearTextureImageSet_(clearedTextureSet);

                ISubScene subScene = _CurrentSubScene;

                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget(subScene);
                subSceneMnp.ResetUndoState();

                SceneManipulator sceneMnp = new SceneManipulator();
                sceneMnp.BindTarget(LECore.LayoutEditorCore.Scene);
                sceneMnp.ResetClipBoard();
            }
        }
        #endregion 選択ペインの消去


        /// <summary>
        /// ダブルクリック
        /// </summary>
        private void Event_IvwTexture_DoubleClick(object sender, System.EventArgs e)
        {
            ListViewItem item;

            // フォーマット変更中のペイン生成を防ぐ。
            var pos = _lvwTexture.PointToClient(new Point(MousePosition.X, MousePosition.Y));
            int idx = ListViewHelper.GetSubItemAt(_lvwTexture, pos.X, pos.Y, out item);
            if (idx == _TextureFormatListViewSubItemIndex)
            {
                return;
            }

            RegisterNewPanes_();
        }

        #region リストビューへのファイルドロップ
        /// <summary>
        /// ドロップ：Enter
        /// </summary>
        private void Event_LvwTexture_DragEnter(object sender, System.Windows.Forms.DragEventArgs e)
        {
            var effect = DragDropEffects.None;
            if (e.Data.GetDataPresent(DataFormats.FileDrop) && _ITextureMgr != null)
            {
                // テクスチャファイルをドロップ可能にします。
                string[] files = (e.Data.GetData(DataFormats.FileDrop) as string[]).Where(x => AppConstants.IsTextureFile(x)).ToArray();
                if (files.Length > 0)
                {
                    effect = DragDropEffects.All;
                }
            }
            e.Effect = effect;
        }

        /// <summary>
        /// ファイルドロップ
        /// </summary>
        private void Event_LvwTexture_DragDrop(object sender, System.Windows.Forms.DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                string[] files = (e.Data.GetData(DataFormats.FileDrop) as string[]).Where(x => AppConstants.IsTextureFile(x)).ToArray();
                List<string> loadFileSet = new List<string>();
                foreach (string fileName in files)
                {
                    if (File.Exists(fileName))
                    {
                        loadFileSet.Add(fileName);
                    }
                    else
                    {
                        // 普通は起きないはず
                        Debug.Assert(false);
                    }
                }

                // 存在するものだけ読み込みます。
                if (loadFileSet.Count != 0)
                {
                    var result = LoadTexturesToManager_(loadFileSet.ToArray());

                    if (result > 0 && _cmbDirectory.SelectedIndex != 0)
                    {
                        _cmbDirectory.SelectedIndex = 0;
                    }
                }

            }
        }
        #endregion リストビューへのファイルドロップ

        /// <summary>
        /// テクスチャのドラッグドロップ用の情報
        /// </summary>
        public class DragDropItem
        {
            // テクスチャ名
            public string TextureName { get; set; }

            // ドロップ先の SubScene と一致するかを比較する
            public ISubScene SubScene { get; set; }

            // サブシーンに含まれるか？
            public bool InSubScne { get; set; }

            // ファイルパス
            public string FilePath { get; set; }
        }

        private void Event_LvwTexture_ItemDrag(object sender, System.Windows.Forms.ItemDragEventArgs e)
        {
            TextureListViewItem listViewItem = e.Item as TextureListViewItem;
            if (listViewItem != null)
            {
                DoDragDrop(new DragDropItem
                {
                    TextureName = listViewItem.Text,
                    SubScene = _CurrentSubScene,
                    InSubScne = listViewItem.InSubScene,
                    FilePath = listViewItem.InternalTexImage.ITextureImage.FilePath,
                }, DragDropEffects.Copy);
            }
        }

        /// <summary>
        /// KeyDown
        /// </summary>
        private void Event_LvwTexture_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyData == Keys.Delete )
            {
                if (_tsbDeleteSelected.Enabled)
                {
                    Event_BtnDeleteSelected_Click(sender, null);
                }
            }
        }

        #endregion イベントハンドラ




        #region ILEToolWindow メンバ

        /// <summary>
        /// シーン更新イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(
                                                  object sender,
                                                  CoreData.SceneModifyEventArgs e)
        {
            switch (e.Target)
            {
                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                    // 親子階層ビューの更新
                    // サブシーン固有のデータを表示しているViewは更新する必要があります。
                    UpdateState();
                    break;
                case SceneModifyEventArgs.Kind.TextureManager:
                    UpdateState();
                    break;
                case SceneModifyEventArgs.Kind.PaneModify:
                case SceneModifyEventArgs.Kind.PaneAnimModify:
                case SceneModifyEventArgs.Kind.PaneAnimAddRemove:
                case SceneModifyEventArgs.Kind.PaneAdd:
                case SceneModifyEventArgs.Kind.PaneRemove:
                    // 頻繁に起こるイベントでレスポンス低下しないために、バックグラウンドでUIを更新する。
                    // 連続でイベントが起こった場合は、スルーしているので原理的にはUI状態が食い違う可能性があるが
                    // 実際には短時間で処理が完了するので問題になっていない。
                    if (!_backgroundUnusedTexUIUpdater.IsBusy)
                    {
                        _backgroundUnusedTexUIUpdater.RunWorkerAsync();
                    }
                    break;
                default:
                    // 何もしません。
                    break;
            }
            base.OnSceneModifyHandler(sender, e);
        }

        /// <summary>
        /// 設定書き出し
        /// </summary>
        public sealed override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            // リスト行の幅の書き出し
            setting.SaveColumnPotisionSettings(_lvwTexture);
            setting.AddUserData("IsDetailView", (_lvwTexture.View == View.Details).ToString());

            if (option.AlsoSaveOtherThanWorkspace)
            {
                setting.AddUserData("LastFileLoadedDirectoryName", _lastFileLoadedDirectoryName);
            }

            var data = (DirectoryDropDownData)((UIListControlItem)_cmbDirectory.SelectedItem).Data;
            setting.AddUserData("Directory", data.GetId());

            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 設定読み込み
        /// </summary>
        public sealed override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            // リスト行の幅の読み込み
            setting.LoadColumnPotisionSettings(_lvwTexture);

            {
                bool isDetailView;
                setting.TryToFindUserBoolDataByName("IsDetailView", out isDetailView);

                _UpdateView(isDetailView);
            }

            if (option.AlsoLoadOtherThanWorkspace)
            {
                string userData;
                if (setting.TryToFindUserDataStrByName("LastFileLoadedDirectoryName", out userData))
                {
                    this._lastFileLoadedDirectoryName = userData;
                }
            }

            string directoryId = null;
            setting.TryToFindUserDataStrByName("Directory", out directoryId);

            base.LoadSetting(setting, option);

            // UI更新
            UpdateState(directoryId);
        }

        /// <summary>
        /// 設定読み込み後の処理
        /// </summary>
        public void InitializeAfterLoadSetting()
        {
            // 設定が設定ファイルにない場合は
            // LoadSetting が呼ばれないが
            // InitializeAfterLoadSetting は呼ばれる
            if (_cmbDirectory.Items.Count == 0)
            {
                UpdateState();
            }
        }

        public override Keys CustomShortcut { get { return Keys.F8; } }
        #endregion ILEToolWindow メンバ

        /// <summary>
        /// キーメッセージを処理します。
        /// 登録されたイベントハンドラを実行します。
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // "元に戻す"、"やり直し"を行う
            if (SubSceneUndoRedoHelper.CanUndoRedo(this, keyData) != false)
            {
                // オーナーが ILECmdKeyAccepter ならメッセージを伝播します。
                if (base.NotifyCmdKeyMessageToOwner(ref msg, keyData))
                {
                    return true;
                }
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_cmsMain_Opening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            bool isSingleSelected = _lvwTexture.SelectedItems.Count == 1
                && ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage.ITextureImage.SourceType == TextureSourceType.File;

            _tmiOpenFromExplorer.Enabled = isSingleSelected || (_lvwTexture.SelectedItems.Count == 0&& !string.IsNullOrEmpty(_DirectoryPath));
            _tmiOpenSrcFileByExplorer.Enabled = isSingleSelected && IsSourceFileOfSigleSelectedTgaFound_();

            _tmiCreatePane.Enabled = _CanCreateNewPane;
            _tmiAddToLayout.Visible = _cmbDirectory.Items.Count > 1;
            _tmiAddToLayout.Enabled = _CurrentSubScene != null && _lvwTexture.SelectedItems.OfType<TextureListViewItem>().Where(x => !x.InSubScene).Any();
        }

        /// <summary>
        /// エクスプローラから開く
        /// </summary>
        private void Event_tmiOpenFromExplorer_Click(object sender, EventArgs e)
        {
            string args;
            if (_lvwTexture.SelectedItems.Count != 1)
            {
                if (!string.IsNullOrEmpty(_DirectoryPath) && Directory.Exists(_DirectoryPath))
                {
                     args = _DirectoryPath;
                }
                else
                {
                    // メニューが無効になっているはず
                    Debug.Assert(false);
                    return;
                }
            }
            else
            {
                InternalTexImage internalTexImg = ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage;
                string filePath = internalTexImg.ITextureImage.FilePath;
                filePath = new FileInfo(filePath).FullName;
                args = $"/select,{filePath}";
            }

            Process p = new Process();
            ProcessStartInfo ps = new ProcessStartInfo();
            ps.FileName = "explorer.exe";
            ps.Arguments = args;
            p.StartInfo = ps;
            p.Start();
        }

        private bool IsSourceFileOfSigleSelectedTgaFound_()
        {
            if (_lvwTexture.SelectedItems.Count != 1)
            {
                return false;
            }

            string searchRootPath = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, ViewManager?.AppSetting?.ProjectSettings?.TextureSourceFileSearchRootPath);
            if (!Directory.Exists(searchRootPath))
            {
                return false;
            }

            InternalTexImage internalTexImg = ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage;
            string srcFilePath = TgaHelper.SearchSourceFileForTga(searchRootPath, internalTexImg.ITextureImage.FilePath);

            return !string.IsNullOrEmpty(srcFilePath) && File.Exists(srcFilePath);
        }

        /// <summary>
        /// エクスプローラからソースファイルを開く
        /// </summary>
        private void Event_tsbOpenSrcFileByExplorer_Click(object sender, EventArgs e)
        {
            if (!IsSourceFileOfSigleSelectedTgaFound_())
            {
                return;
            }

            InternalTexImage internalTexImg = ((TextureListViewItem)_lvwTexture.SelectedItems[0]).InternalTexImage;

            Process p = new Process();
            ProcessStartInfo ps = new ProcessStartInfo();
            ps.FileName = "explorer.exe";

            string searchRootPath = SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, ViewManager?.AppSetting?.ProjectSettings?.TextureSourceFileSearchRootPath);
            string srcFilePath = TgaHelper.SearchSourceFileForTga(searchRootPath, internalTexImg.ITextureImage.FilePath);
            srcFilePath = new FileInfo(srcFilePath).FullName;

            ps.Arguments = string.Format("/select,{0}", srcFilePath);
            p.StartInfo = ps;
            p.Start();
        }

        #region 画像フォーマット指定

        private bool IsEditTargetSubItemIdxValid_(int subItemIdx)
        {
            return _TextureFormatListViewSubItemIndex == subItemIdx;
        }

        private bool CheckTextureFormatChanged_(string newValueString, ListViewItem item)
        {
            try
            {
                TexImagePixelFmt newFmt = (TexImagePixelFmt)Enum.Parse(typeof(TexImagePixelFmt), newValueString);
                InternalTexImage originalTexImage = ((TextureListViewItem)item).InternalTexImage;

                return newFmt != originalTexImage.ITextureImage.PixelFmt;
            }
            catch
            {
                return false;
            }
        }

        private bool TryBeginChangeFormat_(SubItemEventArgs e)
        {
            // 画像フォーマット項目がクリックされ、かつ詳細表示の状態なら処理する。
            if (!IsEditTargetSubItemIdxValid_(e.SubItem) || _lvwTexture.View != View.Details)
            {
                return false;
            }

            // 編集開始
            InternalTexImage texImage = ((TextureListViewItem)e.Item).InternalTexImage;

            // 変更不能指定ならば、処理を中断します。
            if (texImage.ITextureImage.PixelFmtIsFixed || !((TextureListViewItem)e.Item).InSubScene)
            {
                return false;
            }

            // 複数選択した要素が、すべて同じ特性をもっているか確認し、そうではない場合は編集を中止します。
            bool allSrcImageAreSameGrayScale = _lvwTexture.SelectedItems.Cast<TextureListViewItem>().All(
                (item) => item.InternalTexImage.ITextureImage.SrcImageIsGrayScale == texImage.ITextureImage.SrcImageIsGrayScale);

            bool allSrcImageHaveSameAlphaBit = _lvwTexture.SelectedItems.Cast<TextureListViewItem>().All(
                (item) => item.InternalTexImage.ITextureImage.SrcImageHasAlphaBit == texImage.ITextureImage.SrcImageHasAlphaBit);

            bool allSrcImageSourceTypeSame = _lvwTexture.SelectedItems.Cast<TextureListViewItem>().All(
                (item) => item.InternalTexImage.ITextureImage.SourceType == texImage.ITextureImage.SourceType);

            if (!allSrcImageAreSameGrayScale || !allSrcImageHaveSameAlphaBit || !allSrcImageSourceTypeSame)
            {
                // "注意：形式の違うテクスチャ（グレースケール、アルファ情報の有無）の複数同時のフォーマット変更はできません。"
                _lblSelectedItemInfo.Text = StringResMgr.Get("PROP_TEXMGR_CANT_CHANGE_MULTIPLETEX_FMT");
                return false;
            }

            // 動的テクスチャかどうか、ソース画像がグレーか、アルファ情報を持つかどうかで、選択肢を変更します。
            string fmtString = GetPixcelFormatString_(texImage.ITextureImage.PixelFmt);
            if (texImage.ITextureImage.SourceType == TextureSourceType.Dynamic)
            {
                _cmbTextureFmtForDynamic.SelectByCondition((item) => item.ToString() == fmtString);
                _lvwTexture.StartEditing(_cmbTextureFmtForDynamic, e.Item, e.SubItem);
            }
            else
            {
                if (texImage.ITextureImage.SrcImageIsGrayScale)
                {
                    // グレー用フォーマット
                    _cmbTextureFmtForGray.SelectByCondition((item) => item.ToString() == fmtString);
                    _lvwTexture.StartEditing(_cmbTextureFmtForGray, e.Item, e.SubItem);
                }
                else
                {
                    if (texImage.ITextureImage.SrcImageHasAlphaBit)
                    {
                        // アルファ有フォーマットのみ
                        _cmbTextureFmtForGrayAlpha.SelectByCondition((item) => item.ToString() == fmtString);
                        _lvwTexture.StartEditing(_cmbTextureFmtForGrayAlpha, e.Item, e.SubItem);
                    }
                    else
                    {
                        // すべてのフォーマット
                        _cmbTextureFmtForNoGrayNoAplha.SelectByCondition((item) => item.ToString() == fmtString);
                        _lvwTexture.StartEditing(_cmbTextureFmtForNoGrayNoAplha, e.Item, e.SubItem);
                    }
                }
            }

            return true;
        }

        /// <summary>
        /// フォーマット選択コンボ閉じ
        /// </summary>
        void Event_CmbTextureFmtFull_DropDownClosed(object sender, EventArgs e)
        {
            _lvwTexture.Focus();
        }

        /// <summary>
        /// 項目クリックハンドラ：編集開始
        /// </summary>
        private void Event_LvwTexture_SubItemClicked(object sender, SubItemEventArgs e)
        {
            if (LayoutEditorCore.PlatformDetail.TextureFormatEditEnabled)
            {
                TryBeginChangeFormat_(e);
            }
            else
            {
                // CTRでは LayoutEditor からのフォーマット変更はできません。付加情報付 TGA 作成時に指定してください。
                _errorMessageTimer.Start();
            }
        }

        /// <summary>
        /// タイマーでトリガされて、エラーメッセージを表示します。
        /// </summary>
        void ErrorMessageTimer_Tick(object sender, EventArgs e)
        {
            this.BeginInvoke(new Action(() => _lblSelectedItemInfo.Text = StringResMgr.Get("PROP_TEXMGR_CANT_CHANGE_FMT", AppConstants.PlatformName)));
            _errorMessageTimer.Stop();
        }

        private void Event_lvwTexture_BeforeSubItemEndEditing(object sender, EventArgs e)
        {
            _CurrentSubScene.BeginMassiveModify();
        }

        private void Event_lvwTexture_AfterSubItemEndEditing(object sender, EventArgs e)
        {
            _CurrentSubScene.EndMassiveModify();
        }

        /// <summary>
        /// 編集終了：
        /// 設定をデータに反映します。
        /// </summary>
        private void Event_LvwTexture_SubItemEndEditing(object sender, SubItemEndEditingEventArgs e)
        {
            InternalTexImage internalTexImage = ((TextureListViewItem)e.Item).InternalTexImage;
            CoreData.ITextureImage texImg = internalTexImage.ITextureImage;
            Debug.Assert(texImg != null);
            Debug.Assert(_lvwTexture.View == View.Details);

            // 文字列から新しいフォーマットの設定を試みます。
            try
            {
                TexImagePixelFmt newFmt =
                    (TexImagePixelFmt)Enum.Parse(typeof(TexImagePixelFmt), e.DisplayText);

                if (texImg.PixelFmt != newFmt)
                {
                    _TextureMgrManipulator.SetTexturePixelFmt(texImg, newFmt);
                }
                else
                {
                    if (e.SubItem != -1)
                    {
                        e.Item.SubItems[e.SubItem].Text = GetPixcelFormatString_(texImg.PixelFmt);
                    }
                }
            }
            catch
            {
                // 不正な入力が行われたので文字列を元に戻します。
                string texelFmt = texImg.PixelFmt.ToString();
                if (e.SubItem != -1)
                {
                    e.Item.SubItems[e.SubItem].Text = GetPixcelFormatString_(texImg.PixelFmt);
                }
            }
        }

        #endregion 画像フォーマット指定

        /// <summary>
        /// 表示ディレクトリの切り替え
        /// </summary>
        private void _cmbDirectory_SelectedIndexChanged(object sender, EventArgs e)
        {
            // コンボボックスからフォーカスを外す
            _lvwTexture.Focus();
            UpdateState();
        }

        /// <summary>
        /// レイアウトに追加
        /// </summary>
        private void _tmiAddToLayout_Click(object sender, EventArgs e)
        {
            var before = _CurrentSubScene.ITextureMgr.ITextureImageSet.ToArray();
            var errors = new List<string>();
            _CurrentSubScene.BeginMassiveModify();
            foreach (int itemIdx in _lvwTexture.SelectedIndices)
            {
                var item = (TextureListViewItem)_lvwTexture.Items[itemIdx];
                if (!item.InSubScene)
                {
                    var result = _TextureMgrManipulator.RegisterITextureImageSetFromFile(new[] { item.InternalTexImage.ITextureImage.FilePath });
                    if (result.Length == 0)
                    {
                        errors.Add(item.InternalTexImage.ITextureImage.FilePath);
                    }
                }
            }
            _CurrentSubScene.EndMassiveModify();


            if (errors.Count > 0)
            {
                MessageReportDlg dlg = new MessageReportDlg(LayoutEditor.Forms.MessageReportDlg.ButtonTypes.Ok, false);
                dlg.Title = StringResMgr.Get("ADDTEXTURE");
                dlg.Message = StringResMgr.Get("SAMENAMETEXTURE") + "\r\n\r\n" + string.Join("\r\n", errors);
                dlg.ShowDialog(this);
            }
        }

        private void _tsmLargeIcon_Click(object sender, EventArgs e)
        {
            _UpdateView(false);
        }

        private void _tsmDetails_Click(object sender, EventArgs e)
        {
            _UpdateView(true);
        }

        private void _tsbListViewKind_ButtonClick(object sender, EventArgs e)
        {
            _UpdateView(_lvwTexture.View != View.Details);
        }

        /// <summary>
        /// 表示を更新
        /// </summary>
        private void _UpdateView(bool details)
        {
            _lvwTexture.View = details ? View.Details : View.LargeIcon;
            _tsbListViewKind.Image = details ? _tsmDetail.Image : _tsmLargeIcon.Image;
            _pctTexture.Visible = details;
        }

        private void _tmiCreatePane_Click(object sender, EventArgs e)
        {
            RegisterNewPanes_();
        }
    }
}
