﻿using LayoutEditor.Structures.SerializableObject;
using LECore.Manipulator;
using LECore.Structures;
using LECore.Structures.Core;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace LayoutEditor.src.Forms.ToolWindows.StateSettingWindow
{
    public partial class StateTransitionTrackPanel : Control
    {
        /// <summary>
        /// 編集アイテム
        /// </summary>
        class EditItem
        {
            public struct HovewrState
            {
                public MouseEventArgs MouseEventArgs { get; set; }
                public int HoverIndex { get; set; }
                public IStateTransitionTrack HoverTrack { get; set; }

                public bool HoverOnCurrentTime { get; set; }

                public KeyParamater HoverKey { get; set; }

                public bool IsEditKey { get { return this.HoverKey != null; } }

                public void ResetHover()
                {
                    HoverIndex = -1;
                    HoverTrack = null;
                    HoverOnCurrentTime = false;
                    HoverKey = null;
                }

                public bool IsHovering { get { return this.HoverIndex != -1 && this.HoverTrack != null; } }
            }

            public EditItem()
            {
                Reset();
            }

            public void Reset()
            {
                _hoverState.ResetHover();
                _SelectedParamaterList.Clear();
            }

            public HovewrState _hoverState = new HovewrState();


            List<IFeatureParamater> _SelectedParamaterList { get; } = new List<IFeatureParamater>();

            public bool CheckIsParamSelected(IFeatureParamater param)
            {
                return _SelectedParamaterList.Find(p => p.Name ==  param.Name && p.Kind == param.Kind) != null;
            }

            public IEnumerable<IFeatureParamater> SelectedParamaters
            {
                get { return _SelectedParamaterList; }
            }

            public void AddToSelectedParam(IFeatureParamater param, bool adding)
            {
                if(!adding)
                {
                    _SelectedParamaterList.Clear();
                    if(param != null)
                    {
                        _SelectedParamaterList.Add(param);
                    }
                }
                else
                {
                    if (param != null)
                    {
                        if (!CheckIsParamSelected(param))
                        {
                            _SelectedParamaterList.Add(param);
                        }
                        else
                        {
                            _SelectedParamaterList.RemoveAll(p => p.Name == param.Name && p.Kind == param.Kind);
                        }
                    }
                }
            }
        }

        //----------------------------------------------------------

        // レンダラー
        IRenderer _renderer = null;
        // Renderer 生成に使用したウインドウハンドル
        private IntPtr _rendererHandle { get; set; } = IntPtr.Zero;

        public FVec2 ViewScale { get; } = new FVec2(1.0f, 1.0f);
        public FVec2 ViewPosition { get; } = new FVec2(0.0f, 0.0f);

        float _trackTopY;
        float _trackHeight;

        EditItem _editItem = new EditItem();

        IStateMachine _stateMachine = null;
        IStateTransition _stateTransition = null;

        EditTransitionTrackDlg _editTransitionTrackDlg = new EditTransitionTrackDlg();
        EditFeatureParamaterDlg _editFeatureParamaterDlg = new EditFeatureParamaterDlg();
        EditStateTrackKeyEventDlg _editStateTrackKeyEventPanel = new EditStateTrackKeyEventDlg();

        ChoiceFrom2Dlg _choiceFrom2Dlg = new ChoiceFrom2Dlg();

        //----------------------------------------------------------

        public StateTransitionTrackPanel()
        {
            InitializeComponent();
        }

        //----------------------------------------------------------

        private void SetupRendering_(ListView pairedListView)
        {
            Point screenPos = pairedListView.PointToScreen(pairedListView.Items[0].Bounds.Location);

            _trackTopY = this.PointToClient(screenPos).Y;
            _trackHeight = pairedListView.Items[0].Bounds.Height;
        }

        //----------------------------------------------------------

        public void Setup(ListView pairedListView, IStateMachine stateMachine, IStateTransition transition)
        {
            _stateMachine = stateMachine;
            _stateTransition = transition;
            _editItem.Reset();

            if (pairedListView.Items.Count <= 0)
            {
                pairedListView.Items.Add("Dummy");
                SetupRendering_(pairedListView);
                pairedListView.Items.Clear();
            }else
            {
                SetupRendering_(pairedListView);
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing && (components != null))
            {
                components.Dispose();
            }

            if (_renderer != null)
            {
                _renderer.Dispose();
            }

            base.Dispose(disposing);
        }

        //----------------------------------------------------------

        protected override void OnHandleCreated(EventArgs e)
        {
            base.OnHandleCreated(e);

            // レンダラの初期化
            InitializeRenderer();
        }

        //----------------------------------------------------------

        protected override void OnSizeChanged(EventArgs e)
        {
            if (_renderer == null)
            {
                InitializeRenderer();
            }

            base.OnSizeChanged(e);
        }

        //----------------------------------------------------------

        private void InitializeRenderer()
        {
            if (Size.Width > 0 && Size.Height > 0 && IsHandleCreated)
            {
                if (_renderer != null)
                {
                    if (Handle != _rendererHandle)
                    {
                        // ドッキングウインドウを使用しているときにハンドルが変わることがある
                        // ハンドルが切り替わっているので再度初期化
                        _renderer.Dispose();
                        _renderer = null;
                    }
                    else
                    {
                        // 既に初期化されているので何もしない
                        return;
                    }
                }

                _renderer = RendererFactory.Create(AppSetting.RendererType.D3D);
                SetStyle(RendererFactory.GetControlState(AppSetting.RendererType.D3D), true);

                bool bInitSuccess = _renderer.Initialize(this);
                Trace.Assert(bInitSuccess);

                _rendererHandle = Handle;
            }
        }

        //----------------------------------------------------------

        protected override void OnPaint(PaintEventArgs pe)
        {
            if (_renderer == null)
            {
                return;
            }

            if(_stateMachine == null)
            {
                return;
            }

            Graphics g = pe.Graphics;

            _renderer.BeginRendering(g);

            // ビュー行列を設定する
            double halfFovy = (double)(_renderer.FOV) * 0.5;
            float cameraDist = _renderer.PerseNear - (this.Height * 0.5f / (float)Math.Tan(halfFovy));

            _renderer.SetViewTransform(
                1.0f / ViewScale.X,
                1.0f / ViewScale.Y,
                ViewPosition.X,
                ViewPosition.Y,
                cameraDist,
                0.0f);

            // 背景の描画
            DrawEditBackGround_();

            // Transition
            DrawTransitionTracks_();

            // 選択
            DrawEditItem_();

            _renderer.EndRendering();

            base.OnPaint(pe);
        }

        //----------------------------------------------------------

        private void DrawEditBackGround_()
        {
            for (int y = 0; y * _trackHeight < this.Height; y++)
            {
                Color col0 = SystemColors.ControlDarkDark;
                Color col1 = Color.FromArgb(col0.A, col0.R - 3, col0.G - 3, col0.B - 3);
                _renderer.Color = (y % 2) == 0 ? col0 : col1;
                FillRectangleOnScreen_(0, _trackTopY + _trackHeight * y, this.Width, _trackHeight);
            }
        }

        //----------------------------------------------------------

        private void DrawEditItem_()
        {
            if (_editItem._hoverState.IsHovering)
            {
                DrawTrack_(_editItem._hoverState.HoverIndex, _editItem._hoverState.HoverTrack, Color.Transparent, Color.Red);

                if (_editItem._hoverState.HoverOnCurrentTime)
                {
                    DrawOneTrackKey_(_stateTransition.LocalTime.Time, _editItem._hoverState.HoverIndex, Color.Transparent, Color.Red);
                }

                if (_editItem._hoverState.HoverKey != null)
                {
                    DrawOneTrackKey_(_editItem._hoverState.HoverKey.Time, _editItem._hoverState.HoverIndex, Color.Red, Color.White);
                }
            }
        }

        //----------------------------------------------------------

        private void DrawTransitionTracks_()
        {
            if(_stateTransition == null)
            {
                return;
            }

            int trackCount = 0;
            foreach(var featureParam in _stateTransition.GetFreatureParamaters())
            {
                var track = _stateTransition.FindIStateTransitionTrackDataByName(featureParam.Name, featureParam.Kind);
                bool isSelected = _editItem.CheckIsParamSelected(featureParam);

                DrawTrack_(trackCount, track, isSelected ? Color.LightGreen : Color.LightSteelBlue, Color.Blue);
                DrawAllTrackKey_(trackCount, track, Color.Red, Color.Red);
                DrawAllTrackEvent_(trackCount, track, Color.Blue, Color.Red);
                trackCount++;
            }
        }

        //----------------------------------------------------------

        private void DrawOneTrackKey_(float keyTime, int trackIndex, Color fillColor, Color drawColor)
        {
            float keyX = (keyTime / _stateTransition.TotalDuration) * this.Width;
            float trackY = _trackTopY + _trackHeight * trackIndex;
            float keyY = trackY;
            float keyW = 1.0f / _stateTransition.TotalDuration * this.Width;

            _renderer.Color = fillColor;
            FillRectangleOnScreen_(keyX, keyY, keyW, _trackHeight);

            _renderer.Color = drawColor;
            DrawRectangleOnScreen_(keyX, keyY, keyW, _trackHeight);
        }

        //----------------------------------------------------------

        private void DrawAllTrackKey_(int trackIndex, IStateTransitionTrack track, Color fillColor, Color drawColor)
        {
            // キーの描画
            foreach (var key in track.KeyParamaters.Where(k => k.FeatureParamaterData.Kind != FeatureParamaterKind.StateMachineEvent))
            {
                DrawOneTrackKey_(key.Time, trackIndex, fillColor, drawColor);
            }
        }


        //----------------------------------------------------------

        private void DrawAllTrackEvent_(int trackIndex, IStateTransitionTrack track, Color fillColor, Color drawColor)
        {
            // キーの描画
            foreach (var key in track.KeyParamaters.Where(k => k.FeatureParamaterData.Kind == FeatureParamaterKind.StateMachineEvent))
            {
                DrawOneTrackKey_(key.Time, trackIndex, fillColor, drawColor);
            }
        }

        //----------------------------------------------------------

        private void DrawTrack_(int trackCount, IStateTransitionTrack track, Color fillColor, Color drawColor)
        {
            float trackX = (track.Offset / _stateTransition.TotalDuration) * this.Width;
            float trackY = _trackTopY + _trackHeight * trackCount;
            float trackW = track.Duration / _stateTransition.TotalDuration * this.Width;

            _renderer.Color = fillColor;
            FillRectangleOnScreen_(trackX, trackY, trackW - 1, _trackHeight);

            _renderer.Color = drawColor;
            DrawRectangleOnScreen_(trackX, trackY, trackW - 1, _trackHeight);
        }

        //----------------------------------------------------------

        private void FillRectangleOnScreen_(float x, float y, float w, float h)
        {
            var points = new PointF[] { new PointF(x, y) };
            _renderer.TransformPointsToScene(points);

            _renderer.FillRectangle(points[0].X, points[0].Y, 0.0f, w, -h);
        }

        //----------------------------------------------------------

        private void DrawRectangleOnScreen_(float x, float y, float w, float h)
        {
            var points = new PointF[] { new PointF(x, y) };
            _renderer.TransformPointsToScene(points);

            _renderer.DrawRectangle(points[0].X, points[0].Y, 0.0f, w, -h);
        }

        //----------------------------------------------------------

        private bool GetHitHorizontal_(int mouseX, float time, float duration)
        {
            float trackXMin = (time / _stateTransition.TotalDuration) * this.Width;
            float trackXMax = ((time + duration) / _stateTransition.TotalDuration) * this.Width;

            if (trackXMin > mouseX || trackXMax < mouseX)
            {
                return false;
            }
            return true;
        }

        //----------------------------------------------------------

        private bool GetHitTrack_(int mouseX, int mouseY, out int index, out IStateTransitionTrack track)
        {
            // まず縦位置で、どのトラックに相当するか特定する。
            index = (mouseY - (int)_trackTopY) / (int)_trackHeight;
            track = null;
            if (index < 0 || _stateTransition.GetFreatureParamaters().Count() <= index)
            {
                return false;
            }

            // トラックを取得する
            var featureParam = _stateTransition.GetFreatureParamaters().ElementAt(index);
            track = _stateTransition.FindIStateTransitionTrackDataByName(featureParam.Name, featureParam.Kind);

            // 範囲矩形内かチェックする
            if(!GetHitHorizontal_(mouseX, track.Offset, track.Duration))
            {
                return false;
            }

            return true;
        }

        //----------------------------------------------------------

        void UpdateHoverState_(MouseEventArgs e)
        {
            _editItem._hoverState.MouseEventArgs = e;

            int index;
            IStateTransitionTrack track;
            if (GetHitTrack_(e.X, e.Y, out index, out track))
            {
                _editItem._hoverState.HoverIndex = index;
                _editItem._hoverState.HoverTrack = track;
                _editItem._hoverState.HoverOnCurrentTime = GetHitHorizontal_(e.X, _stateTransition.LocalTime.Time, 1);
                _editItem._hoverState.HoverKey = track.KeyParamaters.FirstOrDefault(key => GetHitHorizontal_(e.X, key.Time, 1));
            }
            else
            {
                _editItem._hoverState.ResetHover();
            }
        }

        //----------------------------------------------------------

        private void StateTransitionTrackPanel_MouseMove(object sender, MouseEventArgs e)
        {
            UpdateHoverState_(e);
            Invalidate();
        }

        //----------------------------------------------------------

        private void StateTransitionTrackPanel_MouseDown(object sender, MouseEventArgs e)
        {
            {
                IFeatureParamater featureParam = null;
                if (_editItem._hoverState.IsHovering)
                {
                    featureParam = _stateTransition.GetFreatureParamaters().ElementAt(_editItem._hoverState.HoverIndex);
                }

                // 選択済みアイテムの上での右クリックは、選択変更しない（コンテキストメニューを実行する意図と判断しています）
                if (e.Button == MouseButtons.Right)
                {
                    if (featureParam == null || _editItem.CheckIsParamSelected(featureParam))
                    {
                        return;
                    }
                }

                bool adding = (Control.ModifierKeys & Keys.Control) != 0;
                _editItem.AddToSelectedParam(featureParam, adding);
            }
            Invalidate();
        }

        //----------------------------------------------------------

        private void AddKey_(IEnumerable<IFeatureParamater> featureParams, EditItem.HovewrState hoverState)
        {
            IFeatureParamater featureParam = featureParams.FirstOrDefault();
            if (featureParam == null)
            {
                if(hoverState.HoverIndex  != -1)
                {
                    featureParam = _stateTransition.FindFreatureParamaterByIndex(hoverState.HoverIndex);
                }
            }

            var targetParams = featureParams.Any() ? featureParams : new[] { featureParam };
            bool isStateMachineTrack = targetParams.All(tp => tp.Kind == FeatureParamaterKind.StateMachineEvent);

            var onOK = new Action(() =>
            {
                if (_stateMachine == null)
                {
                    return;
                }

                IStateLayer selectedLayer = _stateMachine.GetSelectedIStateLayer();
                if (selectedLayer == null)
                {
                    return;
                }

                StateMachineManipulator mnp = new StateMachineManipulator();
                mnp.BindTarget(_stateMachine);

                ISubScene subScene = _stateMachine.OwnerSubScene;

                subScene.BeginMassiveModify();

                foreach (var selectedParam in targetParams)
                {
                    if (!isStateMachineTrack)
                    {
                        mnp.AddTransitionTrackKey(
                        selectedLayer,
                        _stateTransition,
                        selectedParam.Name, selectedParam.Kind,
                        _editFeatureParamaterDlg.Time,
                        IStateEasingHelper.Create(_editFeatureParamaterDlg.EasingType, 0.0f),
                        _editFeatureParamaterDlg.X,
                        _editFeatureParamaterDlg.Y,
                        _editFeatureParamaterDlg.Z,
                        _editFeatureParamaterDlg.W);
                    }
                    else
                    {
                        mnp.AddTransitionTrackEvent(
                        selectedLayer,
                        _stateTransition,
                        selectedParam.Name,
                        _stateTransition.LocalTime.Time,
                        StateTrackKeyEventKind.PostEventToChild,
                        _editStateTrackKeyEventPanel.StateMachineEventKind,
                        _editStateTrackKeyEventPanel.Param1,
                        _editStateTrackKeyEventPanel.Param2,
                        _editStateTrackKeyEventPanel.Delay);
                    }

                }
                subScene.EndMassiveModify();
            });

            if (!isStateMachineTrack)
            {
                var state = _stateMachine.GetSelectedIStateLayer().FindStateByName(_stateTransition.StartStateName);
                _editFeatureParamaterDlg.Setup(_stateMachine, state, featureParam, hoverState.HoverTrack.StateEasing, _stateTransition.LocalTime.Time, onOK);
            }else
            {
                _editStateTrackKeyEventPanel.Setup(_stateMachine.OwnerSubScene,
                    0,
                    featureParam.Name,
                    "Base", onOK);
            }

            Form dlg = !isStateMachineTrack ? (Form)_editFeatureParamaterDlg : (Form)_editStateTrackKeyEventPanel;

            dlg.Show();
            dlg.SetLocationToMousePos(this, hoverState.MouseEventArgs.X, hoverState.MouseEventArgs.Y);
        }

        private void EditKey_(EditItem.HovewrState hoverState)
        {
            bool isStateMachineTrack = _editItem.SelectedParamaters.All(tp => tp.Kind == FeatureParamaterKind.StateMachineEvent);

            var onOK = new Action(() =>
            {
                if (_stateMachine == null)
                {
                    return;
                }

                IStateLayer selectedLayer = _stateMachine.GetSelectedIStateLayer();
                if (selectedLayer == null)
                {
                    return;
                }

                StateMachineManipulator mnp = new StateMachineManipulator();
                mnp.BindTarget(_stateMachine);

                ISubScene subScene = _stateMachine.OwnerSubScene;

                subScene.BeginMassiveModify();
                foreach (var selectedParam in _editItem.SelectedParamaters)
                {
                    if(!isStateMachineTrack)
                    {
                        mnp.AddTransitionTrackKey(
                       selectedLayer,
                       _stateTransition,
                       selectedParam.Name, selectedParam.Kind,
                       _editFeatureParamaterDlg.Time,
                       IStateEasingHelper.Create(_editFeatureParamaterDlg.EasingType, _editFeatureParamaterDlg.EasingExtraParamater),
                       _editFeatureParamaterDlg.X,
                       _editFeatureParamaterDlg.Y,
                       _editFeatureParamaterDlg.Z,
                       _editFeatureParamaterDlg.W);
                    }
                    else
                    {
                        mnp.AddTransitionTrackEvent(
                        selectedLayer,
                        _stateTransition,
                        selectedParam.Name,
                        hoverState.HoverKey.Time,
                        StateTrackKeyEventKind.PostEventToChild,
                        _editStateTrackKeyEventPanel.StateMachineEventKind,
                        _editStateTrackKeyEventPanel.Param1,
                        _editStateTrackKeyEventPanel.Param2,
                        _editStateTrackKeyEventPanel.Delay);
                    }

                }
                subScene.EndMassiveModify();
            });

            if (!isStateMachineTrack)
            {
                var state = _stateMachine.GetSelectedIStateLayer().FindStateByName(_stateTransition.StartStateName);
                _editFeatureParamaterDlg.Setup(_stateMachine, state, hoverState.HoverKey.FeatureParamaterData, hoverState.HoverTrack.StateEasing, hoverState.HoverKey.Time, onOK);
            }
            else
            {
                var srcEvent = hoverState.HoverKey.StateMachineEvent;
                _editStateTrackKeyEventPanel.Setup(
                    _stateMachine.OwnerSubScene,
                    srcEvent.delayFrames,
                    srcEvent.param1,
                    srcEvent.param2,
                    onOK);
            }

            Form dlg = !isStateMachineTrack ? (Form)_editFeatureParamaterDlg : (Form)_editStateTrackKeyEventPanel;

            dlg.Show();
            dlg.SetLocationToMousePos(this, hoverState.MouseEventArgs.X, hoverState.MouseEventArgs.Y);
        }

        //----------------------------------------------------------

        private void EditTrack_(IEnumerable<IFeatureParamater> featureParams, EditItem.HovewrState hoverState)
        {
            IFeatureParamater featureParam = featureParams.FirstOrDefault();
            if (featureParam == null)
            {
                if (hoverState.HoverIndex != -1)
                {
                    featureParam = _stateTransition.FindFreatureParamaterByIndex(hoverState.HoverIndex);
                }
            }

            var track = _stateTransition.FindIStateTransitionTrackDataByName(featureParam.Name, featureParam.Kind);
            Debug.Assert(track != null);

            var onOk = new Action(() =>
            {
                if (_stateMachine == null)
                {
                    return;
                }

                IStateLayer selectedLayer = _stateMachine.GetSelectedIStateLayer();
                if (selectedLayer == null)
                {
                    return;
                }

                StateMachineManipulator mnp = new StateMachineManipulator();
                mnp.BindTarget(_stateMachine);

                ISubScene subScene = _stateMachine.OwnerSubScene;

                subScene.BeginMassiveModify();

                var targetParams = featureParams.Any() ? featureParams : new[] { featureParam };
                foreach (var selectedParam in targetParams)
                {
                    mnp.EditTransitionTrack(
                        selectedLayer, _stateTransition,
                        selectedParam.Name, selectedParam.Kind,
                        _editTransitionTrackDlg.Offset,
                        _editTransitionTrackDlg.Duration,
                        IStateEasingHelper.Create(_editTransitionTrackDlg.EasingType, _editTransitionTrackDlg.EasingExtraParamater));
                }
                subScene.EndMassiveModify();
            });

            _editTransitionTrackDlg.Setup(_stateMachine, track, onOk);
            _editTransitionTrackDlg.Show();
            _editTransitionTrackDlg.SetLocationToMousePos(this, hoverState.MouseEventArgs.X, hoverState.MouseEventArgs.Y);
        }

        //----------------------------------------------------------

        private void StateTransitionTrackPanel_MouseUp(object sender, MouseEventArgs e)
        {
            // 右クリックで、編集ダイアログを表示する
            if (e.Button == MouseButtons.Right)
            {
                if (_editItem._hoverState.HoverOnCurrentTime && _editItem._hoverState.HoverKey == null)
                {
                    AddKey_(_editItem.SelectedParamaters, _editItem._hoverState);
                }
                else
                {
                    _tmiDelete.Enabled = _editItem._hoverState.IsEditKey;
                    _cmsEditKey.Show(this, e.X, e.Y);
                    _cmsEditKey.Tag = _editItem._hoverState;
                }
            }
        }

        //----------------------------------------------------------

        private void StateTransitionTrackPanel_MouseLeave(object sender, EventArgs e)
        {
            _editItem._hoverState.ResetHover();
            Invalidate();
        }

        private void _tmiEditKey_Click(object sender, EventArgs e)
        {
            EditItem.HovewrState hoverState = (EditItem.HovewrState)_cmsEditKey.Tag;
            if (hoverState.IsEditKey)
            {
                EditKey_(hoverState);
            }
            else
            {
                Debug.Assert(!hoverState.HoverOnCurrentTime);
                EditTrack_(_editItem.SelectedParamaters, hoverState);
            }
        }

        private void _tmiDelete_Click(object sender, EventArgs e)
        {
            Debug.Assert(_stateMachine != null);
            Debug.Assert(_stateTransition != null);

            // マウス下キーの削除
            EditItem.HovewrState hoverState = (EditItem.HovewrState)_cmsEditKey.Tag;

            if (hoverState.HoverIndex == -1)
            {
                return;
            }

            if (hoverState.HoverKey == null)
            {
                return;
            }

            StateMachineManipulator mnp = new StateMachineManipulator();
            mnp.BindTarget(_stateMachine);

            IFeatureParamater featureParam = _stateTransition.FindFreatureParamaterByIndex(hoverState.HoverIndex);
            mnp.RemoveTransitionTrackKey(_stateMachine.GetSelectedIStateLayer(), _stateTransition, featureParam.Name, featureParam.Kind, hoverState.HoverKey.Time);
        }
    }

    //----------------------------------------------------------

    internal static class DialogHelper
    {
        internal static void SetLocationToToolStripButtonPos(this Control form, object sender)
        {
            Debug.Assert(sender is ToolStripButton);
            var btn = sender as ToolStripButton;
            SetLocationToMousePos(form, btn.GetCurrentParent(), btn.Bounds.Right, btn.Bounds.Top);
        }

        internal static void SetLocationToMousePos(this Control form, Control control, int x, int y)
        {
            var posScrn = control.PointToScreen(new Point(x, y));
            form.Location = new Point(posScrn.X - form.Width / 2, Math.Max(0, posScrn.Y - form.Height));
        }
    }
}
