﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.Windows.Forms;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Manipulator;

    using LayoutEditor.Controls;
    using LayoutEditor.Utility;
    using Structures.SerializableObject;

    using Adapters;
    using System.Collections.Generic;

    /// <summary>
    /// オブジェクトプロパティダイアログクラス。
    /// </summary>
    public sealed class PropertyWindow : LEToolWindow
    {
        //-------------------------------------------------------
        #region デザイナ変数

        private LayoutEditor.Controls.DockContentToolTip ttpHint;
        private System.ComponentModel.IContainer components;
        private LayoutEditor.Controls.ToolButton btnSubDialog;
        private System.Windows.Forms.ImageList imgButtons;
        private LayoutEditor.Controls.ToolButton chkFreeze;
        private PropWndTargetBox ctlTargetBox;
        private PictureBox _pnlMulti;
        private System.Windows.Forms.Panel pnlClient;
        #endregion

        #region デザイナコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(PropertyWindow));
            LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters.PaneGuiAdapterGroup paneGuiAdapterGroup1 = new LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters.PaneGuiAdapterGroup();
            this.imgButtons = new System.Windows.Forms.ImageList(this.components);
            this.ttpHint = new LayoutEditor.Controls.DockContentToolTip(this.components);
            this.btnSubDialog = new LayoutEditor.Controls.ToolButton();
            this.chkFreeze = new LayoutEditor.Controls.ToolButton();
            this._pnlMulti = new System.Windows.Forms.PictureBox();
            this.pnlClient = new System.Windows.Forms.Panel();
            this.ctlTargetBox = new LayoutEditor.Forms.ToolWindows.PropertyEditWindow.PropWndTargetBox();
            ((System.ComponentModel.ISupportInitialize)(this._pnlMulti)).BeginInit();
            this.SuspendLayout();
            //
            // imgButtons
            //
            this.imgButtons.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imgButtons.ImageStream")));
            this.imgButtons.TransparentColor = System.Drawing.Color.Magenta;
            this.imgButtons.Images.SetKeyName(0, "");
            this.imgButtons.Images.SetKeyName(1, "");
            this.imgButtons.Images.SetKeyName(2, "");
            this.imgButtons.Images.SetKeyName(3, "");
            this.imgButtons.Images.SetKeyName(4, "");
            this.imgButtons.Images.SetKeyName(5, "");
            this.imgButtons.Images.SetKeyName(6, "");
            //
            // ttpHint
            //
            this.ttpHint.ShowAlways = true;
            //
            // btnSubDialog
            //
            resources.ApplyResources(this.btnSubDialog, "btnSubDialog");
            this.btnSubDialog.Image = ((System.Drawing.Image)(resources.GetObject("btnSubDialog.Image")));
            this.btnSubDialog.ImageIndex = 4;
            this.btnSubDialog.ImageList = this.imgButtons;
            this.btnSubDialog.Name = "btnSubDialog";
            this.btnSubDialog.TabStop = false;
            this.ttpHint.SetToolTip(this.btnSubDialog, resources.GetString("btnSubDialog.ToolTip"));
            //
            // chkFreeze
            //
            this.chkFreeze.ButtonStyle = LayoutEditor.Controls.ToolButton.Style.Toggle;
            this.chkFreeze.Image = ((System.Drawing.Image)(resources.GetObject("chkFreeze.Image")));
            this.chkFreeze.ImageIndex = 0;
            this.chkFreeze.ImageList = this.imgButtons;
            resources.ApplyResources(this.chkFreeze, "chkFreeze");
            this.chkFreeze.Name = "chkFreeze";
            this.chkFreeze.TabStop = false;
            this.ttpHint.SetToolTip(this.chkFreeze, resources.GetString("chkFreeze.ToolTip"));
            this.chkFreeze.Click += new System.EventHandler(this.Event_chkFreeze_Click);
            //
            // _pnlMulti
            //
            resources.ApplyResources(this._pnlMulti, "_pnlMulti");
            this._pnlMulti.Name = "_pnlMulti";
            this._pnlMulti.TabStop = false;
            this.ttpHint.SetToolTip(this._pnlMulti, resources.GetString("_pnlMulti.ToolTip"));
            //
            // pnlClient
            //
            resources.ApplyResources(this.pnlClient, "pnlClient");
            this.pnlClient.Name = "pnlClient";
            //
            // ctlTargetBox
            //
            resources.ApplyResources(this.ctlTargetBox, "ctlTargetBox");
            this.ctlTargetBox.BackColor = System.Drawing.SystemColors.Window;
            this.ctlTargetBox.Name = "ctlTargetBox";
            paneGuiAdapterGroup1.Active = null;
            this.ctlTargetBox.Target = paneGuiAdapterGroup1;
            //
            // PropertyWindow
            //
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this._pnlMulti);
            this.Controls.Add(this.chkFreeze);
            this.Controls.Add(this.pnlClient);
            this.Controls.Add(this.ctlTargetBox);
            this.Controls.Add(this.btnSubDialog);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.SizableToolWindow;
            this.HideOnClose = false;
            this.KeyPreview = true;
            this.Name = "PropertyWindow";
            this.ShowHint = WeifenLuo.WinFormsUI.Docking.DockState.Float;
            ((System.ComponentModel.ISupportInitialize)(this._pnlMulti)).EndInit();
            this.ResumeLayout(false);

        }
        #endregion

        //-------------------------------------------------------
        #region フィールド
        // フリーズ
        private PropertyWindow _parentDialog = null;

        // ターゲット
        private PaneGuiAdapterGroup _target = new PaneGuiAdapterGroup();

        // プロパティパネル管理
        Dictionary<PropertyPanelType, PropertyPanel> _panelMap = new Dictionary<PropertyPanelType, PropertyPanel>();
        private PropertyPanel _activePanel = null;

        private PropertyWindowMgr _propertyWindowMgr = null;

        // ダイアログ管理
        static private ArrayList _propDialogs = new ArrayList();

        // プロファイル保存用
        static private int _leftWidth = 168;
        static private Point _location;

        private ISubScene _currentISubScene = null; //@@@
        //private LEWindowPartsID _current = null;  //@@@


        readonly AppSetting _appSetting = null;
        #endregion フィールド

        //-------------------------------------------------------
        #region プロパティ
        /// <summary>
        /// ターゲット。
        /// </summary>
        public PaneGuiAdapterGroup Target
        {
            get { return _target; }
            set
            {
                // コピーを作成して設定
                // 参照をそのまま設定しては駄目
                _target = new PaneGuiAdapterGroup(value);
                this.ctlTargetBox.Target = _target;

                //@@@
                if (_target != null && _target.Active != null)
                {
                    PaneGUIAdapter adapter = _target.Active;
                    IPane pane = adapter.PaneManipulator.IPane;

                    // カレントサブシーンの設定
                    _currentISubScene = pane.OwnerSubScene;
                    _pnlMulti.BackColor = _target.IsMulti ? Color.MintCream : SystemColors.Control;
                    _pnlMulti.BorderStyle = _target.IsMulti ? BorderStyle.FixedSingle : BorderStyle.None;
                }
                else
                {
                    _pnlMulti.BackColor = SystemColors.Control;
                    _pnlMulti.BorderStyle = BorderStyle.None;
                }
                //@@@

                // パネル更新
                if (this.Visible)
                {
                    // アクティブパネルを変更します。
                    ChangeActivePropertyPanel_();

                    // 新たなパネルに対して、ターゲット変更時更新を呼び出します。
                    if (_activePanel != null)
                    {
                        _activePanel.OnTargetChangeCategoryUpdate();
                        _activePanel.OnTargetChangePageUpdate();
                        _activePanel.UpdateForm();
                    }
                }
            }
        }

        /// <summary>
        /// ターゲットをフリーズするか否か
        /// </summary>
        public bool Freeze
        {
            get { return this.chkFreeze.Checked; }
            set
            {
                this.chkFreeze.Checked = value;
                this.chkFreeze.ImageIndex = (value) ? 1 : 0;
                ctlTargetBox.BackColor = (value) ? SystemColors.Control : SystemColors.Window;
            }
        }

        /// <summary>
        /// アプリケーション設定
        /// </summary>
        public AppSetting AppSetting
        {
            get { return _appSetting; }
        }

        // DockPanel Suite と組み合わせたときに例外が起きることがある。
        // 直接触らせない
        [Obsolete]
        public new Form Owner { get { return base.Owner; } set { base.Owner = value; } }
        #endregion プロパティ

        //-------------------------------------------------------
        #region イベント

        public delegate void OnCloseHandler(PropertyWindow closingDlg);
        public event OnCloseHandler OnClose;

        #endregion イベント

        //-------------------------------------------------------
        #region 初期化関連

        /// <summary>
        /// 初期化。
        /// </summary>
        static public void initialize()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public PropertyWindow(PropertyWindowMgr mgr, PropertyWindow dlg, AppSetting appSetting)
        {
            Debug.Assert(appSetting != null);

            InitializeComponent();

            _propDialogs.Add(this);
            _parentDialog = dlg;
            _propertyWindowMgr = mgr;

            btnSubDialog.Click += new EventHandler(Event_BtnSubDialog_Click);

            this.ViewManager = _propertyWindowMgr.ViewManager;
            _appSetting = appSetting;
        }

        /// <summary>
        /// プロパティパネル登録。
        /// </summary>
        private PropertyPanel RegisterPropertyPanel(
            PropertyPanelType id,
            Func<PropertyPanel> panelCreateFunction)
        {
            PropertyPanel panel;
            if (!_panelMap.TryGetValue(id, out panel))
            {
                panel = panelCreateFunction();
                SetupRegisteredPropertyPanel(panel);

                _panelMap.Add(id, panel);
            }

            return panel;
        }

        /// <summary>
        /// プロパティパネル登録。
        /// </summary>
        private void SetupRegisteredPropertyPanel(PropertyPanel panel)
        {
            panel.Owner = this;
            panel.Parent = pnlClient;
            panel.Dock = DockStyle.Fill;
            panel.Visible = false;

            //	アクティブパネルが存在するときは左領域の値を受け取り、
            // それ以外はメインダイアログの左領域を受け取る
            if (_activePanel != null)
                panel.LeftWidth = _activePanel.LeftWidth;
            else if (_parentDialog != null)
                panel.LeftWidth = (_parentDialog._activePanel != null) ?
                    _parentDialog._activePanel.LeftWidth : _leftWidth;
            else
                panel.LeftWidth = _leftWidth;

            //	サブプロパティで、最初に選択するものはメインと同じ
            if (_parentDialog != null && _parentDialog._activePanel != null)
            {
                panel.InitializeSamePage(_parentDialog._activePanel);
            }
            //	最初に初期化できたら親の情報は消しておく
            _parentDialog = null;
        }

        /// <summary>
        /// 操作対象の種類に応じた、パネルを取得します。
        /// </summary>
        /// <param name="target"></param>
        /// <returns></returns>
        PropertyPanel GetPanel_(PaneGuiAdapterGroup target)
        {
            // 対象が存在しない場合は...
            // 何も表示しないパネルを設定します。
            if (target.IsEmpty)
            {
                return RegisterPropertyPanel(PropertyPanelType.Null, () => new PropertyPanel());
            }

            PaneGUIAdapter paneAdapter = target.Active;
            PropertyPanelType id = paneAdapter.ObjectID;
            switch (id)
            {
                // この部分、処理の流れを変える！
                case PropertyPanelType.NullPane:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.PicturePane:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.TextBoxPane:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.WindowPane:
                    return RegisterPropertyPanel(id, () => new WindowPropertyPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.RegionPane:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.PartsPane:
                    // 部品内のウインドウペインを上書きするときのために WindowPropertyPanel を使う
                    return RegisterPropertyPanel(id, () => new WindowPropertyPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.CapturePane:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.Alignment:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                case PropertyPanelType.Scissor:
                    return RegisterPropertyPanel(id, () => new PaneBasicPanel(id, target, this.DoSendMessageToViewManager));
                default:
                    Debug.Assert(false); return null;
            }
        }

        /// <summary>
        /// プロパティパネル更新。
        /// 変更があれば、trueを返します。
        /// </summary>
        private bool ChangeActivePropertyPanel_()
        {
            // パネル選択
            PropertyPanel panel = GetPanel_(_target);

            // アクティブパネル切り替え
            if (_activePanel == null)
            {
                _activePanel = panel;
                _activePanel.Visible = true;
            }
            else if (_activePanel != panel)
            {
                // 前のパネルを非表示→次のパネルを表示にすると
                // 背景のダイアログが一瞬見えてちらつくので
                // 次のパネルを表示→前のパネルを非表示にする（倉橋）
                panel.LeftWidth = _activePanel.LeftWidth;
                panel.Visible = true;
                panel.SetCalculationDialogSize(null);

                _activePanel.OnHide();
                _activePanel.Visible = false;
                _activePanel = panel;
            }
            else
            {
                // 変更なし
                return false;
            }

            return true;
        }

        #endregion 初期化関連

        /// <summary>
        /// プロパティウインドウがペインを編集対象としているか取得します。
        /// </summary>
        public bool IsTargetting(IPane pane)
        {
            PaneGUIAdapter targetPane = Target.Active;
            return targetPane != null && targetPane.PaneManipulator.IPane == pane;
        }

        /// <summary>
        /// レイアウトファイル名を取得します。
        /// </summary>
        public string GetLayoutFileName()
        {
            if (_currentISubScene == null)
            {
                return string.Empty;
            }

            var arg = new ViewManagerMessage(ViewManagerMessageKind.FindLayoutPathBySubScene, _currentISubScene);
            DoSendMessageToViewManager(arg);
            return arg.Result as string;
        }

        /// <summary>
        /// 通知メッセージを書きます
        /// </summary>
        public void WriteNotificationMessage(string message)
        {
            this.DoSendMessageToViewManager(new ViewManagerMessage(ViewManagerMessageKind.WriteMessage, message));
        }

        /// <summary>
        /// 再描画を行います。
        /// </summary>
        public void Redraw()
        {
            LEToolWindow.DbgReportUpdate(this);
            PaneGUIAdapter targetPane = this.Target.Active;

            ctlTargetBox.InvalidateIfChanged();

            if (targetPane != null)
            {
                // アダプタオブジェクトの内容を最新の状態に反映します。
                //targetPane.OnRedraw( this );
                // アダプタオブジェクトの内容を基にGUIを更新します。
                if (_activePanel != null)
                {
                    _activePanel.UpdateForm();
                }
            }
        }

        /// <summary>
        /// シーン更新イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(object sender, SceneModifyEventArgs e)
        {
            switch (e.Target)
            {
                //------------------------------------------------------
                case SceneModifyEventArgs.Kind.PaneModify:
                case SceneModifyEventArgs.Kind.HierarchyMake:
                case SceneModifyEventArgs.Kind.HierarchyRemove:
                case SceneModifyEventArgs.Kind.HierarchyReordr:
                case SceneModifyEventArgs.Kind.TextureManager:
                case SceneModifyEventArgs.Kind.FontManager:
                    {
                        Redraw();
                        break;
                    }
                //------------------------------------------------------
                case SceneModifyEventArgs.Kind.PartsLayout:
                case SceneModifyEventArgs.Kind.MaterialEditMode:
                    {
                        if (_activePanel != null)
                        {
                            // 操作対象の状態を更新します。
                            if (this.Target != null)
                            {
                                PaneGUIAdapter targetPane = this.Target.Active;
                                if (targetPane != null)
                                {
                                    targetPane.Refresh();
                                }
                            }

                            // アクティブパネルを変更します。
                            ChangeActivePropertyPanel_();

                            // 新たなパネルに対して、ターゲット変更時更新を呼び出します。
                            if (_activePanel != null)
                            {
                                _activePanel.OnTargetChangeCategoryUpdate();
                                _activePanel.OnTargetChangePageUpdate();
                                _activePanel.UpdateForm();
                            }
                        }
                        break;
                    }
            }

            //------------------------------------------------------
            // アクティブパネルにイベントを伝播します。
            if (_activePanel != null)
            {
                _activePanel.OnSceneModifyHandler(sender, e);
            }
        }

        //-------------------------------------------------------
        #region オーバーライドメソッド
        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }

                // ウインドウクローズ時ハンドラを呼び出します。
                if (OnClose != null)
                {
                    OnClose(this);
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void WndProc(ref Message m)
        {
            base.WndProc(ref m);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            base.OnGotFocus(e);

            // パネルにフォーカスを移す
            if (_activePanel != null)
            {
                _activePanel.Focus();
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnVisibleChanged(EventArgs e)
        {
            if (this.DesignMode) { return; }

            // パネル更新
            if (this.Visible)
            {
                ChangeActivePropertyPanel_();

                // パネルの変更が起きなかったら更新
                if (_activePanel != null)
                {
                    _activePanel.OnTargetChangeCategoryUpdate();
                    _activePanel.OnTargetChangePageUpdate();
                    _activePanel.UpdateForm();
                }
            }
            else if (_activePanel != null)
            {
                // 注意: 親が非表示なので、プロパティの値は既に false になっている。
                // false を代入しておくと内部のステートが変化し、親が表示されたときに非表示のままになる。
                _activePanel.Visible = false;

                // _activePanel に残っていると不要な更新処理が行われるので null にする
                _activePanel = null;
            }

            base.OnVisibleChanged(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            // キャンセルボタン
            if (e.KeyCode == Keys.Escape)
            {
                this.Hide();
            }
            base.OnKeyDown(e);
        }

        /// <summary>
        /// カレントペインの更新//@@@
        /// </summary>
        public void UpdateCurrentPane()
        {
            if (_currentISubScene != null)
            {
                _currentISubScene.CurrentPane = _target != null && _target.Active != null ?
                    _target.Active.PaneManipulator.IPane : null;
            }
        }

        /// <summary>
        /// カレントペインの無効化//@@@
        /// </summary>
        public void DisableCurrentPane()
        {
            if (_currentISubScene != null)
            {
                _currentISubScene.CurrentPane = null;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnDeactivate(EventArgs e)
        {
            // TODO: 目的を確認したい
            this.ActiveControl = null;
            base.OnDeactivate(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnClosing(CancelEventArgs e)
        {
            if (this._activePanel != null && _propDialogs.IndexOf(this) == 0)
            {
                _leftWidth = _activePanel.LeftWidth;
                _location = this.Location;
            }

            // ウインドウクローズ時ハンドラを呼び出します。
            if (OnClose != null)
            {
                OnClose(this);
            }

            _propDialogs.Remove(this);

            base.OnClosing(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            if (this._activePanel != null)
                _activePanel.SetCalculationPageSize();
            base.OnResize(e);
        }

        #endregion オーバーライドメソッド

        //-------------------------------------------------------
        #region イベントハンドラ。

        /// <summary>
        /// イベントハンドラ。
        /// </summary>
        private void Event_BtnSubDialog_Click(object sender, EventArgs e)
        {
            // 作成が重いのでウェイト表示...
            using (WaitCursor wait = new WaitCursor())
            {
                _propertyWindowMgr.DuplicatePropertyDialog(this);
            }
        }

        /// <summary>
        /// フリーズボタンの外観。
        /// </summary>
        private void Event_chkFreeze_Click(object sender, System.EventArgs e)
        {
            this.chkFreeze.ImageIndex = (chkFreeze.Checked) ? 1 : 0;
            ctlTargetBox.BackColor = (chkFreeze.Checked) ? SystemColors.Control : SystemColors.Window;
        }

        //-------------------------------------------------------

        #endregion イベントハンドラ。
    }
}
