﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Windows.Forms;
using System.Drawing;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using LayoutEditor.Controls;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;

    using LECore.Structures;
    using LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    ///
    /// </summary>
    public class WindowTargetPartsPanel : Panel
    {
        SolidBrush _activeBrush = new SolidBrush(Color.FromArgb(64, Color.Red));

        public event Action<LEWindowPartsID> OnWindowPartsClicked;

        public WindowTargetPartsPanel()
        {
            InitializeComponent();
        }

        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // WindowTargetPartsPanel
            //
            this.Paint += new System.Windows.Forms.PaintEventHandler(this.Event_WindowPaneAdapterRenderingPanel_Paint);
            this.MouseClick += new System.Windows.Forms.MouseEventHandler(this.WindowPaneAdapterRenderingPanel_MouseClick);
            ResizeRedraw = true;
            this.ResumeLayout(false);

        }

        //----------------------------------------------------------

        public IWindowPaneAdapterEntry Adapter { get; set; }

        private WindowKind _TargetWindowKind { get { return Adapter != null ? Adapter.Manipulator.ILEWindow.WindowKind : WindowKind.Around; } }

        //----------------------------------------------------------

        private Rectangle _RectOut
        {
            get
            {
                Rectangle rect = this.ClientRectangle;
                rect.Inflate((int)(-rect.Width * 0.1f), (int)(-rect.Height * 0.1f));
                return rect;
            }
        }

        /// <summary>
        /// 描画処理
        /// </summary>
        private void Event_WindowPaneAdapterRenderingPanel_Paint(object sender, System.Windows.Forms.PaintEventArgs e)
        {
            Rectangle rect = this.ClientRectangle;
            Rectangle rectOut = _RectOut;

            Graphics g = e.Graphics;

            // ウインドウ枠
            {
                if (_TargetWindowKind == WindowKind.Around)
                {
                    g.DrawRectangle(Pens.Black, rectOut);
                    g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.Content));
                    g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.CornerLT));
                    g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.CornerRT));
                    g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.CornerLB));
                    g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.CornerRB));
                }
                else
                {
                    if (_TargetWindowKind == WindowKind.HorizontalNoContent)
                    {
                        Rectangle rectL = GetRectangle_(rectOut, LEWindowPartsID.FrameL);
                        rectL = Rectangle.Union(rectL, GetRectangle_(rectOut, LEWindowPartsID.Content));

                        g.DrawRectangle(Pens.LightGray, GetRectangle_(rectOut, LEWindowPartsID.FrameL));
                        g.DrawRectangle(Pens.Black, rectL);
                        g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.FrameR));
                    }
                    else
                    {
                        g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.FrameL));
                        g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.Content));
                        g.DrawRectangle(Pens.Black, GetRectangle_(rectOut, LEWindowPartsID.FrameR));
                    }

                }
            }

            // 設定部分のハイライト
            if (Adapter != null)
            {
                if (Adapter.Manipulator.ILEWindow.NumTexture == 2 &&
                Adapter.CurrentTargetPartsID != LEWindowPartsID.Content)
                {
                    DrawActiveArea_(g, rectOut, ConvertPartsID_(LEWindowPartsID.CornerLT, _TargetWindowKind));
                    DrawActiveArea_(g, rectOut, ConvertPartsID_(LEWindowPartsID.CornerRT, _TargetWindowKind));
                    DrawActiveArea_(g, rectOut, ConvertPartsID_(LEWindowPartsID.CornerLB, _TargetWindowKind));
                    DrawActiveArea_(g, rectOut, ConvertPartsID_(LEWindowPartsID.CornerRB, _TargetWindowKind));
                }
                else
                {
                    DrawActiveArea_(g, rectOut, ConvertPartsID_(Adapter.CurrentTargetPartsID, _TargetWindowKind));
                }
            }

            // 外枠
            g.DrawRectangle(Pens.Black, rect);
        }

        //----------------------------------------------------------

        void DrawActiveArea_(Graphics g, Rectangle rectOut, LEWindowPartsID id)
        {
            var activeRect = GetRectangle_(rectOut, id);
            g.FillRectangle(_activeBrush, activeRect);
        }

        //----------------------------------------------------------

        private Rectangle GetRectangle_(Rectangle rectOut, LEWindowPartsID partsID, WindowKind kind)
        {
            return GetRectangle_(rectOut, ConvertPartsID_(partsID, kind));
        }

        private Rectangle GetRectangle_(Rectangle rectOut, LEWindowPartsID partsID)
        {
            Rectangle rectIn = rectOut;
            float frameSize = -rectIn.Height * 0.2f;
            rectIn.Inflate(new Size((int)(frameSize), (int)(frameSize)));

            int cornerW = rectIn.X - rectOut.X;
            int cornerH = rectIn.Y - rectOut.Y;

            switch (partsID)
            {
                case LEWindowPartsID.Content: return rectIn;
                case LEWindowPartsID.CornerLT: return new Rectangle(rectOut.Left, rectOut.Top, cornerW, cornerH);
                case LEWindowPartsID.CornerRT: return new Rectangle(rectIn.Right, rectOut.Top, cornerW, cornerH);
                case LEWindowPartsID.CornerLB: return new Rectangle(rectOut.Left, rectIn.Bottom, cornerW, cornerH);
                case LEWindowPartsID.CornerRB: return new Rectangle(rectIn.Right, rectIn.Bottom, cornerW, cornerH);
                case LEWindowPartsID.FrameT: return new Rectangle(rectIn.Left, rectOut.Top, rectIn.Width, cornerH);
                case LEWindowPartsID.FrameB: return new Rectangle(rectIn.Left, rectIn.Bottom, rectIn.Width, cornerH);
                case LEWindowPartsID.FrameL: return new Rectangle(rectOut.Left, rectIn.Top, cornerW, rectIn.Height);
                case LEWindowPartsID.FrameR: return new Rectangle(rectIn.Right, rectIn.Top, cornerW, rectIn.Height);
                default: return Rectangle.Empty;
            }
        }

        LEWindowPartsID ConvertPartsID_(LEWindowPartsID origID, WindowKind kind)
        {
            if (kind == WindowKind.Around)
            {
                return origID;
            }
            else
            {
                switch (origID)
                {
                    case LEWindowPartsID.CornerLT:
                    case LEWindowPartsID.CornerLB:
                        return LEWindowPartsID.FrameL;
                    case LEWindowPartsID.CornerRT:
                    case LEWindowPartsID.CornerRB:
                        return LEWindowPartsID.FrameR;
                    case LEWindowPartsID.FrameT:
                    case LEWindowPartsID.FrameB:
                    case LEWindowPartsID.Content:
                        return LEWindowPartsID.Content;
                    default: return origID;
                }
            }
        }

        /// <summary>
        /// クリック
        /// </summary>
        private void WindowPaneAdapterRenderingPanel_MouseClick(object sender, MouseEventArgs e)
        {
            LEWindowPartsID? newID = null;
            foreach (LEWindowPartsID partsID in Enum.GetValues(typeof(LEWindowPartsID)))
            {
                if (GetRectangle_(_RectOut, ConvertPartsID_(partsID, _TargetWindowKind)).Contains(e.Location))
                {
                    newID = partsID;
                    break;
                }
            }

            if (newID.HasValue && OnWindowPartsClicked != null)
            {
                if (newID.Value == LEWindowPartsID.Content && _TargetWindowKind == WindowKind.HorizontalNoContent)
                {
                    OnWindowPartsClicked(LEWindowPartsID.CornerLT);
                    return;
                }

                OnWindowPartsClicked(newID.Value);
            }
        }
    }
}
