﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow;
using LECore.Structures;
using LECore;
using LECore.Manipulator;
using LayoutEditor.Controls;
using LayoutEditor.Forms.ToolWindows.common;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    public partial class DefaultPartsPanePage : PropertyPage
    {
        /// <summary>
        /// パネル初期化パラメータです。
        /// </summary>
        public class InitializeParamaters
        {
            public readonly IPartsLayout PartsLayout;
            public readonly string TargetName;
            public readonly PaneKind PaneKind;
            public readonly IPartsLayout ParentPartsLayout;

            public InitializeParamaters(
                IPartsLayout partsLayout, IPartsLayout parentPartsLayout, string targetName, PaneKind paneKind)
            {
                this.PartsLayout = partsLayout;
                this.ParentPartsLayout = parentPartsLayout;

                this.TargetName = targetName;
                this.PaneKind = paneKind;
            }
        }

        /// <summary>
        /// 初期化パラメータです。
        /// </summary>
        private InitializeParamaters _initializeParamaters { get; set; }
        private PartsPropertyUsageOptions _selectedOption;

        /// <summary>
        /// 初期オプション
        /// </summary>
        protected PartsPropertyUsageOptions _InitialPropertyUsageOptions
        {
            get
            {
                var targetProp = this._initializeParamaters.PartsLayout.PartsPropaerties.FirstOrDefault(
                    (prop) => prop.TargetName == this._initializeParamaters.TargetName);
                return (targetProp != null) ? targetProp.Option : PartsPropertyUsageOptions.None;
            }
        }

        /// <summary>
        /// 選択オプション
        /// </summary>
        protected virtual PartsPropertyUsageOptions _SelectedOption
        {
            set { _selectedOption = value; }
            get
            {
                return !_selectedOption.HasFlag(PartsPropertyUsageOptions.UsePaneExParamater) ?
                    (_selectedOption & ~PartsPropertyUsageOptions.UsePaneExParamaterColorBlend & ~PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText & ~PartsPropertyUsageOptions.UsePaneExParameterTexture) :
                    _selectedOption;
            }
        }

        //----------------------------------------------------------
        // 生成
        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DefaultPartsPanePage()
        {
            InitializeComponent();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        protected DefaultPartsPanePage(PropertyPageInitializeParamaters paramater)
            : this()
        {
            this._initializeParamaters = paramater.Paramaters;
            _SelectedOption = _InitialPropertyUsageOptions;

            // コメント
            IPartsPropaertySettings setting = this._initializeParamaters.PartsLayout.GetPartsPropaertySettings(this._initializeParamaters.TargetName);
            _lblCommentContent.Text = (setting != null) ? setting.Description : string.Empty;
            _ttpMain.SetToolTip(_lblCommentContent, _lblCommentContent.Text);
        }

        /// <summary>
        /// CreateControlイベントを発生させます。
        /// </summary>
        protected override void OnCreateControl()
        {
            base.OnCreateControl();

            // コメントラベルの縦幅の追加サイズを計算
            // コンストラクタで計算すると間違った値になるのでここで行う
            Size maxSize = new Size(_lblCommentContent.Width, 400);
            int addSize = _lblCommentContent.GetPreferredSize(maxSize).Height - _lblCommentContent.Height;

            // コメントラベルを大きくして、コメントラベルの下にあるコントロールの位置をずらす
            _lblCommentContent.Height += addSize;
            _clbPartialOptionEx.Top += addSize;
            _grpOverwrite.Top += addSize;
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static PropertyPage CreateInstance(object arg)
        {
            Ensure.Argument.True(arg is PropertyPageInitializeParamaters);
            return new DefaultPartsPanePage(arg as PropertyPageInitializeParamaters);
        }

        //----------------------------------------------------------
        //
        //----------------------------------------------------------

        /// <summary>
        /// プロパティ初期化。
        /// </summary>
        protected override void InitializeProperty()
        {
            foreach (var item in this.CollectPartialSettingItems())
            {
                _clbPartialOption.Items.Add(item, true);
            }

            foreach (var item in this.CollectPartialSettingItemsEx())
            {
                _clbPartialOptionEx.Items.Add(item, true);
            }

            if (_clbPartialOption.Items.Count <= 0)
            {
                _pnlOverwrite.Visible = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private IEnumerable<UIListControlItem> CollectPartialSettingItems()
        {
            if (_initializeParamaters.PaneKind != PaneKind.Null &&
                _initializeParamaters.PaneKind != PaneKind.Bounding)
            {
                yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USEPANEEXPARAM", GetPaneLableText_(_initializeParamaters.PaneKind)), PartsPropertyUsageOptions.UsePaneExParamater);
            }

            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USEPOS"), PartsPropertyUsageOptions.UsePanePosition);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USESIZE"), PartsPropertyUsageOptions.UsePaneSize);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USESCALE"), PartsPropertyUsageOptions.UsePaneScale);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USEROTATE"), PartsPropertyUsageOptions.UsePaneRotate);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_TRANSPARENCY"), PartsPropertyUsageOptions.UsePaneTransparency);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USEVISIBLITY"), PartsPropertyUsageOptions.UsePaneVisibility);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USEBASICUSER"), PartsPropertyUsageOptions.UsePaneBasicUserData);
            yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_USEEXUSER"), PartsPropertyUsageOptions.UsePaneExUserData);
        }

        /// <summary>
        ///
        /// </summary>
        private IEnumerable<UIListControlItem> CollectPartialSettingItemsEx()
        {
            if (_initializeParamaters.PaneKind == PaneKind.Textbox)
            {
                yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_TEXTBOX_TEXT"), PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText);
            }

            if (PartsPropertyUsageOptionsHelper.IsColorBlendOverwriteEnabled(_initializeParamaters.PaneKind))
            {
                yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_MATCOLORBLEND"), PartsPropertyUsageOptions.UsePaneExParamaterColorBlend);
            }

            if (PartsPropertyUsageOptionsHelper.IsTextureOverwriteEnabled(_initializeParamaters.PaneKind))
            {
                yield return new UIListControlItem(StringResMgr.Get("PROP_PARTSPROP_MATTEXTURE"), PartsPropertyUsageOptions.UsePaneExParameterTexture);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private string GetPaneLableText_(PaneKind kind)
        {
            switch (kind)
            {
                case PaneKind.Textbox: return StringResMgr.Get("PROP_PANE_CATEGORY_TEXTBOX");
                case PaneKind.Picture: return StringResMgr.Get("PROP_PANE_CATEGORY_PICTURE");
                case PaneKind.Window: return StringResMgr.Get("PROP_PANE_CATEGORY_WINDOW");
                case PaneKind.Parts: return StringResMgr.Get("PROP_PANE_CATEGORY_PARTS");
                case PaneKind.Capture: return StringResMgr.Get("PROP_PANE_CATEGORY_CAPTURE");
                case PaneKind.Alignment: return StringResMgr.Get("PROP_PANE_CATEGORY_ALIGNMENT");
                case PaneKind.Scissor: return StringResMgr.Get("PROP_PANE_CATEGORY_SCISSOR");
                default: return "UnKnown";
            }
        }

        /// <summary>
        /// 読み込み専用でも編集を許可するかどうか？
        /// <remarks>
        /// 例外的に、編集を許可したいページでオーバーライドします。
        /// </remarks>
        /// </summary>
        protected override bool AllowEditReadOnlyTarget()
        {
            return true;
        }

        /// <summary>
        /// プロパティ更新。
        /// </summary>
        protected override void UpdateProperty()
        {
            UpdateProperty_();
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateProperty_ListCheck(CheckedListBox clb, PartsPropertyUsageOptions option)
        {
            List<bool> resetIndex = new List<bool>();

            PartsPropertyUsageOptions targetPropOption = option;
            foreach (UIListControlItem item in clb.Items)
            {
                PartsPropertyUsageOptions flag = (PartsPropertyUsageOptions)item.Data;
                resetIndex.Add((targetPropOption & flag) == flag);
            }

            for (int i = 0; i < resetIndex.Count; i++)
            {
                clb.SetItemChecked(i, resetIndex[i]);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void UpdateProperty_()
        {
            UpdateProperty_ListCheck(_clbPartialOption, this._SelectedOption);
            UpdateProperty_ListCheck(_clbPartialOptionEx, this._SelectedOption);

            _btnSetEnabled.Enabled = this._SelectedOption != this._InitialPropertyUsageOptions;
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_BtnSetEnabled_Click(object sender, EventArgs e)
        {
            PartsLayoutManipulator mnp = new PartsLayoutManipulator();

            mnp.BindTarget(this._initializeParamaters.PartsLayout);

            var targetProp = this._initializeParamaters.PartsLayout.FindPartsPropertyByName(this._initializeParamaters.TargetName);
            PartsPropertyUsageOptions beforeModifyOption = targetProp.Option;

            var subScene = this._initializeParamaters.ParentPartsLayout != null ?
                this._initializeParamaters.ParentPartsLayout.OwnerPane.OwnerSubScene : this._initializeParamaters.PartsLayout.OwnerPane.OwnerSubScene;

            // 拡張ユーザー情報のチェックが変わったか
            bool doResetExUserData = (((uint)_SelectedOption ^ (uint)beforeModifyOption & (uint)PartsPropertyUsageOptions.UsePaneExUserData) == (uint)PartsPropertyUsageOptions.UsePaneExUserData);

            try
            {
                if (subScene != null) { subScene.BeginMassiveModify(); }

                IPane targetPane = null;
                PaneManipulator paneManipulator = null;

                // アンドゥで拡張ユーザー情報の参照が変わらないように、暫定処置として参照復元コマンド積む
                if (subScene != null && doResetExUserData)
                {
                    targetPane = this._initializeParamaters.PartsLayout.PartsSubScene.FindPaneByName(this._initializeParamaters.TargetName);
                    paneManipulator = new PaneManipulator();
                    paneManipulator.BindTarget(targetPane);

                    paneManipulator.CreateUserDataRestorePoint(false);
                }

                mnp.ModifyOverwriteParamaters(
                this._initializeParamaters.TargetName,
                this._initializeParamaters.PaneKind,
                _SelectedOption);

                // 自分自身が孫部品の場合、親が上書きしている値にリセットする
                if (this._initializeParamaters.ParentPartsLayout != null)
                {
                    // 今回新規にONにされた項目
                    PartsPropertyUsageOptions resetParamOption = (PartsPropertyUsageOptions)(((uint)_SelectedOption ^ (uint)beforeModifyOption) & ((uint)_SelectedOption));

                    mnp.SetPropertyToPane(this._initializeParamaters.ParentPartsLayout, this._initializeParamaters.TargetName, resetParamOption);
                }

                // 上書き設定をリセットした際にキャプチャテクスチャの参照情報がファイル読み込み時の状態にリセットされるため
                // ランタイムでの配置状態を考慮した名前に再度更新する。
                if (_SelectedOption == PartsPropertyUsageOptions.None)
                {
                    SubSceneHelper.ResolvePartsLayoutCaptureTextureReference(this._initializeParamaters.PartsLayout.OwnerPane);
                }

                // リドゥで拡張ユーザー情報の参照が変わらないように、暫定処置として参照復元コマンド積む
                if (subScene != null && doResetExUserData)
                {
                    paneManipulator.CreateUserDataRestorePoint(true);
                }
            }
            finally
            {
                if (subScene != null) { subScene.EndMassiveModify(); }
            }

            UpdateProperty_();
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_clbPartialOption_MouseClick(object sender, MouseEventArgs e)
        {
            PartsPropertyUsageOptionsChanged_(sender, e, PartsPropertyUsageOptions.None);
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_clbPartialOptionEx_MouseClick(object sender, MouseEventArgs e)
        {
            PartsPropertyUsageOptionsChanged_(sender, e, PartsPropertyUsageOptions.UsePaneExParamater);
        }

        /// <summary>
        ///
        /// </summary>
        private void PartsPropertyUsageOptionsChanged_(object sender, MouseEventArgs e, PartsPropertyUsageOptions baseOption)
        {
            var clb = sender as CheckedListBox;
            int index = clb.IndexFromPoint(e.Location);
            if (index != -1)
            {
                bool newCheck = !clb.GetItemChecked(index);
                clb.SetItemChecked(index, newCheck);

                PartsPropertyUsageOptions opt = (PartsPropertyUsageOptions)(clb.Items[index] as UIListControlItem).Data;
                if (newCheck)
                {
                    // baseOption は必ず一緒に有効にする。
                    opt |= baseOption;
                    this._SelectedOption |= opt;
                }
                else
                {
                    this._SelectedOption &= ~opt;
                }

                UpdateProperty_();
            }
        }
    }
}
