﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;


namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.TextureControls
{
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore.Util;
    using System.ComponentModel;
    using Win32 = LECore.Win32;
    /// <summary>
    /// テクスチャマップボックスクラス。
    /// </summary>
    public sealed class TexMapBox : Control
    {
        private TexMap _texMap   = null;
        private bool   _selected = false;
        private bool   _isDragging = false;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMapBox()
        {
            // 独自描画用
            base.SetStyle(ControlStyles.Opaque|ControlStyles.DoubleBuffer|ControlStyles.ResizeRedraw, true);
        }

        /// <summary>
        /// テクスチャマップ。
        /// </summary>
        public TexMap TexMap
        {
            get { return _texMap; }
            set
            {
                _texMap = value;
                Invalidate();
            }
        }

        /// <summary>
        /// 選択されているかどうか。
        /// </summary>
        public bool Selected
        {
            get { return _selected; }
            set
            {
                _selected = value;
                Invalidate();
            }
        }

        /// <summary>
        /// TexMapを持っているかどうか。
        /// </summary>
        public bool HasTexMap
        {
            get { return TexMap != null; }
        }

        /// <summary>
        /// ドラッグ可能かどうか。
        /// </summary>
        [Browsable(false)]
        public bool IsDraggable
        {
            get; set;
        }

        /// <summary>
        /// 操作対象のマテリアル。
        /// </summary>
        [Browsable(false)]
        public MaterialGUIAdapter TargetMaterialGUIAdapter
        {
            get; set;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override CreateParams CreateParams
        {
            get
            {
                CreateParams cp = base.CreateParams;
                cp.ExStyle |= Win32.WS_EX.WS_EX_STATICEDGE;
                return cp;
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnPaint(PaintEventArgs e)
        {
            // 背景
            if (_selected) { e.Graphics.Clear(SystemColors.Highlight); }
            else           { e.Graphics.Clear(SystemColors.Control);   }

            // テクスチャマップ
            Rectangle rcImage = this.ClientRectangle;
            rcImage.Inflate(-2, -2);
            if (_texMap != null)
            {
                _texMap.TexImage.DrawImageOnCheckerBoard(e.Graphics, rcImage, HasTexMap);

                Pen pen;
                if (HasTexMap) { pen = SystemPens.ControlText; }
                else           { pen = SystemPens.ControlDark; }

                rcImage.Width  -= 1;
                rcImage.Height -= 1;
                e.Graphics.DrawRectangle(pen, rcImage);
            }
            else
            {
                Pen pen;
                if (HasTexMap) { pen = SystemPens.ControlText; }
                else           { pen = SystemPens.ControlDark; }

                rcImage.Width  -= 1;
                rcImage.Height -= 1;
                e.Graphics.DrawRectangle(pen, rcImage);
                e.Graphics.DrawLine(pen, RectangleUtil.GetTopLeft(rcImage), RectangleUtil.GetBottomRight(rcImage));
                e.Graphics.DrawLine(pen, RectangleUtil.GetBottomLeft(rcImage), RectangleUtil.GetTopRight(rcImage));
            }

            // フォーカス枠
            if (this.Focused)
            {
                GraphicsUtil.DrawFocusRectangle(e.Graphics, this.ClientRectangle);
            }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            // フォーカスを取得する
            if (!this.Focused && HasTexMap)
            {
                Focus();
            }

            // フォーカスがある場合ドラッグ処理を開始する
            if (this.Focused && IsDraggable)
            {
                _isDragging = true;
            }

            base.OnMouseDown(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (_isDragging)
            {
                if (e.X < 0 || e.Y < 0 ||
                this.Bounds.Width < e.X || this.Bounds.Height < e.Y)
                {
                    // ドラッグ開始
                    DoDragDrop(new DragDropItem
                    {
                        TexMapBox = this,
                        TargetMaterialGUIAdapter = TargetMaterialGUIAdapter,
                    }, DragDropEffects.Move);
                }
            }

            base.OnMouseMove(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            _isDragging = false;

            base.OnMouseUp(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnGotFocus(EventArgs e)
        {
            this.Selected = true;
            base.OnGotFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnLostFocus(EventArgs e)
        {
            _isDragging = false;
            Invalidate();
            base.OnLostFocus(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnResize(EventArgs e)
        {
            Invalidate();
            base.OnResize(e);
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void OnEnabledChanged(EventArgs e)
        {
            Invalidate();
            base.OnEnabledChanged(e);
        }

        /// <summary>
        /// ドラッグドロップ用の情報
        /// </summary>
        public class DragDropItem
        {
            public TexMapBox TexMapBox { get; set; }

            public MaterialGUIAdapter TargetMaterialGUIAdapter { get; set; }
        }
    }
}
