﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Drawing;
using System.Linq;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using System.Collections.Generic;
    using LayoutEditor.Controls;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore;
    using LECore.Structures;
    using System.Threading;

    /// <summary>
    /// MaterialPage の概要の説明です。
    /// フォームデザイナへの配慮から、abstract 指定をしていませんが
    /// 抽象基底クラスです。
    /// </summary>
    public class MaterialPage : PropertyPage
    {
        //----------------------------------------------------------
        // コンストラクタです。
        //----------------------------------------------------------

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialPage()
        {
            this._OwnerPane = null;
        }

        /// <summary>
        /// マテリアルページを生成します。
        /// </summary>
        public static MaterialPage CreateMaterialPage<TPage>(object arg)
            where TPage : MaterialPage, new()
        {
            Ensure.Argument.NotNull(arg as IPane);

            var newPage = new TPage();
            newPage._OwnerPane = arg as IPane;

            return newPage;
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// アクティブターゲット。
        /// </summary>
        protected MaterialGUIAdapter ActiveTarget
        {
            get
            {
                MaterialGUIAdapter matAdapter = null;
                PaneGUIAdapter peneAdapter = base.Target.Active;

                if (peneAdapter != null)
                {
                    matAdapter = peneAdapter.MaterialAdapters.FirstOrDefault(
                        (mat) => this._OwnerPane.HasMaterial(mat.IMaterial));
                }

                return matAdapter;
            }
        }

        /// <summary>
        /// 対象マテリアル列を取得します。
        /// </summary>
        protected IEnumerable<MaterialGUIAdapter> ActiveTargets
        {
            get
            {
                foreach (PaneGUIAdapter target in base.Target.Objects)
                {
                    if (target.Target.PaneKind == PaneKind.Parts)
                    {
                        // 一つのペインに、沢山のマテリアルが存在します。
                        // ページが関係する部分だけを対象とします。
                        var materialTargets = target.MaterialAdapters.Where(
                            (mat) => this._OwnerPane.HasMaterial(mat.IMaterial));
                        foreach (var matTarget in materialTargets)
                        {
                            yield return matTarget;
                        }
                    }
                    else
                    {
                        // すべてのマテリアルを対象とします。
                        // (ただし、Windowではこの処理をオーバーライドしており対象パーツのみが対象です)
                        foreach (var matTarget in target.MaterialAdapters)
                        {
                            yield return matTarget;
                        }
                    }
                }
            }
        }

        // コマンドの実行を強制します。
        protected bool ForcePushCommand { get; set; }
        /// <summary>
        /// ページの表示内容と関係しているペインです。
        /// peneAdapterからマテリアル操作クラスを取得する際のキーとして利用します。
        /// </summary>
        protected IPane _OwnerPane
        {
            get;
            set;
        }

        /// <summary>
        /// ページ対象を更新します。
        /// <remarks>
        /// 内部に対象に関する情報を保持してる場合にここで更新処理をします。
        /// </remarks>
        /// </summary>
        public override void UpdateTarget(IPane newTargetPane)
        {
            base.UpdateTarget(newTargetPane);
            this._OwnerPane = newTargetPane;
        }

        /// <summary>
        ///
        /// </summary>
        protected delegate void MultipleEditingCallback(MaterialGUIAdapter adapter);
        protected delegate FloatColor GetColorHandler(MaterialGUIAdapter adapter);
        protected delegate void SetColorHandler(MaterialGUIAdapter adapter, FloatColor color);

        protected void EditColor(
            PaneGuiAdapterGroup target, ColorEditEventArgs args, GetColorHandler getter, SetColorHandler setter)
        {
            PaneGUIAdapter activePaneAdapter = target.Active;

            if (target.IsMulti != false)
            {
                ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                subScene.BeginMassiveModify();

                EditColor(args, getter, setter);

                subScene.EndMassiveModify();
            }
            else
            {
                EditColor(args, getter, setter);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void EditColor(
            ColorEditEventArgs args,
            GetColorHandler getter,
            SetColorHandler setter)
        {
            MultipleEditing(delegate(MaterialGUIAdapter adapter)
                             {
                                 FloatColor color = getter(adapter);
                                 float R = args.IsModifiedR != false ? args.Color.R : color.R;
                                 float G = args.IsModifiedG != false ? args.Color.G : color.G;
                                 float B = args.IsModifiedB != false ? args.Color.B : color.B;
                                 float A = args.IsModifiedA != false ? args.Color.A : color.A;
                                 color = new FloatColor(R, G, B, A);
                                 color.ByteConvert = args.Color.ByteConvert;
                                 setter(adapter, color);
                             });


            this.NotifyPropertyChanged();
        }

        /// <summary>
        /// 変更を通知します。
        /// </summary>
        protected override void NotifyPropertyChanged()
        {
            OnModifyTargetAdapterUpdate_();
        }

        /// <summary>
        /// 変更を通知します。
        /// </summary>
        protected override void NotifyPropertyChanged(IBaseGuiAdapter paneAdapter)
        {
            OnModifyTargetAdapterUpdate_();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnModifyTargetAdapterUpdate_()
        {
            Ensure.Operation.ObjectNotNull(this._OwnerPane);
            LECore.LayoutEditorCore.Scene.CurrentISubScene.BeginMassiveModify();

            foreach (MaterialGUIAdapter adapter in this.ActiveTargets)
            {
                Ensure.Operation.ObjectNotNull(adapter);
                adapter.OnModified(this, ForcePushCommand);
            }

            LECore.LayoutEditorCore.Scene.CurrentISubScene.EndMassiveModify();

            ForcePushCommand = false;
        }

        /// <summary>
        /// 再描画前の対象更新です。
        /// </summary>
        protected override void BeforeUpdateTargetUpdate()
        {
            LECore.LayoutEditorCore.Scene.CurrentISubScene.BeginMassiveModify();
            foreach (MaterialGUIAdapter adapter in this.ActiveTargets)
            {
                Ensure.Operation.ObjectNotNull(adapter);
                adapter.SetCoreDataToGUIMock();
            }
            LECore.LayoutEditorCore.Scene.CurrentISubScene.EndMassiveModify();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void UpdateProperty()
        {
            if (this.IsTextureOverwritingEnabled)
            {
                foreach (MaterialGUIAdapter adapter in this.ActiveTargets)
                {
                    Ensure.Operation.ObjectNotNull(adapter);
                    adapter.IsTextureOverwritingEnabled = true;
                }
            }

            if (this.ActiveTarget != null)
            {
                DoUpdateProperty_();
            }
        }

        /// <summary>
        /// 実際のUI更新処理
        /// </summary>
        protected virtual void DoUpdateProperty_(){}

        /// <summary>
        ///
        /// </summary>
        protected void MultipleEditing(MultipleEditingCallback callback)
        {
            // materialIndex は不要になった。処理対象を paneAdapter 側で選別する。
            foreach (MaterialGUIAdapter adapter in this.ActiveTargets)
            {
                Ensure.Operation.ObjectNotNull(adapter);
                callback(adapter);
            }
        }
    }
}
