﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using LayoutEditor.Controls;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Structures;
    using System.Windows.Forms;
    using System.Drawing;
    using System.Linq;
    using LECore.Structures.Core;
    using System.Collections.Generic;
    using src.Controls;

    public partial class MaterialFragmentOperationPage : MaterialPage
    {
        IAnmAttribute AlphaTestAttribute
        {
            get
            {
                return this.ActiveTarget?.CoreMaterial?.GetAnimationTargetAttributeSet(ParamaterKind.Animation_AlphaTest)?.FirstOrDefault();
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialFragmentOperationPage()
        {
            InitializeComponent();

            lblACRef.BindAnimationMark(ctlACRef.AnimationMarkInst);
        }

        /// <summary>
        /// インスタンス作成。
        /// </summary>
        public static PropertyPage CreateInstance(object arg)
        {
            PropertyPage page = new MaterialFragmentOperationPage();
            return page;
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        public override PropertyPageID PageID
        {
            get { return PropertyPageID.MaterialFragmentOperation; }
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void InitializeProperty()
        {
            // プリセット
            _tmiNone.Tag = PEData.None;
            _tmiNone.Click += Event_CmbPreset_SelectedIndexChanged;
            _tmiMask.Tag = PEData.Mask;
            _tmiMask.Click += Event_CmbPreset_SelectedIndexChanged;
            _tmiAlpha.Tag = PEData.AlphaBlend;
            _tmiAlpha.Click += Event_CmbPreset_SelectedIndexChanged;
            _tmiAdd.Tag = PEData.Add;
            _tmiAdd.Click += Event_CmbPreset_SelectedIndexChanged;
            _tmiMul.Tag = PEData.Mul;
            _tmiMul.Click += Event_CmbPreset_SelectedIndexChanged;
            _tmiSub.Tag = PEData.Sub;
            _tmiSub.Click += Event_CmbPreset_SelectedIndexChanged;
            _tmiAlphaMaxA.Tag = PEData.AlphaBlendMaxA;
            _tmiAlphaMaxA.Click += Event_CmbPreset_SelectedIndexChanged;


            // アルファコンペア
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_NEVER"),   AttrCompareFunc.Never));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_LESS"),    AttrCompareFunc.Less));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_LEQUAL"),  AttrCompareFunc.Lequal));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_EQUAL"),   AttrCompareFunc.Equal));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_NEQUAL"),  AttrCompareFunc.Nequal));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_GEQUAL"),  AttrCompareFunc.Gequal));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_GREATER"), AttrCompareFunc.Greater));
            cmbACComp.Items.Add(new UIListControlItem(StringResMgr.Get("PROP_MAT_PE_ACF_ALWAYS"),  AttrCompareFunc.Always));
            cmbACComp.SelectedIndexChanged += Event_CmbACComp_SelectedIndexChanged;

            ctlACRef.ValueEdit += Event_CtlACRef_ValueEdit;
            ctlACRef.SetViewManagerMessageHandler(this.Owner.SendMessageToViewManagerAction);

            // ブレンド
            ctlBlend.InitializeProperty();
            ctlBlend.DataChanged += Event_CtlBlend_DataChanged;

            ctlBlendAlpha.InitializeProperty();
            ctlBlendAlpha.DataChanged += Event_CtlBlendAlpha_DataChanged;
            ctlBlendAlpha.IsAlpha = true;
        }

        void _tmiNone_Click(object sender, EventArgs e)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// オーバーライド。
        /// </summary>
        protected override void DoUpdateProperty_()
        {
            PEData data = this.ActiveTarget.PEData;

            // アニメーションマークとコンテキストメニューの有効無効
            bool bUse = CanUseAnimationMark(this._OwnerPane.OwnerSubScene);
            IEnumerable<IAnimationMarkUsable> animCtrlList = GetAnimationMarkUsableControl();
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UseAnimationMark = bUse;
            }

            // 現在の設定
            {
                bool isPreset = false;
                foreach (ToolStripMenuItem item in _cmsPreset.Items)
                {
                    PEData preset = item.Tag as PEData;
                    if (preset.Equals(data))
                    {
                        _lblPresetCurrent.Text = item.Text;
                        _lblPresetCurrentDesc.Text = item.ToolTipText;

                        isPreset = true;
                        break;
                    }
                }

                if (!isPreset)
                {
                    _lblPresetCurrent.Text = StringResMgr.Get("TAG_CUSTOM");
                    _lblPresetCurrentDesc.Text = "";
                }

                _lblPresetCurrent.ForeColor = isPreset ? Color.Black : Color.Blue;
            }

            // アルファコンペア
            LECore.Structures.PEACompare ac = data.ACompare;
            {
                ctlACRef.SetValue(ac.Ref);
                cmbACComp.SetSelectedItemData(ac.Comp);
            }

            // ブレンド
            ctlBlend.UpdateProperty();
            ctlBlend.Data = data.Blend;

            ctlBlendAlpha.UpdateProperty();
            ctlBlendAlpha.Data = data.BlendAlpha;

            // デフォルトブレンド設定
            _chkUseBlendDefaultSettings.SetChecked(data.UseDefaultBlendSettings);
            _chkUseAlphaTestDefaultSettings.SetChecked(data.UseDefaultAlphaTestSettings);

            _pnlAComp.Enabled = !data.UseDefaultAlphaTestSettings;
            _lblColorBlend.Enabled = !data.UseDefaultBlendSettings;
            _lblBlendAlpha.Enabled = !data.UseDefaultBlendSettings;
            ctlBlend.Enabled = !data.UseDefaultBlendSettings;
            ctlBlendAlpha.Enabled = !data.UseDefaultBlendSettings && ctlBlend.Data.Type != AttrBlendType.Logic;

            this.Enabled = this.ActiveTarget.IsBlendEditable;
            this.panel2.Enabled = this.Enabled;
            this.grpAComp.Enabled = this.Enabled;
            this.grpBlend.Enabled = this.Enabled;

            this._lblNotice.Visible = !this.ActiveTarget.IsBlendEditable;

            // 同時編集
            Color color = GetMultiEditButtonColor();
            _btnPreset.BackColor = color;

            // アニメーションマークで利用するアトリビュートを関連付け
            ctlACRef.SetTargetAttribute(AlphaTestAttribute);

            // アニメーションマークの更新
            foreach (IAnimationMarkUsable ctrl in animCtrlList)
            {
                ctrl.UpdateAnimationMark();
            }
        }

        /// <summary>
        /// アニメーションマークが利用可能なコントロールの一覧を取得する。
        /// </summary>
        protected override IEnumerable<IAnimationMarkUsable> GetAnimationMarkUsableControl()
        {
            List<IAnimationMarkUsable> list = new List<IAnimationMarkUsable>();

            list.Add(lblACRef);
            list.Add(ctlACRef);

            return list;
        }

        #region コピー＆ペースト
        /// <summary>
        /// コピーが可能か。
        /// </summary>
        public override bool CanCopy(){ return true; }

        /// <summary>
        /// コピー。
        /// </summary>
        public override Object Copy()
        {
            return Copy(ActiveTarget);
        }

        /// <summary>
        /// コピー。
        /// </summary>
        public static Object Copy(MaterialGUIAdapter target)
        {
            var clone = new PEData(null);
            clone.Set(target.PEData);

            return clone;
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public override void Paste(Object pasteObject)
        {
            Paste(ActiveTarget, pasteObject);
            using (new GUIUpdating(this))
            {
                UpdateProperty();
            }
            NotifyPropertyChanged();
        }

        /// <summary>
        /// ペースト。
        /// </summary>
        public static void Paste(MaterialGUIAdapter destination, Object pasteObject)
        {
            Debug.Assert(pasteObject is PEData);
            PEData source = (PEData)pasteObject;
            destination.PEData.Set(source);

            // MCS
            //			MaterialPEData.Send(destination, destination.PEData);
        }
        #endregion

        #region イベントハンドラ
        /// <summary>
        /// ブレンド。
        /// </summary>
        private void Event_CtlBlend_DataChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.ActiveTarget.PEData.Blend.Set(ctlBlend.Data);
            NotifyPropertyChanged();

            // 無効もしくは論理演算に設定されたら、アルファ設定も連動して設定します。
            if (ctlBlend.Data.Type == AttrBlendType.None ||
                ctlBlend.Data.Type == AttrBlendType.Logic)
            {
                ctlBlendAlpha.Data.Type = AttrBlendType.None;
                Event_CtlBlendAlpha_DataChanged(null, null);
            }
        }

        /// <summary>
        /// ブレンド(アルファ)。
        /// </summary>
        private void Event_CtlBlendAlpha_DataChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.ActiveTarget.PEData.BlendAlpha.Set(ctlBlendAlpha.Data);
            NotifyPropertyChanged();
        }

        /// <summary>
        /// アルファコンペア
        /// </summary>
        private void Event_CtlACRef_ValueEdit(object sender, ValueEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (e.EditFixed)
            {
                this.ActiveTarget.PEData.ACompare.Ref = ctlACRef.Value;
                NotifyPropertyChanged();
            }
        }

        /// <summary>
        /// アルファコンペア
        /// </summary>
        private void Event_CmbACComp_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.ActiveTarget.PEData.ACompare.Comp = (AttrCompareFunc)cmbACComp.SelectedItemData;
            NotifyPropertyChanged();
        }

        /// <summary>
        /// プリセット
        /// </summary>
        private void Event_CmbPreset_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            PEData peData = (sender as ToolStripMenuItem).Tag as PEData;
            MultiEdit_((obj) => {
                MaterialGUIAdapter adapter = obj as MaterialGUIAdapter;
                adapter.PEData.Set(peData);
                NotifyPropertyChanged();
            }, ActiveTarget, ActiveTargets);
        }

        /// <summary>
        /// デフォルト設定
        /// </summary>
        private void Event_ChkUseDefaultSettings_CheckedChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.ActiveTarget.PEData.UseDefaultBlendSettings = _chkUseBlendDefaultSettings.Checked;
            NotifyPropertyChanged();
        }

        /// <summary>
        /// デフォルト設定
        /// </summary>
        private void Event_chkUseAlphaTestDefaultSettings_CheckedChanged(object sender, EventArgs e)
        {
            if (IsGUIUpdating) { return; }

            this.ActiveTarget.PEData.UseDefaultAlphaTestSettings = _chkUseAlphaTestDefaultSettings.Checked;
            NotifyPropertyChanged();
        }
        #endregion


    }
}
