﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Collections.Specialized;
using System.Text;
using System.Windows.Forms;
using LayoutEditor.Controls;
using CombinerEditor.Service;
using LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow
{
    using LECore;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.Nsrif.Attributes;
    using Utility;

    public partial class MaterialCombinerUserShaderPage : MaterialPage
    {
        /// <summary>
        /// コンスタントカラー
        /// </summary>
        /// <remarks>
        /// LECore.Structures.Nsrif.Attributes.AttrTevKonst が存在するが順序が違うので、別途定義したものを利用します。
        /// </remarks>
        enum ConstantColorIndex
        {
            BlackColor = 0,
            WhiteColor,
            C0,
            C1,
            C2,
            C3,
            C4,
        }

        /// <summary>
        /// 標準のシェーダパラメータの名前変換テーブルです。
        /// </summary>
        private readonly Dictionary<string, string> baseParameterNameTable = new Dictionary<string, string>()
        {
            // Textures
            { "_a0", "テクスチャー0" },
            { "_a1", "テクスチャー1" },
            { "_a2", "テクスチャー2" },
        };

        /// <summary>
        /// コンスタントカラー用シェーダパラメータの名前変換テーブルです。
        /// </summary>
        private readonly Dictionary<string, string> constantColorParameterNameTable = new Dictionary<string, string>()
        {
            { "m_VectorParameter6",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST_BLACKCOLOR") },
            { "m_VectorParameter0",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST_WHITECOLOR") },
            { "m_VectorParameter1",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST0") },
            { "m_VectorParameter2",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST1") },
            { "m_VectorParameter3",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST2") },
            { "m_VectorParameter4",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST3") },
            { "m_VectorParameter5",  StringResMgr.Get("PROP_MAT_TEV_STAGE_KONST4") },
        };

        private struct ExUserDataInfo
        {
            public string propertyName;
            public string text;
            public UserDataKind kind;
            public string name;
            public int pri;

            public ExUserDataInfo(string _propertyName, string _text, UserDataKind _kind, string _name, int _pri)
            {
                propertyName = _propertyName;
                text = _text;
                kind = _kind;
                name = _name;
                pri = _pri;
            }
        };

        /// <summary>
        /// 拡張ユーザー情報パラメータの名前変換テーブルです。
        /// </summary>
        static private readonly ExUserDataInfo[] extUserDataParameterTable = new []
        {
            // Float
            new ExUserDataInfo("m_VectorParameter8.x", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT") + "-" + "ExUserDataFloat_0", UserDataKind.AnmFloat, "__CUS_Float_0", 0),
            new ExUserDataInfo("m_VectorParameter8.y", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT") + "-" + "ExUserDataFloat_1", UserDataKind.AnmFloat, "__CUS_Float_1", 1),
            new ExUserDataInfo("m_VectorParameter8.z", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT") + "-" + "ExUserDataFloat_2", UserDataKind.AnmFloat, "__CUS_Float_2", 2),
            new ExUserDataInfo("m_VectorParameter8.w", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT") + "-" + "ExUserDataFloat_3", UserDataKind.AnmFloat, "__CUS_Float_3", 3),

            // Vec2
            new ExUserDataInfo("m_VectorParameter9.xy", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC2") + "-" + "ExUserDataVec2_0", UserDataKind.AnmFloatVec2, "__CUS_Vec2_0", 4),
            new ExUserDataInfo("m_VectorParameter9.zw", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC2") + "-" + "ExUserDataVec2_1", UserDataKind.AnmFloatVec2, "__CUS_Vec2_1", 5),
            new ExUserDataInfo("m_VectorParameter10.xy", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC2") + "-" + "ExUserDataVec2_2", UserDataKind.AnmFloatVec2, "__CUS_Vec2_2", 6),
            new ExUserDataInfo("m_VectorParameter10.zw", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC2") + "-" + "ExUserDataVec2_3", UserDataKind.AnmFloatVec2, "__CUS_Vec2_3", 7),

            // Vec3
            new ExUserDataInfo("m_VectorParameter11.xyz", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC3") + "-" + "ExUserDataVec3_0", UserDataKind.AnmFloatVec3, "__CUS_Vec3_0", 8),
            new ExUserDataInfo("m_VectorParameter12.xyz", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC3") + "-" + "ExUserDataVec3_1", UserDataKind.AnmFloatVec3, "__CUS_Vec3_1", 9),
            new ExUserDataInfo("m_VectorParameter13.xyz", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC3") + "-" + "ExUserDataVec3_2", UserDataKind.AnmFloatVec3, "__CUS_Vec3_2", 10),
            new ExUserDataInfo("m_VectorParameter14.xyz", StringResMgr.Get("PROP_USERDATA_KIND_ANM_FLOAT_VEC3") + "-" + "ExUserDataVec3_3", UserDataKind.AnmFloatVec3, "__CUS_Vec3_3", 11),

            // Int
            new ExUserDataInfo("m_IVectorParameter0.x", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT") + "-" + "ExUserDataInt_0", UserDataKind.AnmInt, "__CUS_Int_0", 12),
            new ExUserDataInfo("m_IVectorParameter0.y", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT") + "-" + "ExUserDataInt_1", UserDataKind.AnmInt, "__CUS_Int_1", 13),
            new ExUserDataInfo("m_IVectorParameter0.z", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT") + "-" + "ExUserDataInt_2", UserDataKind.AnmInt, "__CUS_Int_2", 14),
            new ExUserDataInfo("m_IVectorParameter0.w", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT") + "-" + "ExUserDataInt_3", UserDataKind.AnmInt, "__CUS_Int_3", 15),

            // IVec2
            new ExUserDataInfo("m_IVectorParameter1.xy", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT_VEC2") + "-" + "ExUserDataIvec2_0", UserDataKind.AnmIntVec2, "__CUS_IVec2_0", 16),
            new ExUserDataInfo("m_IVectorParameter1.zw", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT_VEC2") + "-" + "ExUserDataIvec2_1", UserDataKind.AnmIntVec2, "__CUS_IVec2_1", 17),
            new ExUserDataInfo("m_IVectorParameter2.xy", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT_VEC2") + "-" + "ExUserDataIvec2_2", UserDataKind.AnmIntVec2, "__CUS_IVec2_2", 18),
            new ExUserDataInfo("m_IVectorParameter2.zw", StringResMgr.Get("PROP_USERDATA_KIND_ANM_INT_VEC2") + "-" + "ExUserDataIvec2_3", UserDataKind.AnmIntVec2, "__CUS_IVec2_3", 19),

            // Rgba
            new ExUserDataInfo("m_IVectorParameter3", StringResMgr.Get("PROP_USERDATA_KIND_ANM_COLOR_RGBA") + "-" + "ExUserDataRgba_0", UserDataKind.AnmByteRGBA4, "__CUS_Rgba_0", 20),
            new ExUserDataInfo("m_IVectorParameter4", StringResMgr.Get("PROP_USERDATA_KIND_ANM_COLOR_RGBA") + "-" + "ExUserDataRgba_1", UserDataKind.AnmByteRGBA4, "__CUS_Rgba_1", 21),
            new ExUserDataInfo("m_IVectorParameter5", StringResMgr.Get("PROP_USERDATA_KIND_ANM_COLOR_RGBA") + "-" + "ExUserDataRgba_2", UserDataKind.AnmByteRGBA4, "__CUS_Rgba_2", 22),
            new ExUserDataInfo("m_IVectorParameter6", StringResMgr.Get("PROP_USERDATA_KIND_ANM_COLOR_RGBA") + "-" + "ExUserDataRgba_3", UserDataKind.AnmByteRGBA4, "__CUS_Rgba_3", 23),
        };

        #region フィールド

        private string _loadedCombinerFilePath = string.Empty;
        private BlockElementInfo[] _combinerFileBlockElements = null;

        #endregion

        #region プロパティ
        private string[] BlockDefinitionFolderPaths
        {
            get
            {
                return AppSetting.ProjectSettings.UserCombinerShaderSettings.CombinerBlockDefinitionDirectories
                    .Select(s => SceneHelper.GetAbsolutePathFromPartsRootBasedPath(LayoutEditorCore.Scene, s))
                    .ToArray();
            }
        }

        private string CombinerFile
        {
            get { return this.ActiveTarget.CombinerUserShader.FileName; }
            set { this.ActiveTarget.CombinerUserShader.FileName = value; }
        }

        /// <summary>
        /// 操作対象のマテリアルを持つペインを取得
        /// </summary>
        /// <remarks>
        /// 部品ペインで利用する場合は 操作対象のマテリアルは部品レイアウト内のマテリアルを指します。
        /// 部品ペインそのものは PaneGUIAdapter から Target.Active?.PaneManipulator?.IPane で取得します。
        /// </remarks>
        private IPane TargetPaneForMaterial
        {
            get {  return ActiveTarget.IMaterial.OwnerPane; }
        }

        /// <summary>
        /// 操作対象のマテリアルを取得
        /// </summary>
        private IRevHWMaterial TargetRevHWMaterial
        {
            get
            {
                IRevHWMaterial[] revMtlSets = PaneHelper.GetRevHWMatFromPane( TargetPaneForMaterial );
                return revMtlSets.First(s => s.IMaterial.Equals(ActiveTarget.IMaterial));
            }
        }

        #endregion

        public MaterialCombinerUserShaderPage()
        {
            InitializeComponent();
        }
        protected override void InitializeProperty()
        {
            // コンスタントカラー初期値の設定
            _ctlConstColor0.Tag = 0;
            _ctlConstColor1.Tag = 1;
            _ctlConstColor2.Tag = 2;
            _ctlConstColor3.Tag = 3;
            _ctlConstColor4.Tag = 4;
        }

        private BlockElementInfo[] LoadCombinerFile()
        {
            if (!AppSetting.ProjectSettings.IsCombinerUserShaderEnabled)
            {
                return null;
            }

            if (string.IsNullOrEmpty(CombinerFile) || !File.Exists(CombinerFile))
            {
                return null;
            }

            // ブロック定義フォルダパスを列挙
            string[] blockDefinitionFolderPaths = BlockDefinitionFolderPaths;

            try
            {
                // ブロック定義データをロード
                var blockDefinitions = CombinerEditorService.LoadBlockDefinitions(blockDefinitionFolderPaths);

                if (blockDefinitions == null || blockDefinitions.Count() == 0)
                {
                    // ブロック定義データのロードを失敗
                    return null;
                }

                // コンバイナファイルをロード
                var blockElements = CombinerEditorService.LoadCombinerFile(CombinerFile, blockDefinitions);

                return  blockElements.ToArray();

            }
            catch
            {
                return null;
            }
        }

        protected override void UpdateProperty()
        {
            // 上書き設定時は下記の制限が付きます
            //    ・コンバイナモードの切替不可
            //    ・コンバイナファイルの変更不可
            //    ・コンバイナファイルの設定の上書き機能の利用不可
            _grpCombinerFile.Enabled = !IsOverwrite;
            _grpOverWrite.Enabled = !IsOverwrite;
            _btnChangeMaterialMode.Enabled = !IsOverwrite;

            if (!_loadedCombinerFilePath.Equals(CombinerFile))
            {
                _loadedCombinerFilePath = CombinerFile;
                _combinerFileBlockElements = LoadCombinerFile();
            }
            _strFileName.Text = Path.GetFileName(CombinerFile);

            // TEV カラー
            {
                _ctlBlackColor.Color = this.ActiveTarget.IMaterial.BlackColor;
                _ctlWhiteColor.Color = this.ActiveTarget.IMaterial.WhiteColor;
                _ctlConstColor0.Color = this.ActiveTarget.TevData.GetColor(0);
                _ctlConstColor1.Color = this.ActiveTarget.TevData.GetColor(1);
                _ctlConstColor2.Color = this.ActiveTarget.TevData.GetColor(2);
                _ctlConstColor3.Color = this.ActiveTarget.TevData.GetColor(3);
                _ctlConstColor4.Color = this.ActiveTarget.TevData.GetColor(4);
            }

            UpdateCombinerFileInfo();

            // 「ファイルを指定してコンバイナエディタを開く」ボタン
            _btnOpenCombinerEditor.Enabled = !string.IsNullOrEmpty(CombinerFile);

            // 必須項目が足りているか調べ、ユーザーに通知を行う。
            CheckErrorForCombinerFile();
        }

        #region イベントハンドラ
        private void _btnChangeMaterialMode_Click(object sender, EventArgs e)
        {
            this.ActiveTarget.LowLevelCombinerSettingsEnabled = false;
            this.ActiveTarget.CombinerUserShaderSettingsEnabled = false;
            NotifyPropertyChanged();
        }

        private void _btnOpen_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFileDlg = new OpenFileDialog();
            if (!string.IsNullOrEmpty(CombinerFile))
            {
                openFileDlg.FileName = Path.GetFileName(this.ActiveTarget.CombinerUserShader.FileName);
                openFileDlg.InitialDirectory = SaveLoadUIHelper.GetAbsolutePathForExistDirectory(Path.GetDirectoryName(this.ActiveTarget.CombinerUserShader.FileName));
            }
            else
            {
                if (AppSetting != null && !string.IsNullOrEmpty(AppSetting.LastLoadedCombinerFilePath))
                {
                    openFileDlg.InitialDirectory = Path.GetDirectoryName(AppSetting.LastLoadedCombinerFilePath);
                }
            }
            openFileDlg.Filter = string.Format(
                "{0} ({1};{2})|{1};{2};| all files (*.*)|*.*",
                StringResMgr.Get("COMBINERFILE"),LECore.AppConstants.FCMBFileExtPattern, LECore.AppConstants.ECMBFileExtPattern);

            //ダイアログを表示する
            if (openFileDlg.ShowDialog() == DialogResult.OK)
            {
                int fileNameLength = Path.GetFileNameWithoutExtension(openFileDlg.FileName).Length;
                if (fileNameLength > LECore.AppConstants.CMBFileNameLength)
                {
                    MessageBox.Show(this, string.Format(StringResMgr.Get("ERROR_COMBINERFILE_LENGTH"), LECore.AppConstants.CMBFileNameLength), LECore.AppConstants.AppName, MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                if (AppSetting != null)
                {
                    AppSetting.LastLoadedCombinerFilePath = openFileDlg.FileName;
                }
                CombinerFile = openFileDlg.FileName;
                _strFileName.Text = Path.GetFileName(CombinerFile);

                NotifyPropertyChanged();
            }
        }

        private void _btnRemove_Click(object sender, EventArgs e)
        {
            this.ActiveTarget.CombinerUserShader.FileName = string.Empty;
            _combinerFileBlockElements = null;
            _loadedCombinerFilePath = string.Empty;
            _strFileName.Text = string.Empty;
            _txtboxCombinerFileInfo.Text = string.Empty;
            NotifyPropertyChanged();
        }

        private void _btnReload_Click(object sender, EventArgs e)
        {
            _loadedCombinerFilePath = string.Empty;
            UpdateProperty();
        }

        private void _btnOpenCombinerEditor_Click(Object sender, EventArgs e)
        {
            string launchOptions = string.Format("\"{0}\"", CombinerFile);
            CombinerEditorHelper.LaunchCombinerEditor(AppSetting.ProjectSettings.UserCombinerShaderSettings.CombinerEditorPath, true, launchOptions);
        }

        private void Event_ctlBlackColor_ColorEdit(Object sender, Controls.ColorEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (!e.EditFixed)
            {
                return;
            }

            EditColor(Target, e,
                (adapter) => adapter.BlackColor,
                (adapter, color) => adapter.BlackColor = color);

            NotifyPropertyChanged();
        }

        private void Event_ctlWhiteColor_ColorEdit(Object sender, Controls.ColorEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (!e.EditFixed)
            {
                return;
            }

            EditColor(Target, e,
                (adapter) => adapter.WhiteColor,
                (adapter, color) => adapter.WhiteColor = color);

            NotifyPropertyChanged();
        }

        private void Event_ctlConstColor0_ColorEdit(Object sender, Controls.ColorEditEventArgs e)
        {
            if (IsGUIUpdating) { return; }

            if (!e.EditFixed)
            {
                return;
            }

            ColorEditPanel control = (ColorEditPanel)sender;
            int index = (int)control.Tag;

            EditColor(Target, e,
                (adapter) => adapter.TevData.GetColor(index),
                (adapter, color) => adapter.TevData.SetColor(color, index));

            NotifyPropertyChanged();
        }

        private void _btnOverWriteTextures_Click(Object sender, EventArgs e)
        {
            if (_combinerFileBlockElements == null || _combinerFileBlockElements.Count() == 0)
            {
                return;
            }

            var textureBlockPropertyInfos = _combinerFileBlockElements
                                    .SelectMany(s => s.BlockProperties)
                                    .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                && baseParameterNameTable.ContainsKey(t.PropertyName))
                                    .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                    .OrderBy(t => GetTexturePropertyToTexMapNo(t.PropertyName));     // _a0~_a2 に並べ直す

            // ペインにテクスチャーが設定されているか調べる
            MaterialGUIAdapter target = this.ActiveTarget;
            TexMapList texMapList = target.TexMapList;

            // 更新結果をユーザに通知します。
            // 以下のファイルが処理されました。
            StringBuilder sb = new StringBuilder(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_BELOW") + Environment.NewLine);
            sb.AppendLine("-------------------------------------------------");

            // テクスチャー検索
            {
                ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;

                subScene.BeginMassiveModify();

                foreach(var blockProperty in textureBlockPropertyInfos)
                {
                    if (String.IsNullOrEmpty(blockProperty.ValuesText))
                    {
                        continue;
                    }

                    var fileName = blockProperty.ValuesText;
                    var filePath = GetTextureFilePath(fileName);
                    int _texMapNo = GetTexturePropertyToTexMapNo(blockProperty.PropertyName);

                    IPane pane = this.TargetPaneForMaterial;
                    IRevHWMaterial revMtlSet = this.TargetRevHWMaterial;

                    if (!File.Exists(filePath))
                    {
                        sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_NOTFIND_TEXTURE_FOR_MANAGER",
                            fileName,
                            _texMapNo,
                            filePath) + Environment.NewLine);
                    }

                    if (revMtlSet.IMaterialTexMapSet.Count() <= _texMapNo)
                    {
                        // 未設定項目のテクスチャーを挟んで登録する事は出来ない。
                        if (revMtlSet.IMaterialTexMapSet.Count() != _texMapNo)
                        {
                            sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_INCORRECT_TEXMAP_FOR_PANE",
                                pane.PaneName,
                                _texMapNo,
                                fileName) + Environment.NewLine);
                            continue;
                        }

                        // ペインのテクスチャーマップを新規追加
                        Texture texture = TextureManager.Search(fileName);
                        if (texture == null)
                        {
                            // テクスチャーマネージャに存在しない。
                            RegisterTextureForTextureManager(subScene.ITextureMgr, filePath);
                            sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_ADD_TEXTURE_FOR_MANAGER",fileName)); // ※ペインへのテクスチャー登録が続くので間隔をさらにあけない
                        }
                        else
                        {
                            // テクスチャーマネージャに登録されているが、ファイルパスの違う
                            if (!texture.FilePath.Equals(filePath))
                            {
                                // パスも同じなので、更新処理は行わない。
                                sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_DIFFERENCE_FILEPATH_FOR_MANAGER",
                                    fileName,
                                    filePath,
                                    texture.FilePath));     // ※ペインへのテクスチャー登録が続くので間隔をさらにあけない
                            }
                        }
                    }
                    else
                    {
                        // ペインのテクスチャーマップを入れ替え

                        IMaterialTexMap materialTexMap = revMtlSet.IMaterialTexMapSet[_texMapNo];
                        if (materialTexMap.TexImgName.Equals(fileName))
                        {
                            Texture texture = TextureManager.Search(fileName);

                            // 既に同名のファイル名がペインに設定されている
                            if (texture.FilePath.Equals(filePath))
                            {
                                // パスも同じなので、更新処理は行わない。
                                continue;
                            }
                            else
                            {
                                // パスの違うテクスチャーが設定されている場合は、更新は行わず差異のみ通知
                                sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_DIFFERENCE_FILEPATH_FOR_PANE",
                                    pane.PaneName,
                                    _texMapNo,
                                    filePath,
                                    texture.FilePath) + Environment.NewLine);
                                continue;
                            }
                        }
                        else
                        {
                            // 目的のものではないファイル名がペインに設定されている

                            // テクスチャーマネージャに登録されているか
                            Texture texture = TextureManager.Search(fileName);
                            if (texture == null)
                            {
                                // テクスチャーマネージャに存在しない。
                                RegisterTextureForTextureManager(subScene.ITextureMgr, filePath);
                                sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_ADD_TEXTURE_FOR_MANAGER",fileName)); // ※ペインへのテクスチャー登録が続くので間隔をさらにあけない
                            }
                            else
                            {
                                // テクスチャーマネージャに存在する

                                string oldFileName = materialTexMap.TexImgName;

                                // 入れ替え
                                RegisterReplaceTexMapForPane(revMtlSet, _texMapNo, fileName);

                                if (!texture.FilePath.Equals(filePath))
                                {
                                    // 入れ替えと、ファイルパスが違う事を通知
                                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_CHANGE_TEXMAP_AND_DIFFERENCE_FOR_PANE",
                                        pane.PaneName,
                                        _texMapNo,
                                        oldFileName,
                                        fileName,
                                        filePath,
                                        texture.FilePath) + Environment.NewLine);
                                    continue;
                                }
                                else
                                {
                                    // 入れ替えを行った事を通知
                                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_CHANGE_TEXMAP_FOR_PANE",
                                        pane.PaneName,
                                        _texMapNo,
                                        oldFileName,
                                        fileName) + Environment.NewLine);
                                    continue;
                                }
                            }
                        }
                    }

                    // 挿入
                    {
                        RegisterInsertTexMapForPane(revMtlSet, _texMapNo, fileName);
                        sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_TEXTURE_MSG_ADD_TEXMAP_FOR_PANE",
                            pane.PaneName,
                            _texMapNo,
                            fileName) + Environment.NewLine);
                    }
                }

                subScene.EndMassiveModify();
            }

            // 結果の表示
            // 結果をユーザに報告します。
            {
                string reportMsg = sb.ToString();
                MessageReportDlg messageReportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                messageReportDlg.Title = this.Text;
                messageReportDlg.Message = reportMsg;
                messageReportDlg.ShowDialog();
            }
        }

        private object GetObjectFromValueText(string valueTypeName, string valueText)
        {
            switch(valueTypeName)
            {
            case "float":
            {
                float[] value = valueText.Split(';').Select(t => float.Parse(t)).ToArray();
                Debug.Assert(value.Length == 1);
                return value[0];
            }
            case "vec2":
            {
                float[] value = valueText.Split(';').Select(t => float.Parse(t)).ToArray();
                Debug.Assert(value.Length == 2);
                return new FVec2(value[0], value[1]);
            }
            case "vec3":
            {
                float[] value = valueText.Split(';').Select(t => float.Parse(t)).ToArray();
                Debug.Assert(value.Length == 3);
                return new FVec3(value[0], value[1], value[2]);
            }
            case "vec4":
            {
                float[] value = valueText.Split(';').Select(t => float.Parse(t)).ToArray();
                Debug.Assert(value.Length == 4);
                return new FVec4(value[0], value[1], value[2], value[3]);
            }
            case "int":
            {
                int[] value = valueText.Split(';').Select(t => int.Parse(t)).ToArray();
                Debug.Assert(value.Length == 1);
                return value[0];
            }
            case "ivec2":
            {
                int[] value = valueText.Split(';').Select(t => int.Parse(t)).ToArray();
                Debug.Assert(value.Length == 2);
                return new IVec2(value[0], value[1]);
            }
            case "ivec3":
            {
                int[] value = valueText.Split(';').Select(t => int.Parse(t)).ToArray();
                Debug.Assert(value.Length == 3);
                return new IVec3(value[0], value[1], value[2]);
            }
            // TODO: RGBA 型は現状ではコンバイナファイルから取得出来ない為、必要時に実装を行います。

            default: throw new InvalidOperationException();
            }
        }

        private void _btnOverWriteExtUserData_Click(Object sender, EventArgs e)
        {
            if (_combinerFileBlockElements == null || _combinerFileBlockElements.Count() == 0)
            {
                return;
            }

            var extUserDataBlockPropertyInfos = _combinerFileBlockElements
                                    .SelectMany(s => s.BlockProperties)
                                    .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                        && extUserDataParameterTable.Any(p => p.propertyName == t.PropertyName))
                                    .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                    .Select((info, pri) => new { pri = extUserDataParameterTable.First(k => k.propertyName == info.PropertyName).pri, info }) // ソートを行う為、配列の順を加える
                                    .OrderBy(z => z.pri);

            // 更新結果をユーザに通知します。
            // 以下のファイルが処理されました。
            StringBuilder sb = new StringBuilder(StringResMgr.Get("COMBINERFILE_OVERWRITE_EXUSERDATA_BELOW") + Environment.NewLine);
            sb.AppendLine("-------------------------------------------------");

            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            subScene.BeginMassiveModify();
            {
                IPane pane = this.TargetPaneForMaterial;
                foreach(var blockProperty in extUserDataBlockPropertyInfos)
                {
                    ExUserDataInfo parameter = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));
                    object value = GetObjectFromValueText(blockProperty.info.ValueTypeName, blockProperty.info.ValuesText);
                    var exUserDataInfo = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));
                    var dataElement = UserDataElementHelper.FindByName(exUserDataInfo.name, pane.IUserDataHolder.UserDataElementSet);
                    if (dataElement == null)
                    {
                        RegisterExUserData(pane, exUserDataInfo.kind, exUserDataInfo.name, value);

                        sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_EXTUSERDATA_MSG_ADD_VALUE",
                            exUserDataInfo.text,
                            exUserDataInfo.name));
                    }
                    else
                    {
                        // 同じ場合は何もしない
                        if (!dataElement.Value.Equals(value))
                        {
                            ChangeExUserData(pane, exUserDataInfo.kind, exUserDataInfo.name, value);

                            sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_EXTUSERDATA_MSG_CHANGE_VALUE",
                                exUserDataInfo.text,
                                exUserDataInfo.name,
                                blockProperty.info.ValuesText));
                        }
                    }
                }

                // 拡張ユーザー情報に登録するものがあれば、ペイン基本の「拡張ユーザー情報のアニメーションを有効にする」も有効にします。
                if (extUserDataBlockPropertyInfos.Any() && !pane.ExtUserDataAnimEnabled)
                {
                    RegisterExtUserDataAnimEnabled(pane, true);
                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_OVERWRITE_EXTUSERDATA_MSG_ADD_ANIMENABLED"));

                }
            }
            subScene.EndMassiveModify();

            // 結果の表示
            // 結果をユーザに報告します。
            {
                string reportMsg = sb.ToString();
                MessageReportDlg messageReportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                messageReportDlg.Title = this.Text;
                messageReportDlg.Message = reportMsg;
                messageReportDlg.ShowDialog();
            }
        }

        private void _btnOverWriteConstantColors_Click(Object sender, EventArgs e)
        {
            if (_combinerFileBlockElements == null || _combinerFileBlockElements.Count() == 0)
            {
                return;
            }

            var constantcolorBlockPropertyInfos = _combinerFileBlockElements
                                    .SelectMany(s => s.BlockProperties)
                                    .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                && constantColorParameterNameTable.ContainsKey(t.PropertyName))
                                    .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                    .OrderBy(t => GetConstantColorPropertyToIndex(t.PropertyName));

            // 更新結果をユーザに通知します。
            // 以下のファイルが処理されました。
            StringBuilder sb = new StringBuilder(StringResMgr.Get("COMBINERFILE_OVERWRITE_CONSTANTCOLOR_BELOW") + Environment.NewLine);
            sb.AppendLine("-------------------------------------------------");

            ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            subScene.BeginMassiveModify();
            {
                foreach(var blockProperty in constantcolorBlockPropertyInfos)
                {
                    IRevHWMaterial revMtlSet = this.TargetRevHWMaterial;
                    ConstantColorIndex index = GetConstantColorPropertyToIndex(blockProperty.PropertyName);
                    object value =GetObjectFromValueText(blockProperty.ValueTypeName, blockProperty.ValuesText);
                    Debug.Assert(value is FVec4);
                    FVec4 fvec = (FVec4)value;
                    FloatColor newColor = new FloatColor(fvec.X, fvec.Y, fvec.Z, fvec.W);

                    RGBAColor color = GetRgbaForConstantColorIndex(revMtlSet, index);
                    if (color == newColor.ToRGBAColor())
                    {
                        continue;
                    }

                    RegisterChangeConstantColorForMaterial(revMtlSet, newColor, index);

                    sb.AppendLine(string.Format("{0} = {1}", constantColorParameterNameTable[blockProperty.PropertyName], blockProperty.ValuesText) + Environment.NewLine);
                }
            }
            subScene.EndMassiveModify();

            // 結果の表示
            // 結果をユーザに報告します。
            {
                string reportMsg = sb.ToString();
                MessageReportDlg messageReportDlg = new MessageReportDlg(MessageReportDlg.ButtonTypes.Ok, false);
                messageReportDlg.Title = this.Text;
                messageReportDlg.Message = reportMsg;
                messageReportDlg.ShowDialog();
            }
        }

        #endregion

        private int GetTexturePropertyToTexMapNo(string propertyName)
        {
            int _texMapNo = -1;
            if (propertyName.Equals("_a0")) _texMapNo = 0;
            else if (propertyName.Equals("_a1")) _texMapNo = 1;
            else if (propertyName.Equals("_a2")) _texMapNo = 2;
            else throw new InvalidOperationException();
            return _texMapNo;
        }

        private string GetTextureFilePath(string fileName)
        {
            // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=83955519
            // 4.参照元ファイルが含まれるフォルダ内の textures フォルダ（参照先がテクスチャーの場合のみ）
            // が利用されている為、コンバイナファイル内にはファイル名のみが保存されます。
            // 但し、4. のフォルダ以外からファイルの設定がされた場合はファイル名のみが入り、textures フォルダにテクスチャーが
            // 存在しないケースが出来ます。

            string fulPath = Path.Combine(Path.GetDirectoryName(CombinerFile), "..\\Textures", fileName) + ".tga";
            return Path.GetFullPath(fulPath); // パスの正規化を行う。
        }

        /// <summary>
        /// テクスチャーの差異を検出
        /// </summary>
        /// <returns></returns>
        private bool DetectDifferenceForTexture(IEnumerable<BlockPropertyInfo> textureBlockPropertyInfos)
        {
            foreach(var blockProperty in textureBlockPropertyInfos)
            {
                if (String.IsNullOrEmpty(blockProperty.ValuesText))
                {
                    continue;
                }

                int _texMapNo = GetTexturePropertyToTexMapNo(blockProperty.PropertyName);
                if (this.ActiveTarget.TexMapList.TexMaps.Count <= _texMapNo)
                {
                    return true;
                }

                string fileName = blockProperty.ValuesText;
                string filePath = GetTextureFilePath(fileName);

                // コンバイナエディタは一つ上の固定テクスチャーフォルダにファイルが存在する事として、テクスチャーファイル名を設定しますが、
                // そこ以外から設定されたテクスチャーはファイル名のみ設定される事がある為、テクスチャーファイルが必ずあるとは限りません。
                if (!File.Exists(filePath))
                {
                    return true;
                }

                var texMap = this.ActiveTarget.TexMapList.TexMaps[_texMapNo] as TexMap;
                if (!(texMap.TexImage.Texture.Name.Equals(fileName) ||        // 登録されているテクスチャーの名前が同じ
                    texMap.TexImage.Texture.FilePath.Equals(filePath)))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// マテリアルのコンスタントカラー値をインデックスで取得する。
        /// </summary>
        /// <param name="revMtlSet"></param>
        /// <param name="index"></param>
        /// <returns></returns>
        private RGBAColor GetRgbaForConstantColorIndex(IRevHWMaterial revMtlSet, ConstantColorIndex index)
        {
            RGBAColor color;
            if (index == ConstantColorIndex.BlackColor) // 黒カラー
            {
                color = revMtlSet.IMaterial.BlackColor.ToRGBAColor();
            }
            else if (index == ConstantColorIndex.WhiteColor) // 白カラー
            {
                color = revMtlSet.IMaterial.WhiteColor.ToRGBAColor();
            }
            else
            {
                color = revMtlSet.ITevData.GetTevColor((int)index - (int)ConstantColorIndex.C0).ToRGBAColor();
            }
            return color;
        }

        private bool CheckDifferenceForConstantColorIndex(IEnumerable<BlockPropertyInfo>constantColorBlockPropertyInfos, ConstantColorIndex index)
        {
            IRevHWMaterial revMtlSet = this.TargetRevHWMaterial;
            var blockProperty = constantColorBlockPropertyInfos.FirstOrDefault(s => GetConstantColorPropertyToIndex(s.PropertyName) == index);
            if (blockProperty == null)
            {
                return false;
            }

            object value = GetObjectFromValueText(blockProperty.ValueTypeName, blockProperty.ValuesText);
            Debug.Assert(value is FVec4);
            FVec4 fvec = (FVec4)value;
            RGBAColor color = GetRgbaForConstantColorIndex(revMtlSet, index);
            RGBAColor newColor = new FloatColor(fvec.X, fvec.Y, fvec.Z, fvec.W).ToRGBAColor();
            if (color != newColor)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 拡張ユーザー情報の差異を検出
        /// </summary>
        /// <param name="constantColorBlockPropertyInfos"></param>
        /// <returns></returns>
        private bool DetectDifferenceForExtUserData(IEnumerable<BlockPropertyInfo>constantColorBlockPropertyInfos)
        {
            if (!constantColorBlockPropertyInfos.Any())
            {
                return false;
            }

            IPane pane = this.TargetPaneForMaterial;
            foreach(var blockProperty in constantColorBlockPropertyInfos)
            {
                object value = GetObjectFromValueText(blockProperty.ValueTypeName, blockProperty.ValuesText);
                var exUserDataInfo = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.PropertyName));
                var dataElement = UserDataElementHelper.FindByName(exUserDataInfo.name, pane.IUserDataHolder.UserDataElementSet);
                if (dataElement == null || !dataElement.Value.Equals(value))
                {
                    return true;
                }
            }

            // 拡張ユーザー情報が存在する場合は、「拡張ユーザー情報のアニメーションを有効にする」も有効にならなければならない。
            if (!pane.ExtUserDataAnimEnabled)
            {
                return true;
            }
            return false;
        }

        private void SetConstantColorLabel(System.Windows.Forms.Label label, bool enabled)
        {
            // Enable では色の違いがあまり分からないのでカラー値を変更
            if (enabled)
            {
                label.ForeColor = SystemColors.ControlText;
            }
            else
            {
                label.ForeColor = SystemColors.ControlDark;
            }
        }

        private void UpdateCombinerFileInfo()
        {
            bool overWriteTextureEnable = false;
            bool overWriteExtUserDataEnable = false;

            bool useWriteConstantColorBlack = false;
            bool useWriteConstantColorWhite = false;
            bool useWriteConstantColor0 = false;
            bool useWriteConstantColor1 = false;
            bool useWriteConstantColor2 = false;
            bool useWriteConstantColor3 = false;
            bool useWriteConstantColor4 = false;


            bool overWriteConstBlack = false;
            bool overWriteConstWhite = false;
            bool overWriteC0 = false;
            bool overWriteC1 = false;
            bool overWriteC2 = false;
            bool overWriteC3 = false;
            bool overWriteC4 = false;

            // 追加順を維持するディクショナリを利用。
            OrderedDictionary shaderParameters = new OrderedDictionary();

            StringBuilder fileInfo = new StringBuilder();

            try
            {
                if (_combinerFileBlockElements == null || _combinerFileBlockElements.Count() == 0)
                {
                    return;
                }

                // コメントをファイル情報に追加
                foreach (BlockElementInfo blockElement in _combinerFileBlockElements)
                {
                    if (blockElement.DefinitionGuid == CombinerEditorService.CommentBlockDefinitionGuid)
                    {
                        BlockPropertyInfo commentProperty = blockElement.BlockProperties.First();
                        fileInfo.AppendLine(commentProperty.ValuesText);
                    }
                }

                // テクスチャー
                var textureBlockPropertyInfos = _combinerFileBlockElements
                                        .SelectMany(s => s.BlockProperties)
                                        .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                && baseParameterNameTable.ContainsKey(t.PropertyName));

                // テクスチャー差異を検出
                overWriteTextureEnable = DetectDifferenceForTexture(textureBlockPropertyInfos);

                foreach(var blockProperty in textureBlockPropertyInfos)
                {
                    if (!shaderParameters.Contains(blockProperty.PropertyName))
                    {
                        string propertyName = baseParameterNameTable[blockProperty.PropertyName];
                        string propertyValueText = blockProperty.ValuesText;

                        // 視認性をよくするため、パラメータの区切りを ", " に置換
                        if (blockProperty.ValueTypeName != "file")
                        {
                            propertyValueText = propertyValueText.Replace(";", ", ");
                        }

                        string propertyInfo = $"{propertyName} = {propertyValueText}";

                        shaderParameters.Add(blockProperty.PropertyName, propertyInfo);
                    }
                }

                // コンスタントカラー
                var constantColorBlockPropertyInfos = _combinerFileBlockElements
                                        .SelectMany(s => s.BlockProperties)
                                        .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                && constantColorParameterNameTable.ContainsKey(t.PropertyName))
                                        .OrderBy(q => GetConstantColorPropertyToIndex(q.PropertyName));

                foreach(var blockProperty in constantColorBlockPropertyInfos)
                {
                    if (!shaderParameters.Contains(blockProperty.PropertyName))
                    {
                        string propertyName = constantColorParameterNameTable[blockProperty.PropertyName];
                        string propertyValueText = blockProperty.ValuesText;
                        string propertyInfo = $"{propertyName} = {propertyValueText}";

                        shaderParameters.Add(blockProperty.PropertyName, propertyInfo);

                        var index = GetConstantColorPropertyToIndex(blockProperty.PropertyName);
                        switch(index)
                        {
                        case ConstantColorIndex.BlackColor:
                            useWriteConstantColorBlack = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteConstBlack = true;
                            }
                            break;
                        case ConstantColorIndex.WhiteColor:
                            useWriteConstantColorWhite = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteConstWhite = true;
                            }
                            break;
                        case ConstantColorIndex.C0:
                            useWriteConstantColor0 = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteC0 = true;
                            }
                            break;
                        case ConstantColorIndex.C1:
                            useWriteConstantColor1 = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteC1 = true;
                            }
                            break;
                        case ConstantColorIndex.C2:
                            useWriteConstantColor2 = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteC2 = true;
                            }
                            break;
                        case ConstantColorIndex.C3:
                            useWriteConstantColor3 = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteC3 = true;
                            }
                            break;
                        case ConstantColorIndex.C4:
                            useWriteConstantColor4 = true;
                            if (CheckDifferenceForConstantColorIndex(constantColorBlockPropertyInfos, index))
                            {
                                overWriteC4 = true;
                            }
                            break;
                        default: throw new InvalidOperationException();
                        }
                    }
                }

                // 拡張ユーザー情報(priority 収集用に匿名型の利用)
                var extUserDataBlockPropertyInfos = _combinerFileBlockElements
                                        .SelectMany(s => s.BlockProperties)
                                        .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                            && extUserDataParameterTable.Any(p => p.propertyName == t.PropertyName))
                                        .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                        .Select((info, pri) => new { pri = extUserDataParameterTable.First(k => k.propertyName == info.PropertyName).pri, info }) // ソートを行う為、配列の順を加える
                                        .OrderBy(z => z.pri);

                overWriteExtUserDataEnable = DetectDifferenceForExtUserData(extUserDataBlockPropertyInfos.Select(s => s.info));

                foreach(var blockProperty in extUserDataBlockPropertyInfos)
                {
                    if (!shaderParameters.Contains(blockProperty.info.PropertyName))
                    {
                        ExUserDataInfo parameter = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));
                        string propertyName = parameter.text;
                        string propertyValueText = blockProperty.info.ValuesText;
                        string propertyInfo = $"{propertyName} = {propertyValueText}";

                        shaderParameters.Add(blockProperty.info.PropertyName, propertyInfo);
                    }
                }
            }
            finally
            {
                SetConstantColorLabel(_lblConstBlack, useWriteConstantColorBlack);
                SetConstantColorLabel(_lblConstWhite, useWriteConstantColorWhite);
                SetConstantColorLabel(_lblConstC0, useWriteConstantColor0);
                SetConstantColorLabel(_lblConstC1, useWriteConstantColor1);
                SetConstantColorLabel(_lblConstC2, useWriteConstantColor2);
                SetConstantColorLabel(_lblConstC3, useWriteConstantColor3);
                SetConstantColorLabel(_lblConstC4, useWriteConstantColor4);

                _toolTip.SetToolTip(_lblWarningConstBlack, overWriteConstBlack ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);
                _toolTip.SetToolTip(_lblWarningConstWhite, overWriteConstWhite ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);
                _toolTip.SetToolTip(_lblWarningC0, overWriteC0 ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);
                _toolTip.SetToolTip(_lblWarningC1, overWriteC1 ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);
                _toolTip.SetToolTip(_lblWarningC2, overWriteC2 ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);
                _toolTip.SetToolTip(_lblWarningC3, overWriteC3 ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);
                _toolTip.SetToolTip(_lblWarningC4, overWriteC4 ? StringResMgr.Get("COMBINERFILE_WARNING_DIFF_COLOR"): string.Empty);

                _btnOverWriteTextures.Enabled = overWriteTextureEnable;
                _btnOverWriteExtUserData.Enabled = overWriteExtUserDataEnable;
                _btnOverWriteConstantColors.Enabled = overWriteConstBlack || overWriteConstWhite || overWriteC0 || overWriteC1 || overWriteC2 || overWriteC3 || overWriteC4;

                _lblWarningConstBlack.Visible = overWriteConstBlack;
                _lblWarningConstWhite.Visible = overWriteConstWhite;
                _lblWarningC0.Visible = overWriteC0;
                _lblWarningC1.Visible = overWriteC1;
                _lblWarningC2.Visible = overWriteC2;
                _lblWarningC3.Visible = overWriteC3;
                _lblWarningC4.Visible = overWriteC4;

                // シェーダパラメータをファイル情報に追加
                foreach (var shaderParameter in shaderParameters.Values)
                {
                    fileInfo.AppendLine((string)shaderParameter);
                }

                // テキストボックスへ反映
                _txtboxCombinerFileInfo.Text = fileInfo.ToString();
            }
        }

        private void CheckErrorForCombinerFile()
        {
            StringBuilder sb = new StringBuilder();

            try
            {
                if (string.IsNullOrEmpty(_loadedCombinerFilePath))
                {
                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_ERROR_COMBINERFILE"));
                    return;
                }

                if (!File.Exists(_loadedCombinerFilePath))
                {
                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_ERROR_NOTEXIST_COMBINERFILE"));
                    return;
                }

                if (!AppSetting.ProjectSettings.IsCombinerUserShaderEnabled)
                {
                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_ERROR_DIFINITIONS"));
                    return;
                }

                if (_combinerFileBlockElements == null || _combinerFileBlockElements.Count() == 0)
                {
                    return;
                }

                IPane pane = this.TargetPaneForMaterial;
                IRevHWMaterial revMtlSet = this.TargetRevHWMaterial;

                // テクスチャーの設定確認
                var textureBlockPropertyInfos = _combinerFileBlockElements
                                        .SelectMany(s => s.BlockProperties)
                                        .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                                    && baseParameterNameTable.ContainsKey(t.PropertyName))
                                        .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                        .OrderBy(t => GetTexturePropertyToTexMapNo(t.PropertyName));     // _a0~_a2 に並べ直す

                foreach(var blockProperty in textureBlockPropertyInfos)
                {
                    string propertyName = baseParameterNameTable[blockProperty.PropertyName];
                    string propertyValueText = blockProperty.ValuesText;

                    int texMapNo = GetTexturePropertyToTexMapNo(blockProperty.PropertyName);

                    if (texMapNo >= revMtlSet.IMaterialTexMapSet.Count())
                    {
                        sb.AppendLine(StringResMgr.Get("COMBINERFILE_ERROR_TEXTURE"));
                        break;
                    }
                }

                // 拡張ユーザー情報の設定確認
                var extUserDataBlockPropertyInfos = _combinerFileBlockElements
                                        .SelectMany(s => s.BlockProperties)
                                        .Where(t => t.PropertyType == BlockPropertyType.ShaderParameter
                                            && extUserDataParameterTable.Any(p => p.propertyName == t.PropertyName))
                                        .GroupBy(i => i.PropertyName).Select(t => t.First())            // 同一のプロパティ名を除外 ※Distinct() 相当。 override して GetHashCode, Equals 用意しない為。
                                        .Select((info, pri) => new { pri = extUserDataParameterTable.First(k => k.propertyName == info.PropertyName).pri, info }) // ソートを行う為、配列の順を加える
                                        .OrderBy(z => z.pri);

                foreach(var blockProperty in extUserDataBlockPropertyInfos)
                {
                    ExUserDataInfo parameter = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));
                    var exUserDataInfo = extUserDataParameterTable.First(s => s.propertyName.Equals(blockProperty.info.PropertyName));
                    var dataElement = UserDataElementHelper.FindByName(exUserDataInfo.name, pane.IUserDataHolder.UserDataElementSet);
                    if (dataElement == null)
                    {
                        sb.AppendLine(StringResMgr.Get("COMBINERFILE_ERROR_EXTUSERDATA"));
                        break;
                    }
                }

                // 拡張ユーザー情報に登録するものがあれば、ペイン基本の「拡張ユーザー情報のアニメーションを有効にする」が必要になる。
                if (extUserDataBlockPropertyInfos.Any() && !pane.ExtUserDataAnimEnabled)
                {
                    sb.AppendLine(StringResMgr.Get("COMBINERFILE_ERROR_ANIMENABLED"));
                }
            }
            finally
            {
                _toolTip.SetToolTip(_lblError, sb.ToString());
                _lblError.Visible = sb.Length > 0;
            }
        }

        /// <summary>
        /// テクスチャーマネージャに追加の登録を行う
        /// </summary>
        /// <param name="textureMgr"></param>
        /// <param name="filePath"></param>
        private void RegisterTextureForTextureManager(ITextureMgr textureMgr, string filePath)
        {
            Debug.Assert(textureMgr != null);
            Debug.Assert(!string.IsNullOrEmpty(filePath));

            LECore.Manipulator.TextureMgrManipulator textureMgrManipulator = new LECore.Manipulator.TextureMgrManipulator();
            textureMgrManipulator.BindTarget( textureMgr );
            var results = textureMgrManipulator.RegisterITextureImageSetFromFile( new string[] { filePath } );

            // 事前に同名のテクスチャーが存在していない事。
            Debug.Assert(results.Length == 1);
        }

        /// <summary>
        ///  ペインのマテリアルにテクスチャーに追加の登録を行う
        /// </summary>
        /// <param name="revMtlSet"></param>
        /// <param name="texMapNo"></param>
        /// <param name="fileName"></param>
        private void RegisterInsertTexMapForPane(IRevHWMaterial revMtlSet, int texMapNo, string fileName)
        {
            Debug.Assert(revMtlSet != null);
            Debug.Assert(texMapNo >= 0 && texMapNo < MaterialDefinition.MaxNumMatTexture);
            Debug.Assert(!string.IsNullOrEmpty(fileName));

            // 追加を行う場合は、間に未設定のテクスチャが存在しない事。
            Debug.Assert(revMtlSet.IMaterialTexMapSet.Count() >= texMapNo);

            RevMaterialManipulator materialMnp = new RevMaterialManipulator();
            materialMnp.BindTarget( revMtlSet );
            materialMnp.RenameMatReferencedTexture(
                fileName,
                texMapNo,
                AttrTextureResourceType.LocalFile);
        }

        /// <summary>
        /// ペインのマテリアルにテクスチャーに入れ替えの登録を行う
        /// </summary>
        /// <param name="revMtlSet"></param>
        /// <param name="texMapNo"></param>
        /// <param name="fileName"></param>
        private void RegisterReplaceTexMapForPane(IRevHWMaterial revMtlSet, int texMapNo, string fileName)
        {
            Debug.Assert(revMtlSet != null);
            Debug.Assert(!string.IsNullOrEmpty(fileName));
            Debug.Assert(texMapNo >= 0 && texMapNo < MaterialDefinition.MaxNumMatTexture);

            // 入れ替えを行うので、入れ替え基が存在していなければならない。
            Debug.Assert(revMtlSet.IMaterialTexMapSet.Count() > texMapNo);

            RevMaterialManipulator materialMnp = new RevMaterialManipulator();
            materialMnp.BindTarget( revMtlSet );
            materialMnp.RegisterMatTexture(
                fileName,
                texMapNo,
                AttrTextureResourceType.LocalFile);
        }

        /// <summary>
        /// コンスタントカラーのインデックス番号を取得する。
        /// </summary>
        /// <param name="propertyName"></param>
        /// <returns></returns>
        private ConstantColorIndex GetConstantColorPropertyToIndex(string propertyName)
        {
            switch(propertyName)
            {
            case "m_VectorParameter6": return ConstantColorIndex.BlackColor;
            case "m_VectorParameter0": return ConstantColorIndex.WhiteColor; // 白
            case "m_VectorParameter1": return ConstantColorIndex.C0;
            case "m_VectorParameter2": return ConstantColorIndex.C1;
            case "m_VectorParameter3": return ConstantColorIndex.C2;
            case "m_VectorParameter4": return ConstantColorIndex.C3;
            case "m_VectorParameter5": return ConstantColorIndex.C4;
            default: throw new InvalidOperationException();
            }
        }

        /// <summary>
        ///   ペインのマテリアルにテクスチャーに入れ替えの登録を行う
        /// </summary>
        /// <param name="revMtlSet"></param>
        /// <param name="color"></param>
        /// <param name="index"></param>
        private void RegisterChangeConstantColorForMaterial(IRevHWMaterial revMtlSet, FloatColor color, ConstantColorIndex index)
        {
            Debug.Assert(revMtlSet != null);

            RevMaterialManipulator materialMnp = new RevMaterialManipulator();
            materialMnp.BindTarget( revMtlSet );
            switch(index)
            {
            case ConstantColorIndex.BlackColor:
                color.ByteConvert = revMtlSet.IMaterial.BlackColor.ByteConvert;
                materialMnp.BlackColor = color;
                break;
            case ConstantColorIndex.WhiteColor:
                color.ByteConvert = revMtlSet.IMaterial.WhiteColor.ByteConvert;
                materialMnp.WhiteColor = color;
                break;
            case ConstantColorIndex.C0:
            case ConstantColorIndex.C1:
            case ConstantColorIndex.C2:
            case ConstantColorIndex.C3:
            case ConstantColorIndex.C4:
                color.ByteConvert = true;
                materialMnp.SetTevColors(color, (int)index - (int)ConstantColorIndex.C0);
                break;

            default: throw new InvalidOperationException();
            }
        }

        /// <summary>
        /// ペインの拡張ユーザー情報に追加を登録する。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="kind"></param>
        /// <param name="name"></param>
        /// <param name="val"></param>
        private void RegisterExUserData(IPane pane, UserDataKind kind, string name, object val)
        {
            Debug.Assert(pane != null);
            Debug.Assert(UserDataElementHelper.CheckParamatersValid(name, kind, val));

            if (UserDataElementHelper.FindByName(name, pane.IUserDataHolder.UserDataElementSet) != null)
            {
                return;
            }

            UserDataHolderManipulator userDataHolderMnp = new UserDataHolderManipulator();
            userDataHolderMnp.BindTarget(pane.IUserDataHolder);

            userDataHolderMnp.AddUserDataElement(name, kind, val);
        }

        /// <summary>
        /// ペインの拡張ユーザー情報のパラメータ変更を登録する。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="kind"></param>
        /// <param name="name"></param>
        /// <param name="val"></param>
        private void ChangeExUserData(IPane pane, UserDataKind kind, string name, object val)
        {
            Debug.Assert(pane != null);
            Debug.Assert(UserDataElementHelper.CheckParamatersValid(name, kind, val));
            Debug.Assert(UserDataElementHelper.FindByName(name, pane.IUserDataHolder.UserDataElementSet) != null);

            UserDataHolderManipulator userDataHolderMnp = new UserDataHolderManipulator();
            userDataHolderMnp.BindTarget(pane.IUserDataHolder);

            userDataHolderMnp.SetUserDataPamaters(name, name, kind, val);
        }

        /// <summary>
        /// ペインの「拡張ユーザー情報のアニメーションを有効にする」を登録する。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="extUserDataAnimEnabled"></param>
        private void RegisterExtUserDataAnimEnabled(IPane pane, bool extUserDataAnimEnabled)
        {
            Debug.Assert(pane != null);

            if (pane.ExtUserDataAnimEnabled == extUserDataAnimEnabled)
            {
                return;
            }

            PaneManipulator paneMnp = new PaneManipulator();
            paneMnp.BindTarget(pane);
            paneMnp.ExtUserDataAnimEnabled = extUserDataAnimEnabled;
        }
    }
}
