﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Linq;
using System.Collections.Generic;

namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters
{
    using LECore.Structures;
    using LECore.Manipulator;
    using LayoutEditor.Forms.ToolWindows.PropertyEditWindow;
    using LECore;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    //----------------------------------------------------------
    public interface IWindowPaneAdapterEntry
    {
        /// <summary>
        /// 操作対象のウインドウペインの WindowManipulator を得ます。
        /// </summary>
        WindowManipulator Manipulator { get; }

        /// <summary>
        /// 操作対象のウインドウペイン部位を取得、設定します。
        /// </summary>
        LEWindowPartsID CurrentTargetPartsID { get; set; }

        /// <summary>
        /// 操作対象のウインドウペイン部位のマテリアルに対応した MaterialGUIAdapter を得ます。
        /// </summary>
        MaterialGUIAdapter GetMaterialTargetAdaputer();

        /// <summary>
        /// 操作対象のウインドウペイン部位のマテリアルに対応した IRevHWMaterial を得ます。
        /// </summary>
        IRevHWMaterial GetTargetCoreMaterial();
    }

    /// <summary>
    /// ----------------------------------------------------------
    /// PaneAdapterEntry について
    /// ----------------------------------------------------------
    /// PaneGUIAdapter 内部の Manipulator 管理用クラス
    /// AdapterEntry は一つのペインに関連づく、複数の Manipulator を束ねて管理します。
    /// また、UI の操作対象についてのステート（例えば、ウインドウペインの この部分のマテリアル、、、という情報）を持ちます。
    /// MVVM アーキテクチャーにおける VM のような役割を持っています（View とのバインディングはしていませんが）。
    ///
    /// 現在は、WindowPane でのみ有効に活用されている状況で、WindowPaneAdapterEntry として付加情報込みで外部に公開されています。
    /// 今後、UI の状態を保持する必要が生じた場合には、AdapterEntry にメンバを追加し、IWindowPaneAdapterEntry のように、ビューに公開するようにしてください。
    /// </summary>
    public partial class PaneGUIAdapter
    {
        //----------------------------------------------------------

        private class PaneAdapterEntry<TPaneManipulator> where TPaneManipulator : BaseManipulator
        {
            public TPaneManipulator Manipulator { get; set; }
            public List<MaterialGUIAdapter> MaterialAdapter { get; set; }

            public PaneAdapterEntry(IPaneExParamater paneParamater)
            {
                // mat
                this.MaterialAdapter = new List<MaterialGUIAdapter>();
                if (paneParamater is IRevHWMaterialHolder)
                {
                    foreach (var material in (paneParamater as IRevHWMaterialHolder).IRevHWMaterial)
                    {
                        this.MaterialAdapter.Add(new MaterialGUIAdapter(material));
                    }
                }
            }
        }

        //----------------------------------------------------------

        private class PicturePaneAdapterEntry : PaneAdapterEntry<PictureManipulator>
        {
            public PicturePaneAdapterEntry(IPicture picture)
                : base(picture)
            {
                this.Manipulator = new PictureManipulator();
                this.Manipulator.BindTarget(picture);
            }
        }

        //----------------------------------------------------------

        private class TextBoxPaneAdapterEntry : PaneAdapterEntry<TextBoxManipulator>
        {
            public TextBoxPaneAdapterEntry(ITextBox textBox)
                : base(textBox)
            {
                this.Manipulator = new TextBoxManipulator();
                this.Manipulator.BindTarget(textBox);
            }
        }

        //----------------------------------------------------------

        private class PartsLayoutPaneAdapterEntry : PaneAdapterEntry<PartsLayoutManipulator>
        {
            public PartsLayoutPaneAdapterEntry(IPartsLayout partsLayout)
                : base(partsLayout)
            {
                this.Manipulator = new PartsLayoutManipulator();
                this.Manipulator.BindTarget(partsLayout);
            }
        }

        //----------------------------------------------------------

        private class CapturePaneAdapterEntry : PaneAdapterEntry<CaptureManipulator>
        {
            public CapturePaneAdapterEntry(ICapture capture)
                : base(capture)
            {
                this.Manipulator = new CaptureManipulator();
                this.Manipulator.BindTarget(capture);
            }
        }

        //----------------------------------------------------------

        private class AlignmentPaneAdapterEntry : PaneAdapterEntry<AlignmentManipulator>
        {
            public AlignmentPaneAdapterEntry(IAlignment alignment)
                : base(alignment)
            {
                this.Manipulator = new AlignmentManipulator();
                this.Manipulator.BindTarget(alignment);
            }
        }

        //----------------------------------------------------------

        private class ScissorPaneAdapterEntry : PaneAdapterEntry<ScissorManipulator>
        {
            public ScissorPaneAdapterEntry(IScissor alignment)
                : base(alignment)
            {
                this.Manipulator = new ScissorManipulator();
                this.Manipulator.BindTarget(alignment);
            }
        }

        //----------------------------------------------------------

        private class WindowPaneAdapterEntryImpl : PaneAdapterEntry<WindowManipulator>, IWindowPaneAdapterEntry
        {
            private LEWindowPartsID _currentTragetPartsID = LEWindowPartsID.Content;

            //----------------------------------------------------------

            public WindowPaneAdapterEntryImpl(ILEWindow window)
                : base(window)
            {
                this.Manipulator = new WindowManipulator();
                this.Manipulator.BindTarget(window);
            }

            //----------------------------------------------------------

            /// <summary>
            /// マテリアル数を取得します。
            /// </summary>
            int NumMaterials
            {
                get { return this.Manipulator.ILEWindow.NumTexture; }
            }

            /// <summary>
            /// 操作対象マテリアル種類を設定します。
            /// </summary>
            public LEWindowPartsID CurrentTargetPartsID
            {
                get
                {
                    if (this.Manipulator.ILEWindow.WindowKind == FileFmt.WindowKind.HorizontalNoContent)
                    {
                        if (_currentTragetPartsID != LEWindowPartsID.CornerRT)
                        {
                            return LEWindowPartsID.CornerLT;
                        }
                    }

                    return _currentTragetPartsID;
                }

                set
                {
                    if (_currentTragetPartsID != value)
                    {
                        if (NumMaterials == 2)
                        {
                            if (value > LEWindowPartsID.CornerLT)
                            {
                                value = LEWindowPartsID.CornerLT;
                            }
                        }
                        else if (NumMaterials == 5)
                        {
                            switch (value)
                            {
                                case LEWindowPartsID.FrameT: value = LEWindowPartsID.CornerLT; break;
                                case LEWindowPartsID.FrameR: value = LEWindowPartsID.CornerRT; break;
                                case LEWindowPartsID.FrameB: value = LEWindowPartsID.CornerRB; break;
                                case LEWindowPartsID.FrameL: value = LEWindowPartsID.CornerLB; break;
                            }
                        }

                        _currentTragetPartsID = value;
                    }
                }
            }

            //----------------------------------------------------------
            /// <summary>
            /// 編集対象のマテリアルを取得します。
            /// </summary>
            public IRevHWMaterial GetTargetCoreMaterial()
            {
                return this.MaterialAdapter[(int)_currentTragetPartsID].CoreMaterial;
            }

            //----------------------------------------------------------

            /// <summary>
            /// 操作対処のマテリアルの GUI アダプタを取得します。
            /// </summary>
            public MaterialGUIAdapter GetMaterialTargetAdaputer()
            {
                var materialAdapter = this.MaterialAdapter.ElementAtOrDefault((int)CurrentTargetPartsID);

                bool IsTextureMappingEditable;
                bool IsMaterialColorEditable;
                bool IsBlendEditable;
                bool IsTextureStageEditable;

                ILEWindowHelper.GetMaterialEditableCondition(
                    this.Manipulator.ILEWindow.WindowKind,
                    CurrentTargetPartsID,
                    this.Manipulator.ILEWindow.UseLTMaterial,
                    out IsTextureMappingEditable,
                    out IsMaterialColorEditable,
                    out IsBlendEditable,
                    out IsTextureStageEditable);

                materialAdapter.IsTextureMappingEditable = IsTextureMappingEditable;
                materialAdapter.IsMaterialColorEditable = IsMaterialColorEditable;
                materialAdapter.IsBlendEditable = IsBlendEditable;
                materialAdapter.IsTextureStageEditable = IsTextureStageEditable;

                return materialAdapter;
            }
        }
    }


    /// <summary>
    /// PaneGUIAdapter の概要の説明です。
    /// </summary>
    public partial class PaneGUIAdapter : IBaseGuiAdapter
    {
        readonly IPane _corePane;
        readonly PaneManipulator _corePaneMnpl = new PaneManipulator();
        readonly MaskManipulator _coreMaskMnpl = new MaskManipulator();
        readonly DropShadowManipulator _coreDropShadowMnpl = new DropShadowManipulator();
        ProceduralShapeManipulator _coreProceduralShapeMnpl = null;

        /// <summary>
        /// プロパティパネルが操作対象とするモノすべてに対する、Manipulator を管理します。
        /// 上書きを含む、部品ペインを編集している状態では、複数個、複数種類のペインが同時に編集対象になります。
        /// </summary>
        readonly List<PaneManipulator> _subPaneMnpls = new List<PaneManipulator>();
        readonly List<PicturePaneAdapterEntry> _corePictureMnpls = new List<PicturePaneAdapterEntry>();
        readonly List<TextBoxPaneAdapterEntry> _coreTextBoxMnpls = new List<TextBoxPaneAdapterEntry>();
        readonly List<PartsLayoutPaneAdapterEntry> _corePartsLayoutMnpls = new List<PartsLayoutPaneAdapterEntry>();
        readonly List<IWindowPaneAdapterEntry> _coreWindowMnpls = new List<IWindowPaneAdapterEntry>();
        readonly List<CapturePaneAdapterEntry> _coreCaptureMnpls = new List<CapturePaneAdapterEntry>();
        readonly List<AlignmentPaneAdapterEntry> _coreAlignmentMnpls = new List<AlignmentPaneAdapterEntry>();
        readonly List<ScissorPaneAdapterEntry> _coreScissorMnpls = new List<ScissorPaneAdapterEntry>();
        MaskGUIAdapter _coreMaskGUIAdapter = null;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneGUIAdapter( IPane corePane )
        {
            Debug.Assert( corePane != null );
            _corePane = corePane;
            Refresh_(_corePane);
        }

        /// <summary>
        /// 更新します。
        /// </summary>
        public void Refresh()
        {
            this.Refresh_(this._corePane);
        }

        /// <summary>
        /// 更新します。
        /// </summary>
        private void Refresh_(IPane corePane)
        {
            _subPaneMnpls.Clear();
            _corePictureMnpls.Clear();
            _coreTextBoxMnpls.Clear();
            _coreWindowMnpls.Clear();
            _coreCaptureMnpls.Clear();
            _coreAlignmentMnpls.Clear();
            _coreScissorMnpls.Clear();
            _corePartsLayoutMnpls.Clear();
            _coreMaskGUIAdapter = null;
            _coreProceduralShapeMnpl = null;

            // すべての種類のペインに共通して、
            // 基本的なパラメータ操作クラスを初期化します。
            _corePaneMnpl.BindTarget(corePane);
            _coreMaskMnpl.BindTarget(corePane.IMask);
            _coreMaskGUIAdapter = new MaskGUIAdapter(corePane.IMask);
            _coreDropShadowMnpl.BindTarget(corePane.IDropShadow);

            // ペインの種類に応じて、内部の操作クラスを初期化します。
            switch (corePane.PaneKind)
            {
                case PaneKind.Picture:
                    {
                        _corePictureMnpls.Add(new PicturePaneAdapterEntry(corePane.IPicture));
                        _coreProceduralShapeMnpl = new ProceduralShapeManipulator();
                        _coreProceduralShapeMnpl.BindTarget(corePane.IPicture.IProceduralShape);
                        break;
                    }
                case PaneKind.Textbox: _coreTextBoxMnpls.Add(new TextBoxPaneAdapterEntry(corePane.ITextBox)); break;
                case PaneKind.Null:
                case PaneKind.Bounding:
                    {
                        // DO  NOTIHNG
                        break;
                    }
                case PaneKind.Window: _coreWindowMnpls.Add(new WindowPaneAdapterEntryImpl(corePane.ILEWindow)); break;
                case PaneKind.Parts: _corePartsLayoutMnpls.Add(new PartsLayoutPaneAdapterEntry(corePane.IPartsLayout)); SetupPartsManipulator_(corePane); break;
                case PaneKind.Capture: _coreCaptureMnpls.Add(new CapturePaneAdapterEntry(corePane.ICapture)); break;
                case PaneKind.Alignment: _coreAlignmentMnpls.Add(new AlignmentPaneAdapterEntry(corePane.IAlignment)); break;
                case PaneKind.Scissor: _coreScissorMnpls.Add(new ScissorPaneAdapterEntry(corePane.IScissor)); break;

                default:
                    {
                        Debug.Assert(false);
                        break;
                    }
            }
        }

        /// <summary>
        /// 部品ペインの操作クラスを初期化します。
        /// </summary>
        /// <param name="corePane"></param>
        private void SetupPartsManipulator_(IPane corePane)
        {
            foreach (var prop in corePane.IPartsLayout.PartsPropaerties)
            {
                // まったく上書き指定が使われていない
                if (prop.Option == PartsPropertyUsageOptions.None)
                {
                    continue;
                }

                var paneMnp = new PaneManipulator();
                paneMnp.BindTarget(prop.Paramater.OwnerPane);
                _subPaneMnpls.Add(paneMnp);

                // ピクチャや、テキストなどの上書きがおこなれていない
                if (!PartsPropertyUsageOptionsHelper.IsPaneExDataOverwrote(prop.Option))
                {
                    continue;
                }

                if (prop.PaneKind == PaneKind.Picture)
                {
                    _corePictureMnpls.Add(new PicturePaneAdapterEntry(prop.Paramater as IPicture));
                }
                else if (prop.PaneKind == PaneKind.Textbox)
                {
                    _coreTextBoxMnpls.Add(new TextBoxPaneAdapterEntry(prop.Paramater as ITextBox));
                }
                else if (prop.PaneKind == PaneKind.Window)
                {
                    _coreWindowMnpls.Add(new WindowPaneAdapterEntryImpl(prop.Paramater as ILEWindow));
                }
                else if (prop.PaneKind == PaneKind.Capture)
                {
                    _coreCaptureMnpls.Add(new CapturePaneAdapterEntry(prop.Paramater as ICapture));
                }
                else if (prop.PaneKind == PaneKind.Alignment)
                {
                    _coreAlignmentMnpls.Add(new AlignmentPaneAdapterEntry(prop.Paramater as IAlignment));
                }
                else if (prop.PaneKind == PaneKind.Scissor)
                {
                    _coreScissorMnpls.Add(new ScissorPaneAdapterEntry(prop.Paramater as IScissor));
                }
                else if (prop.PaneKind == PaneKind.Parts)
                {
                    _corePartsLayoutMnpls.Add(new PartsLayoutPaneAdapterEntry(prop.Paramater as IPartsLayout));
                    SetupPartsManipulator_(prop.Paramater.OwnerPane);
                }
            }
        }

        #region プロパティ

        /// <summary>
        /// 対象ペインを取得します。
        /// </summary>
        public IPane Target
        {
            get { return _corePaneMnpl.IPane; }
        }

        /// <summary>
        /// ペイン操作クラスを取得します。
        /// </summary>
        public PaneManipulator PaneManipulator
        {
            get{ return _corePaneMnpl;}
        }

        /// <summary>
        /// ペイン操作クラスを取得します。
        /// </summary>
        public IEnumerable<PaneManipulator> PartsSubPaneManipulators
        {
            get { return _subPaneMnpls; }
        }

        public PictureManipulator FindTargetPictureManipulator(int targetIndex)
        {
            return _corePictureMnpls.ElementAtOrDefault(targetIndex)?.Manipulator;
        }

        public TextBoxManipulator FindTargetTextBoxManipulator(int targetIndex)
        {
            return _coreTextBoxMnpls.ElementAtOrDefault(targetIndex)?.Manipulator;
        }

        public PartsLayoutManipulator FindTargetPartsLayoutManipulator(IPartsLayout key)
        {
            return _corePartsLayoutMnpls.FirstOrDefault((mnp) => mnp?.Manipulator?.IPartsLayout == key)?.Manipulator;
        }

        public CaptureManipulator FindTargetCaptureManipulator(int targetIndex)
        {
            return _coreCaptureMnpls.ElementAtOrDefault(targetIndex)?.Manipulator;
        }

        public AlignmentManipulator FindTargetAlignmentManipulator(int targetIndex)
        {
            return _coreAlignmentMnpls.ElementAtOrDefault(targetIndex)?.Manipulator;
        }

        public ScissorManipulator FindTargetScissorManipulator(int targetIndex)
        {
            return _coreScissorMnpls.ElementAtOrDefault(targetIndex)?.Manipulator;
        }

        public IWindowPaneAdapterEntry FindTargetWindowPaneAdapterEntry(int targetIndex)
        {
            return _coreWindowMnpls.ElementAtOrDefault(targetIndex);
        }

        public void UpdateWindowPaneAdaptorCurrentTargetParts(LEWindowPartsID partsID)
        {
            foreach (var mnp in _coreWindowMnpls)
            {
                mnp.CurrentTargetPartsID = partsID;
            }
        }

        public MaskGUIAdapter MaskGUIAdapter
        {
            get
            {
                return _coreMaskGUIAdapter;
            }
        }

        /// <summary>
        /// 対象の持つマテリアル操作クラスを列挙します。
        /// TODO : これは廃止したい。GUI側の修正と、
        /// UseDetailedMaterial のPaneAdapterEntryへの引っ越しが必要。
        /// </summary>
        public IEnumerable<MaterialGUIAdapter> MaterialAdapters
        {
            get
            {
                foreach (var mnp in this._corePictureMnpls)
                {
                    foreach (var mat in mnp.MaterialAdapter)
                    {
                        yield return mat;
                    }
                }

                foreach (var mnp in this._coreTextBoxMnpls)
                {
                    foreach (var mat in mnp.MaterialAdapter)
                    {
                        yield return mat;
                    }
                }

                foreach (var mnp in this._coreWindowMnpls)
                {
                    yield return mnp.GetMaterialTargetAdaputer();
                }

                foreach (var mnp in _corePartsLayoutMnpls)
                {
                    foreach (var mat in mnp.MaterialAdapter)
                    {
                        yield return mat;
                    }
                }
            }
        }

        /// <summary>
        /// 詳細なマテリアル設定を行うか判定します。
        /// </summary>
        public virtual bool UseDetailedMaterial
        {
            get
            {
                return true; // TODO : ひとまず常に表示。
            }
        }

        // マスクを操作するクラス
        public MaskManipulator MaskManipulator
        {
            get
            {
                return _coreMaskMnpl;
            }
        }

        // ドロップシャドウ効果を操作するクラス
        public DropShadowManipulator DropShadowManipulator
        {
            get
            {
                return _coreDropShadowMnpl;
            }
        }

        // プロシージャル形状および効果を操作するクラス
        public ProceduralShapeManipulator ProceduralShapeManipulator
        {
            get
            {
                return _coreProceduralShapeMnpl;
            }
        }


        #endregion プロパティ

        #region IGuiAdapter メンバ

        // FIXME
        public string Name
        {
            get{return _corePaneMnpl.IPane.PaneName;}
        }

        /// <summary>
        /// 識別子を取得します。
        /// GUIの種類を決定するために使用されます。
        /// </summary>
        public PropertyPanelType ObjectID
        {
            get
            {
                switch( _corePaneMnpl.IPane.PaneKind )
                {
                    case PaneKind.Picture: return PropertyPanelType.PicturePane;
                    case PaneKind.Null: return PropertyPanelType.NullPane;
                    case PaneKind.Textbox: return PropertyPanelType.TextBoxPane;
                    case PaneKind.Bounding: return PropertyPanelType.RegionPane;
                    case PaneKind.Window: return PropertyPanelType.WindowPane;
                    case PaneKind.Parts: return PropertyPanelType.PartsPane;
                    case PaneKind.Capture: return PropertyPanelType.CapturePane;
                    case PaneKind.Alignment: return PropertyPanelType.Alignment;
                    case PaneKind.Scissor: return PropertyPanelType.Scissor;
                    // 意図しないケース
                    default: Debug.Assert( false );return PropertyPanelType.NullPane;
                }
            }
        }

        /// <summary>
        /// 指定インスタンスが、操作対象となっているか調査します。
        /// </summary>
        public bool HasSameTargets( object another )
        {
            PaneGUIAdapter targetPane = another as PaneGUIAdapter;
            return ( targetPane != null && targetPane.PaneManipulator.IPane == this.PaneManipulator.IPane);
        }
        #endregion IGuiAdapter メンバ
    }
}
