﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
// 以下のコードは3Dエディタからの移植コードです。
//
// LECoreモジュールの同内容のクラスと、GUIコードのインタフェース統合を図るクラス軍です。
// PropartyDlgの再描画タイミングで、LECoreデータとGUIデータの同期を取っています。
//
// データ操作時の具体的なメッセージの流れを以下に示します。
//
// ●ユーザによるGUI操作
// ↓
// ●GUIハンドラ内で、ドキュメント更新イベントの発生
// ↓
// ●ドキュメント更新イベントハンドラ内で、GUIデータ => LECoreデータ のデータコピー
// ↓
// ●LECoreシーン更新イベントの発生 => ProrpertyDlgの更新処理発生
// ↓
// ●ProrpertyDlgの更新ハンドラ内で、再描画直前に LECoreデータ => GUIデータ のデータコピー。
//
//
// オーナーモデル 、ドキュメント要素、MCSに関する記述は消去されています。
// ドキュメントクラスが保持していた情報で重要なものは、
// ドキュメントクラスの持ち主であるクラスのメンバ変数へと移動しました。

using System;
using System.Collections;
using System.Diagnostics;
using System.Text;


namespace LayoutEditor.Forms.ToolWindows.PropertyEditWindow.Adapters
{
    using LECore.Structures;
    using LECore.Structures.Nsrif.Elements;
    using LECore.Structures.Nsrif.Attributes;
    using LayoutEditor.Utility;
    using LECore.Manipulator;

    using SystemCoreAttribute = LECore.Structures.Nsrif.Attributes;
using System.Drawing;
    using LECore;
using System.Windows.Forms;

    #region マテリアル関連

    /// <summary>
    /// マテリアルクラス。
    /// </summary>
    /// <summary>
    /// マテリアルクラス。
    /// GUIとのインタフェース調整のための、アダプタクラス。
    ///
    /// 3D Editor では Material というクラスでした。
    /// 同名のクラスが多く紛らわしかったので、改名しました。
    /// </summary>
    public class MaterialGUIAdapter : IBaseGuiAdapter
    {
        // 操作対象(追加メンバ)
        IRevHWMaterial        _coreMaterial;

        bool     _compress_enable   = false;
        bool     _tev_share_enable  = false;

        // ポリゴンリスト
        private readonly ArrayList _polygons = new ArrayList();
        // テクスチャマップリスト
        private readonly TexMapList _texMapList;
        // ＴＥＶデータ
        private readonly TevData _tevData;
        // ＰＥデータ
        private readonly PEData _peData;
        // ユーザーデータ
        private readonly UserDataHolder _userData;

        // UV座標配列
        private readonly TexCoord4[] _texCoord4Set = new TexCoord4[ MaterialDefinition.MaxNumTexCoord ];

        private readonly CombinerUserShader _combinerUserShader;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MaterialGUIAdapter( IRevHWMaterial srcMaterial)
        {
            _coreMaterial    = srcMaterial;

            _texMapList = new TexMapList();
            _tevData = new TevData();
            _peData = new PEData(null);
            _userData = new UserDataHolder();
            _combinerUserShader = new CombinerUserShader();

            Refresh();
        }

        public IPane Target
        {
            get { return null; }
        }

        /// <summary>
        /// ポリゴンリスト。
        /// </summary>
        public ArrayList Polygons
        {
            get { return _polygons; }
        }

        #region アトリビュート
        /// <summary>
        /// 圧縮フラグ。
        /// </summary>
        public bool CompressEnable
        {
            get { return _compress_enable;  }
            set { _compress_enable = value; }
        }

        /// <summary>
        /// ＴＥＶ設定圧縮フラグ。
        /// </summary>
        public bool TevShareEnable
        {
            get { return _tev_share_enable;  }
            set { _tev_share_enable = value; }
        }

        /// <summary>
        /// テクスチャマップリスト。
        /// </summary>
        public TexMapList TexMapList
        {
            get { return _texMapList; }
        }

        /// <summary>
        /// ＴＥＶデータ。
        /// </summary>
        public TevData TevData
        {
            get { return _tevData; }
        }

        /// <summary>
        /// 白カラー
        /// </summary>
        public FloatColor WhiteColor
        {
            get;
            set;
        }

        /// <summary>
        /// 黒カラー
        /// </summary>
        public FloatColor BlackColor
        {
            get;
            set;
        }

        /// <summary>
        /// コンバイナユーザーシェーダー。
        /// </summary>
        public CombinerUserShader CombinerUserShader
        {
            get { return _combinerUserShader; }
        }

        public bool IsThresholdingAlphaInterpolationEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// 黒カラーアルファが有効か？
        /// </summary>
        public bool IsBlackColorAlphaEnabled
        {
            get;
            private set;
        }

        /// <summary>
        /// マテリアル詳細編集が有効かどうか
        /// </summary>
        public bool LowLevelCombinerSettingsEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// コンバイナユーザーシェーダー編集が有効かどうか
        /// </summary>
        public bool CombinerUserShaderSettingsEnabled
        {
            get;
            set;
        }

        /// <summary>
        /// ＰＥデータ。
        /// </summary>
        public PEData PEData
        {
            get { return _peData; }
        }

        /// <summary>
        /// ユーザーデータ。
        /// </summary>
        public UserDataHolder UserData
        {
            get { return _userData; }
        }

        /// <summary>
        /// 操作対象マテリアル
        /// </summary>
        public virtual IMaterial IMaterial
        {
            get { return _coreMaterial != null ? _coreMaterial.IMaterial : null; }
        }

        /// <summary>
        /// 操作対象マテリアル
        /// </summary>
        public int NumTevStages
        {
            get
            {
                if (!this.LowLevelCombinerSettingsEnabled)
                {
                    return Math.Max(0, this.IMaterial.IMaterialTexMapSet.Length - 1);
                }
                else
                {
                    return this.TevData.Stages.NumStages;
                }
            }
        }

        /// <summary>
        /// 操作対象マテリアル
        /// </summary>
        public IRevHWMaterial CoreMaterial
        {
            get{ return _coreMaterial;}
        }

        /// <summary>
        /// テクスチャ座標配列
        /// </summary>
        public TexCoord4[]  TexCoord4Set
        {
            get{ return _texCoord4Set;}
        }

        /// <summary>
        /// テクスチャマッピング設定が変更可能か？
        /// </summary>
        public bool IsTextureMappingEditable
        {
            get;
            set;
        }

        /// <summary>
        /// マテリアルカラー設定が変更可能か？
        /// </summary>
        public bool IsMaterialColorEditable
        {
            get;
            set;
        }

        /// <summary>
        /// ブレンド設定が変更可能か？
        /// </summary>
        public bool IsBlendEditable
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャステージ設定が変更可能か？
        /// </summary>
        public bool IsTextureStageEditable
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャだけ上書きをしているか
        /// </summary>
        public bool IsTextureOverwritingEnabled
        {
            get;
            set;
        }
        #endregion

        #region 頂点情報参照
        /// <summary>
        /// 保持している法線数。
        /// </summary>
        public int NumVtxNormals
        {
            get { return 0; }
        }

        /// <summary>
        /// 保持している頂点カラー数。
        /// (2Dではひとまず、1固定とします。)
        /// </summary>
        public int NumVtxColors
        {
            get { return 1; }
        }

        /// <summary>
        /// 保持している頂点テクスチャ座標数。
        /// (2Dではひとまず、使用テクスチャ枚数と同一とします。)
        ///
        /// NOTICE：ペインから、マテリアルに対して、
        /// テクスチャ座標の個数を設定するように変更になりました。
        /// </summary>
        public int NumVtxTexCoords
        {
            get
            {
                return _coreMaterial.UserMaxNumTexCoord;
            }
        }

        /// <summary>
        /// 法線を保持しているかどうか。
        /// </summary>
        public bool HasVtxNormals
        {
            get { return this.NumVtxNormals > 0; }
        }

        /// <summary>
        /// 頂点カラーを保持しているかどうか。
        /// </summary>
        public bool HasVtxColors
        {
            get { return this.NumVtxColors > 0; }
        }

        /// <summary>
        /// 頂点テクスチャ座標を保持しているかどうか。
        ///
        /// 2Dレイアウトではテクスチャ枚数によって、
        /// NumVtxTexCoordsが決定されます。
        /// したがって、HasVtxTexCoordsがテクスチャ枚数がゼロ枚の時に
        /// false を返すようになり、最初の1枚を登録する際に、
        /// 通常のテクスチャを正しく登録できない現象が発生しました
        /// (HasVtxTexCoordsが true を返す必要がある)。
        /// そこで対策として、常に true を返すように変更しました。
        /// </summary>
        public bool HasVtxTexCoords
        {
            //get { return this.NumVtxTexCoords > 0; }
            get { return true; }
        }
        #endregion

        #region ＵＩテキスト
        /// <summary>
        /// 圧縮フラグテキスト。
        /// </summary>
        public string CompressEnableText
        {
            get { return UIDataBinder.GetFlagTextOnOff(this.CompressEnable); }
        }

        /// <summary>
        /// ＴＥＶ設定圧縮フラグテキスト。
        /// </summary>
        public string TevShareEnableText
        {
            get { return UIDataBinder.GetFlagTextOnOff(this.TevShareEnable); }
        }

        /// <summary>
        /// ユーザーデータテキスト。
        /// </summary>
        public string UserDataText
        {
            get { return UIDataBinder.GetFlagTextExistOrNot(!_userData.IsEmpty); }
        }
        #endregion

        #region IGuiAdapter メンバ

        #region -------------- プロパティ --------------
        public string        Name      { get{ return _coreMaterial.MaterialName;} }
        #endregion -------------- プロパティ --------------

        #region -------------- アダプタクラスと実体クラスの値のコピー処理 --------------

        #region LECore => GUI へのパラメータのコピー
        /// <summary>
        /// GUI-TexMapを新規作成します。
        /// </summary>
        TexMap CreateNewGUITexMap_( IMaterialTexMap srcTex )
        {
            TexMap    newTexMap  = null;

            TexImage newTexImage = TexImage.Create( srcTex.TexImgName, srcTex.PaletteName );
            Debug.Assert( newTexImage != null );
            newTexMap  =
                new TexMap(
                newTexImage,
                (LETexGenMethod)srcTex.ITexGen.Method,
                (TexMapKind)srcTex.TexMapKind,
                srcTex.ResourceType);

            return newTexMap;
        }

        /// <summary>
        /// GUI-TexMapにパラメータを設定します。
        /// </summary>
        void SetCoreTexMapToGUITexMap_( TexMap dstTexMap, IMaterialTexMap srcTexMap )
        {
            // パターンアニメーションによって変更された TexImage を更新します。
            if( dstTexMap.TexImage.Name != srcTexMap.TexImgName )
            {
                TexMap   texMap      = _texMapList.TexMaps[srcTexMap.SlotIdx] as TexMap;
                texMap.TexImage      = TexImage.Create( srcTexMap.TexImgName, srcTexMap.PaletteName );
            }
            // TODO:パレットの変更

            // 必要なパラメータに限ってコピーします(残りは規定値)
            dstTexMap.WrapS      = srcTexMap.WrapS;
            dstTexMap.WrapT      = srcTexMap.WrapT;
            dstTexMap.MinFilter  = srcTexMap.MinFilter;
            dstTexMap.MagFilter  = srcTexMap.MagFilter;

            // ----------- TexGenについて
            dstTexMap.TexGen.Method = (LETexGenMethod)srcTexMap.ITexGen.Method;
            dstTexMap.TexGen.MtxID  = srcTexMap.ITexGen.MtxID;
            dstTexMap.TexGen.ProjectionScale = srcTexMap.ITexGen.ProjectionScale;
            dstTexMap.TexGen.ProjectionOffset = srcTexMap.ITexGen.ProjectionOffset;
            dstTexMap.TexGen.ProjectionRotate = srcTexMap.ITexGen.ProjectionRotate;
            dstTexMap.TexGen.ProjectionFovy = srcTexMap.ITexGen.ProjectionFovy;
            dstTexMap.TexGen.IsFittingLayoutSizeEnabled = srcTexMap.ITexGen.IsFittingLayoutSizeEnabled;
            dstTexMap.TexGen.IsFittingPaneSizeEnabled = srcTexMap.ITexGen.IsFittingPaneSizeEnabled;
            dstTexMap.TexGen.IsAdjustProjectionPaneSREnabled = srcTexMap.ITexGen.IsAdjustProjectionPaneSREnabled;

            // ----------- TexGen-TexMtxについて
            TexMtx    dstTexMtx = dstTexMap.TexGen.TexMtx;
            ITexMtx   srcTexMtx = srcTexMap.ITexGen.ITexMtx;
            dstTexMtx.TransS = srcTexMtx.Trans.X;
            dstTexMtx.TransT = srcTexMtx.Trans.Y;
            dstTexMtx.ScaleS = srcTexMtx.Scale.X;
            dstTexMtx.ScaleT = srcTexMtx.Scale.Y;
            dstTexMtx.Rotate = srcTexMtx.Rotate;

            // ----------- TextureSubListについて
            // テクスチャパターンアニメーションに使用されます。
            dstTexMap.ClearTextureFromSubList();
            string[] texNames = srcTexMap.ITextureChanger.ITextureSubSet.TextureSubSetNameSet;
            foreach( string texName in texNames )
            {
                dstTexMap.AddTextureToSubList( texName );
            }
        }

        /// <summary>
        /// GUI-MaterialTextureを設定します。
        /// </summary>
        protected void SetMaterialTexture_( IMaterialTexMap srcTex )
        {
            // 有効か？
            if( srcTex.IsActive && TextureManager.Search( srcTex.TexImgName ) != null )
            {
                // 現在のGUITexMapが存在しない場合は作成します。
                TexImage dstImg     = _texMapList.SearchTexImageByTexMapID( srcTex.SlotIdx );
                if( dstImg == null )
                {
                    // 新規に登録します
                    TexMap   newTexMap = CreateNewGUITexMap_( srcTex );
                    _texMapList.TexMaps.Add(newTexMap);
                }

                //---------- パラメータの更新
                TexMap   dstTexMap  = _texMapList.SearchTexMapByTexMapID( srcTex.SlotIdx );
                if(dstTexMap != null)
                {
                    SetCoreTexMapToGUITexMap_(dstTexMap, srcTex);
                }
            }
        }

        /// <summary>
        /// GUI-PEDataを設定します。
        /// </summary>
        void SetCorePEDataToGUIPEData_( IPEData srcCorePEData )
        {
            // PE データ
            this._peData.ACompare.Set( srcCorePEData.ACompare );
            this._peData.Blend.Set( srcCorePEData.Blend );
            this._peData.BlendAlpha.Set(srcCorePEData.BlendAlpha);
            this._peData.BlendAlpha.Set(srcCorePEData.BlendAlpha);
            this._peData.DrawMode   = srcCorePEData.DrawMode;
            this._peData.UseDefaultBlendSettings = srcCorePEData.UseDefaultBlendSettings;
            this._peData.UseDefaultAlphaTestSettings = srcCorePEData.UseDefaultAlphaTestSettings;
        }

        /// <summary>
        /// GUI-TevDataを設定します。
        /// </summary>
        void SetCoreTevDataToGUITevData_( ITevData srcCoreTevData )
        {
            //-------- TEVステージ
            _tevData.Stages.NumStages = srcCoreTevData.NumStages;
            for( int i = 0; i < srcCoreTevData.NumStages ; i++ )
            {
                ITevStage    srcStage = srcCoreTevData.GetITevStage( i );

                _tevData.Stages[i].RasOrder = srcStage.RasOrder;
                _tevData.Stages[i].RasSwap = srcStage.RasSwap;

                _tevData.Stages[i].TexCoordOrder = srcStage.TexCoordOrder;
                _tevData.Stages[i].TexMapOrder = srcStage.TexMapOrder;
                _tevData.Stages[i].TexMapSwap = srcStage.TexMapSwap;

                _tevData.Stages[i].ColorStage.Set( srcStage.ColorStage );
                _tevData.Stages[i].AlphaStage.Set( srcStage.AlphaStage );
            }

            // TEVカラー
            for (int i = 0; i < srcCoreTevData.NumTevColors; i++)
            {
                _tevData.SetColor(srcCoreTevData.GetTevColor(i), i);
            }
        }

        void SetCoreCombinerUserShaderToGUICombinerUserShader(ICombinerUserShader srcCoreCombinerUserShader)
        {
            this.CombinerUserShader.FileName = srcCoreCombinerUserShader.FileName;
        }

        /// <summary>
        /// LECoreのデータをGUI用OBJに設定します。
        /// LECoreシーン更新イベントから駆動される再描画イベントの際に
        /// 呼び出されます。
        /// </summary>
        public virtual void SetCoreDataToGUIMock()
        {
            //-------------- マテリアルテクスチャ
            _texMapList.TexMaps.Clear();
            foreach( IMaterialTexMap srcTex in _coreMaterial.IMaterialTexMapSet )
            {
                SetMaterialTexture_( srcTex );
            }

            // 白黒補間
            this.WhiteColor = _coreMaterial.IMaterial.WhiteColor;
            this.BlackColor = _coreMaterial.IMaterial.BlackColor;
            this.IsThresholdingAlphaInterpolationEnabled = _coreMaterial.IMaterial.IsThresholdingAlphaInterpolationEnabled;
            this.IsBlackColorAlphaEnabled = _coreMaterial.IMaterial.BlackColorIAnmAttr.FindSubAttributeByIdx(3).IsActiveAttribute;

            this.LowLevelCombinerSettingsEnabled = _coreMaterial.LowLevelCombinerSettingsEnabled;
            this.CombinerUserShaderSettingsEnabled = _coreMaterial.CombinerUserShaderSettingsEnabled;

            //-------------- PEData
            SetCorePEDataToGUIPEData_( _coreMaterial.IPEData );

            //-------------- TevData
            SetCoreTevDataToGUITevData_( _coreMaterial.ITevData );

            //-------------- CombinerUserShader
            SetCoreCombinerUserShaderToGUICombinerUserShader( _coreMaterial.ICombinerUserShader );
        }

        #endregion LECore => GUI へのパラメータのコピー

        #region GUI => LECore へのパラメータのコピー
        bool CoreMatarialHasTexture_( string texName, int idx )
        {
            foreach( IMaterialTexMap coreTexMap in _coreMaterial.IMaterialTexMapSet )
            {
                if( coreTexMap.TexImgName == texName && coreTexMap.SlotIdx == idx )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// GUI-TexMapのパラメータを LECore-TexMap に設定します。
        /// </summary>
        protected void GUITexMapToCoreTexMap_( IMaterialTexMap dstTexMap, TexMap  srcTexMap )
        {
            //---------- MaterialTexMap について
            MaterialTexMapManipulator   dstTexMapMnp = new MaterialTexMapManipulator();
            dstTexMapMnp.BindTarget( dstTexMap );
            dstTexMapMnp.WrapS      = srcTexMap.WrapS;
            dstTexMapMnp.WrapT      = srcTexMap.WrapT;
            dstTexMapMnp.MinFilter  = srcTexMap.MinFilter;
            dstTexMapMnp.MagFilter  = srcTexMap.MagFilter;

            //---------- TexGen について
            if (this.IsTextureMappingEditable)
            {
                TexGenManipulator texGenMnp = new TexGenManipulator();
                texGenMnp.BindTarget(dstTexMap);
                texGenMnp.Method = (LECore.Structures.LETexGenMethod)srcTexMap.TexGen.Method; // TODO:型を統一すべきか？
                texGenMnp.ProjectionScale = srcTexMap.TexGen.ProjectionScale;
                texGenMnp.ProjectionOffset = srcTexMap.TexGen.ProjectionOffset;
                texGenMnp.ProjectionRotate = srcTexMap.TexGen.ProjectionRotate;
                texGenMnp.ProjectionFovy = srcTexMap.TexGen.ProjectionFovy;
                texGenMnp.IsFittingLayoutSizeEnabled = srcTexMap.TexGen.IsFittingLayoutSizeEnabled;
                texGenMnp.IsFittingPaneSizeEnabled = srcTexMap.TexGen.IsFittingPaneSizeEnabled;
                texGenMnp.IsAdjustProjectionPaneSREnabled = srcTexMap.TexGen.IsAdjustProjectionPaneSREnabled;


                //---------- TexGen-TexMtx について
                TexMtx srcTexMtx = srcTexMap.TexGen.TexMtx;
                TexMtxManipulator texMtxMnp = new TexMtxManipulator();
                texMtxMnp.BindTarget(dstTexMap.ITexGen.ITexMtx);
                texMtxMnp.Trans = new FVec2(srcTexMtx.TransS, srcTexMtx.TransT);
                texMtxMnp.Scale = new FVec2(srcTexMtx.ScaleS, srcTexMtx.ScaleT);
                texMtxMnp.Rotate = srcTexMtx.Rotate;
            }

            //---------- TextureSubList について
            // (テクスチャパターンアニメーションに使用する、テクスチャサブリスト)
            TextureChangerManipulator texChangerMnp = new TextureChangerManipulator();
            texChangerMnp.BindTarget(dstTexMap.ITextureChanger);
            texChangerMnp.SetAll(srcTexMap.TextureSubList);
        }

        /// <summary>
        /// TevStageの値をLECoreのデータに反映します。
        /// </summary>
        void GUITevStagesToCoreTevStages_( RevMaterialManipulator  materialMnp )
        {
            Ensure.Operation.True(this.IsTextureStageEditable);
            int srcNumStages = this.NumTevStages;
            int dstNumStages = materialMnp.Target.ITevData.NumStages;

            // ステージ数減少を反映する場合
            if( srcNumStages != dstNumStages )
            {
                // コピー先のステージ数を同一に設定します。
                materialMnp.NumTevStages = srcNumStages;
            }

            // すべてのステージを更新します。
            for( int i = 0; i < srcNumStages; i++ )
            {
                TevStage  srcStage = _tevData.Stages[i] as TevStage;

                materialMnp.ModifyTevStage(
                    srcStage.RasOrder,
                    srcStage.TexMapOrder,
                    srcStage.TexCoordOrder,
                    srcStage.RasSwap,
                    srcStage.TexMapSwap,
                    srcStage.ColorStage,
                    srcStage.AlphaStage,
                    i );
            }

            // TEVカラー
            for( int i = 0; i < this.TevData.GetColorCount(); i++ )
            {
                materialMnp.SetTevColors(this.TevData.GetColor(i), i);
            }

            // CombinerUserShader
            materialMnp.ModifyCombinerUserShader(this.CombinerUserShader.FileName);
        }

        /// <summary>
        /// 編集対象にテクスチャーが登録されているか調べて、未登録なら登録します。
        /// </summary>
        protected void RegisterTextureToTargetIfNeeded_(TexImage srcTexImg)
        {
            var textureManager = _coreMaterial?.IMaterial?.OwnerPane?.OwnerSubScene?.ITextureMgr;
            if (textureManager == null)
            {
                return;
            }

            if (textureManager.FindITextureImageByName(srcTexImg.Name) != null)
            {
                return;
            }

            // 画像下キャプチャ用のダミーテクスチャは登録しない。
            if (srcTexImg.Name == PaneHelper.FramebufferCaptureDummyTextureName)
            {
                return;
            }

            // キャプチャテクスチャは登録しない
            if (srcTexImg.Texture.SourceType != LECore.Structures.LECoreInterface.TextureSourceType.File)
            {
                return;
            }

            TextureMgrManipulator mnp = new TextureMgrManipulator();
            mnp.BindTarget(textureManager);

            var results = mnp.RegisterITextureImageSetFromFile(new string[] { srcTexImg.Texture.FilePath });

            // 同名のテクスチャが見つかっていないので登録されるはず
            Debug.Assert(results.Length == 1);

            mnp.SetTexturePixelFmt(results[0], srcTexImg.Texture.PixelFormat);
        }

        /// <summary>
        /// GUIで行われた編集をLECoreのデータに反映します。
        /// LECoreのデータに反映が行われた結果、
        /// シーン変更イベントが発生します。
        ///
        /// GUIは変更イベントを監視しているので、再描画が行われます。
        /// </summary>
        void SetGUIMockDataToCoreData_(bool forcePushCommand)
        {
            // 多数の変更を行うことをシーンに通知します。
            LECore.LayoutEditorCore.Scene.CurrentISubScene.BeginMassiveModify();

            //-------- マテリアル関連の情報を反映します。
            RevMaterialManipulator         materialMnp = new RevMaterialManipulator();
            materialMnp.BindTarget( _coreMaterial );

            if (forcePushCommand)
            {
                materialMnp.PushEmptyCommand(SceneModifyEventArgs.Kind.PaneModify);
            }

            // すべてのTexMapについて...
            for( int i = 0; i < 8; i++ )
            {
                // TexMapが設定されていれば...(FIXME)
                if( _texMapList.SearchTexImageByTexMapID( i ) != null )
                {
                    TexMap   srcTexMap = _texMapList.SearchTexMapByTexMapID(i);
                    if(srcTexMap != null)
                    {
                        TexImage srcTexImg = srcTexMap.TexImage;

                        // テクスチャの登録が無ければ新規に登録します。
                        RegisterTextureToTargetIfNeeded_(srcTexImg);

                        // ----------- TexImage の登録
                        // 新規登録テクスチャを発見したら、登録します。
                        IMaterialTexMap matTexMap = null;
                        if (srcTexImg != null && !CoreMatarialHasTexture_(srcTexImg.Name, i))
                        {
                            AttrTextureResourceType resType = MaterialTexMapHelper.GetAttrTextureResourceType(_coreMaterial.IMaterial, srcTexImg.Texture.SourceType);
                            Debug.Assert(i < 3);   //@@
                            materialMnp.RegisterMatTexture(
                                srcTexImg.Name,
                                i,
                                resType);
                        }

                        // 各 GUI-TexMap の値をCoreSystem-TexMapにコピーします。
                        matTexMap = _coreMaterial.IMaterialTexMapSet[i];
                        Debug.Assert(matTexMap != null);
                        GUITexMapToCoreTexMap_(matTexMap, srcTexMap);
                    }
                }
                else
                {
                    // マテリアル・テクスチャを消去します。
                    materialMnp.RemoveMatTexture( i );
                }
            }

            materialMnp.LowLevelCombinerSettingsEnabled = this.LowLevelCombinerSettingsEnabled;
            materialMnp.CombinerUserShaderSettingsEnabled = this.CombinerUserShaderSettingsEnabled;

            if (this.IsBlendEditable)
            {
                //-------- PEData関連の情報を反映します。
                materialMnp.ModifyPEData(
                    _peData.ACompare,
                    _peData.Blend,
                    _peData.BlendAlpha,
                    /*_peData.ZCompare,*/
                    _peData.DrawMode,
                    _peData.UseDefaultBlendSettings,
                    _peData.UseDefaultAlphaTestSettings);
            }

            if (this.IsMaterialColorEditable)
            {
                // 白黒補間
                materialMnp.WhiteColor = this.WhiteColor;
                materialMnp.BlackColor = this.BlackColor;
                materialMnp.IsThresholdingAlphaInterpolationEnabled = this.IsThresholdingAlphaInterpolationEnabled;
            }

            //-------- TevStageパラメータ
            if (this.IsTextureStageEditable)
            {
                GUITevStagesToCoreTevStages_(materialMnp);
            }

            // 変更の終了をシーンに通知します。(変更イベントが発生します)
            LECore.LayoutEditorCore.Scene.CurrentISubScene.EndMassiveModify();
        }
        #endregion GUI => LECore へのパラメータのコピー

        #endregion -------------- アダプタクラスと実体クラスの値のコピー処理 --------------

        /// <summary>
        /// GUIによるパラメータ変更時に、モデル変更ハンドラから呼ばれます。
        /// GUIデータ => LECoreデータの同期を行います。
        /// </summary>
        /// <param name="sender"></param>
        public virtual void OnModified(PropertyPage sender, bool forcePushCommand)
        {
            SetGUIMockDataToCoreData_(forcePushCommand);
        }

        /// <summary>
        /// 指定した、インスタンスが
        /// アダプタクラスの操作対象か調査します。
        /// </summary>
        public virtual bool HasSameTargets( object another )
        {
            MaterialGUIAdapter    anotherMat = another as MaterialGUIAdapter;
            return ( anotherMat != null && anotherMat.CoreMaterial == this._coreMaterial );
        }

        /// <summary>
        /// 更新します。
        /// </summary>
        public void Refresh()
        {
            // テクスチャ座標の初期化
            for (int i = 0; i < _texCoord4Set.Length; i++)
            {
                _texCoord4Set[i] = new TexCoord4();
            }

            this.IsTextureMappingEditable = true;
            this.IsMaterialColorEditable = true;
            this.IsBlendEditable = true;
            this.IsTextureStageEditable = true;
            this.IsTextureOverwritingEnabled = false;
        }

        #endregion IGuiAdapter メンバ
    }



    #region テクスチャデータ

    #region TexData
    /// <summary>
    /// テクスチャデータクラス。
    /// </summary>
    public sealed class TexData
    {
        private readonly TexDataType _dataType;
        private readonly TexMap[]    _texMaps = new TexMap[3];

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexData(TexDataType dataType, TexMap texMap0, TexMap texMap1, TexMap texMap2)
        {
            _dataType   = dataType;
            _texMaps[0] = texMap0;
            _texMaps[1] = texMap1;
            _texMaps[2] = texMap2;
        }
    }

    /// <summary>
    /// テクスチャデータタイプ。
    /// </summary>
    public enum TexDataType
    {
        /// <summary>通常（１枚使用）。</summary>
        Standard,
        /// <summary>トゥーン（１枚使用）。</summary>
        Toon,
    }
    #endregion

    #region TexMapKind(追加)
    /// <summary>
    /// テクスチャの利用種類(追加)
    /// </summary>
    public enum TexMapKind
    {
        Normal,
        IndirectWarpTexture,
        IndirectBumpMap
    };
    #endregion TexMapKind(追加)

    #region TexMap




    /// <summary>
    /// テクスチャマップクラス。
    /// </summary>
    public sealed class TexMap
    {
        // テクスチャイメージ
        private TexImage         _texImage  = null;
        private AttrTexWrap      _wrapS     = AttrTexWrap.Clamp;
        private AttrTexWrap      _wrapT     = AttrTexWrap.Clamp;
        private bool             _mipmap    = false;
        private AttrTexFilterMin _minFilter = AttrTexFilterMin.Linear;
        private AttrTexFilterMag _magFilter = AttrTexFilterMag.Linear;
        private float            _lodBias   = 0.0f;
        private bool             _biasClamp = false;
        private bool             _doEdgeLod = false;
        private AttrMaxAniso     _maxAniso  = AttrMaxAniso.Aniso1;
        private AttrTextureResourceType _resType = AttrTextureResourceType.LocalFile;

        // テクスジェン
        private readonly TexGen _texGen = new TexGen();

        // テクスチャパターンアニメーション用のサブリスト(追加)
        ArrayList                   _textureSubList = new ArrayList();

        /// <summary>
        /// テクスチャの利用種類(追加)
        /// 現状では変更できないパラメータとして考えています。
        /// </summary>
        readonly TexMapKind          _texMapKind = TexMapKind.Normal;

        #region コンストラクタ
        public TexMap(TexImage texImage, LETexGenMethod texGenMethod, TexMapKind kind)
            : this(texImage, texGenMethod, kind, AttrTextureResourceType.LocalFile)
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMap(TexImage texImage, LETexGenMethod texGenMethod, TexMapKind kind, AttrTextureResourceType resType)
        {
            // テクスチャイメージ
            _texImage = texImage;
            if (texImage.Texture.IsMipmapAvailable)
            {
                // ミップマップ可能なら設定しておく
                _mipmap    = true;
                _minFilter = AttrTexFilterMin.LinearMipLinear;
            }

            // テクスジェン
            _texGen = new TexGen(texGenMethod);

            // テクスチャ種類
            _texMapKind = kind;

            // テクスチャのリソースタイプ
            _resType = resType;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMap(TexMap src)
        {
            Set(src);
        }
        #endregion

        #region プロパティ
        /// <summary>
        /// テクスチャイメージ。
        /// </summary>
        public TexImage TexImage
        {
            get { return _texImage;  }
            set
            {
                _texImage = value;

                // ミップマップ設定調整
                if (_texImage.Texture.IsMipmapAvailable)
                {
                    if (!_mipmap)
                    {
                        _mipmap    = true;
                        _minFilter = AttrTexFilterMin.LinearMipLinear;
                    }
                }
                else
                {
                    if (_mipmap)
                    {
                        _mipmap    = false;
                        _minFilter = AttrTexFilterMin.Linear;
                    }
                }
            }
        }

        /// <summary>
        /// Ｓ方向の繰り返し方。
        /// </summary>
        public AttrTexWrap WrapS
        {
            get { return _wrapS;  }
            set { _wrapS = value; }
        }

        /// <summary>
        /// Ｔ方向の繰り返し方。
        /// </summary>
        public AttrTexWrap WrapT
        {
            get { return _wrapT;  }
            set { _wrapT = value; }
        }

        /// <summary>
        /// ミップマップ。
        /// </summary>
        public bool Mipmap
        {
            get { return _mipmap;  }
            set
            {
                _mipmap = value;

                // 縮小時フィルタ調整
                if (value)
                {
                    if (_minFilter <= AttrTexFilterMin.Linear)
                    {
                        _minFilter = AttrTexFilterMin.LinearMipLinear;
                    }
                }
                else
                {
                    if (_minFilter >= AttrTexFilterMin.NearMipNear)
                    {
                        _minFilter = AttrTexFilterMin.Linear;
                    }
                }
            }
        }

        /// <summary>
        /// 縮小時フィルタ。
        /// </summary>
        public AttrTexFilterMin MinFilter
        {
            get { return _minFilter;  }
            set { _minFilter = value; }
        }

        /// <summary>
        /// 拡大時フィルタ。
        /// </summary>
        public AttrTexFilterMag MagFilter
        {
            get { return _magFilter;  }
            set { _magFilter = value; }
        }

        /// <summary>
        /// ＬＯＤバイアス。
        /// </summary>
        public float LodBias
        {
            get { return _lodBias;  }
            set { _lodBias = value; }
        }

        /// <summary>
        /// バイアスクランプ。
        /// </summary>
        public bool BiasClamp
        {
            get { return _biasClamp;  }
            set { _biasClamp = value; }
        }

        /// <summary>
        /// エッジＬＯＤ。
        /// </summary>
        public bool DoEdgeLod
        {
            get { return _doEdgeLod;  }
            set { _doEdgeLod = value; }
        }

        /// <summary>
        /// 最大アニソトロピックフィルタ。
        /// </summary>
        public AttrMaxAniso MaxAniso
        {
            get { return _maxAniso;  }
            set { _maxAniso = value; }
        }

        /// <summary>
        /// 参照しているテクスチャのリソースタイプ
        /// </summary>
        public AttrTextureResourceType TextureResourceType
        {
            get { return _resType; }
            set { _resType = value; }
        }

        /// <summary>
        /// テクスジェン。
        /// </summary>
        public TexGen TexGen
        {
            get { return _texGen; }
        }

        /// <summary>
        /// テクスチャ利用目的種類
        /// </summary>
        public TexMapKind TexMapKind
        {
            get{ return _texMapKind; }
        }

        /// <summary>
        /// インダイレクトテクスチャか？
        /// </summary>
        public bool IsUsedForIndirectTexture
        {
            get{ return this.TexMapKind != TexMapKind.Normal; }
        }

        /// <summary>
        /// テクスチャパターンアニメーション用のサブリストを取得します。(追加)
        /// </summary>
        public string[] TextureSubList
        {
            get{ return _textureSubList.ToArray( typeof( string ) ) as string[]; }
        }

        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TexMap src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _texImage  = src._texImage;
            _wrapS     = src._wrapS;
            _wrapT     = src._wrapT;
            _mipmap    = src._mipmap;
            _minFilter = src._minFilter;
            _magFilter = src._magFilter;
            _lodBias   = src._lodBias;
            _biasClamp = src._biasClamp;
            _doEdgeLod = src._doEdgeLod;
            _maxAniso  = src._maxAniso;
            _resType   = src._resType;

            _texGen.Set(src._texGen);
        }

        #region テクスチャパターンアニメーションに使用するサブリスト
        /// <summary>
        /// サブリストにテクスチャを登録します。(追加)
        /// </summary>
        /// <param name="texName"></param>
        public void AddTextureToSubList( string texName)
        {
            if( !_textureSubList.Contains( texName ) )
            {
                _textureSubList.Add( texName );
            }
        }
        /// <summary>
        /// サブリストにテクスチャを登録します。(追加)
        /// </summary>
        /// <param name="texName"></param>
        public void InsertTextureToSubList( int index, string texName )
        {
            if( !_textureSubList.Contains( texName ) )
            {
                _textureSubList.Insert( index , texName );
            }
        }

        /// <summary>
        /// サブリスト内の位置を取得します。
        /// </summary>
        public int GetTextureIndexOfSubList( string texName )
        {
            return _textureSubList.IndexOf( texName );
        }

        /// <summary>
        /// サブリストの要素数を取得します。
        /// </summary>
        public int CountOfSubList
        {
            get{ return _textureSubList.Count;}
        }

        /// <summary>
        /// サブリストからテクスチャを削除します。(追加)
        /// </summary>
        /// <param name="texName"></param>
        public void RemoveTextureFromSubList( string texName )
        {
            if( _textureSubList.Contains( texName ) )
            {
                _textureSubList.Remove( texName );
            }
        }

        /// <summary>
        /// サブリストからすべてのテクスチャを削除します。(追加)
        /// </summary>
        /// <param name="texName"></param>
        public void ClearTextureFromSubList()
        {
            _textureSubList.Clear();
        }
        #endregion テクスチャパターンアニメーションに使用するサブリスト


        #endregion


    }
    #endregion

    #region TexGen
    /// <summary>
    /// テクスチャ座標生成クラス。
    /// (TODO:2重化しているので後々統合したい。)
    /// </summary>
    public sealed class TexGen
    {
        // アトリビュート
        private LETexGenMethod _method = LETexGenMethod.UV0;
        private AttrTexCoordMtx _mtxID  = AttrTexCoordMtx.Identity;

        // テクスチャ行列
        private readonly TexMtx _texMtx = new TexMtx();

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexGen()
        {
            this.ProjectionScale = FVec2.One;
            this.ProjectionOffset = FVec2.Empty;
            this.ProjectionRotate = FVec3.Empty;
            this.ProjectionFovy = 40.0f;
            this.IsFittingLayoutSizeEnabled = false;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexGen(TexGen src)
            : this()
        {
            Set(src);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexGen(LETexGenMethod method)
            : this()
        {
            _method = method;
        }
        #endregion

        #region プロパティ
        /// <summary>
        /// 座標生成方法。
        /// </summary>
        public LETexGenMethod Method
        {
            get { return _method;  }
            set { _method = value; }
        }

        /// <summary>
        /// マトリクスＩＤ。
        /// </summary>
        public AttrTexCoordMtx MtxID
        {
            get { return _mtxID;  }
            set { _mtxID = value; }
        }

        /// <summary>
        /// テクスチャ行列。
        /// </summary>
        public TexMtx TexMtx
        {
            get { return _texMtx; }
        }

        /// <summary>
        /// ＵＶマッピング設定かどうか。
        /// </summary>
        public bool IsUVMethod
        {
            get { return LETexGenMethod.UV0 <= _method && _method <= LETexGenMethod.UV2; }
        }

        /// <summary>
        /// 投影スケール
        /// </summary>
        public FVec2 ProjectionScale { get; set; }

        /// <summary>
        /// 投影オフセット
        /// </summary>
        public FVec2 ProjectionOffset { get; set; }

        /// <summary>
        /// 投影回転
        /// </summary>
        public FVec3 ProjectionRotate { get; set; }

        /// <summary>
        /// 投影画角
        /// </summary>
        public float ProjectionFovy { get; set; }

        /// <summary>
        /// レイアウトサイズに一致するように自動スケールするか
        /// </summary>
        public bool IsFittingLayoutSizeEnabled { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool IsFittingPaneSizeEnabled { get; set; }

        /// <summary>
        ///
        /// </summary>
        public bool IsAdjustProjectionPaneSREnabled { get; set; }

        #endregion

        #region 設定

        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TexGen src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _method = src._method;
            _mtxID  = src._mtxID;
            _texMtx.Set(src._texMtx);

            this.ProjectionScale.Set(src.ProjectionScale);
            this.ProjectionOffset.Set(src.ProjectionOffset);
            this.ProjectionRotate.Set(src.ProjectionRotate);
            this.ProjectionFovy = src.ProjectionFovy;
            this.IsFittingLayoutSizeEnabled = src.IsFittingLayoutSizeEnabled;
            this.IsFittingPaneSizeEnabled = src.IsFittingPaneSizeEnabled;
            this.IsAdjustProjectionPaneSREnabled = src.IsAdjustProjectionPaneSREnabled;
        }
        #endregion
    }
    #endregion

    #region TexMtx
    /// <summary>
    /// テクスチャ行列クラス。
    /// </summary>
    public sealed class TexMtx
    {
        private float _scaleS = 1.0f;
        private float _scaleT = 1.0f;
        private float _rotate = 0.0f;
        private float _transS = 0.0f;
        private float _transT = 0.0f;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMtx() {}

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMtx(Elem_tex_matrix element)
        {
            _scaleS = element.tex_scale[0];
            _scaleT = element.tex_scale[1];
            _rotate = element.tex_rotate;
            _transS = element.tex_translate[0];
            _transT = element.tex_translate[1];
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMtx(TexMtx src)
        {
            Set(src);
        }
        #endregion

        #region プロパティ
        /// <summary>
        /// スケールＳ。
        /// </summary>
        public float ScaleS
        {
            get { return _scaleS;  }
            set { _scaleS = value; }
        }

        /// <summary>
        /// スケールＴ。
        /// </summary>
        public float ScaleT
        {
            get { return _scaleT;  }
            set { _scaleT = value; }
        }

        /// <summary>
        /// ローテーション。
        /// </summary>
        public float Rotate
        {
            get { return _rotate;  }
            set { _rotate = value; }
        }

        /// <summary>
        /// トランスレーションＳ。
        /// </summary>
        public float TransS
        {
            get { return _transS;  }
            set { _transS = value; }
        }

        /// <summary>
        /// トランスレーションＴ。
        /// </summary>
        public float TransT
        {
            get { return _transT;  }
            set { _transT = value; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TexMtx src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _scaleS = src._scaleS;
            _scaleT = src._scaleT;
            _rotate = src._rotate;
            _transS = src._transS;
            _transT = src._transT;
        }
        #endregion

        #region ドキュメント更新
        /// <summary>
        /// ドキュメント要素更新。
        /// </summary>
        public void UpdateDocElement(Elem_tex_matrix element)
        {
            element.tex_scale[0]     = _scaleS;
            element.tex_scale[1]     = _scaleT;
            element.tex_rotate       = _rotate;
            element.tex_translate[0] = _transS;
            element.tex_translate[1] = _transT;

            // エフェクト行列（単位行列固定）
            for (int i = 0; i < 4; i++)
            {
                for (int j = 0; j < 4; j++)
                {
                    if (i == j)
                    {
                        element.tex_effect_mtx0[i * 4 + j] = 1.0f;
                        element.tex_effect_mtx1[i * 4 + j] = 1.0f;
                    }
                    else
                    {
                        element.tex_effect_mtx0[i * 4 + j] = 0.0f;
                        element.tex_effect_mtx1[i * 4 + j] = 0.0f;
                    }
                }
            }
        }
        #endregion
    }
    #endregion

    #endregion

    #region ＴＥＶデータ

    #region TevData
    /// <summary>
    /// ＴＥＶデータクラス。
    /// </summary>
    public sealed class TevData
    {
        private readonly FloatColor[] _colors = new FloatColor[LECore.Structures.TevColorsConstants.NumTevKColorRegs];
        private readonly TevStages     _stages;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevData()
        {
            _stages     = new TevStages();
        }

        #endregion

        #region アトリビュート

        /// <summary>
        /// ステージ。
        /// </summary>
        public TevStages Stages
        {
            get { return _stages; }
        }

        /// <summary>
        /// カラーの設定
        /// </summary>
        public void SetColor(FloatColor col, int index)
        {
            _colors[index] = col;
        }

        /// <summary>
        /// カラーの取得
        /// </summary>
        public FloatColor GetColor(int index)
        {
            return _colors[index];
        }

        /// <summary>
        /// カラー数の取得
        /// </summary>
        public int GetColorCount()
        {
            return _colors.Length;
        }

        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TevData src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _stages.Set(src._stages);
        }
        #endregion

        #region ドキュメント
        /// <summary>
        /// ドキュメント要素更新。
        /// </summary>
        public void UpdateDocElement(Elem_material element)
        {
            _stages.UpdateDocElement(element.tev_stage_array);
        }
        #endregion
    }
    #endregion

    #region TevStages
    /// <summary>
    /// ＴＥＶステージ管理クラス。
    /// </summary>
    public sealed class TevStages
    {
        private int                 _numStages = 0;
        private readonly TevStage[] _stages    = new TevStage[LayoutEditorCore.PlatformDetail.MaxTevStageCount];

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevStages()
        {
            for (int i = 0; i < _stages.Length; i++)
            {
                _stages[i] = new TevStage(i);
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevStages(Elem_tev_stage_array element)
        {
            _numStages = element.size;

            // 既設定ステージ
            for (int i = 0; i < element.size; i++)
            {
                _stages[i] = new TevStage( i, (Elem_tev_stage)element.tev_stage[i]);
            }
            // 未設定ステージ
            for (int i = element.size; i < _stages.Length; i++)
            {
                _stages[i] = new TevStage( i);
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevStages(TevStages src) : this()
        {
            Set(src);
        }

        /// <summary>
        /// インダイレクト行列の更新
        /// (本クラスが廃止されるまでの暫定対処処理です)
        /// </summary>
        public void SetIndirectMatrixAll(TevDirectStage srcColorStage)
        {
            foreach(TevStage stage in _stages)
            {
                stage.ColorStage.IndirectRotate = srcColorStage.IndirectRotate;
                stage.ColorStage.IndirectScale = srcColorStage.IndirectScale;
            }
        }

        #endregion

        #region アトリビュート
        /// <summary>
        /// ステージ数。
        /// </summary>
        public int NumStages
        {
            get { return _numStages;  }
            set { _numStages = value; }
        }

        /// <summary>
        /// ステージ（インデクサ）。
        /// </summary>
        public TevStage this[int stageNo]
        {
            get { return _stages[stageNo]; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TevStages src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _numStages = src._numStages;

            for (int i = 0; i < _stages.Length; i++)
            {
                _stages[i].Set(src._stages[i]);
            }
        }
        #endregion

        #region ドキュメント更新
        /// <summary>
        /// ドキュメント要素更新。
        /// </summary>
        public void UpdateDocElement(Elem_tev_stage_array element)
        {
            element.size = _numStages;
            element.tev_stage.Clear();

            for (int i = 0; i < _numStages; i++)
            {
                Elem_tev_stage elemTevStage = new Elem_tev_stage();
                elemTevStage.index = i;

                _stages[i].UpdateDocElement(elemTevStage);
                element.tev_stage.Add(elemTevStage);
            }
        }
        #endregion
    }
    #endregion

    #region TevStage
    /// <summary>
    /// ＴＥＶステージクラス。
    /// </summary>
    public sealed class TevStage
    {
        // オーダー
        private AttrRasOrder _rasOrder      = AttrRasOrder.Color0A0;
        private int          _texMapOrder   = -1;
        private int          _texCoordOrder = -1;

        // スワップ
        private int _rasSwap    = 0;
        private int _texMapSwap = 0;

        // ステージ
        private readonly TevDirectStage   _colorStage;
        private readonly TevDirectStage   _alphaStage;

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevStage(int stageNo)
        {
            _colorStage = new TevDirectStage(null, stageNo, false);
            _alphaStage = new TevDirectStage(null, stageNo, true);
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevStage(int stageNo, Elem_tev_stage element)
        {
            _rasOrder = element.ras_order;
            _texMapOrder = element.tex_map_order;
            _texCoordOrder = element.tex_coord_order;
            _rasSwap = element.ras_swap;
            _texMapSwap = element.tex_map_swap;
            _colorStage = new TevDirectStage(null, stageNo, false);
            _alphaStage = new TevDirectStage(null, stageNo, true);
        }
        #endregion

        #region アトリビュート
        /// <summary>
        /// ラスタライズカラーオーダー。
        /// </summary>
        public AttrRasOrder RasOrder
        {
            get { return _rasOrder;  }
            set { _rasOrder = value; }
        }

        /// <summary>
        /// テクスチャマップオーダー。
        /// </summary>
        public int TexMapOrder
        {
            get { return _texMapOrder;  }
            set { _texMapOrder = value; }
        }

        /// <summary>
        /// テクスチャ座標オーダー。
        /// </summary>
        public int TexCoordOrder
        {
            get { return _texCoordOrder;  }
            set { _texCoordOrder = value; }
        }

        /// <summary>
        /// ラスタライズドカラースワップ。
        /// </summary>
        public int RasSwap
        {
            get { return _rasSwap;  }
            set { _rasSwap = value; }
        }

        /// <summary>
        /// テクスチャマップスワップ。
        /// </summary>
        public int TexMapSwap
        {
            get { return _texMapSwap;  }
            set { _texMapSwap = value; }
        }

        /// <summary>
        /// カラーステージ。
        /// </summary>
        public TevDirectStage ColorStage
        {
            get { return _colorStage; }
        }

        /// <summary>
        /// アルファステージ。
        /// </summary>
        public TevDirectStage AlphaStage
        {
            get { return _alphaStage; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TevStage src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _rasOrder      = src._rasOrder;
            _texMapOrder   = src._texMapOrder;
            _texCoordOrder = src._texCoordOrder;
            _rasSwap       = src._rasSwap;
            _texMapSwap    = src._texMapSwap;

            _colorStage.Set(src._colorStage);
            _alphaStage.Set(src._alphaStage);
        }
        #endregion

        #region ドキュメント
        /// <summary>
        /// ドキュメント要素更新。
        /// </summary>
        public void UpdateDocElement(Elem_tev_stage element)
        {
        }
        #endregion
    }
    #endregion

    #region TexElements
    /// <summary>
    /// テクスチャ関連要素管理クラス。
    /// </summary>
    public sealed class TexElements
    {
        private Elem_tex_map         _texMap    = null;
        private Elem_tex_coord       _texCoord  = null;
        private Elem_tex_matrix      _texMatrix = null;
        private Elem_indirect_stage  _indStage  = null;
        private Elem_indirect_matrix _indMatrix = null;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexElements(
            Elem_tex_map    texMap,
            Elem_tex_coord  texCoord,
            Elem_tex_matrix texMatrix
            )
        {
            _texMap    = texMap;
            _texCoord  = texCoord;
            _texMatrix = texMatrix;
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexElements(
            Elem_tex_map         texMap,
            Elem_tex_coord       texCoord,
            Elem_tex_matrix      texMatrix,
            Elem_indirect_stage  indStage,
            Elem_indirect_matrix indMatrix
            )
            : this(texMap, texCoord, texMatrix)
        {
            _indStage  = indStage;
            _indMatrix = indMatrix;
        }

        /// <summary>
        /// tex_map要素。
        /// </summary>
        public Elem_tex_map TexMap { get { return _texMap; }}

        /// <summary>
        /// tex_coord要素。
        /// </summary>
        public Elem_tex_coord TexCoord { get { return _texCoord; }}

        /// <summary>
        /// tex_matrix要素。
        /// </summary>
        public Elem_tex_matrix TexMatrix { get { return _texMatrix; }}

        /// <summary>
        /// indirect_stage要素。
        /// </summary>
        public Elem_indirect_stage IndStage { get { return _indStage; }}

        /// <summary>
        /// indirect_matrix要素。
        /// </summary>
        public Elem_indirect_matrix IndMatrix { get { return _indMatrix; }}

        /// <summary>
        /// インダイレクト要素を持つかどうか。
        /// </summary>
        public bool HasIndirectElements
        {
            get { return _indStage != null && _indMatrix != null; }
        }
    }
    #endregion

    #region TexMapList

    /// <summary>
    /// テクスチャマップリストクラス。
    /// </summary>
    public sealed class TexMapList
    {
        private readonly ArrayList _texMaps = new ArrayList();

        // 最大テクスチャ枚数
        public const int NumMaxTexture = 3;

        // インダイレクトテクスチャ最大枚数(=最大インダイレクトステージ数)
        private const int NumMaxIndirectTexture = 4;

        #region コンストラクタ

        public TexMapList()
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TexMapList(TexMapList src)
        {
            Set(src);
        }
        #endregion

        #region アトリビュート
        /// <summary>
        /// テクスチャマップリスト。
        /// </summary>
        public ArrayList TexMaps
        {
            get { return _texMaps; }
        }

        /// <summary>
        /// テクスチャマップ（インデクサ）。
        /// </summary>
        public TexMap this[int texMapNo]
        {
            get { return (TexMap)_texMaps[texMapNo]; }
        }

        /// <summary>
        /// 登録テクスチャリスト内のインダイレクトテクスチャ用テクスチャの枚数を取得します。
        /// </summary>
        int _NumIndirectTexture
        {
            get
            {
                int numIndirectTexMap = 0;
                foreach( TexMap texMap in _texMaps )
                {
                    if( texMap.IsUsedForIndirectTexture )
                    {
                        numIndirectTexMap++;
                    }
                }
                return numIndirectTexMap;
            }
        }

        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TexMapList src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _texMaps.Clear();
            foreach (TexMap texMap in src._texMaps)
            {
                _texMaps.Add(new TexMap(texMap));
            }
        }

        /// <summary>
        /// テクスチャマトリクスＩＤ調節。
        /// </summary>
        public void AdjustTexMtxID()
        {
            for (int i = 0; i < _texMaps.Count; i++)
            {
                TexGen texGen = ((TexMap)_texMaps[i]).TexGen;
                texGen.MtxID = (AttrTexCoordMtx)(AttrTexCoordMtx.TexMtx0 + i);
            }
        }
        #endregion

        #region 状態確認
        /// <summary>
        /// 標準テクスチャが追加可能か。
        /// </summary>
        public bool CanAddStandardTexMap(MaterialGUIAdapter material)
        {
            if (_texMaps.Count < NumMaxTexture)
            {
                return material.HasVtxTexCoords || material.HasVtxNormals;
            }
            return false;
        }

        /// <summary>
        /// インダイレクト用テクスチャが登録可能か。(追加)
        /// </summary>
        public bool CanAddToonIndirectTexMap()
        {
            if (_texMaps.Count < NumMaxTexture)
            {
                if( _NumIndirectTexture < NumMaxIndirectTexture )
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 使用頂点テクスチャ座標数を取得。
        /// </summary>
        public int GetNumUsingVtxTexCoords()
        {
            int result = 0;
            foreach (TexMap texMap in _texMaps)
            {
                if (texMap.TexGen.IsUVMethod)
                {
                    int number = texMap.TexGen.Method - LETexGenMethod.UV0 + 1;
                    if (number > result)
                    {
                        result = number;
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// テクスチャマップの使用するインダイレクトステージのステージ番号を取得します。
        /// </summary>
        public int GetTexMapIndStageIdx( int texMapNo )
        {
            Debug.Assert( texMapNo >= 0 && texMapNo < _texMaps.Count );
            TexMap targetTexMap = _texMaps[texMapNo] as TexMap;

            int numIndirectStage = 0;
            foreach (TexMap texMap in _texMaps)
            {
                if( targetTexMap == texMap )
                {
                    return numIndirectStage;
                }

                // インダイレクトテクスチャマップが発見されたら数えます。
                if( texMap.IsUsedForIndirectTexture )
                {
                    numIndirectStage++;
                }
            }

            // 発見され無いことはありえないはず...
            Debug.Assert( false );
            return -1;
        }

        #endregion



        #region 検索
        /// <summary>
        /// テクスチャイメージをテクスチャマップＩＤで検索。
        /// </summary>
        public TexImage SearchTexImageByTexMapID(int texmapID)
        {
            // TODO: 現状では連続的にテクスチャマップを配置しているので検索の必要は特に無い
            if (0 <= texmapID && texmapID < _texMaps.Count)
            {
                return ((TexMap)_texMaps[texmapID]).TexImage;
            }
            else
            {
                return null;
            }
        }

        public TexMap SearchTexMapByTexMapID(int texmapID)
        {
            if (0 <= texmapID && texmapID < _texMaps.Count)
            {
                return _texMaps[texmapID] as TexMap;
            }
            else
            {
                return null;
            }
        }
        #endregion

        #region ドキュメント更新

        #endregion
    }
    #endregion

    #region CombinerUserShader
    public sealed class CombinerUserShader
    {
        string _combinerFileName = string.Empty;

        public string FileName { get; set; }
    }
    #endregion

    #endregion

    #endregion
}
