﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Linq;
using System.Windows.Forms;
using LayoutEditor.Forms.ToolWindows;
using LayoutEditor.src.Forms.Dialogs;
using LayoutEditor.Structures.SerializableObject;
using LayoutEditor.Utility;
using LECore;
using LECore.Manipulator;
using LECore.Structures;
using System.Drawing;
using System.Collections.Generic;
using LayoutEditor.Forms.Dialogs;
using LayoutEditor.Controls;
using System.Diagnostics;

namespace LayoutEditor.Forms.ToolWindows.PartsWindow
{
    /// <summary>
    /// 部品ウインドウです。
    /// </summary>
    public sealed partial class PartsWindow : LEToolWindow
    {
        const string _DescStringNotFoundControl = "???";
        const string _DescStringNotFoundElement = "????";

        const int _EditableSubItemIndex = 1;

        Image _cancelImage = null;
        IControlSettings _currentControlSettings = null;

        /// <summary>
        /// 複数コントロールモードかどうか
        /// </summary>
        public bool IsMultiControlMode { get; set; }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsWindow()
        {
            InitializeComponent();

            this._cmhAnimName.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_NAME");
            this._cmhAnimTagName.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_ANIM_ANIMTAG");
            this._cmhAnimDesc.Text = LayoutEditor.StringResMgr.Get("TAG_COMMENT");
            this._cmhAnimRequired.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_REQUIRED");

            this._cmhPaneName.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_NAME");
            this._cmhPanePaneName.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_PANE_NAME");
            this._cmhPaneDesc.Text = LayoutEditor.StringResMgr.Get("TAG_COMMENT");
            this._cmhPaneRequired.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_REQUIRED");

            this._cmhParamPaneName.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_PANE_NAME");
            this._cmhParamPaneKind.Text = LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_KIND");
            this._cmhParamDescription.Text = LayoutEditor.StringResMgr.Get("TAG_COMMENT");

            _pnlParam.TitleString = StringResMgr.Get("PROP_PANE_CATEGORY_PARTSPROPERTY");
            _pnlPane.TitleString = StringResMgr.Get("PROP_PANE_CATEGORY_PROPERTYPANE");
            _pnlAnim.TitleString = StringResMgr.Get("PROP_PANE_CATEGORY_PROPERTYANIM");

            _cancelImage = ImageResMgr.GetManifestResourceBitmap("newicons.Delete.bmp");

            InitializeDlgState();
            UpdateDlgState();
        }

        //----------------------------------------------------------
        // プロパティ
        //----------------------------------------------------------

        /// <summary>
        /// 現在のサブシーンです。
        /// </summary>
        private ISubScene _CurrentSubScene
        {
            get { return LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// 現在の部品設定です。
        /// </summary>
        private IPartsSettings _IPartsSettings
        {
            get { return this._CurrentSubScene != null ?
                this._CurrentSubScene.IPartsSettings : null; }
        }

        /// <summary>
        /// 現在のサブシーンの部品種類です。
        /// </summary>
        private string _CurrentPartsKindName
        {
            get
            {
                return this._CurrentSubScene != null ?
                    this._CurrentSubScene.IPartsSettings.Name : string.Empty;
            }
        }

        /// <summary>
        /// 現在のサブシーンの部品種類です。
        /// </summary>
        private IControlSettings _CurrentControlSettings
        {
            set
            {
                if (_currentControlSettings != value)
                {
                    _currentControlSettings = value;
                    InitializeContorlUI_();
                }
            }

            get
            {
                return _currentControlSettings;
            }
        }

        /// <summary>
        /// 表示されている部品種類です。
        /// </summary>
        private string _ViewPartsKindName
        {
            get;
            set;
        }

        #region LEToolWindow メンバ

        //----------------------------------------------------------
        // LEToolWindow メンバ
        //----------------------------------------------------------

        /// <summary>
        /// シーン変更イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(object sender, SceneModifyEventArgs e)
        {
            switch (e.Target)
            {
                // ペイン更新
                case SceneModifyEventArgs.Kind.PaneModify:
                case SceneModifyEventArgs.Kind.PaneRemove:
                case SceneModifyEventArgs.Kind.PaneAdd:
                case SceneModifyEventArgs.Kind.PaneAnimAddRemove:
                    {
                        InitializeContorlCombo_();
                        UpdateDlgState();
                        break;
                    }
                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                case SceneModifyEventArgs.Kind.AnimSectionTagSet:
                    {
                        this.InitializeDlgState();
                        break;
                    }
                case SceneModifyEventArgs.Kind.PartsSettings:
                    {
                        this.InitializeDlgState();
                        break;
                    }
                // サブシーンが削除された
                case SceneModifyEventArgs.Kind.SubSceneRemoved:
                    {
                        break;
                    }
                // クリップボード状態更新
                case SceneModifyEventArgs.Kind.ClipBoardModify:
                    {
                        // メニューの状態を更新します。
                        break;
                    }
                // 選択セット変更ハンドラ
                case SceneModifyEventArgs.Kind.SelectedSetModify:
                    {
                        // メニューの状態を更新します。
                        break;
                    }
                default:
                    {
                        break;
                    }
            }

            base.OnSceneModifyHandler(sender, e);
        }

        /// <summary>
        /// 可視状態を変更できるかどうかです。
        /// </summary>
        public override bool AllowToChangeVisible { get { return true; } }

        /// <summary>
        /// カスタムショートカットキーです。
        /// </summary>
        public override Keys CustomShortcut { get { return Keys.Control | Keys.F12; } }

        //----------------------------------------------------------
        // データ保存
        //----------------------------------------------------------

        /// <summary>
        /// 保存(固有の情報を保存します。)
        /// </summary>
        public override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            setting.SaveColumnPotisionSettings(_lvwAnim);
            setting.SaveColumnPotisionSettings(_lvwPane);
            setting.SaveColumnPotisionSettings(_lvwParam);
            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 読み込み(固有の情報を読み込みます。)
        /// </summary
        public override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            setting.LoadColumnPotisionSettings(_lvwAnim);
            setting.LoadColumnPotisionSettings(_lvwPane);
            setting.LoadColumnPotisionSettings(_lvwParam);
            base.LoadSetting(setting, option);
        }

        #endregion LEToolWindow メンバ

        #region ウインドウ更新関係

        /// <summary>
        /// 部品パネルを有効にします
        /// </summary>
        private void DiactivePartsPanal()
        {
            this._lblNoPartsNotice.Visible = true;

            this._lblPartsDescriptionName.Text = string.Empty;
            this._tbxDescription.Text = string.Empty;

            this._pnlParts.Visible = false;
            this._pnlParts.Enabled = false;
        }

        /// <summary>
        /// 部品パネルを有効にします
        /// </summary>
        private void DiactiveContorlPanal()
        {
            this._lvwAnim.ResetSelectedItems();
            this._lvwPane.ResetSelectedItems();

            this._lblControlName.Text = string.Empty;
            this._lblControlDesc.Text = string.Empty;
            _ttpMain.SetToolTip(this._lblControlDesc, this._lblControlDesc.Text);

            this._pnlControl.Visible = false;
            this._pnlControl.Enabled = false;
            this._pnlControlTag.Visible = this.IsMultiControlMode;

            this._pnlMultiControl.Visible = this.IsMultiControlMode;
            this._btnCtrlModify.Enabled = false;
            this._tabMultiControl.Visible = false;
            this._lblNoControlNotice.Visible = true;
        }

        /// <summary>
        /// 部品パネルを無効にします
        /// </summary>
        private void ActivePartsPanal()
        {
            this._lblNoPartsNotice.Visible = false;

            this._pnlParts.Visible = true;
            this._pnlParts.Enabled = true;
        }

        /// <summary>
        /// 部品パネルを無効にします
        /// </summary>
        private void ActiveContorlPanal()
        {
            this._pnlControl.Visible = true;
            this._pnlControl.Enabled = true;

            this._pnlControlTag.Visible = true;

            this._pnlMultiControl.Visible = this.IsMultiControlMode;
            this._btnCtrlModify.Enabled = true;
            this._tabMultiControl.Visible = true && this.IsMultiControlMode;
            this._lblNoControlNotice.Visible = false;
        }

        /// <summary>
        /// エラー状態更新
        /// </summary>
        private void UpdateDlgErrorState()
        {
            if (_CurrentControlSettings == null)
            {
                return;
            }

            // TODO:複数コントロール(他たくさん)
            if (_CurrentControlSettings.ContainsUnsetAnimationTag())
            {
                _erpAnimation.SetError(_pnlAnim.HeaderContorol, LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_ANM_NOTSET"));
            }
            else if (_CurrentControlSettings.ContainsNotFoundAnimationTag())
            {
                _erpAnimation.SetError(_pnlAnim.HeaderContorol, LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_ANM_CANTFIND"));
            }
            else if (_CurrentControlSettings.ContainsNotExportAnimationTag())
            {
                _erpAnimation.SetError(_pnlAnim.HeaderContorol, LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_ANM_NOTEXPORT"));
            }
            else
            {
                _erpAnimation.SetError(_pnlAnim.HeaderContorol, "");
            }

            if (_CurrentControlSettings.ContainsUnsetPane())
            {
                _erpAnimation.SetError(_pnlPane.HeaderContorol, LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_PANE_NOTSET"));
            }
            else if (_CurrentControlSettings.ContainsNotFoundPane())
            {
                _erpAnimation.SetError(_pnlPane.HeaderContorol, LECoreStringResMgr.Get("LECORE_VALIDATION_CTRL_PANE_CANTFIND"));
            }
            else
            {
                _erpAnimation.SetError(_pnlPane.HeaderContorol, "");
            }
        }

        /// <summary>
        ///
        /// </summary>
        string GetParamAnimDesc_(string paramAnimName)
        {
            var controlSetting = _CurrentControlSettings != null ? LayoutEditorCore.Scene.FindPartsControlSetting(_CurrentControlSettings.Name) : null;
            if (controlSetting == null)
            {
                return _DescStringNotFoundControl;
            }
            else
            {
                var animTemplate = controlSetting.AnimationNames.FirstOrDefault((anim) => anim.Name == paramAnimName);
                if (animTemplate == null)
                {
                    return _DescStringNotFoundElement;
                }
                else
                {
                    return animTemplate.Description;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        string GetParamPaneDesc_(string paramPaneName)
        {
            var controlSetting = _CurrentControlSettings != null ? LayoutEditorCore.Scene.FindPartsControlSetting(_CurrentControlSettings.Name) : null;
            if (controlSetting == null)
            {
                return _DescStringNotFoundControl;
            }
            else
            {
                var paneTemplate = controlSetting.PaneNames.FirstOrDefault((pane) => pane.Name == paramPaneName);
                if (paneTemplate == null)
                {
                    return _DescStringNotFoundElement;
                }
                else
                {
                    return paneTemplate.Description;
                }
            }
        }

        /// <summary>
        /// 状態更新
        /// </summary>
        private void UpdateControlState_()
        {
            if (_CurrentControlSettings == null)
            {
                return;
            }

            foreach (ListViewItem item in this._lvwAnim.Items)
            {
                var paramAnim = item.Tag as IParamaterAnimaiton;
                item.SubItems[0].Text = paramAnim.Name;
                item.SubItems[1].Text = paramAnim.TagName;
                item.SubItems[2].Text = GetParamAnimDesc_(paramAnim.Name);
                item.SubItems[3].Text = GetRequiredFlagString_(paramAnim.IsRequired);

                item.ForeColor = (
                    item.SubItems[2].Text != _DescStringNotFoundControl &&
                    item.SubItems[2].Text != _DescStringNotFoundElement &&
                    !AnimFrameSectionSetHelper.CheckNotExportAnimationTag(_IPartsSettings.Owner, paramAnim.TagName) &&
                    paramAnim.IsValid(_CurrentSubScene.IAnimFrameSectionSet)
                    ) ? SystemColors.WindowText : Color.Red;
            }

            foreach (ListViewItem item in this._lvwPane.Items)
            {
                var paramPane = item.Tag as IParamaterPane;
                item.SubItems[0].Text = paramPane.Name;
                item.SubItems[1].Text = paramPane.PaneName;
                item.SubItems[2].Text = GetParamPaneDesc_(paramPane.Name);
                item.SubItems[3].Text = GetRequiredFlagString_(paramPane.IsRequired);

                item.ForeColor = (
                    item.SubItems[2].Text != _DescStringNotFoundControl &&
                    item.SubItems[2].Text != _DescStringNotFoundElement &&
                    paramPane.IsValid(_CurrentSubScene)
                    ) ? SystemColors.WindowText : Color.Red;
            }

            Event_LvwAnim_SelectedIndexChanged(null, null);
            Event_LvwPane_SelectedIndexChanged(null, null);

            UpdateSetFuncAnimationControls();
        }

        /// <summary>
        /// 状態更新
        /// </summary>
        private void UpdateDlgState()
        {
            if (_CurrentSubScene == null)
            {
                return;
            }

            UpdateControlState_();

            this.InitializePropertyList();

            // エラー
            this.UpdateDlgErrorState();
        }

        /// <summary>
        /// 上書きプロパティに変更があったか判定します。
        /// </summary>
        private bool AreOverridePorpertiesChaged()
        {
            if (_CurrentSubScene == null)
            {
                return false;
            }

            if (_lvwParam.Items.Count != _CurrentSubScene.IPartsSettings.OverridePorperties.Count())
            {
                return true;
            }

            for (int i = 0; i < _lvwParam.Items.Count; i++)
            {
                var foundPane = _CurrentSubScene.FindPaneByName(_lvwParam.Items[i].Text);
                if (!object.ReferenceEquals(_lvwParam.Items[i].Tag, foundPane))
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 必須フラグを表す文字列を取得します。
        /// </summary>
        private string GetRequiredFlagString_(bool IsRequired)
        {
            return IsRequired ? LayoutEditor.StringResMgr.Get("PARTS_LISTVIEW_REQUIRED_ON") : "---";
        }

        /// <summary>
        /// ウインドウ状態を初期化します。
        /// </summary>
        private void InitializeDlgState()
        {
            // 対象が無ければ、無効にして終わる。
            if (_CurrentSubScene == null)
            {
                this.DiactivePartsPanal();
                this.DiactiveContorlPanal();

                this._pnlPartsInit.Enabled = false;
                this._pnlMultiControl.Enabled = false;

                // ウインドウサイズ再計算
                SetTableLayoutHeight_(null, null);

                return;
            }

            this._pnlPartsInit.Enabled = true;
            this._pnlMultiControl.Enabled = true;

            // 部品関連
            if (_CurrentSubScene.IPartsSettings.IsInitialState)
            {
                this.DiactivePartsPanal();
                this._ViewPartsKindName = _CurrentPartsKindName;
            }
            else
            {
                this.ActivePartsPanal();
                InitializePartsUI_();
                this._ViewPartsKindName = _CurrentPartsKindName;
            }

            // コントロール関連
            if (_CurrentSubScene.IControlSettings.Count() <= 0)
            {
                this.DiactiveContorlPanal();
                _CurrentControlSettings = null;
            }
            else
            {
                this.ActiveContorlPanal();

                if (!_CurrentSubScene.IControlSettings.Contains(_CurrentControlSettings))
                {
                    _CurrentControlSettings = _CurrentSubScene.IControlSettings.First();
                }

                // コントロール情報の初期化
                InitializeCurrentContorolTab_();
                InitializeContorlUI_();
            }

            // 一度更新処理を実行しておく
            this.UpdateDlgState();

            // ウインドウサイズ再計算
            SetTableLayoutHeight_(null, null);
        }

        /// <summary>
        ///
        /// </summary>
        private void InitializePartsUI_()
        {
            this._lblPartsDescriptionName.Text = _CurrentSubScene.IPartsSettings.DescriptionName;
            this._tbxDescription.Text = _CurrentSubScene.IPartsSettings.Description;
        }

        /// <summary>
        /// カレントコントロール選択タブの初期化
        /// </summary>
        private void InitializeCurrentContorolTab_()
        {
            var oldSelectedTab = _CurrentControlSettings;
            TabPage[] oldPages = _tabMultiControl.TabPages.Cast<TabPage>().ToArray();

            List<TabPage> newPages = new List<TabPage>();
            foreach (var ctrlSettings in _CurrentSubScene.IControlSettings)
            {
                TabPage page = new TabPage(ctrlSettings.DescriptionName);
                page.Tag = ctrlSettings;
                newPages.Add(page);
            }

            if (newPages.Count != oldPages.Length)
            {
                _tabMultiControl.TabPages.Clear();
                _tabMultiControl.TabPages.AddRange(newPages.ToArray());
                _tabMultiControl.SelectByData(oldSelectedTab);
            }
            else
            {
                bool changed = false;
                for (int i = 0; i < newPages.Count; i++)
                {
                    var oldCtrl = oldPages[i].Tag as IControlSettings;
                    var newCtrl = newPages[i].Tag as IControlSettings;

                    changed |= oldCtrl != newCtrl || oldPages[i].Text != newPages[i].Text;
                }

                if (changed)
                {
                    _tabMultiControl.TabPages.Clear();
                    _tabMultiControl.TabPages.AddRange(newPages.ToArray());
                    _tabMultiControl.SelectByData(oldSelectedTab);
                }
            }
        }

        /// <summary>
        /// 入力候補コンボボックスを初期化
        /// </summary>
        private void InitializeContorlCombo_()
        {
            if (this._CurrentSubScene == null)
            {
                return;
            }

            // 入力候補コンボボックスを初期化
            {
                // 区間タグコンボボックスの初期化。
                {
                    this._cmbAnimTag.Items.Clear();
                    foreach (var animSection in
                    this._CurrentSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
                    {
                        this._cmbAnimTag.Items.Add(animSection.Name);
                    }
                    this._cmbAnimTag.Items.Add(string.Empty);
                    this._cmbAnimTag.ExpandDropDownWidth();
                }

                // ペインコンボボックスの初期化。
                {
                    this._cmbPanes.Items.Clear();
                    foreach (var pane in this._CurrentSubScene.IPaneArray)
                    {
                        this._cmbPanes.Items.Add(pane.PaneName);
                    }
                    this._cmbPanes.Items.Add(string.Empty);
                    this._cmbPanes.ExpandDropDownWidth();
                }
            }
        }

        /// <summary>
        /// コントロール情報UIの初期化
        /// </summary>
        private void InitializeContorlUI_()
        {
            InitializeContorlCombo_();

            if (_CurrentControlSettings == null)
            {
                return;
            }

            _lblControlName.Text = _CurrentControlSettings.DescriptionName;
            _lblControlDesc.Text = _CurrentControlSettings.Description;
            _ttpMain.SetToolTip(this._lblControlDesc, this._lblControlDesc.Text);

            // Animations
            var animSelectItemCache = new ListViewHelper.SelectItemCache(this._lvwAnim);
            this._lvwAnim.Items.Clear();
            foreach (var anim in _CurrentControlSettings.Animations)
            {
                var animItem = new ListViewItem(new string[] { anim.Name, anim.TagName, GetParamAnimDesc_(anim.Name), GetRequiredFlagString_(anim.IsRequired) });
                animItem.Tag = anim;
                this._lvwAnim.Items.Add(animItem);
            }
            animSelectItemCache.Apply(this._lvwAnim);

            // Panes
            var paneSelectItemCache = new ListViewHelper.SelectItemCache(this._lvwPane);
            this._lvwPane.Items.Clear();
            this._lvwPane.SelectedIndices.Clear();
            foreach (var pane in _CurrentControlSettings.Panes)
            {
                var item = new ListViewItem(new string[] { pane.Name, pane.PaneName, GetParamPaneDesc_(pane.Name), GetRequiredFlagString_(pane.IsRequired) });
                item.Tag = pane;
                this._lvwPane.Items.Add(item);
            }
            paneSelectItemCache.Apply(this._lvwPane);

            // ユーザデータ
            IEnumerable<ExUserDataPreset> presets = ViewManager != null ? ViewManager.AppSetting.ProjectSettings.ExUserDataPresets : new List<ExUserDataPreset>();
            _pnlUserData.Initialize(_CurrentControlSettings.IUserDataHolder, false, false, false, presets, (preset) => preset.IsTragetsContorol(_CurrentControlSettings.Name));
            _pnlUserData.UpdateProperty();

            // エラー
            this.UpdateDlgErrorState();
        }

        /// <summary>
        /// 上書きプロパティリストの初期化をします。
        /// </summary>
        private void InitializePropertyList()
        {
            _lvwParam.Items.Clear();
            if (_CurrentSubScene != null)
            {
                foreach (var property in _CurrentSubScene.IPartsSettings.OverridePorperties)
                {
                    var item = new ListViewItem(new string[]{
                        property.TargetName, property.PaneKind.ToString(), property.Description });
                    item.Tag = _CurrentSubScene.FindPaneByName(property.TargetName);
                    item.ForeColor = (item.Tag != null) ? SystemColors.WindowText : Color.Red;
                    item.ToolTipText = property.Description;
                    _lvwParam.Items.Add(item);
                }
            }
        }

        #endregion

        //----------------------------------------------------------
        // イベント
        //----------------------------------------------------------

        /// <summary>
        /// コメント編集
        /// </summary>
        private void Event_TbxDescription_TextChanged(object sender, EventArgs e)
        {
            if (_IPartsSettings == null) return;

            var partsSettingsMnp = new PartsSettingsManipulator();
            partsSettingsMnp.BindTarget(_IPartsSettings);
            partsSettingsMnp.SetDescriptions(
                        _IPartsSettings.DescriptionName,
                        _tbxDescription.Text);
        }

        /// <summary>
        /// アニメーションリストの選択変更。
        /// </summary>
        private void Event_LvwAnim_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (this._lvwAnim.SelectedIndices.Count > 0)
            {
                // UI文字をセット
                // UIをセット(Enable)
                //   コンボボックス初期化
                var paramAnim = this._lvwAnim.SelectedItems[0].Tag as IParamaterAnimaiton;
                _lblAnimItem.Text = string.Format("[{0}]", paramAnim.Name);

                _cmbAnimTag.SelectByText(paramAnim.TagName);
                _cmbAnimTag.Tag = paramAnim.Name;

                string desc = GetParamAnimDesc_(paramAnim.Name);
                if (desc == _DescStringNotFoundElement)
                {
                    _lblAnimItemDescContent.Text = LayoutEditor.StringResMgr.Get("PARTSWND_SET_PROPERTYPANE_NOTFIND_ELEMENT");
                    _lblAnimItemDescContent.ForeColor = Color.Red;
                }
                else if (desc == _DescStringNotFoundControl)
                {
                    _lblAnimItemDescContent.Text = LayoutEditor.StringResMgr.Get("PARTSWND_SET_PROPERTYPANE_NOTFIND_CONTROL");
                    _lblAnimItemDescContent.ForeColor = Color.Red;
                }
                else
                {
                    _lblAnimItemDescContent.Text = desc;
                    _lblAnimItemDescContent.ForeColor = SystemColors.ControlText;
                }
            }
            else
            {
                // UI文字を無効な文字に、
                // 表示をリセット
                _lblAnimItem.Text = LayoutEditor.StringResMgr.Get("PARTS_TARGET_PARAM_NONE");
                _cmbAnimTag.SelectByText(string.Empty);
                _cmbAnimTag.Tag = null;
                _lblAnimItemDescContent.Text = StringResMgr.Get("PARTSWND_SET_PROPERTYANIM");
                _lblAnimItemDescContent.ForeColor = SystemColors.GrayText;
            }
        }

        /// <summary>
        /// ペインリストの選択変更。
        /// </summary>
        private void Event_LvwPane_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (this._lvwPane.SelectedIndices.Count > 0)
            {
                var paramPane = this._lvwPane.SelectedItems[0].Tag as IParamaterPane;
                _lblPaneItem.Text = string.Format("[{0}]", paramPane.Name);

                string desc = GetParamPaneDesc_(paramPane.Name);
                if (desc == _DescStringNotFoundElement)
                {
                    _lblPaneDescContent.Text = LayoutEditor.StringResMgr.Get("PARTSWND_SET_PROPERTYPANE_NOTFIND_ELEMENT");
                    _lblPaneDescContent.ForeColor = Color.Red;
                }
                else if (desc == _DescStringNotFoundControl)
                {
                    _lblPaneDescContent.Text = LayoutEditor.StringResMgr.Get("PARTSWND_SET_PROPERTYPANE_NOTFIND_CONTROL");
                    _lblPaneDescContent.ForeColor = Color.Red;
                }
                else
                {
                    _lblPaneDescContent.Text = desc;
                    _lblPaneDescContent.ForeColor = SystemColors.ControlText;
                }
            }
            else
            {
                _lblPaneItem.Text = LayoutEditor.StringResMgr.Get("PARTS_TARGET_PARAM_NONE");
                _lblPaneDescContent.Text = StringResMgr.Get("PARTSWND_SET_PROPERTYPANE");
                _lblPaneDescContent.ForeColor = SystemColors.GrayText;
            }
        }

        /// <summary>
        /// ペイン名テキストボックス ドラッグ開始。
        /// </summary>
        private void Event_TbxPaneName_DragEnter(object sender, DragEventArgs e)
        {
            // TreeNode以外のドロップは処理しません。
            if (!e.Data.GetDataPresent(typeof(TreeNode[]).ToString(), false))
            {
                return;
            }

            var nodes = e.Data.GetData(typeof(TreeNode[])) as TreeNode[];
            if (nodes.Count() > 1)
            {
                return;
            }

            var listView = sender as ListView;
            if(listView == null)
            {
                return;
            }

            e.Effect = e.AllowedEffect;
        }

        /// <summary>
        ///
        /// </summary>
        private void _lvwPane_DragOver(object sender, DragEventArgs e)
        {
            var listView = sender as ListView;
            if (listView == null)
            {
                e.Effect = DragDropEffects.None;
                return;
            }

            Point clientPos = listView.PointToClient(new Point(e.X, e.Y));
            var item = listView.GetItemAt(clientPos.X, clientPos.Y);
            if (item == null || item.Tag == null)
            {
                e.Effect = DragDropEffects.None;
                return;
            }

            e.Effect = e.AllowedEffect;
        }

        /// <summary>
        /// ペイン名テキストボックス ドロップ。
        /// </summary>
        private void Evnet_TbxPaneName_DragDrop(object sender, DragEventArgs e)
        {
            var nodes = e.Data.GetData(typeof(TreeNode[])) as TreeNode[];
            Ensure.Argument.NotNull(nodes);
            Ensure.Argument.True(nodes.Count() == 1);

            string paneName = nodes[0].Text;
            if(SubSceneHelper.FindPaneByName(_CurrentSubScene, paneName) == null)
            {
                return;
            }

            Point clientPos = this._lvwPane.PointToClient(new Point(e.X, e.Y));
            var targetItem = this._lvwPane.GetItemAt(clientPos.X, clientPos.Y);
            if (targetItem == null)
            {
                return;
            }

            targetItem.Selected = true;
            var paramPane = targetItem.Tag as IParamaterPane;
            var ctrlSettingsMnp = new ControlSettingsManipulator();
            ctrlSettingsMnp.BindTarget(_CurrentControlSettings);
            ctrlSettingsMnp.ModifyParamaterPane(paramPane, paneName);
        }

        // 新規に作る処理。
        // OFFにする機能。DescriptionName名を編集する機能。

        /// <summary>
        /// パーツ変更ボタン押下
        /// </summary>
        private void Event_BtnPartsChange_Click(object sender, EventArgs e)
        {
            Ensure.Operation.ObjectNotNull(_CurrentSubScene);
            Ensure.Operation.True(!this.IsMultiControlMode);

            // ダイアログを表示する。
            // 結果を受けて、PartsSettingを初期化する処理を実行する。
            SetPartsKindDialog setPartsKindDialog = new SetPartsKindDialog();
            setPartsKindDialog.Owner = this;

            string ctrlName = _CurrentControlSettings != null ? _CurrentControlSettings.Name : string.Empty;
            string ctrlDescriptionName = _IPartsSettings.DescriptionName;
            setPartsKindDialog.Initialize(ctrlName, ctrlDescriptionName);

            DialogResult result = setPartsKindDialog.ShowDialog();
            if (result == DialogResult.OK)
            {
                // 設定変更を実際に反映します。
                var controlSetting = setPartsKindDialog.PartsControlSetting;
                if (controlSetting != null)
                {
                    if (_CurrentControlSettings == null || setPartsKindDialog.IsChanged)
                    {
                        _CurrentSubScene.BeginMassiveModify();
                        {
                            // コントロールが無ければ、追加する
                            if (_CurrentControlSettings == null)
                            {
                                var ctrlSettingsSetMnp = new ControlSettingsSetManipulator();
                                ctrlSettingsSetMnp.BindTarget(_CurrentSubScene.IControlSettingsSet);
                                ctrlSettingsSetMnp.Add(controlSetting, controlSetting.Name, controlSetting.Description);
                                _CurrentControlSettings = _CurrentSubScene.IControlSettingsSet.Items.First();
                            }

                            // 複数コントロールでなければ部品設定も同時に行う
                            var partsSettingsMnp = new PartsSettingsManipulator();
                            partsSettingsMnp.BindTarget(_IPartsSettings);
                            partsSettingsMnp.Initialze(controlSetting, setPartsKindDialog.PartsDescriptionName, _IPartsSettings.Description);

                            var oldProperties = _IPartsSettings.OverridePorperties.ToList();
                            CopyPartsOldSettings(partsSettingsMnp, oldProperties);

                            ResetControlIfNotPartsLayout_();

                            // コントロール設定
                            {
                                var oldAnims = _CurrentControlSettings.Animations.ToList();
                                var oldPanes = _CurrentControlSettings.Panes.ToList();

                                var ctrlSettingsMnp = new ControlSettingsManipulator();
                                ctrlSettingsMnp.BindTarget(_CurrentControlSettings);

                                ctrlSettingsMnp.InitialzeControl(
                                    controlSetting,
                                    controlSetting.Name,
                                    controlSetting.Description);

                                CopyOldSettings(ctrlSettingsMnp, oldAnims, oldPanes);
                            }
                        }
                        _CurrentSubScene.EndMassiveModify();
                    }
                    else
                    {

                        var ctrlSettingsMnp = new ControlSettingsManipulator();
                        ctrlSettingsMnp.BindTarget(_CurrentControlSettings);

                        ctrlSettingsMnp.SetDescriptions(
                            _CurrentControlSettings.Name,
                            _CurrentControlSettings.Description);
                    }
                }
                else
                {
                    var partsSettingsMnp = new PartsSettingsManipulator();
                    partsSettingsMnp.BindTarget(_IPartsSettings);
                    partsSettingsMnp.Reset();

                    ResetControlIfNotPartsLayout_();
                }
            }
        }

        /// <summary>
        /// コントロール変更ボタン押下
        /// </summary>
        private void Event_BtnControlChange_Click(object sender, EventArgs e)
        {
            Ensure.Operation.ObjectNotNull(_CurrentSubScene);

            // ダイアログを表示する。
            // 結果を受けて、PartsSettingを初期化する処理を実行する。
            SetPartsKindDialog setPartsKindDialog = new SetPartsKindDialog();
            setPartsKindDialog.Owner = this;

            string ctrlName = _CurrentControlSettings != null ? _CurrentControlSettings.Name : string.Empty;
            string ctrlDescriptionName = _CurrentControlSettings != null ? _CurrentControlSettings.DescriptionName : string.Empty;
            setPartsKindDialog.InitializeAsControlSelectMode(ctrlName, ctrlDescriptionName, _CurrentSubScene.IControlSettingsSet.GetUniqueControlSettingsName);
            setPartsKindDialog.InitFirstControlItemLabel(StringResMgr.Get("PARTS_ITEMTAG_DELETE_CTRL"));


            // ダイアログ表示
            DialogResult result = setPartsKindDialog.ShowDialog();
            if (result != DialogResult.OK)
            {
                return;

            }

            // 設定変更を実際に反映します。
            var controlSetting = setPartsKindDialog.PartsControlSetting;
            if (controlSetting == null)
            {
                // 削除する
                var ctrlSettingsSetMnp = new ControlSettingsSetManipulator();
                ctrlSettingsSetMnp.BindTarget(_CurrentSubScene.IControlSettingsSet);
                ctrlSettingsSetMnp.Remove(_CurrentControlSettings);

                return;
            }

            // 設定変更
            var ctrlSettingsMnp = new ControlSettingsManipulator();
            ctrlSettingsMnp.BindTarget(_CurrentControlSettings);
            if (controlSetting.Name != _CurrentControlSettings.Name)
            {
                _CurrentSubScene.BeginMassiveModify();
                {
                    ctrlSettingsMnp.InitialzeControl(controlSetting, setPartsKindDialog.PartsDescriptionName, controlSetting.Description);

                    CopyOldSettings(ctrlSettingsMnp, _CurrentControlSettings.Animations.ToList(), _CurrentControlSettings.Panes.ToList());
                }
                _CurrentSubScene.EndMassiveModify();
            }
            else
            {
                ctrlSettingsMnp.SetDescriptions(setPartsKindDialog.PartsDescriptionName, _CurrentControlSettings.Description);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void ResetControlIfNotPartsLayout_()
        {
            if (_IPartsSettings.IsInitialState)
            {
                var ctrlSettingsSetMnp = new ControlSettingsSetManipulator();
                ctrlSettingsSetMnp.BindTarget(_CurrentSubScene.IControlSettingsSet);
                ctrlSettingsSetMnp.Remove(_CurrentControlSettings);
            }
        }

        /// <summary>
        /// コントロール設定をコピーします。
        /// </summary>
        private void CopyOldSettings(ControlSettingsManipulator ctrlSettingsMnp, List<IParamaterAnimaiton> oldAnims, List<IParamaterPane> oldPanes)
        {
            oldAnims.ForEach((anim) =>
            {
                var targetAnim = _CurrentControlSettings.Animations.FirstOrDefault((newAnim) => newAnim.Name == anim.Name);
                if (targetAnim != null)
                {
                    ctrlSettingsMnp.ModifyParamaterAnimation(targetAnim, anim.TagName);
                }
            });
            oldPanes.ForEach((pane) =>
            {
                var targetPane = _CurrentControlSettings.Panes.FirstOrDefault((newPane) => newPane.Name == pane.Name);
                if (targetPane != null)
                {
                    ctrlSettingsMnp.ModifyParamaterPane(targetPane, pane.PaneName);
                }
            });
        }

        /// <summary>
        /// 部品設定をコピーします。
        /// </summary>
        private void CopyPartsOldSettings(PartsSettingsManipulator partsSettingsMnp, List<IPartsPropaertySettings> oldProperties)
        {
            var oldPropertySources = PartsSettingsHelper.GetPartsPropaertySettingsSources(_CurrentSubScene, oldProperties);
            partsSettingsMnp.SetOverrideProperties(oldPropertySources);
        }

        /// <summary>
        /// プロパティ設定ボタン
        /// </summary>
        private void Event_BtnSetParamater_Click(object sender, EventArgs e)
        {
            SetPartsPropaertyDlg dlg = new SetPartsPropaertyDlg();
            dlg.Initialize(_CurrentSubScene);
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                var partsSettingsMnp = new PartsSettingsManipulator();
                partsSettingsMnp.BindTarget(_CurrentSubScene.IPartsSettings);
                partsSettingsMnp.SetOverrideProperties(dlg.OverwritePaneNames);
            }
        }

        /// <summary>
        /// タグ編集開始
        /// </summary>
        private bool TryBeginChangeAnimTag_(InPlaceEditListView lvw, Control editCtrl, SubItemEventArgs e)
        {
            // 画像フォーマット項目がクリックされ、かつ詳細表示の状態なら処理する。
            if (e.SubItem != _EditableSubItemIndex || lvw.View != View.Details)
            {
                return false;
            }

            // 編集開始
            lvw.Controls.Add(editCtrl);
            lvw.StartEditing(editCtrl, e.Item, e.SubItem);

            return true;
        }

        /// <summary>
        /// タグ編集
        /// </summary>
        private void Event_lvwAnim2_SubItemClicked_(object sender, LayoutEditor.Controls.SubItemEventArgs e)
        {
            TryBeginChangeAnimTag_(_lvwAnim, _cmbAnimTag, e);
        }

        /// <summary>
        /// タグ編集
        /// </summary>
        private void Event_lvwPane_SubItemClicked_(object sender, LayoutEditor.Controls.SubItemEventArgs e)
        {
            TryBeginChangeAnimTag_(_lvwPane, _cmbPanes, e);
        }

        /// <summary>
        /// タグ編集
        /// </summary>
        private void Event_lvwAnim2_SubItemEndEditing_(object sender, LayoutEditor.Controls.SubItemEndEditingEventArgs e)

        {
            if (_CurrentSubScene == null)
            {
                return;
            }

            var paramAnim = e.Item.Tag as IParamaterAnimaiton;
            if (paramAnim == null)
            {
                return;
            }

            // 新しいタグ名を設定します。
            var ctrlSettingsMnp = new ControlSettingsManipulator();
            ctrlSettingsMnp.BindTarget(_CurrentControlSettings);
            ctrlSettingsMnp.ModifyParamaterAnimation(paramAnim, _cmbAnimTag.Text);
        }

        /// <summary>
        /// タグ編集
        /// </summary>
        private void Event_lvwPane_SubItemEndEditing_(object sender, LayoutEditor.Controls.SubItemEndEditingEventArgs e)
        {
            if (_CurrentSubScene == null)
            {
                return;
            }

            var paramPane = e.Item.Tag as IParamaterPane;
            if (paramPane == null)
            {
                return;
            }

            // 新しいタグ名を設定します。
            var ctrlSettingsMnp = new ControlSettingsManipulator();
            ctrlSettingsMnp.BindTarget(_CurrentControlSettings);
            ctrlSettingsMnp.ModifyParamaterPane(paramPane, _cmbPanes.Text);
        }

        /// <summary>
        /// パネルの縮小状態が変わったときの処理
        /// </summary>
        private void Event_pnl_CollapsedChanged(object sender, EventArgs e)
        {
            // 同時にスクロールバーの表示が切り替わるときに横幅が小さくなる不具合の対応
            if (sender is CollapsiblePanel)
            {
                CollapsiblePanel panel = (CollapsiblePanel)sender;
                panel.Width = panel.Parent.Width - (panel.Location.X + panel.Margin.Right);
            }

            SetTableLayoutHeight_(this, null);
        }

        /// <summary>
        /// サイズ変更
        /// </summary>
        private void SetTableLayoutHeight_(object sender, EventArgs e)
        {
            this.SuspendLayout();

            // パネルサイズの調整
            _pnlParts.Height = _pnlParam.Bottom + 4;
            _pnlControl.Height = _lblControlDesc.Height + _pnlPane.Height + _pnlAnim.Height + _lblUserData.Height + _pnlUserData.Height;

            // _pnlControl 中のコントロールの位置調整
            _pnlPane.Top = _lblControlDesc.Bottom;
            _pnlAnim.Top = _pnlPane.Bottom;
            _lblUserData.Top = _pnlAnim.Bottom;
            _pnlUserData.Top = _lblUserData.Bottom;

            // 表示状態に合わせて決める。
            // 注意: ウインドウ自体の表示状態の影響も受ける。
            Control bottomPanel = _pnlBase.Controls.Cast<Control>().LastOrDefault((ctrl) => ctrl != null && ctrl.Visible);

            if (bottomPanel != null)
            {
                _pnlBase.Height = bottomPanel.Bottom;
            }

            this.ResumeLayout();
        }

        /// <summary>
        /// 編集コンボ、オーナー描画
        /// </summary>
        private void Event_cmbPanes_DrawItem_(object sender, DrawItemEventArgs e)
        {
            if (e.Index == -1) return;

            // 背景を描画
            e.Graphics.FillRectangle(
                (((e.State & DrawItemState.Selected) != 0) ?
                    SystemBrushes.Highlight :     // 選択時の背景色
                    SystemBrushes.Window),       // 非選択時の背景色
                e.Bounds);

            string itemStr = (string)((ComboBox)sender).Items[e.Index];
            if (string.IsNullOrEmpty(itemStr) && ((e.State & DrawItemState.ComboBoxEdit) == 0))
            {
                int imgW = e.Bounds.Height;
                int imgH = imgW;
                e.Graphics.DrawImage(_cancelImage, e.Bounds.X, e.Bounds.Y, imgW, imgH);
                e.Graphics.DrawString(StringResMgr.Get("TAG_RESET"), e.Font, SystemBrushes.ControlText, e.Bounds.X + imgW + 2, e.Bounds.Y + 2);
            }
            else
            {
                e.Graphics.DrawString(itemStr, e.Font, SystemBrushes.ControlText, e.Bounds.X + 2, e.Bounds.Y + 2);
            }
        }

        /// <summary>
        /// コントロール追加ボタン
        /// </summary>
        private void Event_btnAddCtrl_Click(object sender, EventArgs e)
        {
            // ダイアログを表示する。
            // 結果を受けて、PartsSettingを初期化する処理を実行する。
            SetPartsKindDialog setPartsKindDialog = new SetPartsKindDialog();
            setPartsKindDialog.Owner = this;
            setPartsKindDialog.InitializeAsControlSelectMode(string.Empty, string.Empty, _CurrentSubScene.IControlSettingsSet.GetUniqueControlSettingsName);
            setPartsKindDialog.InitFirstControlItemLabel(StringResMgr.Get("PARTS_ITEMTAG_SELECT_CTRL"));

            DialogResult result = setPartsKindDialog.ShowDialog();
            if (result == DialogResult.OK)
            {
                var ctrlSettingsMnp = new ControlSettingsSetManipulator();
                ctrlSettingsMnp.BindTarget(_CurrentSubScene.IControlSettingsSet);
                ctrlSettingsMnp.Add(
                    setPartsKindDialog.PartsControlSetting,
                    setPartsKindDialog.PartsDescriptionName,
                    setPartsKindDialog.PartsControlSetting.Description);
            }
        }

        /// <summary>
        /// カレントコントロールタブの変更
        /// </summary>
        private void Event_tabControl_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (_tabMultiControl.SelectedTab != null)
            {
                _CurrentControlSettings = _tabMultiControl.SelectedTab.Tag as IControlSettings;
            }
        }

        /// <summary>
        /// 部品設定ダイアログ
        /// </summary>
        private void Event_BtnPartsChange_MultiControlMode_Click(object sender, EventArgs e)
        {
            Ensure.Operation.ObjectNotNull(_CurrentSubScene);

            // ダイアログを表示する。
            SetPartsLayoutDialog setPartsDialog = new SetPartsLayoutDialog();
            setPartsDialog.Owner = this;
            setPartsDialog.Initialize(!_IPartsSettings.IsInitialState, _IPartsSettings.DescriptionName);

            DialogResult result = setPartsDialog.ShowDialog();
            if (result == DialogResult.OK)
            {
                var partsSettingsMnp = new PartsSettingsManipulator();
                partsSettingsMnp.BindTarget(_IPartsSettings);

                if (setPartsDialog.IsPartsLayoutEnabled)
                {
                    partsSettingsMnp.SetDescriptions(setPartsDialog.PartsDescriptionName,_IPartsSettings.Description);
                }
                else
                {
                    partsSettingsMnp.Reset();
                }
            }
        }

        /// <summary>
        /// 部品設定
        /// </summary>
        private void Event_btnPartsChange_Click(object sender, EventArgs e)
        {
            // 複数コントロールが有効かどうかで挙動を変える
            if (this.IsMultiControlMode)
            {
                Event_BtnPartsChange_MultiControlMode_Click(sender, e);
            }
            else
            {
                Event_BtnPartsChange_Click(sender, e);
            }
        }

        /// <summary>
        /// 機能アニメーションと同名のアニメーション区間タグを作成＆設定します。
        /// </summary>
        /// <param name="sender">未使用</param>
        /// <param name="e">未使用</param>
        private void Event_BtnSetFuncAnimation_Click(object sender, EventArgs e)
        {
            // ダイアログに渡す未設定の機能アニメーションリストを作成
            var anims = new List<PartsAnimSetting>();
            foreach (ListViewItem item in this._lvwAnim.Items)
            {
                var paramAnim = item.Tag as IParamaterAnimaiton;
                if (string.IsNullOrEmpty(paramAnim.TagName))
                {
                    var anim = new PartsAnimSetting
                    {
                        Anim = paramAnim,
                        StartFrame = 0,
                        EndFrame = 1,
                        Desc = GetParamAnimDesc_(paramAnim.Name),
                        IsFoundTag = false,
                    };

                    // 同名タグが作成済みだったらタグの設定情報を読み取って設定し、作成済みであることを通知するフラグを立てる
                    var sameNameAnim =
                        this._CurrentSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(
                            t => t.Name == paramAnim.Name);
                    if (sameNameAnim != null)
                    {
                        anim.StartFrame = sameNameAnim.StartFrame;
                        anim.EndFrame = sameNameAnim.EndFrame;
                        anim.Desc = sameNameAnim.Comment;
                        anim.IsFoundTag = true;
                    }

                    anims.Add(anim);
                }
            }

            // ダイアログを開く
            var dlg = new PartsAnimToTagDlg();
            dlg.Initialize(anims);
            if (dlg.ShowDialog() == DialogResult.Cancel)
            {
                return;
            }

            this._CurrentSubScene.BeginMassiveModify();

            // 処理対象としてチェックされたアニメーションに対して操作コマンドを発行する
            foreach (var item in dlg.TargetAnims)
            {
                if (!item.IsFoundTag)
                {
                    // 作成済みタグでない場合はここで作る
                    var animFrameSectionSetMnp = new AnimFrameSectionSetManipulator();
                    animFrameSectionSetMnp.BindTarget(this._CurrentSubScene.IAnimFrameSectionSet);
                    animFrameSectionSetMnp.AddNewFrameSection(item.Anim.Name, item.Desc, item.StartFrame, item.EndFrame, _CurrentSubScene);
                }

                // 同名タグを機能アニメーションに設定
                var ctrlSettingsMnp = new ControlSettingsManipulator();
                ctrlSettingsMnp.BindTarget(_CurrentControlSettings);
                ctrlSettingsMnp.ModifyParamaterAnimation(item.Anim, item.Anim.Name);
            }

            this._CurrentSubScene.EndMassiveModify();
            UpdateSetFuncAnimationControls();
        }

        /// <summary>
        /// 機能アニメーションの設定状態に応じてUIの無効化状態を更新する
        /// </summary>
        private void UpdateSetFuncAnimationControls()
        {
            bool isEnableButton = false;
            foreach (ListViewItem item in this._lvwAnim.Items)
            {
                var paramAnim = item.Tag as IParamaterAnimaiton;
                isEnableButton |= string.IsNullOrEmpty(paramAnim.TagName);
            }

            _btnSetFuncAnimation.Enabled = isEnableButton;
        }

        private void _cmbPanes_SelectedValueChanged(object sender, EventArgs e)
        {
            // フォーカスを外すと編集が完了する
            _lvwPane.Focus();
        }

        private void _cmbAnimTag_SelectedValueChanged(object sender, EventArgs e)
        {
            // フォーカスを外すと編集が完了する
            _lvwAnim.Focus();
        }

        private void PartsWindow_VisibleChanged(object sender, EventArgs e)
        {
            // サイズを調整する
            SetTableLayoutHeight_(this, null);
        }
    }
}
