﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;

namespace LayoutEditor.Forms.ToolWindows.LayoutWindow
{
    using System.Windows.Forms;
    using LayoutEditor.Forms.ToolWindows.common;
    using LECore.Manipulator;
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Util;
    using AxisSnap = LayoutEditor.Forms.ToolWindows.common.DragModifierHelper.AxisSnap;
    using LECore;
    using static PaneDragTargetAdapter;

    /// <summary>
    /// ペインを DragModifier に操作させるための アダプタクラス
    /// </summary>
    class PaneDragTargetAdapter : IDragTarget
    {
        //-------------------------------------------------
        #region
        /// <summary>
        /// 操作対象ペイン
        /// </summary>
        readonly IPane		_pane;

        PointF _transformOrigin = new PointF( 0.0f, 0.0f );
 		PointF _translate = new PointF( 0.0f, 0.0f );

        // サイズ変更時の計算誤差を減らすために PointD を使う
        PointD _scale = new PointD( 1.0, 1.0 );

        /// <summary>
        /// double の点
        /// </summary>
        public struct PointD
        {
            public PointD(double x, double y)
            {
                X = x;
                Y = y;
            }

            public double X { get; set; }
            public double Y { get; set; }
        }

        float _rotateZ = 0.0f;

        /// <summary>
        /// 行列の乗算順序
        ///		ローカル変換の前か、後か
        /// </summary>
        MatrixOrder		_matrixOrder = MatrixOrder.Append;
        #endregion

        #region プロパティ
        /// <summary>
        /// 回転変換が適用されているか？
        /// </summary>
        bool _RotateModified
        {
            get { return _rotateZ != 0.0f; }
        }

        /// <summary>
        /// スケール変換行列を取得します。
        /// </summary>
        bool _ScaleModified
        {
            get { return _scale.X != 1.0f || _scale.Y != 1.0f; }
        }


        PointF TryToConvertToBaseSpace_( Matrix mtxOriginBase, PointF v )
        {
            if( mtxOriginBase != null && mtxOriginBase.IsInvertible )
            {
                Matrix mtxBaseInv = mtxOriginBase.Clone();
                mtxBaseInv.Invert();

                v = MathUtil.MtxTransformPoint( mtxBaseInv, v );
            }
            return v;
        }

        /// <summary>
        /// 回転変換行列を取得します。
        /// 回転中心の基底空間を定義する行列を引数にとります。
        /// </summary>
        Matrix CalcMtxRotate_( Matrix mtxOriginBase )
        {
            Debug.Assert( _RotateModified );
            Matrix mtx = new Matrix();

            PointF posOrigin = TryToConvertToBaseSpace_( mtxOriginBase, _transformOrigin );
            mtx.RotateAt( _rotateZ, posOrigin );

            return mtx;
        }

        /// <summary>
        /// スケール変換行列を取得します。
        /// </summary>
        Matrix CalcMtxScale_( Matrix mtxOriginBase )
        {
            Debug.Assert( _ScaleModified );

            PointF posOrigin = TryToConvertToBaseSpace_( mtxOriginBase, _transformOrigin );
            return MathUtil.GetScaleMtxAt( _scale.X, _scale.Y, posOrigin.X, posOrigin.Y );
        }

        /// <summary>
        /// ペイン
        /// </summary>
        public IPane IPane
        {
            get { return _pane; }
        }

        #endregion プロパティ

        /// <summary>
        /// 親ペイン座標系に変換します。
        /// </summary>
        PointF ConvertToParentSpace_( PointF v )
        {
            Matrix mtxParent = _pane.Parent.WorldMtx;
            if( mtxParent.IsInvertible )
            {
                mtxParent.Invert();
                v = MathUtil.MtxTransformPoint( mtxParent, v );
            }
            return v;
        }

        /// <summary>
        /// スケール変換を適用します。
        /// </summary>
        public void SetScale( PointD vS, PointF vO, MatrixOrder order )
        {
            _scale = vS;
            _transformOrigin = vO;
            _matrixOrder = order;
        }

        /// <summary>
        /// 回転変換を適用します。
        /// </summary>
        public void SetRotate( float vR, PointF vO, MatrixOrder order )
        {
            _rotateZ = vR;
            _transformOrigin = vO;
            _matrixOrder = order;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneDragTargetAdapter( IPane pane )
        {
            _pane = pane;
        }

        /// <summary>
        /// 描画用の行列を計算します
        /// </summary>
        /// <returns></returns>
        public Matrix CalcDrawMatrix()
        {
            Matrix mtxParent = _pane.Parent.WorldMtx;

            Matrix paneLocalMtx;
            if( _matrixOrder == MatrixOrder.Append )
            {
                paneLocalMtx = _pane.LocalMtx;

                if( _RotateModified )
                {
                    paneLocalMtx.Multiply( CalcMtxRotate_( mtxParent ), MatrixOrder.Append );
                }
                else if( _ScaleModified )
                {
                    paneLocalMtx.Multiply( CalcMtxScale_( mtxParent ), MatrixOrder.Append );
                }
            }
            else
            {
                Matrix mtxS = new Matrix();
                Matrix mtxR = new Matrix();

                mtxS.Scale( _pane.Scale.X, _pane.Scale.Y );
                mtxR.Rotate( _pane.RotAngleZ );

                if( _RotateModified )
                {
                    mtxR.Multiply( CalcMtxRotate_( null ) );
                }
                else if( _ScaleModified )
                {
                    mtxS.Multiply( CalcMtxScale_( null ) );
                }


                paneLocalMtx = new Matrix();
                paneLocalMtx.Translate(
                    _pane.GetParentDrawOffsetX() + _pane.Trans.X,
                    _pane.GetParentDrawOffsetY() + _pane.Trans.Y);

                paneLocalMtx.Multiply( mtxR, MatrixOrder.Prepend );
                paneLocalMtx.Multiply( mtxS, MatrixOrder.Prepend );
            }

            mtxParent.Multiply( paneLocalMtx );

            return mtxParent;
        }

        //-------------------------------------------------
        #region 静的メソッド
        /// <summary>
        /// targetSet の中に、指定したペインに関連する
        /// PaneDragTargetAdapterが存在するか調査します。
        /// </summary>
        static bool CheckPaneExists_( PaneDragTargetAdapter[] targetSet, IPane pane )
        {
            return Array.Exists( targetSet, delegate( PaneDragTargetAdapter target )
            {
                return target.IPane == pane;
            } );
        }

        /// <summary>
        /// ターゲットの親ペインを集めます。
        /// </summary>
        public static PaneDragTargetAdapter[] CollectParentSet( PaneDragTargetAdapter[] targetSet )
        {
            return Array.FindAll( targetSet, delegate( PaneDragTargetAdapter target )
            {
                IPane pane = target.IPane;
                IPane parent = pane.Parent as IPane;
                while( parent != null )
                {
                    // 親ペインが選択されている場合
                    if( CheckPaneExists_( targetSet, parent ) )
                    {
                        // 自分は選択セットから外れる
                        return false;
                    }
                    parent = parent.Parent as IPane;
                }
                return true;
            } );
        }

        /// <summary>
        /// 配列の変換
        /// </summary>
        public static PaneDragTargetAdapter[] FromPaneSet( IPane[] paneSet )
        {
            PaneDragTargetAdapter[] adapterSet = new PaneDragTargetAdapter[paneSet.Length];
            for( int i = 0 ; i < paneSet.Length ; i++ )
            {
                adapterSet[i] = new PaneDragTargetAdapter( paneSet[i] );
            }
            return adapterSet;
        }
        #endregion 静的メソッド

        //-------------------------------------------------
        #region IDragTarget メンバ

        public float X { get { return PaneHelper.GetBoundingVolumeInWorld( _pane ).X; } }
        public float Y { get { return PaneHelper.GetBoundingVolumeInWorld( _pane ).Y; } }
        public float Width { get { return PaneHelper.GetBoundingVolumeInWorld( _pane ).Width; } }
        public float Height { get { return PaneHelper.GetBoundingVolumeInWorld( _pane ).Height; } }
        public RectangleF BoundingRectangle{ get { return PaneHelper.GetBoundingVolumeInWorld( _pane ); } }

        #endregion

        //-------------------------------------------------
        #region IDrawable メンバ

        /// <summary>
        /// 描画
        /// </summary>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if( _pane.ValidScaleForDraw )
            {
                Matrix mtx = renderer.CurrentMtx;
                renderer.PushMtx();

                // 編集中のペイン
                mtx.Multiply(CalcDrawMatrix());
                renderer.CurrentMtx = mtx;
                _pane.Draw(renderer, option);

                // 親ペインにおける原点位置
                {
                    renderer.Color = option.SystemDarkRed;

                    renderer.CurrentMtx = _pane.Parent.WorldMtx;
                    int origTransparency = renderer.Transparency;
                    renderer.Transparency = 255;

                    FVec2 parentCenter = new FVec2(
                        _pane.GetParentDrawOffsetX(_pane.Parent as IPane),
                        _pane.GetParentDrawOffsetY(_pane.Parent as IPane));

                    RendererHelper.DrawCrossWithoutTransform(renderer, parentCenter.AsPointF, 10.0f);
                    RendererHelper.DrawCrossWithoutTransform(renderer, PointF.Empty, 5.0f);

                    // 薄く直線
                    renderer.Transparency = 128;
                    renderer.DrawLine(PointF.Empty, parentCenter.AsPointF);

                    renderer.Transparency = origTransparency;
                }

                renderer.PopMtx();
            }
        }

        #endregion
    }

    /// <summary>
    /// ペインドラッグ編集基底クラス
    /// </summary>
    class PaneDragModifierBase : DragModifier<PaneDragTargetAdapter>
    {
        //----------------------------------------
        #region フィールド・プロパティ
        PaneManipulator _paneMnp = new PaneManipulator();

        PointF _transformOrigin = PointF.Empty;
        PointF _translate = new PointF( 0.0f, 0.0f );
        PointD _scale		= new PointD( 1.0, 1.0 );
        float _rotateZ		= 0.0f;


        protected PointF _Translate
        {
            get { return _translate; }
            set { _translate = value; }
        }

        protected PointD _Scale
        {
            get { return _scale; }
            set { _scale = value; }
        }

        protected float _RotateZ
        {
            get { return _rotateZ; }
            set { _rotateZ = value; }
        }

        protected PointF _TransformOrigin
        {
            get { return _transformOrigin; }
            set { _transformOrigin = value; }
        }

        public float Magnity
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        ISubScene _CoreScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }
        #endregion フィールド・プロパティ

        //----------------------------------------
        #region 静的メソッド
        /// <summary>
        ///
        /// </summary>
        public static RectangleF CalcBoundingRect(PaneDragTargetAdapter[] targetSet)
        {
            if (targetSet.Length > 0)
            {
                if (targetSet.Length == 1)
                {
                    return PaneHelper.GetBoundingVolume(targetSet[0].IPane);
                }
                else if (targetSet.Length > 1)
                {
                    // TODO:
                    // 角度を見て、基底座標軸を決定する。
                    // その軸から、OBBを計算する。
                }
            }

            return DragModifierHelper.CalcBoundingRect(targetSet as IDragTarget[]);
        }

        /// <summary>
        ///
        /// </summary>
        public static Matrix CalcTargetLocalMatrix(PaneDragTargetAdapter[] targetSet)
        {
            if (targetSet == null || targetSet.Length <= 0)
            {
                return new Matrix();
            }

            Matrix mtx = new Matrix();
            mtx.Reset();

            // float rotAng;
            if (targetSet.Length == 1)
            {
                return targetSet[0].IPane.WorldMtx;
            }

            return mtx;
        }
        #endregion 静的メソッド

        //----------------------------------------
        #region オーバーライド

        /// <summary>
        /// 境界矩形を計算します。
        /// </summary>
        public override RectangleF CalcBoundingRect()
        {
        return CalcBoundingRect( TargetSet );
        }

        /// <summary>
        /// 操作対象のローカル系への変換行列を計算します。
        /// </summary>
        public override Matrix CalcTargetLocalMatrix()
        {
            return CalcTargetLocalMatrix( TargetSet );
        }

        //----------------------------------------------------
        /// <summary>
        /// 描画します。
        /// </summary>
        public override void Draw( IRenderer renderer, DrawableOption option )
        {
            if( this.Empty )
            {
                return;
            }

            float lineW = renderer.LineWidth;
            renderer.LineWidth = 1.0f;

            //---------------------------------------------
            // 倍率計算
            RectangleF defaultRect = CalcBoundingRect();
            PointF difference;
            CalcDifference_( out difference );

            //---------------------------------------------
            // 矩形
            renderer.PushMtx();

            Matrix mtxDraw = renderer.CurrentMtx;
            if( TargetSet.Length == 1 )
            {
                mtxDraw.Multiply( TargetSet[0].CalcDrawMatrix() );
            }
            else
            {
                mtxDraw.Multiply( CalcModifyMatrix_() );
            }
            renderer.CurrentMtx = mtxDraw;

            DragPaneScaleModifier.DrawAnchorRectangle( renderer, defaultRect, 6.0f );

            renderer.PopMtx();

            //---------------------------------------------
            // 対象
            DrawTargetOutline_( renderer, option );


            renderer.LineWidth = lineW;
        }
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneDragModifierBase()
        {
            this.Magnity = 1.0f;
        }

        #region ドラッグイベントハンドラ
        /// <summary>
        /// 内部パラメータをリセットします。
        /// </summary>
        protected void ResetInternalParamaters_()
        {
            _transformOrigin = PointF.Empty;
            _translate = PointF.Empty;
            _scale = new PointD( 1.0, 1.0 );

            _rotateZ = 0.0f;
        }

        /// <summary>
        /// 変更パラメータを更新します。
        /// 派生操作クラスでオーバーライドして、パラメータを更新して下さい。
        /// </summary>
        protected virtual bool UpdateModifyParamaters_( PointF difference, PointF transformOrigin )
        {
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public override void BeginDragging( PaneDragTargetAdapter[] paneSet, PointF posInScreen, DragModifierHelper.OptionFlag optionFlags )
        {
            base.BeginDragging( paneSet, posInScreen, optionFlags );

            ResetInternalParamaters_();

            DragModifierHelper.Contact modifierContact;
            Matrix mtxLocal = PaneDragModifierBase.CalcTargetLocalMatrix(paneSet);
            DragModifierHelper.IsControlAnchorClicked(
                mtxLocal, DefaultBoundingRect, this.Magnity, this.DragStartPointF, out modifierContact);

            _transformOrigin = modifierContact.TransOrigin;
            AxisSnapKind = modifierContact.AxisSnap;
        }

        /// <summary>
        ///
        /// </summary>
        public override void UpdateDragging( PointF posInScreen )
        {
            base.UpdateDragging( posInScreen );

            PointF difference;
            CalcDifference_( out difference );

            bool bUpdated = UpdateModifyParamaters_( difference, _transformOrigin );
            if( bUpdated )
            {
                Matrix mtxModify = CalcModifyMatrix_();
                //--------------------------------------------
                // 対象に変換行列を設定します。
                MatrixOrder matrixOrder = ( TargetSet.Length == 1 ) ?
                    MatrixOrder.Prepend : MatrixOrder.Append;

                foreach( PaneDragTargetAdapter targetPane in TargetSet )
                {
                    targetPane.SetRotate( _rotateZ, _transformOrigin, matrixOrder );
                    targetPane.SetScale( _scale, _transformOrigin, matrixOrder );
                }
            }
        }

        /// <summary>
        /// 操作を実際に適用する対象を抽出します。
        /// </summary>
        protected virtual PaneDragTargetAdapter[] CollectFixTargetFromAll_( PaneDragTargetAdapter[] tagetSet )
        {
            // 基底の動作では、親ペインのみを抽出します。
            return PaneDragTargetAdapter.CollectParentSet( TargetSet );
        }

        /// <summary>
        /// ドラッグ終了
        /// </summary>
        public override void EndDragging()
        {
            if( _CoreScene == null || Empty )
            {
                return;
            }

            PointF difference;
            CalcDifference_( out difference );

            bool bUpdated = UpdateModifyParamaters_( difference, _transformOrigin );
            if( bUpdated )
            {
                _CoreScene.BeginMassiveModify();

                // 捜査対象の中から、親ペインのみに変更を適用します。
                PaneDragTargetAdapter[] targetParentSet = CollectFixTargetFromAll_( TargetSet );
                foreach( PaneDragTargetAdapter targetPane in targetParentSet )
                {
                    FixModify_( targetPane );
                }

                _CoreScene.EndMassiveModify();
            }

            base.EndDragging();
        }

        #endregion ドラッグイベントハンドラ

        #region 座標変換
        /// <summary>
        /// モデファイアのローカル系に点を変換します。
        /// </summary>
        protected PointF ConvertPointWorldToLocal( PointF srcPoint )
        {
            PointF dstPoint = srcPoint;
            Matrix mtx = CalcTargetLocalMatrix();

            if( mtx.IsInvertible )
            {
                mtx.Invert();
                dstPoint = MathUtil.MtxTransformPoint( mtx, srcPoint );
            }

            return dstPoint;
        }

        /// <summary>
        /// モデファイアのローカル系に点を変換します。
        /// </summary>
        protected PointF ConvertPointLocalToWorld( PointF srcPoint )
        {
            Matrix mtx = CalcTargetLocalMatrix();

            return MathUtil.MtxTransformPoint( mtx, srcPoint );
        }

        /// <summary>
        /// モデファイアのローカル系にベクトルを変換します。
        /// </summary>
        protected PointF ConvertVectorLocalToWorld( PointF srcPoint )
        {
            Matrix mtx = CalcTargetLocalMatrix();

            return MathUtil.MtxTransformVec( mtx, srcPoint );
        }
        #endregion 座標変換

        /// <summary>
        /// アンカーがクリックされたか？
        ///		TODO:CalcScaleOrigin内の処理を分離すること
        /// </summary>
        public bool IsControlAnchorClicked( float magnify, PointF posMouse )
        {
            if( Empty )
            {
                return false;
            }

            return DragModifierHelper.IsControlAnchorClicked(CalcTargetLocalMatrix(), DefaultBoundingRect, magnify, posMouse);
        }

        /// <summary>
        /// 変更行列を計算します。
        /// </summary>
        protected Matrix CalcPaneModifiedMtx_( Matrix mtxBaseWorld )
        {
            //--------------------------------------------
            // 対象に変換行列を設定します。
            MatrixOrder matrixOrder = ( TargetSet.Length == 1 ) ?
                MatrixOrder.Prepend : MatrixOrder.Append;

            Matrix mtxNew = mtxBaseWorld.Clone();

            mtxNew.Multiply( CalcModifyMatrix_(), matrixOrder );

            return mtxNew;
        }

        /// <summary>
        /// 指定原点における、スケール変換
        ///		各種変換クラスで適切にオーバーライドしてください。
        /// </summary>
        protected virtual Matrix CalcModifyMatrix_()
        {
            return new Matrix();
        }

        /// <summary>
        /// 変更を設定します。
        /// Undoコマンドが生成される変更として処理されます。
        /// </summary>
        protected virtual void FixModify_( PaneDragTargetAdapter targetPane )
        {
            Matrix mtx = CalcPaneModifiedMtx_( targetPane.IPane.WorldMtx );

            PointF newPosInWorld = new PointF( mtx.OffsetX, mtx.OffsetY );

            IPane pane = targetPane.IPane;
            _paneMnp.BindTarget( pane );


            _paneMnp.TranslateInWorld( new FVec3(
                newPosInWorld.X - pane.XInWorld,
                newPosInWorld.Y - pane.YInWorld,
                0.0F ) );

            _paneMnp.Scale = new FVec3(
                (float)(pane.Scale.X * _scale.X),
                (float)(pane.Scale.Y * _scale.Y),
                pane.Scale.Z );

            _paneMnp.RotAngleZ = pane.RotAngleZ + _rotateZ;

        }


        /// <summary>
        /// 操作対象のアウトライン描画をおこないます。
        /// </summary>
        void DrawTargetOutline_( IRenderer renderer, DrawableOption option )
        {
            uint flag = renderer.OptionFlag;

            renderer.OptionFlag = flag | IRenderOptionFlag.DrawOutline;

            foreach( PaneDragTargetAdapter targetPane in TargetSet )
            {
                targetPane.Draw( renderer, option );
            }

            renderer.OptionFlag = flag;
        }
    }

    /// <summary>
    /// ペイン操作クラス(平行移動)
    /// </summary>
    class PaneDragTransModifier : PaneDragModifierBase
    {
        readonly MainView _mainView = null;

        /// <summary>
        /// ワールド系に変換された、変移量を取得します。
        /// </summary>
        PointF GetDifferenceInWorld_()
        {
            Matrix mtxLocal = CalcTargetLocalMatrix();

            PointF difference;
            CalcDifference_( out difference );

            difference = MathUtil.MtxTransformVec( mtxLocal, difference );

            if( difference != PointF.Empty )
            {
                // グリッド吸着が有効ならば...
                if( _GridSnapEnable && TargetSet.Length > 0 )
                {
                    IPane pane = TargetSet[0].IPane;
                    PointF posPane = new PointF( pane.XInWorld, pane.YInWorld );

                    difference = CalcAlignedDifference( _GridSize, posPane, difference );
                }
            }

            return difference;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneDragTransModifier( MainView mainView )
        {
            _mainView = mainView;
        }

        #region オーバーライド
        /// <summary>
        ///
        /// </summary>
        /// <returns></returns>
        public override Matrix CalcTargetLocalMatrix()
        {
            if( Empty )
            {
                return new Matrix();
            }

            if( TargetSet.Length == 1 )
            {
                return TargetSet[0].IPane.WorldMtx;
            }
            else
            {
                return new Matrix();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override RectangleF CalcBoundingRect()
        {
            if( !Empty )
            {
                if( this.TargetSet.Length == 1 )
                {
                    return PaneHelper.GetBoundingVolume( TargetSet[0].IPane );
                }
                else if( this.TargetSet.Length > 1 )
                {
                    // TODO:
                    // 角度を見て、基底座標軸を決定する。
                    // その軸から、OBBを計算する。
                }
            }

            return base.CalcBoundingRect();
        }

        /// <summary>
        ///
        /// </summary>
        public override void EndDragging()
        {

            PointF difference = GetDifferenceInWorld_();

            // ドラッグを終了し、移動量を取得します。
            if( Math.Abs( difference.X ) > float.Epsilon ||
                Math.Abs( difference.Y ) > float.Epsilon )
            {
                // 移動量を反映します。
                // シーンに大量の変更を通知します。
                _mainView._TargetActiveSubScene.BeginMassiveModify();

                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget( _mainView._TargetActiveSubScene );
                subSceneMnp.AddOffsetToSelectedSet(
                    difference,
                    _mainView.TransAffectsChildren,
                    SubSceneManipulator.ValueRouwndDownDisable );

                _mainView._TargetActiveSubScene.EndMassiveModify();
            }

            base.EndDragging();
        }

        /// <summary>
        ///
        /// </summary>
        public override void Draw( IRenderer renderer, DrawableOption option )
        {
            if( !Empty )
            {
                // 変移量を算出します。
                PointF difference = GetDifferenceInWorld_();

                renderer.PushMtx();
                renderer.Trans( difference.X, difference.Y );

                // 色の変更
                Color oldColor = renderer.Color;
                renderer.Color = option.SystemGray;

                // 簡易描画フラグの指定
                uint flag = renderer.OptionFlag;
                renderer.OptionFlag = flag | IRenderOptionFlag.DrawOutline;

                // すべての対象を描画します。
                foreach( PaneDragTargetAdapter pane in TargetSet )
                {
                    pane.Draw( renderer, option );
                }

                // 描画フラグの復元
                renderer.OptionFlag = flag;

                // 色の復元
                renderer.Color = oldColor;

                renderer.PopMtx();

            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void HandleConstraint( bool bConstraintEnabled )
        {
            AxisSnapKind = bConstraintEnabled ? AxisSnap.EightDirection : AxisSnap.None;
        }

        #endregion オーバーライド
    }



    /// <summary>
    /// ペイン操作クラス(スケール)
    /// </summary>
    class DragPaneScaleModifier : PaneDragModifierBase
    {
        //---------------------------------------------
        #region フィールド
        readonly MainView _mainView = null;
        bool _bModifySize = false;
        bool _bChangeXYSynchronus = false;
        #endregion フィールド

        /// <summary>
        /// サイズを変更する
        /// </summary>
        public bool ModifySize
        {
            set{ _bModifySize = value;}
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DragPaneScaleModifier( MainView mainView )
        {
            _mainView = mainView;
        }

        #region オーバーライド

        /// <summary>
        /// ドラッグ開始
        /// </summary>
        public override void BeginDragging(
            PaneDragTargetAdapter[] paneSet, PointF posInScreen, DragModifierHelper.OptionFlag optionFlags)
        {
            base.BeginDragging(paneSet, posInScreen, optionFlags);

            // 操作対象による制約
            if (this._bModifySize)
            {
                var paneSizeMode = PaneDragModifier.GetSlectedPanesSizeMode_(paneSet);
                if (paneSizeMode == PaneSizeMode.HEnabled)
                {
                    AxisSnapKind = AxisSnap.V;
                }
                else if (paneSizeMode == PaneSizeMode.VEnabled)
                {
                    AxisSnapKind = AxisSnap.H;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void Clear()
        {
            _bModifySize = false;
            base.Clear();
        }

        #region FixModify_　関連

        /// <summary>
        ///
        /// </summary>
        protected override PaneDragTargetAdapter[] CollectFixTargetFromAll_( PaneDragTargetAdapter[] tagetSet )
        {
            if( _bModifySize )
            {
                return tagetSet;
            }
            else
            {
                return base.CollectFixTargetFromAll_( tagetSet );
            }
        }

        /// <summary>
        /// 変更を設定します。
        /// Undoコマンドが生成される変更として処理されます。
        /// </summary>
        protected override void FixModify_( PaneDragTargetAdapter targetPane )
        {
            if( _bModifySize )
            {
                PaneManipulator paneMnp = new PaneManipulator();
                paneMnp.TransAffectsChildren = _mainView.TransAffectsChildren;

                Matrix mtx = CalcPaneModifiedMtx_( targetPane.IPane.WorldMtx );

                PointF newPosInWorld = new PointF( mtx.OffsetX, mtx.OffsetY );

                IPane pane = targetPane.IPane;
                paneMnp.BindTarget( pane );

                paneMnp.TranslateInWorld( new FVec3(
                    newPosInWorld.X - pane.XInWorld,
                    newPosInWorld.Y - pane.YInWorld,
                    0.0F ) );

                // ほとんど整数なら整数にする
                paneMnp.Trans = new FVec3(
                    RoundIfNearlyInteger(paneMnp.IPane.X, 1.0e-6f),
                    RoundIfNearlyInteger(paneMnp.IPane.Y, 1.0e-6f),
                    paneMnp.IPane.Z);

                float width = RoundIfNearlyInteger((float)(pane.Width * _Scale.X), 1.0e-6f);
                float height = RoundIfNearlyInteger((float)(pane.Height * _Scale.Y), 1.0e-6f);
                paneMnp.Size = new FVec3(
                    width,
                    height,
                    pane.Size.Z );

                paneMnp.RotAngleZ = pane.RotAngleZ + _RotateZ;
            }
            else
            {
                base.FixModify_( targetPane );
            }
        }

        private float RoundIfNearlyInteger(float value, float relativeTolerance)
        {
            if (Math.Abs(Math.Round(value) - value) <= relativeTolerance * Math.Abs(value))
            {
                return (float)Math.Round(value);
            }

            return value;
        }

#endregion FixModify_　関連

        /// <summary>
        /// 変更パラメータを更新します。
        /// </summary>
        protected override bool UpdateModifyParamaters_( PointF difference, PointF transformOrigin )
        {
            PointD scale;

            // 原点が Center のペインを含むときは、サイズの偶奇を変えないように調整
            float unitX = TargetSet.Any(x => x.IPane.BasePosTypeH == HorizontalLocation.Center) ? 2: 1;
            float unitY = TargetSet.Any(x => x.IPane.BasePosTypeV == VerticalLocation.Center) ? 2 : 1;

            var mat = CalcTargetLocalMatrix();

            // 行列がスケールだけの場合に、ワールド座標でサイズが整数になるようにする
            if (mat.Elements[0] != 0.0f &&
                mat.Elements[1] == 0.0f &&
                mat.Elements[2] == 0.0f &&
                mat.Elements[3] != 0.0)
            {
                unitX /= Math.Abs(mat.Elements[0]); // (0,0) 成分
                unitY /= Math.Abs(mat.Elements[3]); // (1,1) 成分
            }

            difference = new PointF((float)Math.Round(difference.X / unitX) * unitX, (float)Math.Round(difference.Y / unitY) * unitY);

            bool bUpdated = CalcScale_( difference, _TransformOrigin, out scale);
            if( bUpdated )
            {
                if( _bChangeXYSynchronus )
                {
                    double min = Math.Min( scale.X, scale.Y );
                    scale.X = min;
                    scale.Y = min;
                }
                _Scale = scale;
            }

            return bUpdated;
        }

        /// <summary>
        /// 指定原点における、スケール変換
        /// </summary>
        protected override Matrix CalcModifyMatrix_()
        {
            return MathUtil.GetScaleMtxAt( _Scale.X, _Scale.Y, _TransformOrigin.X, _TransformOrigin.Y );
        }

        /// <summary>
        ///
        /// </summary>
        public override void HandleConstraint( bool bConstraintEnabled )
        {
            _bChangeXYSynchronus = bConstraintEnabled;
        }

#endregion オーバーライド
    }

    /// <summary>
    /// ペイン操作クラス(回転)
    /// </summary>
    class DragPaneRotateModifier : PaneDragModifierBase
    {
#region フィールド
        readonly MainView _mainView = null;
        bool _bSnapEightDirections = false;
#endregion

        /// <summary>
        /// 90度単位でスナップを行うか設定します。
        /// </summary>
        bool SnapEightDirections_
        {
            set { _bSnapEightDirections = value; }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public DragPaneRotateModifier( MainView mainView )
        {
            _mainView = mainView;
        }

#region オーバーライド
        /// <summary>
        /// 内部状態をリセットします。
        /// </summary>
        public override void Clear()
        {
            _bSnapEightDirections = false;
            base.Clear();
        }

        /// <summary>
        ///
        /// </summary>
        protected override bool UpdateModifyParamaters_(
            PointF difference,
            PointF transformOrigin )
        {
            PointF vOrigInWorld = ConvertPointLocalToWorld( _TransformOrigin );
            PointF vDiffInWorld = ConvertVectorLocalToWorld( difference );


            PointF vDragStart = MathUtil.SubVec( DragStartPointF, vOrigInWorld );
            PointF vDragEnd = MathUtil.SubVec( MathUtil.AddVec( DragStartPointF, vDiffInWorld ), vOrigInWorld );

            _RotateZ = MathUtil.CalcAngleBtweenLines2( vDragEnd, vDragStart );

            // 8方向固定処理
            if( _bSnapEightDirections )
            {
                if( _RotateZ > 0.0f )
                {
                    _RotateZ = _RotateZ - _RotateZ % 45.0f;
                }
                else
                {
                    _RotateZ = _RotateZ - (45.0f + (_RotateZ % 45.0f) );
                }
            }

            return true;
        }

        /// <summary>
        /// 指定原点における、スケール変換
        /// </summary>
        protected override Matrix CalcModifyMatrix_()
        {
            Matrix mtxR = new Matrix();
            mtxR.RotateAt( _RotateZ, _TransformOrigin );
            return mtxR;
        }


#region ドラッグイベントハンドラ

        /// <summary>
        ///
        /// </summary>
        public override void BeginDragging( PaneDragTargetAdapter[] paneSet, PointF posInScreen, DragModifierHelper.OptionFlag optionFlags )
        {
            base.BeginDragging( paneSet, posInScreen, optionFlags );

            ResetInternalParamaters_();

            _TransformOrigin = new PointF(
                ( DefaultBoundingRect.Right + DefaultBoundingRect.Left ) / 2.0f,
                ( DefaultBoundingRect.Top + DefaultBoundingRect.Bottom ) / 2.0f );

            AxisSnapKind = AxisSnap.None;
        }

#endregion ドラッグイベントハンドラ

        /// <summary>
        ///
        /// </summary>
        public override void  HandleConstraint(bool bConstraintEnabled)
        {
            SnapEightDirections_ = bConstraintEnabled;
        }

#endregion オーバーライド
    }

    /// <summary>
    /// ペインドラッグ編集クラス
    /// </summary>
    class PaneDragModifier
    {
        /// <summary>
        /// オプションフラグ
        /// オプションフラグは、最長の場合でも操作完了時にリセットされます。
        /// </summary>
        [Flags]
        public enum Option
        {
            None				= 0x00,
            NeedToSetup		= 0x01,
            TryToRotate		= 0x02,
            ForceToTranslate	= 0x04,
            ModifySize			= 0x08,
            SnapDirections		= 0x10,
        }

        /// <summary>
        /// 動作モード
        /// </summary>
        [Flags]
        public enum Mode
        {
            None,
            Trans,
            Size,
            Scale,
            Rotate
        }

        static readonly Cursor _rotateCursor = new Cursor(ImageResMgr.GetManifestResourcePng("rotate.png").GetHicon());

        readonly PaneDragTransModifier _transModifier;
        readonly DragPaneScaleModifier _scaleModifier;
        readonly DragPaneRotateModifier _roteteModifier;

        PaneDragModifierBase _currentModifier = null;

        Option _option = Option.None;



        /// <summary>
        /// 軸固定状態
        /// </summary>
        public AxisSnap AxisSnapKind
        {
            get
            {
                Debug.Assert( _currentModifier != null );
                return _currentModifier.AxisSnapKind;
            }
            set
            {
                Debug.Assert( _currentModifier != null );
                _currentModifier.AxisSnapKind = value;
            }
        }


        /// <summary>
        /// モードです。
        /// </summary>
        public Mode ModifyMode
        {
            get;
            private set;
        }

        /// <summary>
        ///
        /// </summary>
        public PointF DragStartPointF
        {
            get
            {
                Debug.Assert( _currentModifier != null );
                return _currentModifier.DragStartPointF;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void SetGridSnapEnable( float gridSize )
        {
            if( !Empty )
            {
                Debug.Assert( _currentModifier != null );
                _currentModifier.SetGridSnapEnable( gridSize );
            }
        }



        /// <summary>
        ///
        /// </summary>
        public string ReportMsgString()
        {

            if( _currentModifier == null )
            {
                return string.Empty;
            }
            else
            {
                return _currentModifier.ReportMsgString();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool Empty
        {
            get { return _currentModifier == null || _currentModifier.Empty; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool ContainsReadOnly
        {
            get { return _currentModifier == null || _currentModifier.TargetSet.Any((paneAdp) => paneAdp.IPane.IsReadOnlyLocked); }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public PaneDragModifier( MainView mainView )
        {
            _transModifier = new PaneDragTransModifier( mainView );
            _scaleModifier = new DragPaneScaleModifier( mainView );
            _roteteModifier = new DragPaneRotateModifier( mainView );
        }

        /// <summary>
        /// ヒットテストを行います。
        /// </summary>
        public bool CheckClicked(float magnify, PointF posMouse, out PaneDragTargetAdapter clicked)
        {
            clicked = null;

            if (_currentModifier == null || _currentModifier.Empty)
            {
                return false;
            }

            if (DragModifierHelper.IsControlAnchorClicked(
                _currentModifier.CalcTargetLocalMatrix(), _currentModifier.CalcBoundingRect(), magnify, posMouse))
            {
                return true;
            }
            else
            {
                return _currentModifier.CheckClicked(posMouse, out clicked);
            }
        }

        /// <summary>
        /// オプションフラグを追加します。
        /// オプションフラグは、最長の場合でも操作完了時にリセットされます。
        /// </summary>
        public void AddOptionFlag( Option flag )
        {
            _option |= flag;
        }

        /// <summary>
        /// オプションフラグを削除します。
        /// オプションフラグは、最長の場合でも操作完了時にリセットされます。
        /// </summary>
        public void ResetOptionFlag(Option flag)
        {
            _option &= ~flag;
        }

        /// <summary>
        /// オプションフラグが有効か取得します。
        /// </summary>
        public bool IsOptionFlagEnable( Option flag )
        {
            return ( _option & flag ) != 0;
        }

#region ドラッグイベントハンドラ
        /// <summary>
        ///
        /// </summary>
        public void BeginDragging( PaneDragTargetAdapter[] paneSet, float magnify, PointF posInScreen, DragModifierHelper.OptionFlag optionFlags )
        {
            {
                DragModifierHelper.Contact contant;
                var mode = GetModifyMode(paneSet, magnify, posInScreen, optionFlags, out contant);
                switch (mode)
                {
                    case Mode.Trans: _currentModifier = _transModifier; break;
                    case Mode.Size: _currentModifier = _scaleModifier; _scaleModifier.ModifySize = true; break;
                    case Mode.Scale: _currentModifier = _scaleModifier; break;
                    case Mode.Rotate: _currentModifier = _roteteModifier; break;
                    case Mode.None: _currentModifier = _transModifier; break;
                }

                this.ModifyMode = mode;
            }

            if (_currentModifier != null)
            {
                _currentModifier.Magnity = magnify;
                _currentModifier.BeginDragging(paneSet, posInScreen, optionFlags);
                _option &= ~Option.NeedToSetup;
            }
        }

        /// <summary>
        /// 変更モードを取得します。
        /// </summary>
        public Mode GetModifyMode(
            PaneDragTargetAdapter[] paneSet,
            float magnify, PointF posInScreen, DragModifierHelper.OptionFlag optionFlags, out DragModifierHelper.Contact modifierContact)
        {
            modifierContact = new DragModifierHelper.Contact();
            modifierContact.TransOrigin = PointF.Empty;
            modifierContact.AxisSnap = AxisSnap.None;

            if (IsOptionFlagEnable(Option.ForceToTranslate))
            {
                return Mode.Trans;
            }

            // サイズ変更ができないペインがあれば平行移動モードに。
            var paneSizeMode = GetSlectedPanesSizeMode_(paneSet);
            if (paneSizeMode == PaneSizeMode.Fixed)
            {
                return Mode.Trans;
            }

            // アンカーがクリックされたか？
            RectangleF rect = PaneDragModifierBase.CalcBoundingRect(paneSet);
            Matrix mtxLocal = PaneDragModifierBase.CalcTargetLocalMatrix(paneSet);
            if (!DragModifierHelper.IsControlAnchorClicked(mtxLocal, rect, magnify, posInScreen, out modifierContact))
            {
                return (modifierContact.IsHit) ? Mode.Trans : Mode.None;
            }

            // 回転
            if (IsOptionFlagEnable(Option.TryToRotate))
            {
                return Mode.Rotate;
            }

            // スケール・サイズ変更
            return (IsOptionFlagEnable(Option.ModifySize)) ? Mode.Size : Mode.Scale;
        }

        /// <summary>
        /// 適切なカーソルを取得します。
        /// </summary>
        public Cursor GetCursor(PaneDragTargetAdapter[] paneSet,
            float magnify,
            PointF posInScreen,
            DragModifierHelper.OptionFlag optionFlags)
        {
            if (paneSet.Length <= 0)
            {
                return Cursors.Arrow;
            }

            DragModifierHelper.Contact contant;
            var mode = GetModifyMode(paneSet, magnify, posInScreen, optionFlags, out contant);
            switch (mode)
            {
                case PaneDragModifier.Mode.None: return Cursors.Arrow;
                case PaneDragModifier.Mode.Trans: return Cursors.SizeAll;
                case PaneDragModifier.Mode.Rotate: return _rotateCursor;
                case PaneDragModifier.Mode.Scale:
                case PaneDragModifier.Mode.Size:
                    {
                        if (contant.AxisSnap == AxisSnap.H)
                        {
                            return Cursors.SizeNS;
                        }
                        else if (contant.AxisSnap == AxisSnap.V)
                        {
                            return Cursors.SizeWE;
                        }
                        else
                        {
                            return (contant.IsCornerClicked) ? Cursors.SizeNWSE : Cursors.SizeNESW;
                        }
                    }
            }

            return Cursors.Arrow;
        }

        /// <summary>
        /// サイズを変更できない対象を含むかどうかです。
        /// </summary>
        static public PaneSizeMode GetSlectedPanesSizeMode_(IEnumerable<PaneDragTargetAdapter> paneSet)
        {
            var firstPane = paneSet.FirstOrDefault();
            if(firstPane == null)
            {
                return PaneSizeMode.Fixed;
            }

            if(!paneSet.All((pane) => pane.IPane.PaneSizeMode == firstPane.IPane.PaneSizeMode))
            {
                return PaneSizeMode.Fixed;
            }

            // 選択ペインの変更モードがすべて一致している。
            return firstPane.IPane.PaneSizeMode;
        }

        /// <summary>
        /// Option.SnapDirections が指定された場合に、
        /// 各操作クラスに適切な設定をおこないます。
        /// </summary>
        void HandleSnapDirections_()
        {
            bool bSnapEnabled = IsOptionFlagEnable( Option.SnapDirections );

            _currentModifier.HandleConstraint( bSnapEnabled );

            // フラグをリセットします。
            _option &= ~Option.SnapDirections;
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateDragging( float magnify, PointF posInScreen )
        {
            if (this.ContainsReadOnly)
            {
                return;
            }

            // いきなりドラッグからスタートした場合は、平行移動操作として初期化します
            if( IsOptionFlagEnable( Option.NeedToSetup ) )
            {
                _option |= PaneDragModifier.Option.ForceToTranslate;
                ResetDragging( magnify, posInScreen );
            }

            if( _currentModifier != null )
            {
                HandleSnapDirections_();

                _currentModifier.UpdateDragging( posInScreen );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void EndDragging()
        {
            if( !IsOptionFlagEnable( Option.NeedToSetup ) )
            {
                if( _currentModifier != null )
                {
                    _currentModifier.EndDragging();
                    _currentModifier = null;
                }
            }

            Clear();
        }

        /// <summary>
        ///
        /// </summary>
        public void ResetDragging( float magnify, PointF posInScreen )
        {
            if( _currentModifier != null )
            {
                BeginDragging( _currentModifier.TargetSet, magnify, posInScreen, _currentModifier.Option );
            }
        }

#endregion ドラッグイベントハンドラ

        /// <summary>
        /// 描画
        /// </summary>
        public void Draw(IRenderer renderer, DrawableOption option)
        {
            if (_currentModifier == null || _currentModifier.Empty)
            {
                return;
            }

            // アンカー点の描画
            {
                renderer.PushMtx();

                Matrix mtx = renderer.CurrentMtx;


                RectangleF rect = PaneDragModifierBase.CalcBoundingRect(_currentModifier.TargetSet);
                Matrix mtxLocal = PaneDragModifierBase.CalcTargetLocalMatrix(_currentModifier.TargetSet);

                mtx.Multiply(mtxLocal);
                renderer.CurrentMtx = mtx;

                // サイズ変更ができるかどうかで表示を変える
                if (GetSlectedPanesSizeMode_(_currentModifier.TargetSet) != PaneSizeMode.Fixed)
                {
                    DragPaneRotateModifier.DrawAnchorRectangle(renderer, rect, 4.0f);
                }
                else
                {
                    RendererHelper.DrawRectangleWithoutTransform(renderer, rect);
                }

                renderer.PopMtx();
            }


            if (_currentModifier != null)
            {
                _currentModifier.Draw(renderer, option);
            }
        }

        /// <summary>
        /// 削除
        /// </summary>
        public void Clear()
        {
            _transModifier.Clear();
            _scaleModifier.Clear();
            _roteteModifier.Clear();

            _currentModifier = null;

            _option = Option.None;
        }
    }

}
