﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Drawing;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.ComponentModel;
using System.Windows.Forms;
using System.Drawing.Drawing2D;
using System.Text.RegularExpressions;
using System.Linq;

namespace LayoutEditor.Forms.ToolWindows.HierarchyWindow
{
    using LECore;
    using LECore.Structures;
    using LECore.Manipulator;
    using Structures.SerializableObject;

    using common;
    using Utility;
    using Controls;

    using TreeNodeDropMode = common.TreeNodeDragDropMgr.TreeNodeDropHint;

    /// <summary>
    /// 階層ウインドウ
    /// </summary>
    public class HierarchyWindow : LEToolWindow
    {
        private PaneTreeStatePanel.PaneHierarchyTreeView _tvwHierarchy;
        private IContainer components = null;

        #region デザイナ生成コード

        private PaneTreeStatePanel _pnlState;
        private MenuStrip _mspMain;
        private ToolStripMenuItem _tmiDelete;
        private ToolStripMenuItem _tmiCopy;
        private ToolStripMenuItem _tmiPaste;
        private ToolStripMenuItem _tmiCut;
        private ToolStripMenuItem _tmiSetPaneLocked;
        private ToolStripMenuItem _tmiSetPaneUnLocked;
        private ToolStripMenuItem _tmiSetPaneHidden;
        private ToolStripMenuItem _tmiSetPaneNotHidden;
        private ToolStripMenuItem _tmiSelectHier;
        private ToolStripMenuItem _tmiSetHierLocked;
        private ToolStripMenuItem _tmiSetHierUnLocked;
        private ToolStripMenuItem _tmiSetHierHidden;
        private ToolStripMenuItem _tmiSetHierNotHidden;
        private ToolStripMenuItem _tmiShowSelectedEnable;
        private ToolStripSeparator toolStripSeparator1;
        private ToolStripSeparator toolStripSeparator2;
        private ToolStripSeparator toolStripSeparator3;
        private ToolStripSeparator toolStripSeparator4;
        private ToolStripMenuItem _tmiProperty;
        private ToolStripMenuItem _tmiShowSelectedEnable2;
        private ToolStripSeparator toolStripSeparator5;
        private ToolStripMenuItem _tmiCreateGroup;
        private ToolStripMenuItem _tmiEditGroup;
        private ToolStripMenuItem _tmiExpandAll;
        private ToolStripMenuItem _tmiCollopseAll;
        private ToolStripMenuItem _tmiChangeToSlectedParts;
        private ToolStripMenuItem _tmiOpenSlectedParts;
        private ToolStripSeparator toolStripSeparator6;
        private ToolStripSeparator toolStripMenuItem1;
        private ToolStripMenuItem _tmiAsChild;
        private ToolStripMenuItem _tmiAsParent;
        private ToolStripMenuItem _tmiCreateNullPaneAndAddChild;
        private ToolStripMenuItem _tmiDuplicate;
        private ToolStripMenuItem _tmiMain;
        #region Windows フォーム デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(HierarchyWindow));
            this._tvwHierarchy = new LayoutEditor.Forms.ToolWindows.common.PaneTreeStatePanel.PaneHierarchyTreeView();
            this._pnlState = new LayoutEditor.Forms.ToolWindows.common.PaneTreeStatePanel();
            this._mspMain = new System.Windows.Forms.MenuStrip();
            this._tmiMain = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCut = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCopy = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiPaste = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiDelete = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator1 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiSetPaneLocked = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetPaneUnLocked = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetPaneHidden = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetPaneNotHidden = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator2 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiSelectHier = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator3 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiSetHierLocked = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetHierUnLocked = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetHierHidden = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiSetHierNotHidden = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator5 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiCreateGroup = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiEditGroup = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator4 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiChangeToSlectedParts = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiOpenSlectedParts = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripSeparator6 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiProperty = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiShowSelectedEnable2 = new System.Windows.Forms.ToolStripMenuItem();
            this.toolStripMenuItem1 = new System.Windows.Forms.ToolStripSeparator();
            this._tmiAsChild = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiAsParent = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCreateNullPaneAndAddChild = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiShowSelectedEnable = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiExpandAll = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiCollopseAll = new System.Windows.Forms.ToolStripMenuItem();
            this._tmiDuplicate = new System.Windows.Forms.ToolStripMenuItem();
            this._mspMain.SuspendLayout();
            this.SuspendLayout();
            //
            // _tvwHierarchy
            //
            this._tvwHierarchy.AllowDrop = true;
            resources.ApplyResources(this._tvwHierarchy, "_tvwHierarchy");
            this._tvwHierarchy.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this._tvwHierarchy.ItemHeight = 16;
            this._tvwHierarchy.LabelEdit = true;
            this._tvwHierarchy.Name = "_tvwHierarchy";
            this._tvwHierarchy.ShowNodeToolTips = true;
            this._tvwHierarchy.ShowRootLines = false;
            this._tvwHierarchy.BeforeLabelEdit += new System.Windows.Forms.NodeLabelEditEventHandler(this.Event_TvwHierarchy_BeforeLabelEdit);
            this._tvwHierarchy.AfterLabelEdit += new System.Windows.Forms.NodeLabelEditEventHandler(this.Event_TvwHierarchy_AfterLabelEdit);
            this._tvwHierarchy.AfterCollapse += new System.Windows.Forms.TreeViewEventHandler(this.Event_TvwHierarchy_AfterCollapse);
            this._tvwHierarchy.AfterExpand += new System.Windows.Forms.TreeViewEventHandler(this.Event_TvwHierarchy_AfterExpand);
            this._tvwHierarchy.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Event_TvwHierarchy_MouseDown);
            this._tvwHierarchy.MouseUp += new System.Windows.Forms.MouseEventHandler(this.Event_TvwHierarchy_MouseUp);
            //
            // _pnlState
            //
            resources.ApplyResources(this._pnlState, "_pnlState");
            this._pnlState.BackColor = System.Drawing.SystemColors.Window;
            this._pnlState.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this._pnlState.Name = "_pnlState";
            //
            // _mspMain
            //
            this._mspMain.Items.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiMain,
            this._tmiShowSelectedEnable,
            this._tmiExpandAll,
            this._tmiCollopseAll});
            resources.ApplyResources(this._mspMain, "_mspMain");
            this._mspMain.Name = "_mspMain";
            //
            // _tmiMain
            //
            this._tmiMain.DropDownItems.AddRange(new System.Windows.Forms.ToolStripItem[] {
            this._tmiCut,
            this._tmiCopy,
            this._tmiPaste,
            this._tmiDuplicate,
            this._tmiDelete,
            this.toolStripSeparator1,
            this._tmiSetPaneLocked,
            this._tmiSetPaneUnLocked,
            this._tmiSetPaneHidden,
            this._tmiSetPaneNotHidden,
            this.toolStripSeparator2,
            this._tmiSelectHier,
            this.toolStripSeparator3,
            this._tmiSetHierLocked,
            this._tmiSetHierUnLocked,
            this._tmiSetHierHidden,
            this._tmiSetHierNotHidden,
            this.toolStripSeparator5,
            this._tmiCreateGroup,
            this._tmiEditGroup,
            this.toolStripSeparator4,
            this._tmiChangeToSlectedParts,
            this._tmiOpenSlectedParts,
            this.toolStripSeparator6,
            this._tmiProperty,
            this._tmiShowSelectedEnable2,
            this.toolStripMenuItem1,
            this._tmiAsChild,
            this._tmiAsParent,
            this._tmiCreateNullPaneAndAddChild});
            this._tmiMain.Name = "_tmiMain";
            resources.ApplyResources(this._tmiMain, "_tmiMain");
            this._tmiMain.DropDownOpening += new System.EventHandler(this.Event_TmiMain_DropDownOpening);
            //
            // _tmiCut
            //
            this._tmiCut.Name = "_tmiCut";
            resources.ApplyResources(this._tmiCut, "_tmiCut");
            this._tmiCut.Click += new System.EventHandler(this.Event_MniCut_Click);
            //
            // _tmiCopy
            //
            this._tmiCopy.Name = "_tmiCopy";
            resources.ApplyResources(this._tmiCopy, "_tmiCopy");
            this._tmiCopy.Click += new System.EventHandler(this.Event_MniCopy_Click);
            //
            // _tmiPaste
            //
            this._tmiPaste.Name = "_tmiPaste";
            resources.ApplyResources(this._tmiPaste, "_tmiPaste");
            this._tmiPaste.Click += new System.EventHandler(this.Event_MniPaste_Click);
            //
            // _tmiDelete
            //
            this._tmiDelete.Name = "_tmiDelete";
            resources.ApplyResources(this._tmiDelete, "_tmiDelete");
            this._tmiDelete.Click += new System.EventHandler(this.Event_MniDelete_Click);
            //
            // toolStripSeparator1
            //
            this.toolStripSeparator1.Name = "toolStripSeparator1";
            resources.ApplyResources(this.toolStripSeparator1, "toolStripSeparator1");
            //
            // _tmiSetPaneLocked
            //
            this._tmiSetPaneLocked.Name = "_tmiSetPaneLocked";
            resources.ApplyResources(this._tmiSetPaneLocked, "_tmiSetPaneLocked");
            this._tmiSetPaneLocked.Click += new System.EventHandler(this.Event_MniSetHierToLocked_Click);
            //
            // _tmiSetPaneUnLocked
            //
            this._tmiSetPaneUnLocked.Name = "_tmiSetPaneUnLocked";
            resources.ApplyResources(this._tmiSetPaneUnLocked, "_tmiSetPaneUnLocked");
            this._tmiSetPaneUnLocked.Click += new System.EventHandler(this.Event_MniSetHierToLocked_Click);
            //
            // _tmiSetPaneHidden
            //
            this._tmiSetPaneHidden.Name = "_tmiSetPaneHidden";
            resources.ApplyResources(this._tmiSetPaneHidden, "_tmiSetPaneHidden");
            this._tmiSetPaneHidden.Click += new System.EventHandler(this.Event_MniSetHierToHidden_Click);
            //
            // _tmiSetPaneNotHidden
            //
            this._tmiSetPaneNotHidden.Name = "_tmiSetPaneNotHidden";
            resources.ApplyResources(this._tmiSetPaneNotHidden, "_tmiSetPaneNotHidden");
            this._tmiSetPaneNotHidden.Click += new System.EventHandler(this.Event_MniSetHierToHidden_Click);
            //
            // toolStripSeparator2
            //
            this.toolStripSeparator2.Name = "toolStripSeparator2";
            resources.ApplyResources(this.toolStripSeparator2, "toolStripSeparator2");
            //
            // _tmiSelectHier
            //
            this._tmiSelectHier.Name = "_tmiSelectHier";
            resources.ApplyResources(this._tmiSelectHier, "_tmiSelectHier");
            this._tmiSelectHier.Click += new System.EventHandler(this.Event_MniSelectChildren_Click);
            //
            // toolStripSeparator3
            //
            this.toolStripSeparator3.Name = "toolStripSeparator3";
            resources.ApplyResources(this.toolStripSeparator3, "toolStripSeparator3");
            //
            // _tmiSetHierLocked
            //
            this._tmiSetHierLocked.Name = "_tmiSetHierLocked";
            resources.ApplyResources(this._tmiSetHierLocked, "_tmiSetHierLocked");
            this._tmiSetHierLocked.Click += new System.EventHandler(this.Event_MniSetHierToLocked_Click);
            //
            // _tmiSetHierUnLocked
            //
            this._tmiSetHierUnLocked.Name = "_tmiSetHierUnLocked";
            resources.ApplyResources(this._tmiSetHierUnLocked, "_tmiSetHierUnLocked");
            this._tmiSetHierUnLocked.Click += new System.EventHandler(this.Event_MniSetHierToLocked_Click);
            //
            // _tmiSetHierHidden
            //
            this._tmiSetHierHidden.Name = "_tmiSetHierHidden";
            resources.ApplyResources(this._tmiSetHierHidden, "_tmiSetHierHidden");
            this._tmiSetHierHidden.Click += new System.EventHandler(this.Event_MniSetHierToHidden_Click);
            //
            // _tmiSetHierNotHidden
            //
            this._tmiSetHierNotHidden.Name = "_tmiSetHierNotHidden";
            resources.ApplyResources(this._tmiSetHierNotHidden, "_tmiSetHierNotHidden");
            this._tmiSetHierNotHidden.Click += new System.EventHandler(this.Event_MniSetHierToHidden_Click);
            //
            // toolStripSeparator5
            //
            this.toolStripSeparator5.Name = "toolStripSeparator5";
            resources.ApplyResources(this.toolStripSeparator5, "toolStripSeparator5");
            //
            // _tmiCreateGroup
            //
            this._tmiCreateGroup.Name = "_tmiCreateGroup";
            resources.ApplyResources(this._tmiCreateGroup, "_tmiCreateGroup");
            this._tmiCreateGroup.Click += new System.EventHandler(this.Event_MniCreateGroup);
            //
            // _tmiEditGroup
            //
            this._tmiEditGroup.Name = "_tmiEditGroup";
            resources.ApplyResources(this._tmiEditGroup, "_tmiEditGroup");
            this._tmiEditGroup.Click += new System.EventHandler(this.Event_MniEditGroup);
            //
            // toolStripSeparator4
            //
            this.toolStripSeparator4.Name = "toolStripSeparator4";
            resources.ApplyResources(this.toolStripSeparator4, "toolStripSeparator4");
            //
            // _tmiChangeToSlectedParts
            //
            this._tmiChangeToSlectedParts.Name = "_tmiChangeToSlectedParts";
            resources.ApplyResources(this._tmiChangeToSlectedParts, "_tmiChangeToSlectedParts");
            //
            // _tmiOpenSlectedParts
            //
            this._tmiOpenSlectedParts.Name = "_tmiOpenSlectedParts";
            resources.ApplyResources(this._tmiOpenSlectedParts, "_tmiOpenSlectedParts");
            //
            // toolStripSeparator6
            //
            this.toolStripSeparator6.Name = "toolStripSeparator6";
            resources.ApplyResources(this.toolStripSeparator6, "toolStripSeparator6");
            //
            // _tmiProperty
            //
            this._tmiProperty.Name = "_tmiProperty";
            resources.ApplyResources(this._tmiProperty, "_tmiProperty");
            this._tmiProperty.Click += new System.EventHandler(this.Event_MniProperty_Click);
            //
            // _tmiShowSelectedEnable2
            //
            this._tmiShowSelectedEnable2.Name = "_tmiShowSelectedEnable2";
            resources.ApplyResources(this._tmiShowSelectedEnable2, "_tmiShowSelectedEnable2");
            this._tmiShowSelectedEnable2.Click += new System.EventHandler(this.Event_MniShowSelectedEnable_Click);
            //
            // toolStripMenuItem1
            //
            this.toolStripMenuItem1.Name = "toolStripMenuItem1";
            resources.ApplyResources(this.toolStripMenuItem1, "toolStripMenuItem1");
            //
            // _tmiAsChild
            //
            this._tmiAsChild.Name = "_tmiAsChild";
            resources.ApplyResources(this._tmiAsChild, "_tmiAsChild");
            this._tmiAsChild.Click += new System.EventHandler(this.Event_AsChild_Click);
            //
            // _tmiAsParent
            //
            this._tmiAsParent.Name = "_tmiAsParent";
            resources.ApplyResources(this._tmiAsParent, "_tmiAsParent");
            this._tmiAsParent.Click += new System.EventHandler(this.Event_AsParent_Click);
            //
            // _tmiCreateNullPaneAndAddChild
            //
            this._tmiCreateNullPaneAndAddChild.Name = "_tmiCreateNullPaneAndAddChild";
            resources.ApplyResources(this._tmiCreateNullPaneAndAddChild, "_tmiCreateNullPaneAndAddChild");
            this._tmiCreateNullPaneAndAddChild.Click += new System.EventHandler(this.Event_tmiCreateNullPaneAndAddChild_Click);
            //
            // _tmiShowSelectedEnable
            //
            resources.ApplyResources(this._tmiShowSelectedEnable, "_tmiShowSelectedEnable");
            this._tmiShowSelectedEnable.DisplayStyle = System.Windows.Forms.ToolStripItemDisplayStyle.Image;
            this._tmiShowSelectedEnable.Name = "_tmiShowSelectedEnable";
            this._tmiShowSelectedEnable.Click += new System.EventHandler(this.Event_MniShowSelectedEnable_Click);
            //
            // _tmiExpandAll
            //
            this._tmiExpandAll.AutoToolTip = true;
            this._tmiExpandAll.DisplayStyle = System.Windows.Forms.ToolStripItemDisplayStyle.Image;
            resources.ApplyResources(this._tmiExpandAll, "_tmiExpandAll");
            this._tmiExpandAll.Name = "_tmiExpandAll";
            this._tmiExpandAll.Click += new System.EventHandler(this.Event_tmiExpandAll_Click);
            //
            // _tmiCollopseAll
            //
            this._tmiCollopseAll.AutoToolTip = true;
            this._tmiCollopseAll.DisplayStyle = System.Windows.Forms.ToolStripItemDisplayStyle.Image;
            resources.ApplyResources(this._tmiCollopseAll, "_tmiCollopseAll");
            this._tmiCollopseAll.Name = "_tmiCollopseAll";
            this._tmiCollopseAll.Click += new System.EventHandler(this.Event_tmiCollopseAll_Click);
            //
            // _tmiDuplicate
            //
            this._tmiDuplicate.Name = "_tmiDuplicate";
            resources.ApplyResources(this._tmiDuplicate, "_tmiDuplicate");
            this._tmiDuplicate.Click += new System.EventHandler(this.Event_MniDuplicate_Click);
            //
            // HierarchyWindow
            //
            resources.ApplyResources(this, "$this");
            this.Controls.Add(this._pnlState);
            this.Controls.Add(this._tvwHierarchy);
            this.Controls.Add(this._mspMain);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.SizableToolWindow;
            this.MainMenuStrip = this._mspMain;
            this.Name = "HierarchyWindow";
            this.Activated += new System.EventHandler(this.HierarchyWindow_Activated);
            this._mspMain.ResumeLayout(false);
            this._mspMain.PerformLayout();
            this.ResumeLayout(false);
            this.PerformLayout();

        }
        #endregion


        #endregion デザイナ生成コード


        #region フィールド・プロパティ

        readonly SubSceneManipulator _subSceneMnp = new SubSceneManipulator();
        readonly HierarchyManipulator _hierarchyMnp = new HierarchyManipulator();
        readonly TreeNodeFactory _treeNodeFactory = new TreeNodeFactory(new PaneStateCheckPolicy());
        const int _AnimationOverrayIconIdx = 1;
        const int _CommentOverrayIconIdx = 2;
        const int _AnimAndCommentOverrayIconIdx = 3;
        const int _WarningOverrayIconIdx = 4;
        const int _PrunedPaneOverrayIconIndex = 5;

        /// <summary>
        /// 操作モード
        /// </summary>
        enum ControlMode
        {
            Tree,
            Pane,
        }

        /// <summary>
        /// 複数選択処理モジュール
        /// </summary>
        readonly MultiSelectTreeNodeMgr _multiSelectTreeNodeMgr;

        /// <summary>
        /// ドラッグドロップ処理モジュール
        /// </summary>
        readonly TreeNodeDragDropMgr _treeNodeDragDropMgr;

        readonly TreeNodeStateHolder<IPane> _treeNodeStateHolder = new TreeNodeStateHolder<IPane>(
            (rootPane) => LayoutEditorCore.Scene.ContainsOwnerSubSceneOf(rootPane));

        /// <summary>
        /// 選択ペインを自動表示するか？永続化されるパラメータです。
        /// </summary>
        bool _bEnsureVisibleSelectedNode = true;

        private ShortcutHandler _shortCutHandler = null;

        /// <summary>
        /// 環境設定
        /// </summary>
        AppSetting _AppSetting
        {
            get
            {
                return ViewManager.AppSetting;
            }
        }

        /// <summary>
        /// 選択ノードセット
        /// </summary>
        TreeNode[] _SelectedNodeSet
        {
            get { return _multiSelectTreeNodeMgr.SelectedNodeSet; }
        }

        /// <summary>
        /// 操作中のサブシーン
        /// </summary>
        ISubScene _CurrentSubScene
        {
            get { return LECore.LayoutEditorCore.Scene.CurrentISubScene; }
        }

        /// <summary>
        /// クリップボード
        /// </summary>
        ILEClipboard _ILEClipboard
        {
            get { return LECore.LayoutEditorCore.Scene.ILEClipboard; }
        }

        /// <summary>
        /// 操作モード
        /// </summary>
        ControlMode _ControlMode
        {
            get { return _AppSetting.HierarchyWindowPaneControl ? ControlMode.Pane : ControlMode.Tree; }
        }
        #endregion フィールド

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HierarchyWindow()
        {
            InitializeComponent();
            SetStyle(ControlStyles.DoubleBuffer, true);

            _tvwHierarchy.ImageList = new ImageList();
            _tvwHierarchy.ImageList.ColorDepth = ColorDepth.Depth32Bit;
            PaneItemImageHelper.CopyPaneItemImage(_tvwHierarchy.ImageList);

            _tvwHierarchy.ImageList.Images.Add(PaneItemImageHelper.GetAnimationOverrayIcon());
            int animationOverrayIconIndex = _tvwHierarchy.ImageList.Images.Count - 1;
            OverlayIconHelper.ImageList_SetOverlayImage(_tvwHierarchy.ImageList.Handle, animationOverrayIconIndex, _AnimationOverrayIconIdx);

            _tvwHierarchy.ImageList.Images.Add(PaneItemImageHelper.GetCommentOverrayIcon());
            int commentOverrayIconIndex = _tvwHierarchy.ImageList.Images.Count - 1;
            OverlayIconHelper.ImageList_SetOverlayImage(_tvwHierarchy.ImageList.Handle, commentOverrayIconIndex, _CommentOverrayIconIdx);

            _tvwHierarchy.ImageList.Images.Add(PaneItemImageHelper.GetAnimAndCommentOverrayIcon());
            int animAndCommentOverrayIconIndex = _tvwHierarchy.ImageList.Images.Count - 1;
            OverlayIconHelper.ImageList_SetOverlayImage(_tvwHierarchy.ImageList.Handle, animAndCommentOverrayIconIndex, _AnimAndCommentOverrayIconIdx);

            _tvwHierarchy.ImageList.Images.Add(PaneItemImageHelper.GetWarningOverrayIcon());
            int warningOverrayIconIndex = _tvwHierarchy.ImageList.Images.Count - 1;
            OverlayIconHelper.ImageList_SetOverlayImage(_tvwHierarchy.ImageList.Handle, warningOverrayIconIndex, _WarningOverrayIconIdx);

            _tvwHierarchy.ImageList.Images.Add(PaneItemImageHelper.GetPrunedPaneOverrayIcon());
            int prunedPaneOverrayIconIndex = _tvwHierarchy.ImageList.Images.Count - 1;
            OverlayIconHelper.ImageList_SetOverlayImage(_tvwHierarchy.ImageList.Handle, prunedPaneOverrayIconIndex, _PrunedPaneOverrayIconIndex);

            _multiSelectTreeNodeMgr = new MultiSelectTreeNodeMgr(_tvwHierarchy);
            _multiSelectTreeNodeMgr.ChangeSelecetItemByMouse += SelectPaneOfNode_;
            _multiSelectTreeNodeMgr.CheckNodeContentSelected += CheckNodeContentSelected_;

            _multiSelectTreeNodeMgr.SetNodeApperanceSelected += SetNodeAppearenceSelected_;
            _multiSelectTreeNodeMgr.ResetNodeApperanceSelected += ResetNodeAppearenceSelected_;

            _multiSelectTreeNodeMgr.TreeViewKeyDown += Event_TvwHierarchy_KeyDown;

            _treeNodeDragDropMgr = new TreeNodeDragDropMgr(
                _tvwHierarchy,
                CheckValidToDrop_,
                OnBeginDragHandler_,
                OnDropHandler_);
            _treeNodeDragDropMgr.EnableDragDropDuplicate = true;

            _pnlState.Initialize(_tvwHierarchy);

            // ダイアログの状態を最新にします。
            InitializeDlgState();

            // メッセージフィルタ初期化
            ToolStripMenuItemHelper.ToolStripMessageFilter.BindMessageFilter(_mspMain);

            // ドッキングウインドウ用のショートカットの自前処理の設定
            _shortCutHandler = new ShortcutHandler(_mspMain);
        }


        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>
        /// ペインアイテムの画像番号を取得します。
        /// </summary>
        int GetPaneItemImageIndex_(IPane pane)
        {
            string keyString = PaneItemImageHelper.GetPaneImageKindAsString(pane);
            return _tvwHierarchy.ImageList.Images.IndexOfKey(keyString);
        }

        /// <summary>
        /// ペイン階層をTreeNodeとして加えます。
        /// </summary>
        void AddPaneAsTreeItemNode_(IPane pane, TreeNode treeNode)
        {
            UpdateNodeState_(treeNode, false);

            // 子ペインに対して再帰的に処理をします。
            foreach (IPane childPane in pane.Children)
            {
                TreeNode childNode = _treeNodeFactory.MakeNewNode(childPane);

                AddPaneAsTreeItemNode_(childPane, childNode);
                treeNode.Nodes.Add(childNode);
            }
        }

        /// <summary>
        /// ペイン参照から、ペインに対応するTreeNodeを検索します。
        /// </summary>
        TreeNode FindTreeNodeByPaneRef_(TreeNode treeNode, IPane pane)
        {
            if (treeNode.Tag == pane)
            {
                return treeNode;
            }

            foreach (TreeNode subNode in treeNode.Nodes)
            {
                // 子ノードについても再帰的に調べます。
                TreeNode serchResult = FindTreeNodeByPaneRef_(subNode, pane);
                if (serchResult != null)
                {
                    return serchResult;
                }
            }
            // 子ノード以下では発見できなかった
            return null;
        }

        /// <summary>
        /// ノードの表示状態を更新します。
        /// </summary>
        void UpdateNodeState_(TreeNode treeNode, bool bSelected)
        {
            IPane pane = treeNode.Tag as IPane;
            bool hasAnyTrouble = !string.IsNullOrEmpty(pane.WarningMsg);

            treeNode.Text = PaneTreeViewHelper.GetPaneNodeText(pane);
            treeNode.ToolTipText = hasAnyTrouble ? pane.WarningMsg : PaneTreeViewHelper.GetUICommentText(pane.UserCommentString);
            treeNode.ImageIndex = GetPaneItemImageIndex_(pane);
            treeNode.SelectedImageIndex = treeNode.ImageIndex;

            // アニメーションオーバーレイアイコン
            if (treeNode.TreeView != null)
            {
                if (hasAnyTrouble)
                {
                    OverlayIconHelper.TreeViewOverlay(treeNode, (uint)_WarningOverrayIconIdx);
                }
                else
                {
                    bool paneHasAnimation = pane.HasAnyAnimation();

                    // ノード圧縮の対象か判断する
                    // 状況が整うまでOFFにしておきます。
                    //if (PaneTreeCompressionHelper.CheckPaneTreeCompressionTarget(pane, paneHasAnimation))
                    //{
                    //    OverlayIconHelper.TreeViewOverlay(treeNode, (uint)_PrunedPaneOverrayIconIndex);
                    //    treeNode.ToolTipText = StringResMgr.Get("PANETREE_COMP_HIERTREE_TOOLTIP", pane.PaneName);
                    //}
                    //else
                    {
                        if (paneHasAnimation && pane.UserCommentString != string.Empty)
                        {
                            OverlayIconHelper.TreeViewOverlay(treeNode, (uint)_AnimAndCommentOverrayIconIdx);
                        }
                        else if (paneHasAnimation && pane.UserCommentString == string.Empty)
                        {
                            OverlayIconHelper.TreeViewOverlay(treeNode, (uint)_AnimationOverrayIconIdx);
                        }
                        else if (!paneHasAnimation && pane.UserCommentString != string.Empty)
                        {
                            OverlayIconHelper.TreeViewOverlay(treeNode, (uint)_CommentOverrayIconIdx);
                        }
                        else
                        {
                            OverlayIconHelper.TreeViewOverlay(treeNode, 0);
                        }
                    }
                }
            }


            bool active = !pane.Locked && !pane.Hidden;

            treeNode.ForeColor = !bSelected && pane.IsReadOnlyLocked ? LECore.AppConstants.InactiveForeColor : PaneTreeViewHelper.GetPaneNodeForeColor(active, bSelected);
            treeNode.BackColor = !bSelected && pane.IsReadOnlyLocked ? LECore.AppConstants.InactiveBackColor : PaneTreeViewHelper.GetPaneNodeBackColor(bSelected);

            // UIによる選択変更ではない事を確認します。
            if (!_multiSelectTreeNodeMgr.Updating)
            {
                if (this.Visible & _bEnsureVisibleSelectedNode && bSelected)
                {
                    treeNode.EnsureVisible();
                }
            }
        }

        /// <summary>
        /// すべてのノードの表示状態を更新します。
        /// </summary>
        void UpdateNodeStateAll_(TreeNode treeNode)
        {
            bool bSelected = Array.IndexOf(_multiSelectTreeNodeMgr.SelectedNodeSet, treeNode) != -1;
            UpdateNodeState_(treeNode, bSelected);

            foreach (TreeNode child in treeNode.Nodes)
            {
                UpdateNodeStateAll_(child);
            }
        }

        /// <summary>
        /// ダイアログの状態を最新にします。
        /// </summary>
        public void InitializeDlgState()
        {
            LEToolWindow.DbgReportUpdate(this);

            // スクロール位置の保存。
            int scrollPos_V = LECore.Win32.User32.GetScrollPos(_tvwHierarchy.Handle, LECore.Win32.SB.SB_VERT);


            _treeNodeFactory.Clear();
            _subSceneMnp.BindTarget(_CurrentSubScene);

            TreeNode rootNode = null;
            if (_CurrentSubScene != null)
            {
                IPane rootPane = _CurrentSubScene.RootIPane;

                rootNode = _treeNodeFactory.MakeNewNode(rootPane);
                rootNode.Text = PaneTreeViewHelper.GetPaneNodeText(rootPane);

                // 再帰的に、シーンを走査し、TreeViewNodeを構築します。
                // すべてのノードを展開状態にします。
                AddPaneAsTreeItemNode_(rootPane, rootNode);
                rootNode.ExpandAll();

                // 以前の状態が存在すれば、それを新規作成したデータにコピーします。
                if (_tvwHierarchy.Nodes.Count > 0)
                {
                    // _tvwHierarchy には、RootNode のみが要素として存在するはず。
                    Debug.Assert(_tvwHierarchy.Nodes.Count == 1);

                    TreeNode oldRoot = _tvwHierarchy.Nodes[0] as TreeNode;

                    _treeNodeStateHolder.PreserveNodeState(oldRoot, oldRoot.Tag as IPane);
                    _treeNodeStateHolder.RestoreNodeState(rootNode, rootPane);
                }
            }

            _tvwHierarchy.BeginUpdate();
            // 以前の情報を消去します。
            _tvwHierarchy.Nodes.Clear();

            // 新規データを設定します。
            if (rootNode != null)
            {
                _tvwHierarchy.Nodes.Add(rootNode);

                // オーバーレイアイコン更新は、Treeに設定した後でないとできない。
                UpdateNodeStateAll_(rootNode);

                // ルートは必ず開いておく。
                rootNode.Expand();
            }

            // スクロール位置の復元。
            LECore.Win32.User32.SetScrollPos(_tvwHierarchy.Handle, LECore.Win32.SB.SB_VERT, scrollPos_V, true);

            _tvwHierarchy.EndUpdate();

            // 目玉アイコンの更新
            _pnlState.Invalidate();

            UpdateMenuItemEnableState_();

            if (!_multiSelectTreeNodeMgr.Updating)
            {
                _multiSelectTreeNodeMgr.UpdateSelectedNodeByNodeContent();
            }
        }

        #region MultiSelectTreeNodeMgrハンドラ
        /// <summary>
        /// ペインを選択します。
        /// </summary>
        void SelectPaneOfNode_(TreeNode[] selectedNode)
        {
            _subSceneMnp.BindTarget(_CurrentSubScene);
            _subSceneMnp.BeginSelectSetChange();
            _subSceneMnp.ResetSelectedSet();

            foreach (TreeNode treeNode in selectedNode)
            {

                IPane selsectedPane = treeNode.Tag as IPane;
                if (selsectedPane != null)
                {
                    _subSceneMnp.SelectPanesByPaneRef(selsectedPane);
                }
            }
            _subSceneMnp.EndSelectSetChange();
        }

        /// <summary>
        ///
        /// </summary>
        bool CheckNodeContentSelected_(TreeNode treeNode)
        {
            if (treeNode.Tag is IPane)
            {
                return (treeNode.Tag as IPane).IsSelected;
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        void SetNodeAppearenceSelected_(List<TreeNode> selectedSet)
        {
            // 選択セットの順番を並び替えます
            _multiSelectTreeNodeMgr.SortSelectedSet(_CurrentSubScene?.ISelectedSet?.IPaneArray);

            foreach (TreeNode node in selectedSet)
            {
                UpdateNodeState_(node, true);
            }
        }

        /// <summary>
        ///
        /// </summary>
        void ResetNodeAppearenceSelected_(List<TreeNode> selectedSet)
        {
            foreach (TreeNode node in selectedSet)
            {
                UpdateNodeState_(node, false);
            }
        }
        #endregion MultiSelectTreeNodeMgrハンドラ

        #region TreeNodeDragDropMgr ハンドラ関連
        /// <summary>
        /// 親ペインを取得します。
        /// </summary>
        TreeNode GetParentNode_(TreeNodeDropMode dropMode, TreeNode target)
        {
            switch (dropMode)
            {
                case TreeNodeDropMode.AddChild: return target;
                case TreeNodeDropMode.InsertNext:
                case TreeNodeDropMode.InsertPrev:
                    // 挿入動作の場合、ドロップ先のペインは、兄弟になる。
                    // 親が見つからない場合(Rootが指定されている場合)は自分自身を返す
                    return (target.Parent != null) ? target.Parent : target;
            }

            Debug.Assert(false);
            return null;
        }

        /// <summary>
        /// targetNode が childCandidateNode のドロップ先として正しいか判定します。
        /// </summary>
        TreeNodeDropMode CheckValidToDrop_(TreeNode targetNode, TreeNode[] childCandidateNodeSet)
        {
            TreeNodeDropMode validTypeForAll = TreeNodeDropMode.DropModeMask;


            foreach (TreeNode childNode in childCandidateNodeSet)
            {
                TreeNodeDropMode validType = TreeNodeDropMode.None;

                IPane childCandidate = childNode.Tag as IPane;
                IPane target = targetNode.Tag as IPane;

                // 対象の親ノードが親として有効なら
                if (targetNode.Parent != null)
                {
                    IPane targetParent = targetNode.Parent.Tag as IPane;

                    if (childCandidate.IsValidForChildOf(targetParent))
                    {
                        // 対象の近傍に挿入可能である
                        validType = TreeNodeDropMode.InsertNext | TreeNodeDropMode.InsertPrev;

                        // ただし、対象が読み取り専用の場合は、対象が最終子ノードの場合だけ
                        // 順番変更がおこらず挿入が可能。それ以外では近傍挿入を禁止する。
                        if (target.IsReadOnlyLocked)
                        {
                            int targetIndex = targetParent.IndexOf(target);
                            if (targetParent.Children.Count() - 1 == targetIndex)
                            {
                                validType = TreeNodeDropMode.InsertNext;
                            }
                            else
                            {
                                validType = TreeNodeDropMode.None;
                            }
                        }
                    }
                }

                // 対象ノード自身が親ノードとして、有効なら
                if (childCandidate.IsValidForChildOf(targetNode.Tag as IPane) &&
                    !childCandidate.IsChildrenOf(targetNode.Tag as IPane))
                {
                    // 対象の子供として追加可能である
                    validType |= TreeNodeDropMode.AddChild;
                }
                validTypeForAll &= validType;
            }

            return validTypeForAll;
        }

        /// <summary>
        /// selectedNodeSet が 階層化解除可能か判定します。
        /// 対象ノードの親がRootPaneでなければ階層化解除可能と判定します。
        /// </summary>
        bool CheckValidToReleaseHierarchy(TreeNode[] selectedNodeSet)
        {
            foreach (TreeNode node in selectedNodeSet)
            {
                IPane pane = node?.Parent?.Tag as IPane;
                if (pane != null && !pane.IsRootPane())
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        bool OnBeginDragHandler_(out TreeNode[] selectedNodeSet)
        {
            if (_SelectedNodeSet.Any((node) => (node.Tag as IPane).IsReadOnlyLocked))
            {
                selectedNodeSet = new TreeNode[0];
                return false;
            }

            selectedNodeSet = _SelectedNodeSet;
            return true;
        }

        /// <summary>
        /// ドロップ処理ハンドラ
        /// </summary>
        void OnDropHandler_(TreeNode targetNode, TreeNode[] childNodeSet, TreeNodeDropMode mode)
        {
            if (targetNode == null)
            {
                // 余白へのドロップは、ペインを複製します
                SendViewManagerMessageDuplicate_(null);
            }
            else
            {
                // ノードへのドロップ
                TreeNode parentNode = GetParentNode_(mode, targetNode);
                IPane parentPane = parentNode.Tag as IPane;

                // 親側にキャプチャしているペインがある場合は、キャプチャしているペイン、キャプチャしたテクスチャを使用したペインに依存関係が発生するため子供になれないように制限する。
                if (PaneHelper.CheckHaveCapturedPaneInParentTree(targetNode.Tag as IPane) != null)
                {
                    foreach (var droppedNode in childNodeSet)
                    {
                        IPane result = PaneHelper.CheckHaveCaptureRelativePaneInSubTree(droppedNode.Tag as IPane);
                        if (result != null)
                        {
                            MessageBox.Show(this, LayoutEditor.StringResMgr.Get("TREEVIEW_MSG_DROP_CAPTURE_PANE"), LayoutEditor.StringResMgr.Get("TREEVIEW_MSG_DLG_TITLE"), MessageBoxButtons.OK);
                            return;
                        }
                    }
                }

                if (parentNode != null &&
                    childNodeSet != null &&
                    targetNode != null)
                {
                    // Alt+ノードへのドロップでペインを複製します
                    if ((Control.ModifierKeys & Keys.Alt) != 0)
                    {
                        SendViewManagerMessageDuplicate_(targetNode.Tag as IPane);
                        parentNode.ExpandAll();
                        return;
                    }

                    _CurrentSubScene.BeginMassiveModify();

                    foreach (TreeNode childNode in childNodeSet)
                    {
                        IPane childPane = childNode.Tag as IPane;

                        // HierarchyManipulator.ResetHierarchy( childPane );

                        // 親子関係が設定されていない場合はまず、親子関係を設定します。
                        if (!childPane.IsChildrenOf(parentPane))
                        {
                            HierarchyManipulator.SetHierarchyToPanePair(parentPane, childPane);
                            parentNode.Expand();

                            // ドラッグノードを選択ノードとします。
                            PaneTreeViewHelper.SelectNodeByNodeText(_tvwHierarchy, childNode.Text);
                        }
                    }

                    // 必要があれば、適切な位置に挿入します。
                    if (mode != TreeNodeDropMode.AddChild)
                    {
                        // 挿入インデックスを算出します。
                        int insertIndex = 0;
                        IPane pane = targetNode.Tag as IPane;

                        switch (mode)
                        {
                            case TreeNodeDropMode.InsertPrev:
                                {
                                    insertIndex = parentPane.IndexOf(pane);
                                    break;
                                }
                            case TreeNodeDropMode.InsertNext:
                                {
                                    insertIndex = Math.Min(
                                        parentPane.IndexOf(pane) + 1,
                                        parentPane.Children.Length);
                                    break;
                                }
                            default:
                                {
                                    Debug.Assert(false); break;
                                }
                        }

                        foreach (TreeNode childNode in childNodeSet)
                        {
                            IPane childPane = childNode.Tag as IPane;
                            // 子供の順番を変更します。
                            insertIndex = HierarchyManipulator.ChangeChildOrder(parentPane, childPane, insertIndex);
                        }
                    }

                    _CurrentSubScene.EndMassiveModify();
                }
            }
        }
        #endregion TreeNodeDragDropMgr ハンドラ関連


        #region イベントハンドラ

        /// <summary>
        /// ウインドウアクティベート
        /// </summary>
        private void HierarchyWindow_Activated(object sender, EventArgs e)
        {
            // アクティブになる時にメニューの状態を更新します。
            // 逐一更新しなくても、ショートカット動作で問題は起こりません。
            UpdateMenuItemEnableState_();
        }

        /// <summary>
        /// コンテキストメニューの表示を処理します。
        /// クリック位置から、メニュー内容を変更します。
        /// </summary>
        void OnContextMenuShow_(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            // メニュー項目の状態を更新します。
            UpdateMenuItemEnableState_();


            ToolStripItem[] itemSet = new ToolStripItem[_tmiMain.DropDownItems.Count];
            _tmiMain.DropDownItems.CopyTo(itemSet, 0);

            // コンテキストメニューにメニューを複製
            ContextMenuStrip cm =
                ToolStripMenuItemHelper.MakeContextMenuStripFromToolStripMenuItem(itemSet);

            Control ctrl = sender as Control;
            Debug.Assert(ctrl != null);

            // メッセージ送り主の子供として、表示します。
            Point pntMouse = new Point(e.X, e.Y);
            cm.Show(ctrl, pntMouse);
        }

        /// <summary>
        /// マウスダウン
        /// </summary>
        private void Event_TvwHierarchy_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            // 左クリック
            if (e.Button == MouseButtons.Left)
            {
                // Ctrl Shiftが押されていない
                if ((Control.ModifierKeys & (Keys.Control | Keys.Shift)) == 0)
                {
                    // ダブルクリック
                    if (e.Clicks == 2)
                    {

                        TreeNode node = _tvwHierarchy.GetNodeAt(new Point(e.X, e.Y));
                        if (node != null &&
                            node.Bounds.Contains(new Point(e.X, e.Y)))
                        {
                            // プロパティを開く
                            DoSendMessageToViewManager(ViewManagerMessage.ShowPropertyWndMsg);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// マウスアップ
        /// </summary>
        private void Event_TvwHierarchy_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                // 右クリック
                // コンテキストメニューの表示
                OnContextMenuShow_(sender, e);
            }
        }

        /// <summary>
        /// ノード編集前
        /// </summary>
        private void Event_TvwHierarchy_BeforeLabelEdit(object sender, System.Windows.Forms.NodeLabelEditEventArgs e)
        {
            IPane tagetPane = e.Node.Tag as IPane;
            if (tagetPane.IsRootPane() || tagetPane.IsReadOnlyLocked)
            {
                e.CancelEdit = true;
            }
        }

        /// <summary>
        /// ノードラベル編集によるペイン名の変更。
        /// </summary>
        private void Event_TvwHierarchy_AfterLabelEdit(object sender, System.Windows.Forms.NodeLabelEditEventArgs e)
        {
            if (e.Label != null && e.Node.Tag != null)
            {
                ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                IPane pane = e.Node.Tag as IPane;

                string uniqueName = subScene.GetUniqueNewPaneName(pane, pane.PaneName, e.Label);
                if (PaneHelper.CheckPaneNameValid(uniqueName))
                {
                    if (pane != subScene.RootIPane)
                    {
                        PaneManipulator paneMnp = new PaneManipulator();
                        paneMnp.BindTarget(pane);

                        paneMnp.PaneName = uniqueName;
                    }
                }
            }
            e.CancelEdit = true;
        }

        /// <summary>
        /// キーダウン
        /// </summary>
        private void Event_TvwHierarchy_KeyDown(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Up:
                case Keys.Down:
                case Keys.Left:
                case Keys.Right:
                    {
                        if (_ControlMode == ControlMode.Tree)
                        {
                            // ツリー操作
                            Keys key = e.KeyData & ~Keys.Shift;
                            ControlNodeTree(e.KeyCode);
                        }
                        else if (_ControlMode == ControlMode.Pane)
                        {
                            if (e.Control)
                            {
                                // ノードツリーを操作します
                                ControlNodeTree(e.KeyCode);
                            }
                            else
                            {
                                // ペイン操作
                                Keys key = e.KeyData & ~Keys.Control;
                                var arg = new KeyCmdViewManagerMessage(ViewManagerMessageKind.KeyCmd, key, null);
                                DoSendMessageToViewManager(arg);
                            }

                        }

                        e.Handled = true; // 親子階層ウインドウに割り当てられている
                    }
                    break;
            }
        }

        /// <summary>
        /// ノードツリーを操作します
        /// </summary>
        private void ControlNodeTree(Keys keyCode)
        {
            switch (keyCode)
            {
                case Keys.Up: _multiSelectTreeNodeMgr.UpKeyAction(); break;
                case Keys.Down: _multiSelectTreeNodeMgr.DownKeyAction(); break;
                case Keys.Left: _multiSelectTreeNodeMgr.LeftKeyAction(); break;
                case Keys.Right: _multiSelectTreeNodeMgr.RightKeyAction(); break;
            }

            // 独自実装しているツリービューの管理情報と、ツリビュー実体の選択位置を同期します
            _tvwHierarchy.SelectedNode = _multiSelectTreeNodeMgr.FirstSelectedNode;
        }

        #region キー処理
        /// <summary>
        /// コマンドキーの処理
        /// </summary>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            // ラベル編集中のノードがあれば、固有の処理(削除など）を実行しません。
            foreach (TreeNode node in _SelectedNodeSet)
            {
                if (node.IsEditing)
                {
                    return false;
                }
            }

            if (_shortCutHandler.ProcessCmdKey(keyData))
            {
                return true;
            }

            switch (keyData)
            {
                case Keys.F2:
                    {
                        // ラベル編集
                        if (_SelectedNodeSet.Length == 1)
                        {
                            _SelectedNodeSet[0].BeginEdit();
                        }
                    }
                    break;
                case Keys.Control | Keys.A:
                    // メッセージを伝播します。
                    if (base.NotifyCmdKeyMessageToOwner(ref msg, keyData))
                    {
                        return true;
                    }
                    break;
                case Keys.Shift | Keys.P:
                    {
                        ReleaseHierarchy();
                    }
                    break;
                case Keys.P:
                    {
                        SetHierarchy();
                    }
                    break;
                case Keys.A:
                case Keys.F:
                case Keys.G:
                    {
                        KeyCmdOperation_(keyData);
                    }
                    break;
                case Keys.ControlKey:
                    return true;
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        #endregion キー処理

        #endregion イベントハンドラ

        #region LEToolWindow メンバ

        /// <summary>
        /// シーン変更イベントハンドラ
        /// </summary>
        public override void OnSceneModifyHandler(
            object sender,
            SceneModifyEventArgs e)
        {
            switch (e.Target)
            {
                // ペイン更新
                case SceneModifyEventArgs.Kind.PaneModify:
                    {
                        foreach (IPane changedPane in e.Paramaters)
                        {
                            // 選択ノードの編集ならノード状態を必ず更新します。
                            List<TreeNode> nodeSet = _treeNodeFactory.FindUpdatedNodeByContents(changedPane, !changedPane.IsSelected);
                            if (nodeSet.Count > 0)
                            {
                                _tvwHierarchy.BeginUpdate();
                                nodeSet.ForEach((node) => UpdateNodeState_(node, changedPane.IsSelected));
                                _tvwHierarchy.EndUpdate();
                            }
                        }
                        _pnlState.Invalidate();

                        break;
                    }
                case SceneModifyEventArgs.Kind.PaneAnimAddRemove:
                    {
                        foreach (IPane changedPane in e.Paramaters)
                        {
                            // アニメーションの場合キャッシュ状態にかかわらず表示更新する。
                            List<TreeNode> nodeSet = _treeNodeFactory.FindUpdatedNodeByContents(changedPane, false);
                            if (nodeSet.Count > 0)
                            {
                                nodeSet.ForEach((node) => UpdateNodeState_(node, changedPane.IsSelected));
                            }
                        }
                        _pnlState.Invalidate();
                        break;
                    }
                case SceneModifyEventArgs.Kind.PaneRemove:
                    {
                        _pnlState.Invalidate();
                        break;
                    }
                // 階層の更新
                case SceneModifyEventArgs.Kind.HierarchyModify:
                    {
                        // 親子階層ビューの更新
                        InitializeDlgState();
                        break;
                    }
                // カレントサブシーンの更新
                case SceneModifyEventArgs.Kind.CurrentSubSceneChanged:
                    {
                        // 親子階層ビューの更新
                        // サブシーン固有のデータを表示しているViewは更新する必要があります。
                        InitializeDlgState();
                        break;
                    }
                // サブシーンが削除された
                case SceneModifyEventArgs.Kind.SubSceneRemoved:
                    {
                        _treeNodeStateHolder.RemoveUnnecessaryNodeState();
                        break;
                    }
                // クリップボード状態更新
                case SceneModifyEventArgs.Kind.ClipBoardModify:
                    {
                        // メニューの状態を更新します。
                        UpdateMenuItemEnableState_();
                        break;
                    }
                // 選択セット変更ハンドラ
                case SceneModifyEventArgs.Kind.SelectedSetModify:
                    {
                        IPane[] selSet = e.Paramaters[0] as IPane[];
                        Debug.Assert(selSet != null);

                        if (!_multiSelectTreeNodeMgr.Updating)
                        {
                            _multiSelectTreeNodeMgr.UpdateSelectedNodeByNodeContent();
                        }

                        // メニューの状態を更新します。
                        UpdateMenuItemEnableState_();

                        break;
                    }
                default:
                    {
                        break;
                    }
            }
            base.OnSceneModifyHandler(sender, e);
        }

        public override Keys CustomShortcut { get { return Keys.F10; } }
        #region データ保存
        /// <summary>
        /// 保存(固有の情報を保存します。)
        /// </summary>
        public override void SaveSetting(LEToolFormSetting setting, SaveSettingOption option)
        {
            if (option.AlsoSaveOtherThanWorkspace)
            {
                LEControlUserDataChunk data = null;
                data = new LEControlUserDataChunk("ShowSelectNodeEnable", this._bEnsureVisibleSelectedNode.ToString());
                setting.UserDataSet.Add(data);
            }

            base.SaveSetting(setting, option);
        }

        /// <summary>
        /// 読み込み(固有の情報を読み込みます。)
        /// </summary
        public override void LoadSetting(LEToolFormSetting setting, LoadSettingOption option)
        {
            base.LoadSetting(setting, option);

            if (option.AlsoLoadOtherThanWorkspace)
            {
                setting.TryToFindUserBoolDataByName("ShowSelectNodeEnable", out _bEnsureVisibleSelectedNode);
            }
        }

        #endregion データ保存


        #endregion LEToolWindow メンバ

        #region メニュー項目

        /// <summary>
        /// 再帰的に選択します。
        /// </summary>
        void SelectPaneRecursive_(SubSceneManipulator subSceneMnp, IPane pane)
        {
            subSceneMnp.SelectPanesByPaneRef(pane);
            foreach (IPane child in pane.Children)
            {
                SelectPaneRecursive_(subSceneMnp, child);
            }
        }

        /// <summary>
        /// 階層以下を選択
        /// </summary>
        private void Event_MniSelectChildren_Click(object sender, System.EventArgs e)
        {
            SubSceneManipulator subSceneMnp = new SubSceneManipulator();
            subSceneMnp.BindTarget(_CurrentSubScene);
            subSceneMnp.BeginSelectSetChange();

            foreach (TreeNode treeNode in _SelectedNodeSet)
            {
                if (treeNode.Tag is IPane)
                {
                    SelectPaneRecursive_(subSceneMnp, treeNode.Tag as IPane);
                }
            }
            subSceneMnp.EndSelectSetChange();
        }

        /// <summary>
        /// ペインセットのロック状態を設定します。
        /// </summary>
        void SetPaneSetLockedState_(PaneManipulator paneMnp, bool bLocked, bool bRecursive)
        {
            paneMnp.Locked = bLocked;

            if (bRecursive)
            {
                IPane pane = paneMnp.IPane;
                foreach (IPane childPane in pane.Children)
                {
                    paneMnp.BindTarget(childPane);
                    SetPaneSetLockedState_(paneMnp, bLocked, bRecursive);
                }
                paneMnp.BindTarget(pane);
            }
        }

        /// <summary>
        /// ペインセットの隠された状態を設定します。
        /// </summary>
        void SetPaneSetHiddenState_(PaneManipulator paneMnp, bool bHidden, bool bRecursive)
        {
            paneMnp.Hidden = bHidden;

            if (bRecursive)
            {
                IPane pane = paneMnp.IPane;
                foreach (IPane childPane in pane.Children)
                {
                    paneMnp.BindTarget(childPane);
                    SetPaneSetHiddenState_(paneMnp, bHidden, bRecursive);
                }
                paneMnp.BindTarget(pane);
            }
        }

        /// <summary>
        /// メニューアイテムが等しいか判定します。
        /// sender は コピーである可能性があるため、文字列で判定します。
        /// </summary>
        static bool IsMenuItemEqual_(ToolStripMenuItem mniL, ToolStripMenuItem mniR)
        {
            return mniL.Text == mniR.Text;
        }

        /// <summary>
        /// 選択ペインをロック、ロック解除する
        /// </summary>
        private void Event_MniSetHierToLocked_Click(object sender, System.EventArgs e)
        {
            ToolStripMenuItem mi = sender as ToolStripMenuItem;
            PaneManipulator paneMnp = new PaneManipulator();
            bool bLocked = (IsMenuItemEqual_(mi, _tmiSetPaneLocked) ||
                                IsMenuItemEqual_(mi, _tmiSetHierLocked));
            bool bRecursive = (IsMenuItemEqual_(mi, _tmiSetHierLocked) ||
                                IsMenuItemEqual_(mi, _tmiSetHierUnLocked));

            _CurrentSubScene.BeginMassiveModify();
            foreach (TreeNode treeNode in _SelectedNodeSet)
            {
                if (treeNode.Tag is IPane)
                {
                    paneMnp.BindTarget(treeNode.Tag as IPane);
                    SetPaneSetLockedState_(paneMnp, bLocked, bRecursive);
                }
            }
            _CurrentSubScene.EndMassiveModify();
        }

        /// <summary>
        /// 選択ペインを隠す、あらわす
        /// </summary>
        private void Event_MniSetHierToHidden_Click(object sender, System.EventArgs e)
        {
            ToolStripMenuItem mi = sender as ToolStripMenuItem;
            PaneManipulator paneMnp = new PaneManipulator();

            bool bHidden = (IsMenuItemEqual_(mi, _tmiSetPaneHidden) ||
                                IsMenuItemEqual_(mi, _tmiSetHierHidden));
            bool bRecursive = (IsMenuItemEqual_(mi, _tmiSetHierHidden) ||
                                IsMenuItemEqual_(mi, _tmiSetHierNotHidden));

            _CurrentSubScene.BeginMassiveModify();
            foreach (TreeNode treeNode in _SelectedNodeSet)
            {
                if (treeNode.Tag is IPane)
                {
                    paneMnp.BindTarget(treeNode.Tag as IPane);
                    SetPaneSetHiddenState_(paneMnp, bHidden, bRecursive);
                }
            }
            _CurrentSubScene.EndMassiveModify();
        }

        /// <summary>
        /// プロパティ
        /// </summary>
        private void Event_MniProperty_Click(object sender, System.EventArgs e)
        {
            // プロパティを開く
            DoSendMessageToViewManager(ViewManagerMessage.ShowPropertyWndMsg);
        }

        /// <summary>
        /// 選択ペインを自動的に表示
        /// </summary>
        private void Event_MniShowSelectedEnable_Click(object sender, EventArgs e)
        {
            ToolStripMenuItem mi = sender as ToolStripMenuItem;
            mi.Checked = !mi.Checked;

            _bEnsureVisibleSelectedNode = mi.Checked;
        }

        /// <summary>
        /// メニューアイテムの状態を更新します。
        /// </summary>
        void UpdateMenuItemEnableState_()
        {
            TreeNode[] selectedNodeSet = _SelectedNodeSet;

            // 子供を持つか調査し、メニュー項目の状態を変更します。
            // ロック、表示の有効状態を設定します。
            bool bHasChildren = false;
            bool bHiddenOr = false;
            bool bLockedOr = false;
            bool bHiddenAnd = true;
            bool bLockedAnd = true;
            bool bSceneExist = _CurrentSubScene != null;

            bool bPaneExist = false;
            bool bReadOnlyPaneExist = false;
            bool bPartsPaneExist = false;
            foreach (TreeNode node in selectedNodeSet)
            {
                IPane pane = node.Tag as IPane;
                bHasChildren |= pane.HasChildren;
                if (PaneHelper.IsRootPane(pane) == false)
                {
                    bHiddenOr |= pane.Hidden;
                    bLockedOr |= pane.Locked;
                    bHiddenAnd &= pane.Hidden;
                    bLockedAnd &= pane.Locked;
                    bPaneExist = true;

                    if (pane.IsReadOnlyLocked)
                    {
                        bReadOnlyPaneExist = true;
                    }

                    if (pane.PaneKind == PaneKind.Parts)
                    {
                        bPartsPaneExist = true;
                    }
                }
            }

            bPaneExist &= bSceneExist;

            _tmiMain.Enabled = bSceneExist;
            _tmiSetHierHidden.Enabled = bHasChildren;
            _tmiSetHierLocked.Enabled = bHasChildren;
            _tmiSetHierNotHidden.Enabled = bHasChildren;
            _tmiSetHierUnLocked.Enabled = bHasChildren;
            _tmiSelectHier.Enabled = bHasChildren;

            _tmiSetPaneHidden.Enabled = !bHiddenAnd;
            _tmiSetPaneNotHidden.Enabled = bHiddenOr;
            _tmiSetPaneLocked.Enabled = !bLockedAnd;
            _tmiSetPaneUnLocked.Enabled = bLockedOr;
            _tmiProperty.Enabled = bPaneExist;

            // 選択を表示チェックを更新します。
            if (_tmiShowSelectedEnable.Checked != _bEnsureVisibleSelectedNode)
            {
                _tmiShowSelectedEnable.Checked = _bEnsureVisibleSelectedNode;
            }

            // 貼り付けが有効か判定します。
            _tmiPaste.Enabled = _ILEClipboard.CanPaste(typeof(ClipBoardPane)) && bSceneExist && !bPartsPaneExist;
            _tmiCopy.Enabled = bPaneExist;
            _tmiCut.Enabled = bPaneExist & !bReadOnlyPaneExist;
            _tmiDelete.Enabled = bPaneExist & !bReadOnlyPaneExist;

            _tmiShowSelectedEnable2.Checked = _bEnsureVisibleSelectedNode;

            //
            _tmiCreateGroup.Enabled = bPaneExist;
            _tmiEditGroup.Enabled = bPaneExist && _CurrentSubScene.ILEGroupMgr.GroupSet.Length != 0;

            // 部品レイアウトを開くメニュー
            {
                _tmiChangeToSlectedParts.Enabled = _tmiOpenSlectedParts.Enabled = toolStripSeparator6.Visible = false;
                if (bPartsPaneExist)
                {
                    // 選択ペイン
                    TreeNode treeNode = _SelectedNodeSet.First();
                    var targetPane = treeNode.Tag as IPane;
                    if (targetPane != null && targetPane.PaneKind == PaneKind.Parts)
                    {
                        _tmiChangeToSlectedParts.Enabled = _tmiOpenSlectedParts.Enabled = toolStripSeparator6.Visible = true;

                        DoUpdateSubPartsDropDownItems_(_tmiChangeToSlectedParts, targetPane, Event_TmiChangeCurrentToSlectedParts_Click);
                        DoUpdateSubPartsDropDownItems_(_tmiOpenSlectedParts, targetPane, Event_tmiOpenSlectedParts_Click);
                    }
                }
            }

            _tmiAsChild.Enabled = AsChildEnable_();
            _tmiAsParent.Enabled = AsParentEnable_();
            _tmiCreateNullPaneAndAddChild.Enabled = _SelectedNodeSet.Where(node => node.Parent != null).Count() > 0;
        }

        private bool AsChildEnable_()
        {
            return _SelectedNodeSet.Count() >= 2;
        }

        private bool AsParentEnable_()
        {
            return _SelectedNodeSet.Where(node => node.Parent?.Parent != null).Count() > 0;
        }

        /// <summary>
        /// メニューポップアップ
        /// </summary>
        private void Event_MniMain_Popup(object sender, EventArgs e)
        {
            UpdateMenuItemEnableState_();
        }
        #endregion メニュー項目

        #region コピー・貼り付け
        /// <summary>
        /// 切り取り
        /// </summary>
        private void Event_MniCut_Click(object sender, EventArgs e)
        {
            if (CopyToClipboard_())
            {
                // コピーが成功したら削除。
                Event_MniDelete_Click(sender, e);
            }
        }

        /// <summary>
        /// クリップボードにコピーします。
        /// </summary>
        bool CopyToClipboard_()
        {
            // 選択ノードから、ペイン列を取り出し、コピーします。
            IPane[] selectedSet = _CurrentSubScene?.ISelectedSet?.IPaneArray;
            if(selectedSet == null || selectedSet.Length <= 0)
            {
                return false;
            }

            ClipBoardPane clipBoardPane = new ClipBoardPane(
                selectedSet,
                PaneSetDupulicator.Option.CopyAnimation |
                PaneSetDupulicator.Option.CopyHierarchy |
                PaneSetDupulicator.Option.ConvertTransToWorld);

            // クリップボードへ登録。
            _ILEClipboard.Copy(clipBoardPane);
            return true;
        }

        /// <summary>
        /// DropDownOpening
        /// </summary>
        private void Event_TmiMain_DropDownOpening(object sender, EventArgs e)
        {
            UpdateMenuItemEnableState_();
        }

        /// <summary>
        /// コピー
        /// </summary>
        private void Event_MniCopy_Click(object sender, EventArgs e)
        {
            CopyToClipboard_();
        }

        /// <summary>
        /// paneSet に parentPaneの子が存在するか調査します。
        /// </summary>
        static bool ExsistChildInPaneSet_(IPane parentPane, List<IPane> paneSet)
        {
            return paneSet.Exists(delegate(IPane memberPane)
                {
                    return memberPane.IsChildrenOf(parentPane);
                });
        }

        /// <summary>
        /// 貼り付け
        /// </summary>
        private void Event_MniPaste_Click(object sender, EventArgs e)
        {
            // 貼り付け対象ペインセットを計算します。
            TreeNode[] nodeSet = _multiSelectTreeNodeMgr.SelectedNodeSet;
            List<IPane> pasteTargetPaneSet = new List<IPane>();
            foreach (TreeNode node in nodeSet)
            {
                IPane newPane = node.Tag as IPane;
                if (newPane != null)
                {
                    pasteTargetPaneSet.Add(newPane);
                }
            }

            // 選択ペインのうち、もっとも子階層にあるペインを収集します。
            pasteTargetPaneSet = pasteTargetPaneSet.FindAll(delegate(IPane pasteTarget)
            {
                // ペインの親が、選択されていないペインを抽出します。
                return !ExsistChildInPaneSet_(pasteTarget, pasteTargetPaneSet);
            });

            // クリップボードにコピーされたデータの中にキャプチャテクスチャ関連のペインが存在する場合は
            // 不正な依存関係を作成しないために親になるペインのツリーにキャプチャしているペインが含まれていないかチェックする。
            if (SubSceneHelper.DoesClipboardHaveCaptureTextureRelativePane())
            {
                foreach(IPane pane in pasteTargetPaneSet)
                {
                    IPane result = PaneHelper.CheckHaveCapturedPaneInParentTree(pane);
                    if (result != null)
                    {
                        MessageBox.Show(this, LayoutEditor.StringResMgr.Get("TREEVIEW_MSG_PASTE_CAPTURE_PANE"), LayoutEditor.StringResMgr.Get("TREEVIEW_MSG_DLG_TITLE"), MessageBoxButtons.OK);
                        return;
                    }
                }
            }

            if (pasteTargetPaneSet.Count <= 0)
            {
                return;
            }

            _subSceneMnp.ISubScene.BeginMassiveModify();


            // すべての貼り付け対象ペインについて
            foreach (IPane pasteTargetPane in pasteTargetPaneSet)
            {
                // 貼り付けキーメッセージを送信し、貼り付けを実行します。
                Message dummyMsg = new Message();
                base.NotifyCmdKeyMessageToOwner(ref dummyMsg, Keys.Control | Keys.V);

                // 貼り付けたペインが選択されているはず。
                //
                // 貼り付けたペインセットから、
                // もっとも親子階層、上位のペイン（親がセット内に含まれないペイン）
                // のみを収集します。
                IPane[] paneSet = _CurrentSubScene.ISelectedSet.IPaneArray;
                paneSet = Array.FindAll(paneSet, delegate(IPane memberPane)
                {
                    return Array.IndexOf(paneSet, memberPane.Parent) == -1;
                });

                // すべてのペインを貼り付け対象の子供に設定します。
                foreach (IPane pane in paneSet)
                {
                    HierarchyManipulator.SetHierarchyToPanePair(pasteTargetPane, pane);
                }
            }

            _subSceneMnp.ISubScene.EndMassiveModify();
        }

        /// <summary>
        /// 複製
        /// </summary>
        private void Event_MniDuplicate_Click(object sender, EventArgs e)
        {
            SendViewManagerMessageDuplicate_(null);
        }
        private void SendViewManagerMessageDuplicate_(IPane parent)
        {
            var arg = new ViewManagerMessage(ViewManagerMessageKind.DuplicatePanes, parent);
            DoSendMessageToViewManager(arg);
        }

        /// <summary>
        /// 削除
        /// </summary>
        private void Event_MniDelete_Click(object sender, EventArgs e)
        {
            // 削除キーメッセージを送信します。
            Message dummyMsg = new Message();
            base.NotifyCmdKeyMessageToOwner(ref dummyMsg, Keys.Delete);
        }
        #endregion コピー・貼り付け

        /// <summary>
        /// グループを作成
        /// </summary>
        private void Event_MniCreateGroup(object sender, EventArgs e)
        {
            Message dummyMsg = new Message();
            base.NotifyCmdKeyMessageToOwner(ref dummyMsg, Keys.Control | Keys.G);
        }

        /// <summary>
        /// グループを編集
        /// </summary>
        private void Event_MniEditGroup(object sender, EventArgs e)
        {
            Message dummyMsg = new Message();
            base.NotifyCmdKeyMessageToOwner(ref dummyMsg,
                                             Keys.Control | Keys.Alt | Keys.Shift | Keys.G);
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TvwHierarchy_AfterExpand(object sender, TreeViewEventArgs e)
        {
            if ((Control.ModifierKeys & Keys.Shift) != 0)
            {
                e.Node.ExpandAll();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void Event_TvwHierarchy_AfterCollapse(object sender, TreeViewEventArgs e)
        {
            if ((Control.ModifierKeys & Keys.Shift) != 0)
            {
                CollapseNodeRecursive(e.Node);
            }
        }

        ///
        private void CollapseNodeRecursive(TreeNode node)
        {
            node.Collapse();

            foreach (TreeNode childNode in node.Nodes)
            {
                CollapseNodeRecursive(childNode);
            }
        }

        /// <summary>
        /// すべて展開
        /// </summary>
        private void Event_tmiExpandAll_Click(object sender, EventArgs e)
        {
            _tvwHierarchy.BeginUpdate();
            _tvwHierarchy.ExpandAll();
            _tvwHierarchy.EndUpdate();
        }

        /// <summary>
        /// すべて閉じる
        /// </summary>
        private void Event_tmiCollopseAll_Click(object sender, EventArgs e)
        {
            _tvwHierarchy.BeginUpdate();
            _tvwHierarchy.CollapseAll();
            if (_tvwHierarchy.Nodes.Count > 0)
            {
                _tvwHierarchy.Nodes[0].Expand();
            }
            _tvwHierarchy.EndUpdate();
        }

        /// <summary>
        /// 配下のパーツ用編集対象を切り替えるのクリック時ハンドラーです。
        /// </summary>
        private void Event_TmiChangeCurrentToSlectedParts_Click(object sender, EventArgs e)
        {
            var item = sender as ToolStripDropDownItem;
            IPartsSubScene ps = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(item.Text);
            if(ps == null || !ps.IsLoaded)
            {
                LayoutEditorCore.MsgReporter.ReportError(
                    StringResMgr.Get("ERROR_PARTS_NOT_LOAD_TITLE"),
                    StringResMgr.Get("ERROR_PARTS_NOT_LOADED", item.Text)
                );
                return;
            }

            var lw = ViewManager.FindLayoutWindowBySubScene(_CurrentSubScene);
            lw.OpenPartsAsCurrentEditTargetByFilePath(ps.FilePath);
        }

        /// <summary>
        /// 配下のパーツ用編集対象を切り替えるのクリック時ハンドラーです。
        /// </summary>
        private void Event_tmiOpenSlectedParts_Click(object sender, EventArgs e)
        {
            var item = sender as ToolStripDropDownItem;
            IPartsSubScene ps = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(item.Text);
            if (ps == null || !ps.IsLoaded)
            {
                LayoutEditorCore.MsgReporter.ReportError(
                    StringResMgr.Get("ERROR_PARTS_NOT_LOAD_TITLE"),
                    StringResMgr.Get("ERROR_PARTS_NOT_LOADED", item.Text)
                );
                return;
            }

            try
            {
                LayoutEditorCore.MsgReporter.BeginPacking(LECore.LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSLAYOUTLOADING"));
                var args = new ViewManagerMessage(ViewManagerMessageKind.OpenLayout, ps.FilePath);
                this.DoSendMessageToViewManager(args);
            }
            finally
            {
                LayoutEditorCore.MsgReporter.EndPacking();
            }
        }

        /// <summary>
        /// 階層化のクリック時ハンドラーです。
        /// </summary>
        private void Event_AsChild_Click(object sender, EventArgs e)
        {
            SetHierarchy();
        }

        /// <summary>
        /// 階層化の解除のクリック時ハンドラーです。
        /// </summary>
        private void Event_AsParent_Click(object sender, EventArgs e)
        {
            ReleaseHierarchy();
        }

        /// <summary>
        /// ヌルペインを作成して選択ペインの子供にするのクリック時ハンドラーです。
        /// </summary>
        private void Event_tmiCreateNullPaneAndAddChild_Click(object sender, EventArgs e)
        {
            KeyCmdOperation_(Keys.G);
        }

        /// <summary>
        /// 配下のパーツ用ドロップダウンリストの更新をします。
        /// </summary>
        private void DoUpdateSubPartsDropDownItems_(ToolStripMenuItem rootItemNode, IPane partsPane, Action<object, EventArgs> onClick)
        {
            IPartsLayout partsLayout = partsPane.IPartsLayout;

            // 参照レイアウト
            // 依存階層が深い順に列挙されるので、ひっくり返す。
            List<string> relatedFiles = new List<string>(partsLayout.EnumlateReferencingLayoutFiles());
            relatedFiles.Reverse();

            // 基底部品
            IPartsSubScene partsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsPane.IPartsLayout.PartsLayoutName);
            if (partsSubScene != null && partsSubScene.IsDerivativeParts())
            {
                relatedFiles.Add(partsSubScene.BasePartsName);
            }

            rootItemNode.DropDownItems.Clear();
            foreach (var relatedFile in relatedFiles.Distinct())
            {
                ToolStripMenuItem item = new ToolStripMenuItem(relatedFile);
                item.Click += new EventHandler(onClick);

                rootItemNode.DropDownItems.Add(item);
            }
        }

        /// <summary>
        /// 親子階層を設定します。
        /// </summary>
        public void SetHierarchy()
        {
            if (!AsChildEnable_())
            {
                return;
            }

            var childNodes = _multiSelectTreeNodeMgr.SelectedNodeSet.Reverse().Skip(1).ToArray();
            if (CheckValidToDrop_(_multiSelectTreeNodeMgr.FirstSelectedNode, childNodes) == TreeNodeDropMode.DropModeMask)
            {
                // 親子階層を設定します
                // 最後に選択されたノードが親になります
                OnDropHandler_(
                    _multiSelectTreeNodeMgr.FirstSelectedNode,
                    childNodes,
                    TreeNodeDropMode.AddChild);
            }
            else
            {
                // エラーを表示する
                LayoutEditorMessageArgs args = new LayoutEditorMessageArgs()
                {
                    title = StringResMgr.Get("TAG_ERROR"),
                    msg = string.Format(StringResMgr.Get("HIERARCHY_ADDCHILD_ERROR"), _multiSelectTreeNodeMgr.FirstSelectedNode.Text),
                    messageType = MessageReportDlg.MessageTypes.Always,
                    chkIgnoreSimilarMessageEnabled = false,
                    buttonType = MessageReportDlg.ButtonTypes.Ok
                };
                LayoutEditorCore.MsgReporter.ReportError(this, args);
            }
        }

        /// <summary>
        /// 階層化を解除します。
        /// </summary>
        public void ReleaseHierarchy()
        {
            if (!AsParentEnable_())
            {
                return;
            }

            // 選択ペインセットからRootPaneを除外します
            var targetNodeSet = _multiSelectTreeNodeMgr.SelectedNodeSet.Where(node => node.Parent != null).ToArray();
            if (targetNodeSet.Count() == 0)
            {
                return;
            }

            if (CheckValidToReleaseHierarchy(targetNodeSet))
            {
                // 階層化を解除します
                OnDropHandler_(
                    _tvwHierarchy.Nodes[0],
                    targetNodeSet,
                    TreeNodeDropMode.AddChild);
            }
        }

        /// <summary>
        /// VieManagerにKeyCmdを通知します。
        /// </summary>
        private void KeyCmdOperation_(Keys key)
        {
            var paneSet = _multiSelectTreeNodeMgr.SelectedNodeSet.Select<TreeNode, IPane>(node => node.Tag as IPane);
            var arg = new KeyCmdViewManagerMessage(ViewManagerMessageKind.KeyCmd, key, paneSet);
            DoSendMessageToViewManager(arg);
        }
    }
}
