﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Diagnostics;
using System.Data;
using System.Windows.Forms;

using LECore.Structures.Core;
using LECore;
using LECore.Structures;

namespace LayoutEditor.Forms.ToolWindows.CurveEditWindow
{
    /// <summary>
    /// 時間目盛りを表示するコントロールです。
    /// </summary>
    public class TimeScaleControl : System.Windows.Forms.Control
    {

        #region ---------------- フィールド ----------------
        /// <summary>
        /// 必要なデザイナ変数です。
        /// </summary>
        private System.ComponentModel.Container components = null;

        ITime _time = null;
        IAnmKeyFrame[] _targetKeyframes = null;

        // タグ区間の範囲
        int _sectionStartTime = 0;
        int _sectionEndTime = 0;

        // マウス状態
        bool             _mouseClicked = false;
        // 時間区間 開始位置
        int              _beginTime    = 0;
        // 時間区間 終了位置
        int              _endTime      = 100;
        // 時間区間 現在位置(現在位置にはポインタが表示されます。)
        int              _currentTime  = 0;

        Brush _brush = new SolidBrush(Color.FromArgb(255, 230, 230, 255));
        Brush _disableBrush = new SolidBrush(Color.FromArgb(200, Color.LightGray));
        Brush _sectionBrush = new SolidBrush(Color.FromArgb(255, Color.White));
        Brush _keyBrush = Brushes.Red;
        Pen _opqPen = new Pen(Color.FromArgb(32, 255, 127, 63));
        Pen _pen = new Pen(Color.FromArgb(255, 255, 127, 63));
        Pen _keyOutlinePen = new Pen(Color.LightSlateGray);
        Pen _blackPen = new Pen(Color.Black);
        Pen _grayPen = new Pen(Color.Gray);
        Font _font = new Font("Tahoma", 8.25F);

        const int KeyFrameHilightW = 5;


        #endregion // ---------------- フィールド ----------------

        #region ---------------- プロパティ ----------------

        /// <summary>
        /// 時間区間 現在位置を設定します。
        /// </summary>
        public int CurrrentTime
        {
            set
            {
                _currentTime = value;
                Invalidate();
            }
        }

        #endregion // ---------------- プロパティ ----------------

        #region イベント
        public event EventHandler BeginDrag;
        public event EventHandler EndDrag;
        #endregion

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TimeScaleControl()
        {
            // この呼び出しは、Windows.Forms フォーム デザイナで必要です。
            InitializeComponent();

            SetStyle(ControlStyles.DoubleBuffer, true);
        }

        public void Initialize(ITime time)
        {
            _time = time;

            _beginTime = _time.AnimPlayStartTime;
            _endTime = _time.AnimPlayEndTime;
            _currentTime = _time.Time;
        }

        /// <summary>
        /// 表示対象を設定します。
        /// </summary>
        /// <param name="targetAnimatable"></param>
        public void SetTraget(IAnmCurve[] targetAnimatables, IAnimFrameSection animFrameSection)
        {
            List<IAnmKeyFrame> keies = new List<IAnmKeyFrame>();
            foreach(var curve in targetAnimatables)
            {
                foreach(var key in curve.IKeyFrameSet)
                {
                    if (!keies.Exists((k) => k.Time == key.Time))
                    {
                        keies.Add(key);
                    }
                }
            }

            keies.Sort(delegate (IAnmKeyFrame k1, IAnmKeyFrame k2)
            {
                if (k2.Time - k1.Time > 0)
                {
                    return -1;
                }
                else if (k2.Time - k1.Time < 0)
                {
                    return 1;
                }
                else
                {
                    return 0;
                }
            });

            _targetKeyframes = keies.ToArray();

            _sectionStartTime = animFrameSection != null ? animFrameSection.StartFrame : 0;
            _sectionEndTime = animFrameSection != null ? animFrameSection.EndFrame : 0;

            Invalidate();
        }

        /// <summary>
        /// 表示対象設定を解除します。
        /// </summary>
        public void ResetTraget()
        {
            _targetKeyframes = null;
            Invalidate();
        }

        /// <summary>
        /// フレーム範囲を設定します。
        /// </summary>
        public void SetFrameRange( int start, int End )
        {
            if( start < End )
            {
                if( _beginTime != start || _endTime != End )
                {
                    _beginTime = start;
                    _endTime = End;

                    Invalidate();
                }
            }
            else
            {
                Debug.Assert( false );
            }
        }

        /// <summary>
        /// 使用されているリソースに後処理を実行します。
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if( components != null )
                    components.Dispose();
            }
            base.Dispose( disposing );
        }

        #region 外部非公開メソッド
        void ChangeCurrentTimeAccordingToClickPos_( int mouseX )
        {
            if( ClientRectangle.Contains( new Point( mouseX, 0 ) ) )
            {
                NumberLineDrawer     numberLineDrawer = new NumberLineDrawer( ClientSize.Width , _beginTime, _endTime );
                int newTime = numberLineDrawer.CalcNumberLineValFromPixPosition( mouseX );

                if (_time.Time != newTime)
                {
                    if (!_time.IsPlaying)
                    {
                        _time.IsPlaying = true;
                    }
                    _time.Time = newTime;

                    Invalidate();
                }
            }
        }

        /// <summary>
        /// ScaleDrawer から呼ばれる描画コールバック
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="diffPix"></param>
        /// <param name="val"></param>
        /// <returns></returns>
        bool OnNumberLineDrawHandler_( NumberLineDrawer sender, int diffPix, int val )
        {

            Graphics g = sender.UserDrawParams as Graphics;

            g.DrawLine(_blackPen, new Point(diffPix, 0), new Point(diffPix, ClientSize.Height));
            g.DrawString(val.ToString(), _font, Brushes.Black, diffPix, 0);
            return true;
        }
        #endregion

        #region コンポーネント デザイナで生成されたコード
        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード］エディタで変更しないでください。
        /// </summary>
        private void InitializeComponent()
        {
            this.SuspendLayout();
            //
            // TimeScaleControl
            //
            this.BackColor = System.Drawing.SystemColors.ButtonFace;
            this.SizeChanged += new System.EventHandler(this.Event_TimeScaleControl_SizeChanged);
            this.ResumeLayout(false);

        }
        #endregion

        #region 再描画関連
        protected override void OnPaint(PaintEventArgs pe)
        {
            NumberLineDrawer numberLineDrawer = new NumberLineDrawer(this.ClientSize.Width, _beginTime, _endTime);
            int keyFrmW = (int)numberLineDrawer.PixelPerVal;
            Graphics g = pe.Graphics;

            // 目盛りと現在フレーム
            {
                // 全体を灰色で塗りつぶす
                g.FillRectangle(_disableBrush, 0, 0, this.ClientSize.Width, this.ClientSize.Height);

                // タグ区間の範囲を塗りつぶす
                int sectionStartPos = numberLineDrawer.CalcPixPosFromNumberLineVal(_sectionStartTime);
                int sectionWidth = numberLineDrawer.CalcPixPosFromNumberLineVal(_sectionEndTime - _sectionStartTime);
                g.FillRectangle(_sectionBrush, sectionStartPos, 0, sectionWidth, this.ClientSize.Height);

                if (_targetKeyframes != null && _targetKeyframes.Length > 0)
                {
                    // キーフレームの存在する区間を別の色でハイライト表示します。
                    IAnmKeyFrame[] keyFrameSet = _targetKeyframes;
                    int keyFirst = numberLineDrawer.CalcPixPosFromNumberLineVal(keyFrameSet[0].TimeAsInt);
                    int keyLast = numberLineDrawer.CalcPixPosFromNumberLineVal(keyFrameSet[_targetKeyframes.Length - 1].TimeAsInt);

                    int keyHeight = this.ClientSize.Height / 3;
                    g.FillRectangle(_brush, keyFirst, this.ClientSize.Height - keyHeight, keyLast + KeyFrameHilightW - keyFirst, keyHeight);
                }

                numberLineDrawer.MinNumberLineSize = 64;
                numberLineDrawer.UserDrawParams = pe.Graphics;
                numberLineDrawer.DoDraw(new NumberLineDrawer.OnNumberLineDrawHandler(OnNumberLineDrawHandler_));

                // カレントフレームの位置を塗りつぶす
                int currFrmPos = numberLineDrawer.CalcPixPosFromNumberLineVal(_time != null ? _time.Time : 0);
                int currFrmW = (int)Math.Ceiling((double)numberLineDrawer.PixelPerVal);

                pe.Graphics.FillRectangle(new SolidBrush(Color.FromArgb(128, 0, 0, 255)), currFrmPos, 0, currFrmW, this.ClientSize.Height);
            }

            // 対象が設定されていなければ、何もしません。
            // キーフレームを持たない場合も何もしません。
            if (_targetKeyframes != null && _targetKeyframes.Length > 0)
            {
                // キーフレームの存在するフレームに矩形を描画します。
                for (int i = 0; i < _targetKeyframes.Length; i++)
                {
                    IAnmKeyFrame kf = _targetKeyframes[i];
                    int keyFrmPos = numberLineDrawer.CalcPixPosFromNumberLineVal(kf.TimeAsInt);

                    // this.ClientSize.Height - 1 は領域の内側に矩形を描画するため...
                    int height = this.ClientSize.Height / 3;
                    int width = Math.Min((int)Math.Ceiling((double)numberLineDrawer.PixelPerVal), KeyFrameHilightW);
                    g.FillRectangle(_keyBrush, keyFrmPos, this.ClientSize.Height - height, width, height - 1);
                    g.DrawRectangle(_keyOutlinePen, keyFrmPos, this.ClientSize.Height - height, width, height - 1);
                }
            }

            g.DrawRectangle(_grayPen, 0, 0, this.ClientSize.Width - 1, this.ClientSize.Height - 1);

            if (!this.Parent.Enabled)
            {
                g.FillRectangle(_disableBrush, 0, 0, this.ClientSize.Width - 1, this.ClientSize.Height - 1);
            }

            base.OnPaint(pe);
        }
        #endregion

        #region イベントハンドラ関連
        private void Event_TimeScaleControl_SizeChanged(object sender, System.EventArgs e)
        {
            Invalidate();
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);
            BeginDrag?.Invoke(this, new EventArgs());

            this.Focus();
            _mouseClicked = true;
            ChangeCurrentTimeAccordingToClickPos_( e.X );
        }

        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
            OnEndDrag();
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if( _mouseClicked )
            {
                ChangeCurrentTimeAccordingToClickPos_( e.X );
            }
        }

        protected override void OnLostFocus(EventArgs e)
        {
            // ドッキングされていると LostFocus が呼ばれない場合がある
            // 現状は対応できていない

            base.OnLostFocus(e);
            OnEndDrag();
        }

        /// <summary>
        /// ドラッグ終了処理
        /// </summary>
        private void OnEndDrag()
        {
            if (_mouseClicked)
            {
                if (_time.IsPlaying)
                {
                    _time.Stop();
                }
                _mouseClicked = false;

                EndDrag?.Invoke(this, new EventArgs());
            }
        }
        #endregion




    }
}


